package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	Assign.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////


/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

        //-----------------------------------------------//
    	//Note on atom naming scheme: 
    	//Suppose we have following four different atom naming schemes:
    	//PDB-NEW : eg HA1, HA2 (Gly);   PDB-OLD : eg 1HA, 2HA (Gly);
    	//BMRB-NEW: eg HA2, HA3 (Gly);   BMRB-OLD: eg 2HA, 3HA (Gly).
    	//BMRB-NEW-METHYL: the last number of methyl protons are removed, eg HG21,HG22,HG23=>HG2
    	//We assume that input files use following different (but kind of standard) naming schemes:
    	//(Small) Rotamer library: PDB-NEW (Different rotamer libary may use differnt scheme);  
    	//Resonance assignment: BMRB-NEW-METHYL ;
    	//Input backbone PDB structure: PDB-OLD;
    	//
    	//Our strategy in this code: 
    	//(1). Convert backbone PDB structure from PDB-OLD to BMRB-NEW format (in Pdb.nameConvert());
    	//(2). Convert rotamer library from PDB-NEW to BMRB-NEW ;
    	//(3). Initial ambiguous NOE assignments are in BMRB-NEW-METHYL format.
    	//(4). Place rotamer in (2) into backbone (1), and prune ambigous noe assignment. Final NOE assignments are in BMRB-NEW-METHYL format.
    	//(5). Output the final noe assignment in xplor format, namely label methyl protons with #
    	//     and change back to PDB-NEW format.
    	    	    	
import java.io. *;
import java.util. *;
import java.text.NumberFormat;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.lang. *;
import Jampack.JampackException;

/** 
*   This class provides functions that involves NOE assignment, such initial NOE assignment using only 
*   chemical shift information, pruning ambiguous NOE assignments using given structural information,etc.
*   Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
*/

public class Assign
{
    
    /** The residue no of the chemical shift. */
    int residueNo;
    
    /** The residue type of the chemical shift. */
    String residueType;
    
    /** all the resonancs assignments for a residue. */
    Map assignMap = new TreeMap(); 
    
    /** list of HN NOEs. */
    Vector hnNoeVec = new Vector(); //
    
    /**  list of HA NOEs. */
    Vector haNoeVec = new Vector(); //
    
    /**  list for other protons other than HN and HA. */
    Vector h1NoeVec = new Vector(); //
   
    /** The maximum matching score. */
    double maxValue=0.0;
    
    /** The missing index. */
    int [] missingIndex;
    
    /**
     * Sets the max value.
     * 
     * @param maxval the new max value
     */
    public void setMaxValue(double maxval)
    {
    	maxValue=maxval;
    }
    
    /**
     * Sets the missing index.
     * 
     * @param index the new missing index
     */
    public void setMissingIndex(int [] index)
    {
    	missingIndex=index;
    }
    
    /**
     * Gets the missing index.
     * 
     * @return the missing index
     */
    public int[] getMissingIndex()
    {
    	return missingIndex;
    }
    
    /**
     * Gets the max value.
     * 
     * @return the max value
     */
    public double getMaxValue()
    {
    	return maxValue;
    }
    
    /**
     * Instantiates a new assign.
     */
    public Assign(){}

    /**
     * Instantiates a new assign.
     * 
     * @param no the no
     */
    public Assign(int no) 
    {
	residueNo = no;
	residueType = null;
    }

    /**
     * Instantiates a new assign.
     * 
     * @param no the no
     * @param type the type
     */
    public Assign(int no, String type)
    {
    	residueNo = no;
    	residueType = type;
    }

    /**
     * Instantiates a new assign.
     * 
     * @param no the no
     * @param type the type
     * @param aMap the map
     */
    public Assign(int no, String type, Map aMap)
    {
		residueNo = no;
		residueType = type;
		assignMap = aMap;
    }

    /**
     * Instantiates a new assign.
     * 
     * @param no the no
     * @param type the type
     * @param aMap the a map
     * @param hnVec the hn vec
     * @param haVec the ha vec
     */
    public Assign(int no, String type, Map aMap, Vector hnVec, Vector haVec)
    {
		residueNo = no;
		residueType = type;
		assignMap = aMap;
		hnNoeVec  = hnVec;
		haNoeVec  = haVec;
    }

    /**
     * Instantiates a new assign.
     * 
     * @param no the no
     * @param type the type
     * @param aMap the a map
     * @param hnVec the hn vec
     * @param haVec the ha vec
     * @param h1Vec the h1 vec
     */
    public Assign(int no, String type, Map aMap, Vector hnVec, Vector haVec, Vector h1Vec)
    {
		residueNo = no;
		residueType = type;
		assignMap = aMap;
		hnNoeVec = hnVec;
		haNoeVec = haVec;
		h1NoeVec = h1Vec;
    }

    /**
     * Gets the residue no.
     * 
     * @return the residue no
     */
    public int getResidueNo(){
	return residueNo;
    }

    /**
     * Gets the residue type.
     * 
     * @return the residue type
     */
    public String getResidueType(){
	return residueType;
    }

    /**
     * Gets the map.
     * 
     * @return the map
     */
    public Map getMap(){
	return assignMap;
    }
    
    /**
     * Gets the hn noe vec.
     * 
     * @return the hn noe vec
     */
    public Vector  getHnNoeVec(){
	return hnNoeVec;
    }
    
    /**
     * Gets the ha noe vec.
     * 
     * @return the ha noe vec
     */
    public Vector  getHaNoeVec(){
	return haNoeVec;
    }
    
    /**
     * Gets the h1 noe vec.
     * 
     * @return the h1 noe vec
     */
    public Vector  getH1NoeVec(){
	return h1NoeVec;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
	double csValue = 0.0;
	String desc = String.valueOf(residueNo) +" "+ residueType+"\n";
	String str = "";
	for (Iterator i = assignMap.entrySet().iterator(); i.hasNext(); ){
	    Map.Entry e = (Map.Entry) i.next();
	    csValue =  ((Double)e.getValue()).doubleValue();
	    if( Math.abs(csValue+999.99) > 0.01)
		str += (String)e.getKey()+" = "+ (Double)e.getValue()+"\n";
	}
	desc +=  str;
	return desc;	
    }

    /**
     * The Class assignComparator.
     */
    public static class assignComparator implements Comparator
    {
        
        /* (non-Javadoc)
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object o1, Object o2){
            Assign n1 = (Assign)o1;
            Assign n2 = (Assign)o2;
            int d1 = n1.getResidueNo();
            int d2 = n2.getResidueNo();
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }

    /**
     * The Class hnComparator.
     */
    public static class hnComparator implements Comparator{
        
        /* (non-Javadoc)
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object o1, Object o2){
            Assign n1 = (Assign)o1;
            Assign n2 = (Assign)o2;
	    Map map1 =  n1.getMap();
	    Map map2 =  n2.getMap();
	    double d1 = ((Double)map1.get("H")).doubleValue();
	    double d2 = ((Double)map2.get("H")).doubleValue();
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }

    /**
     * The Class caComparator.
     */
    public static class caComparator implements Comparator{
        
        /* (non-Javadoc)
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object o1, Object o2){
            Assign n1 = (Assign)o1;
            Assign n2 = (Assign)o2;
	    Map map1 =  n1.getMap();
	    Map map2 =  n2.getMap();
	    double d1 = ((Double)map1.get("CA")).doubleValue();
	    double d2 = ((Double)map2.get("CA")).doubleValue();
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }

    /**
     * The Class haComparator.
     */
    public static class haComparator implements Comparator{
        
        /* (non-Javadoc)
         * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
         */
        public int compare(Object o1, Object o2){
            Assign n1 = (Assign)o1;
            Assign n2 = (Assign)o2;
	    Map map1 =  n1.getMap();
	    Map map2 =  n2.getMap();
	    double d1 = ((Double)map1.get("HA")).doubleValue();
	    double d2 = ((Double)map2.get("HA")).doubleValue();
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }
    
    /**
     * A very simple reader for secondary structure identification file.
     * 
     * @param strSSES the str sses
     * 
     * @return a vector of IDof2aryStructure objects
     */
    public Vector getOrder(String strSSES)
    {
    	Vector vecOrder=new Vector();
    	String fileName="d";///
    	String ss ="", str ="";
        Vector inputs = new Vector();
        int index = -1;
        
        if ( strSSES.indexOf("-") <0 )
        		System.out.println("We need at lease two secondary structure elements.");
        int preIndex=0;
        for(int i=1;i<strSSES.length()+1;i++)
        {
        	str=strSSES.substring(i-1,i);
        	if(str.equalsIgnoreCase("-"))
        	{
        		ss=strSSES.substring(preIndex, i-1);
        		preIndex=i;
        		vecOrder.add(ss);
        	}  	
        	
        }//for(int i=0;i<strSSES.length();i++)   
        
        ss=strSSES.substring(preIndex, strSSES.length());
        vecOrder.add(ss);
        
        return vecOrder;
    }
    
  /**
   * Read the resonance assignment.
   * 
   * @param pdbFile the pdb file
   * @param h1Ref the chemical shift correction (?)
   * 
   * @return a vector of Assign object
   */
    public Vector assignReader(String pdbFile, double h1Ref){	
	String ss="", str="";
	int no=0, noTmp=0;
	String aaType="";
	String atomName="";
	String nucleusType;
	StringTokenizer st = new StringTokenizer("");
	Vector assignVec = new Vector();
	Map mapAssign = new TreeMap();
	Residue res = new Residue();
	double csValue = 0.0;
	try{
	    BufferedReader in = new BufferedReader(new FileReader(pdbFile));	
	    ss = in.readLine();
	    stop:
	    while(true) {
		st = new StringTokenizer(ss);
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    no = Integer.parseInt(st.nextToken());
		if (st.hasMoreTokens())
		    aaType = st.nextToken();
		if (st.hasMoreTokens())
		    atomName = st.nextToken();
		if (st.hasMoreTokens())
		    nucleusType  = st.nextToken();
		if (st.hasMoreTokens())
		    csValue = (new Double(st.nextToken()).doubleValue()) + h1Ref;
		mapAssign = res.nucluesByAA(aaType);
		mapAssign.put(atomName, new Double(csValue));
		if ((ss = in.readLine()) == null){
		    assignVec.add(new Assign(no, aaType, mapAssign));
		    break stop;
		}
		st = new StringTokenizer(ss);
		st.nextToken();
		if (st.hasMoreTokens())
		    noTmp = Integer.parseInt(st.nextToken());
		while (noTmp == no){
		    if (st.hasMoreTokens())
			aaType = st.nextToken();
		    if (st.hasMoreTokens())
			atomName = st.nextToken();
		    if (st.hasMoreTokens())
			nucleusType  = st.nextToken();
		    if (st.hasMoreTokens())
			csValue = (new Double(st.nextToken()).doubleValue()) + h1Ref;
		    mapAssign.put(atomName, new Double(csValue)); 
		    if ((ss = in.readLine()) == null){
			assignVec.add(new Assign(no, aaType, mapAssign));
			break stop;
		    }
		    st = new StringTokenizer(ss);
		    st.nextToken();
		    if (st.hasMoreTokens())
			noTmp = Integer.parseInt(st.nextToken());
		}
		assignVec.add(new Assign(no, aaType, mapAssign));
	    }
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + pdbFile);
     	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	Collections.sort(assignVec, new assignComparator() );
	return assignVec;
    }


    /**
     * Read the resonance assignment, meanwhile update the residue name for each
     * chemical shift.
     * 
     * @param hncacbFile the hncacb file
     * @param assignVec the assign vec
     * @param chsqcVec the chsqc vec
     * 
     * @return a vector of Assign object
     */
    public Vector assignReader2(String hncacbFile, Vector assignVec, Vector chsqcVec){	
	String ss="", str="";
	int no=0, noTmp=0;
	String aaType="";
	String aaTypeS="";
	String atomName="";
	String nucleusType;
	StringTokenizer st = new StringTokenizer("");
	Map mapAssign = new TreeMap();
	Residue res = new Residue();
	double hnValue = 0.0, nValue = 0.0, caValue = 0.0, cbValue = 0.0;
	double intensity = 1.00;
	double cs = 1.00;
	Assign assign = new Assign();
	Hsqc hsqc = new Hsqc();
	Map aMap = new TreeMap();
	Vector assignVecN = new Vector();
	double haCs = 0.0;
	double caCs = 0.0;
	boolean haveIt  = false;
	try{
	    BufferedReader in = new BufferedReader(new FileReader(hncacbFile));	
	    ss = in.readLine();
	    stop:
	    while(true) {
		st = new StringTokenizer(ss);
		if (st.hasMoreTokens())
		    no = Integer.parseInt(st.nextToken());
		if ((ss = in.readLine()) == null)
		    break stop;
		st = new StringTokenizer(ss);
		if (st.hasMoreTokens())
		    nValue = 100*(new Double(st.nextToken()).doubleValue());
		if (st.hasMoreTokens())
		    hnValue = new Double(st.nextToken()).doubleValue();
		if ((ss = in.readLine()) == null)
		    break stop;
		st = new StringTokenizer(ss);
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    cs = new Double(st.nextToken()).doubleValue();
		if (st.hasMoreTokens())
		    intensity =  new Double(st.nextToken()).doubleValue();
		if (intensity > 0)
		    caValue = cs;
		else cbValue = cs;
		if ((ss = in.readLine()) == null)
		    break stop;
		st = new StringTokenizer(ss);
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    st.nextToken();
		if (st.hasMoreTokens())
		    cs = new Double(st.nextToken()).doubleValue();
		if (st.hasMoreTokens())
		    intensity =  new Double(st.nextToken()).doubleValue();
		if (intensity > 0)
		    caValue = cs;
		else cbValue = cs;
		for (int i=0; i<assignVec.size(); i++){
		    assign = (Assign)assignVec.elementAt(i);
		    noTmp = assign.getResidueNo();
		    aaType = assign.getResidueType();
		    if (no == noTmp){

			aMap = assign.getMap();
			aMap.put("HN", new Double(hnValue));
			aMap.put("N", new Double(nValue));
 			aMap.put("CA", new Double(caValue));
			if (!aaType.equalsIgnoreCase("GLY"))
			    aMap.put("CB", new Double(cbValue));

			aaTypeS = aaType;
		    }
		}
		haveIt  = false;
		for (int k=0; k<chsqcVec.size(); k++){
		    hsqc = (Hsqc)chsqcVec.elementAt(k);
		    haCs = hsqc.getHN();
		    caCs = hsqc.getN15();
		    noTmp = hsqc.getResidueNo();
		    if (no == noTmp){
			aMap.put("ha", new Double(haCs));
 			aMap.put("ca", new Double(caCs));
			haveIt = true;
		    }
		}
		if (!haveIt){
		    aMap.put("ha", new Double(-999.9));
		    aMap.put("ca", new Double(-999.9));
		}
		assignVecN.add(new Assign(no, aaTypeS, aMap));
		aMap = new TreeMap();
		if ((ss = in.readLine()) == null)
		    break stop;
	    }
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + hncacbFile);
     	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return assignVecN;
    }
  
    /**
     * Read protein sequence.
     * 
     * @param filename the name of the file
     * 
     * @return a vector of residues and corresponding residue numbers (IDs)
     */
    public Vector <Assign> ReaderSeq(String filename)
    {
    	Vector inputs = new Vector();
    	
    	Vector csVec = new Vector();
    	StringTokenizer st = new StringTokenizer("");
    	String str = "";
    	int index1=0, index2=0;
    	String order="";
    	double value=0.0;
    	int no =0;
    	String ss ="";    	
    	String resName="";
    	int resNo=0;
    	
    	try
    	{
    		BufferedReader in = new BufferedReader(new FileReader(filename));	
    		ss = in.readLine();    		
    		stop:
    			while(true) 
    			{   			    
    				st = new StringTokenizer(ss);
    				if (st.hasMoreTokens())
    					resName = st.nextToken();		
    				
    				if (st.hasMoreTokens())
    					resNo =  Integer.parseInt(st.nextToken());			
    				
    				
    				inputs.add(new Assign(resNo, resName));
    				if ((ss = in.readLine()) == null )
    				{
    					break stop;
    				}
    				if(ss.equalsIgnoreCase(""))
    					break stop;
    			}
    	}catch (FileNotFoundException e) {
    		System.out.println("File not found: " + filename);
    	}catch (IOException e) 
    	{
    		System.out.println("IOException: the stack trace is:");
    		e.printStackTrace();
    	}
    	return inputs;
    }

    /**
     * Read the parameters such as the number of refinement cycle
     * etc for running the program. This function was only used in Lincong's
     * previous code.
     * 
     * @param fileName the name of the parameter file
     * 
     * @return an array of Map with the name of the parameter and the corresponding value
     */
    public Vector<Map<String, String>> ParamReader(String fileName)
    {
    	Vector<Map<String, String>> parameterVec = new Vector<Map<String, String>>();
    	String ss ="", str ="";
    	String ss2   = "";
    	int    index = -1;
    	int    ind2  = -1;
    	String key   = "";
    	String value = "";
	
		try
		{
			BufferedReader in = new BufferedReader(new FileReader(fileName));	
			if ((ss = in.readLine()) != null)
				
			while( ss != null ) 
			{				
				index = ss.indexOf("=");
				
				
				while (index == -1)
				{
					if ((ss = in.readLine()) != null)
					{
						
					}
					else
						break;
					index = ss.indexOf("=");
				}
				while ( ss != null && index  > -1 )
				{
					key = ss.substring(0, index).trim();
					key = key.toUpperCase();
					ind2 = ss.indexOf(";", index);
					if (ind2 < 0)
					{
						System.out.println("Error in parameter file: lines need to be ended by \";\", see Example");
						System.exit(1);
					}
					value = ss.substring(index+1, ind2).trim();
					
					Map<String, String> mapParam = new TreeMap<String, String>();
					
					mapParam.put(key, value);
					parameterVec.add(mapParam);
					if ((ss = in.readLine()) != null)
					{
						
						index = ss.indexOf("=");
					}
				}//while ( ss != null && index  > -1 )
				if ((ss = in.readLine()) != null)
				{
					
				}
					
			}//while( ss != null ) 
			in.close();
	}
	catch (FileNotFoundException e) 
	{
	    System.out.println("File not found: " + fileName);
	}
	catch (IOException e) 
	{
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}

	return parameterVec;
    }
  
    /**
     * Written by Lincong, only used in his previous functions.
     * Read peak list generated by NMRView, just for ubq peaks picked by Lincong
     * 
     * @param filename the name of the file
     * 
     * @return a vector of RDC object
     */
    public Vector asgReader3(String filename){
	Vector inputs = new Vector();
	double cs = 0.0;
	StringTokenizer st = new StringTokenizer("");
	String str = "";
	int no =0;
	int cnt = 0;
	String aaType = "", ss ="";
	Map aMap = new TreeMap();
	try{
	    BufferedReader in = new BufferedReader(new FileReader(filename));	
	    ss = in.readLine();
	    st = new StringTokenizer(ss);
	    no =  Integer.parseInt(st.nextToken());
	    if (st.hasMoreTokens())
		aaType = st.nextToken();
	    stop:
	    while((ss = in.readLine()) != null) {
		st = new StringTokenizer(ss);
		cnt = st.countTokens();
		if (cnt == 2){
		    inputs.add(new Assign(no, aaType, aMap));
		    aMap = new TreeMap();
		    no =  Integer.parseInt(st.nextToken());
		    aaType = st.nextToken();
		    continue stop;
		}else if(cnt == 3){
		    str = st.nextToken();
		    cs = new Double(st.nextToken()).doubleValue();
		    aMap.put(str, new Double(cs));
		}
	    }
	    inputs.add(new Assign(no, aaType, aMap));
	}catch (FileNotFoundException e) {
	    System.out.println("File not found: " + filename);
	}catch (IOException e) {
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
	}
	return inputs;
    }
 
    /**
     * Written by Lincong, only used in his previous functions.
     * 
     * @param assignVec the assign vec
     * @param hsqcVec the hsqc vec
     */
    public void assignHsqc(Vector assignVec, Vector hsqcVec)
    {
		double hErr = 0.02;
		double nErr = 0.05;
		int no=0, noTmp=0;
		String aaType="";
		double hnValue = 0.0, nValue = 0.0, csHN = 0.0, csN = 0.0;
		Assign assign = new Assign();
		Map aMap = new TreeMap();
		Vector assignVecN = new Vector();
		Hsqc qc = new Hsqc();
		for (int i=0; i<assignVec.size(); i++)
		{
		    assign = (Assign)assignVec.elementAt(i);
		    noTmp = assign.getResidueNo();
		    aaType = assign.getResidueType();
		    aMap = assign.getMap();
		    hnValue = ((Double)aMap.get("HN")).doubleValue();
		    nValue = ((Double)aMap.get("N")).doubleValue();
		    for(int j=0; j<hsqcVec.size(); j++)
		    {
				qc = (Hsqc)hsqcVec.elementAt(j);
				csHN = qc.getHN();
				csN  = qc.getN15();
				if ( Math.abs(csHN - hnValue) < hErr && Math.abs(csN-nValue) < nErr)
				{
				    qc.setResidueNo(noTmp);
				    qc.setResidueType(aaType);
				}
		    }
		}
    }
    
    /**
     * Written by Lincong, only used in his previous functions.
     * 
     * @param foldHsqcVec the fold hsqc vec
     * @param hsqcVec the hsqc vec
     */
    public void matchC13Hsqc(Vector foldHsqcVec, Vector hsqcVec){
	double hErr = 0.02;
	double nErr = 0.05;
	int no=0, noTmp=0;
	String aaType="";
	double hnValue = 0.0, nValue = 0.0, csHN = 0.0, csN = 0.0;
	Hsqc qc = new Hsqc();
	Hsqc qc2 = new Hsqc();
	for (int i=0; i<foldHsqcVec.size(); i++){
	    qc = (Hsqc)foldHsqcVec.elementAt(i);
	    hnValue = qc.getHN();
	    nValue = qc.getN15();
	    for(int j=0; j<hsqcVec.size(); j++){
		qc2 = (Hsqc)hsqcVec.elementAt(j);
		csHN = qc2.getHN();
		csN  = qc2.getN15();
		noTmp = qc2.getResidueNo();
		aaType = qc2.getResidueType();
		if ( Math.abs(csHN - hnValue) < hErr && Math.abs(csN-nValue) < nErr){
		    qc.setResidueNo(noTmp);
		    qc.setResidueType(aaType);
		}
	    }
	}
    }

    /**
     * Written by Lincong, only used in his previous functions.
     * 
     * @param hnNoeVec the hn noe vec
     * @param hsqcVec the hsqc vec
     * @param hnErr the hn err
     * @param nErr the n err
     * 
     * @return the vector
     */
    public Vector mergeNoe(Vector hnNoeVec, Vector hsqcVec, double hnErr, double nErr){
	int no=0, noTmp=0;
	String aaType="";
	double hnValue = 0.0, nValue = 0.0, csHN = 0.0, csN = 0.0;
	Assign assign = new Assign();
	HnNoe hnNoe = new HnNoe();
	Map aMap = new TreeMap();
	Vector noeVecN = new Vector();	
	Vector oneVec = new Vector();	
	for (int i=0; i<hsqcVec.size(); i++){
	    csHN = ((Hsqc)hsqcVec.elementAt(i)).getHN();
	    csN = ((Hsqc)hsqcVec.elementAt(i)).getN15();
	    for(int j=0; j<hnNoeVec.size(); j++){
		hnNoe = (HnNoe)hnNoeVec.elementAt(j);
		hnValue = hnNoe.getHN();
		nValue =  hnNoe.getN15();
		if ( Math.abs(csHN - hnValue) < hnErr && Math.abs(csN-nValue) < nErr){
		    oneVec.add(hnNoe);
		}
	    }		
	    noeVecN.add(oneVec);
	    oneVec = new Vector();
	}
	return noeVecN;
    }

    /**
     * Written by Lincong, only used in his previous functions.
     * 
     * @param hcchVec a C13 NOE vector which has the same members as HcchTocsy object
     * @param hsqcVec the assigned C13 Hsqc vector (in fact, the folded one since C13 NOESY is folded)
     * @param haErr the ha err
     * @param cErr the c err
     * 
     * @return the vector
     */
    public Vector mergeC13Noe(Vector hcchVec, Vector hsqcVec, double haErr, double cErr){
	int no=0, noTmp=0;
	String aaType="";
	double haValue = 0.0, cValue = 0.0, csHa = 0.0, csC = 0.0;
	double h1Value = 0.0;
	HcchTocsy hcch = new HcchTocsy();
	Vector allVec = new Vector();	
	Vector oneVec = new Vector();	
	Vector pkVec = new Vector();	
	Residue resid = new Residue();
	Hsqc qc = new Hsqc();
	for (int i=0; i<hsqcVec.size(); i++){
	    qc =  (Hsqc)hsqcVec.elementAt(i);
	    csHa = qc.getHN();
	    csC  = qc.getN15();
	    no = qc.getResidueNo();
	    aaType = qc.getResidueType();
	    pkVec.add(new Peak("ha", csHa));
	    pkVec.add(new Peak("ca", csC));
	    for(int j=0; j<hcchVec.size(); j++){
		hcch = (HcchTocsy)hcchVec.elementAt(j);
		haValue = hcch.getHA();
		cValue =  hcch.getC13();
		h1Value = hcch.getH1();
		if (Math.abs(csHa - haValue) < haErr  && Math.abs(csC-cValue) < cErr){
		    oneVec.add(hcch);
		}
	    }		
	    allVec.add(new Residue(no, aaType, pkVec, oneVec));
	    pkVec = new Vector();
	    oneVec = new Vector();
	}
	return allVec;
    }

    /**
     * Written by Lincong, only used in his previous functions.
     * 
     * @param hnNoeVec the hn noe vec
     * @param hsqcVec the hsqc vec
     * @param assignVec the assign vec
     * @param hnErr the hn err
     * @param nErr the n err
     * 
     * @return the vector
     */
  public Vector mergeNoeAndAssign(Vector hnNoeVec, Vector hsqcVec, Vector assignVec, 
				  double hnErr, double nErr){
	int no=0, noTmp=0;
	String aaType="";
	double hnValue = 0.0, nValue = 0.0, csHN = 0.0, csN = 0.0;
	Assign assign = new Assign();
	HnNoe hnNoe = new HnNoe();
	Map aMap = new TreeMap();
	Vector noeVecN = new Vector();	
	Vector oneVec = new Vector();	
	Hsqc qc = new Hsqc();
	for (int i=0; i<hsqcVec.size(); i++){
	    qc =  (Hsqc)hsqcVec.elementAt(i);
	    csHN = qc.getHN();
	    csN = qc.getN15();
	    noTmp = qc.getResidueNo();
	    if (noTmp != 0){
		aaType = qc.getResidueType();
		for(int j=0; j<hnNoeVec.size(); j++){
		    hnNoe = (HnNoe)hnNoeVec.elementAt(j);
		    hnValue = hnNoe.getHN();
		    nValue =  hnNoe.getN15();
		    if ( Math.abs(csHN - hnValue) < hnErr && Math.abs(csN-nValue) < nErr){
			oneVec.add(hnNoe);
		
		    }
		}	
		for (int k=0; k<assignVec.size(); k++){
		    assign = (Assign)assignVec.elementAt(k);
		    aMap = assign.getMap();
		    hnValue = ((Double)aMap.get("HN")).doubleValue();
		    nValue = ((Double)aMap.get("N")).doubleValue();
		    if ( Math.abs(csHN - hnValue) < hnErr && Math.abs(csN-nValue) < nErr 
			 && (!oneVec.isEmpty())){
			hnNoe = (HnNoe)oneVec.elementAt(0);
			hnNoe.setResidueNo(noTmp);
			hnNoe.setResidueType(aaType);
			hnNoe.setAssignMap(aMap);
		    }
		}	
		noeVecN.add(oneVec);
		oneVec = new Vector();
	    }
	}
	return noeVecN;
    }

  /**
   * Written by Lincong, only used in his previous functions.
   * 
   * @param asgVec the asg vec
   * @param hnNoeVec the hn noe vec
   * @param cNoeVec the c noe vec
   * @param hnErr the hn err
   * @param nErr the n err
   * @param haErr the ha err
   * @param cErr the c err
   * 
   * @return the vector
   */
    public Vector mergeAssignAndNoe( Vector asgVec, Vector hnNoeVec, Vector cNoeVec,
				     double hnErr, double nErr, double haErr, double cErr){
	int no=0, noTmp=0;
	String aaType="";
	double hnValue = 0.0, nValue = 0.0, csHN = 0.0, csN = 0.0;
	double he1 = 0.0, he2 = 0.0, ne = 0.0;
	double haValue = 0.0, caValue = 0.0, csHA = 0.0, csHA2 = 0.0, csHA3 = 0.0, csCa = 0.0;
	Assign assign = new Assign();
	HnNoe hnNoe = new HnNoe();
	HcchTocsy hcch = new HcchTocsy();
	Map aMap = new TreeMap();
	Vector hnNoeVec2 = new Vector();	
	Vector haNoeVec = new Vector();	
	Vector h1NoeVec = new Vector();	
	Vector asgVec2   = new Vector();
	Residue res = new Residue();
	Vector pkVec = new Vector();
	Vector hcchVec2 = new Vector();
	int i, j, k;
	for (i=0; i<asgVec.size(); i++){
	    assign =  (Assign)asgVec.elementAt(i);
	    no = assign.getResidueNo();
	    aaType = assign.getResidueType();
	    aMap = assign.getMap();
	    csHN = ((Double)aMap.get("HN")).doubleValue();

	    if ( aMap.get("N") != null)
		csN  = ((Double)aMap.get("N")).doubleValue();
	    else csN = -999.99;
	    if(aMap.get("ca") == null)
		System.out.println(no+" "+aaType);
	    else 
		csCa = ((Double)aMap.get("ca")).doubleValue();
	    for(j=0; j<hnNoeVec.size(); j++){
		hnNoe = (HnNoe)hnNoeVec.elementAt(j);
		hnValue = hnNoe.getHN();
		nValue =  hnNoe.getN15();
		if ( Math.abs(csHN - hnValue) < hnErr && Math.abs(csN-nValue) < nErr)
		    hnNoeVec2.add(hnNoe);
	    }
	    if(aMap.get("NE") != null){
		ne  = ((Double)aMap.get("NE")).doubleValue();
		he1 = ((Double)aMap.get("HE21")).doubleValue();
		he2 = ((Double)aMap.get("HE22")).doubleValue();
		for(j=0; j<hnNoeVec.size(); j++){
		    hnNoe = (HnNoe)hnNoeVec.elementAt(j);
		    hnValue = hnNoe.getHN();
		    nValue =  hnNoe.getN15();
		    if ( (Math.abs(he1 - hnValue) < 0.50*hnErr || Math.abs(he2 - hnValue) < 0.50*hnErr)
			 && Math.abs(ne-nValue) < 0.50*nErr)
			h1NoeVec.add(hnNoe);
		}

	    }else if (aMap.get("ND") != null){
		ne  = ((Double)aMap.get("ND")).doubleValue();
		he1 = ((Double)aMap.get("HD21")).doubleValue();
		he2 = ((Double)aMap.get("HD22")).doubleValue();
		for(j=0; j<hnNoeVec.size(); j++){
		    hnNoe = (HnNoe)hnNoeVec.elementAt(j);
		    hnValue = hnNoe.getHN();
		    nValue =  hnNoe.getN15();
		    if ( (Math.abs(he1 - hnValue) < 0.50*hnErr || Math.abs(he2 - hnValue) < 0.50*hnErr)
			 && Math.abs(ne-nValue) < 0.50*nErr)
			h1NoeVec.add(hnNoe);
		}

	    }

	    for(j=0; j<cNoeVec.size(); j++){
		hcch = (HcchTocsy)cNoeVec.elementAt(j);
		noTmp = hcch.getResidueNo();

		if ( no == noTmp)
		    haNoeVec.add(hcch);
	    }
	    asgVec2.add(new Assign(no, aaType, aMap, hnNoeVec2, haNoeVec, h1NoeVec));
	    hnNoeVec2 = new Vector();	
	    haNoeVec = new Vector();	
	    h1NoeVec = new Vector();	
	}
	return asgVec2;
    }

    /**
     * Delete repeated chemical shifts.
     * 
     * @param allVec original chemical shift list;
     * @param epsH the eps h
     * @param epsN the eps n
     * @param epsH1 the eps h1
     * 
     * @return new list of chemical shifts
     */
    public Vector deleteRepeat(Vector allVec, double epsH, double epsN, double epsH1){    
	Vector oneVec = new Vector();
	Vector oneVecNew = new Vector();
	Vector allVecNew = new Vector();
	double csHN1 = 0.0,  csHN2 = 0.0;
	double csH1 = 0.0,  csH2 = 0.0;
	double csN1 = 0.0, csN2 = 0.0;
	double intensity1 = 0.0, intensity2 = 0.0;
	HnNoe hncc = new HnNoe();
	HnNoe hncc1 = new HnNoe();
	HnNoe hncc2 = new HnNoe();
	HnNoe hnNoe1 = new HnNoe();
	HnNoe hnNoe2 = new HnNoe();
	boolean bigger = false;
	int i, j;
	int no = 0;
	String residue ="";
	Map aMap = new TreeMap();
	Assign assign = new Assign();
	for(i=0; i<allVec.size(); i++){
	    oneVec = (Vector)allVec.elementAt(i);
	    if (!oneVec.isEmpty()){
		hnNoe1 = (HnNoe)oneVec.elementAt(0);
		no =  hnNoe1.getResidueNo();
		residue =  hnNoe1.getResidueType();
		aMap =   hnNoe1.getAssignMap();
	    }
	    for(j=0; j<oneVec.size(); j++){
		hncc1 = (HnNoe)oneVec.elementAt(j);
		csHN1 = hncc1.getHN();
		csH1 = hncc1.getH1();
		csN1 = hncc1.getN15();
		intensity1 = hncc1.getIntensity();
 		bigger = false;
		for(int k=0; k<oneVec.size(); k++){
		    hncc2 = (HnNoe)oneVec.elementAt(k);
		    csHN2 = hncc2.getHN();
		    csH2 = hncc2.getH1();
		    csN2  = hncc2.getN15();
		    intensity2 = hncc2.getIntensity();
		    if ( Math.abs(csHN2 - csHN1) < epsH && Math.abs(csN2 - csN1) < epsN
			 && Math.abs(csH2 - csH1) < epsH && Math.abs(intensity2) >= Math.abs(intensity1) ){
 			bigger = true;
			hncc = hncc2;
			csH1 = hncc.getH1();
			csHN1 = hncc.getHN();
			csN1 = hncc.getN15();
			intensity1 = hncc.getIntensity();
		    }
		}
		if (!bigger){
 		    if (oneVecNew.indexOf(hncc1) < 0 )
			oneVecNew.add(hncc1);
		}else {
 		    if (oneVecNew.indexOf(hncc) < 0 )
			oneVecNew.add(hncc);  //if (bigger && !oneVecNew.contains(hncc));
		}
	    }
	    if (!oneVecNew.isEmpty()){
		hnNoe2 = (HnNoe)oneVecNew.elementAt(0);
		hnNoe2.setResidueNo(no);
		hnNoe2.setResidueType(residue);
		hnNoe2.setAssignMap(aMap);
	    }
	    allVecNew.add(oneVecNew);
	    oneVecNew = new Vector();
	}
	return allVecNew;
    }
    
    /**
     * Written by Lincong.
     * Please note that in order to sort the map structure we first converted it into a vector
     * IT's a stupid ans slow but I am in a hurry
     * 
     * @param allVec the all vec
     * 
     * @return the vector
     */
    public Vector sortAssign(Vector allVec){
	Vector oneVec = new Vector();
	double csHN2 = 0.0;
	double csH2 = 0.0;
	double csN2 = 0.0;
	HnNoe hncc2 = new HnNoe();
	double cs = 0.0;
	double inten = 0.0;
	HnNoe hnNoe1 = new HnNoe();
	int i, j;
	int no = 0;
	String residue ="";
	String nucleus ="";
	Map aMap = new TreeMap();
	Vector pkVec = new Vector();
	Vector oneNoeVec = new Vector();
	Vector allPkVec = new Vector();
	Map noeMap = new TreeMap();
	Residue resid = new Residue();
	for(i=0; i<allVec.size(); i++){
	    oneVec = (Vector)allVec.elementAt(i);
	    for(int k=0; k<oneVec.size(); k++){
		hncc2 = (HnNoe)oneVec.elementAt(k);
		aMap = hncc2.getAssignMap();
		if (!aMap.isEmpty()){
		    no =  hncc2.getResidueNo();

		    residue = hncc2.getResidueType();
		    for (Iterator it = aMap.entrySet().iterator(); it.hasNext(); ){
			Map.Entry e = (Map.Entry) it.next();
			nucleus = (String)e.getKey();
			cs = ((Double)e.getValue()).doubleValue();
			if( Math.abs(cs+999.99) > 0.01)
			    pkVec.add(new Peak(nucleus, cs));
		    }
		}
		csHN2 = hncc2.getHN();
		csH2 = hncc2.getH1();
		csN2  = hncc2.getN15();
		inten = hncc2.getIntensity();
		oneNoeVec.add(new HnNoe(no, residue, csHN2, csN2,  csH2, inten));
	    }
	    noeMap.put("H", oneNoeVec);
	    allPkVec.add(new Residue(no, residue, pkVec, noeMap));
	    oneNoeVec = new Vector();
	    noeMap = new TreeMap();
	    pkVec = new Vector();
	}

	return allPkVec;
    }

   /**
    * Written by Lincong.
    * Please note that in order to sort the map structure we first converted it into a vector
    * IT's a stupid ans slow but I am in a hurry
    * 
    * @param allVec the all vec
    * @param hcchVec the hcch vec
    * @param haErr the ha err
    * @param c13Err the c13 err
    * 
    * @return the vector
    */
    public Vector mergeAll(Vector allVec, Vector hcchVec, double haErr, double c13Err){
	Vector oneVec = new Vector();
	double csHa = 0.0;
	double csHA = 0.0;
	double csHA2 = 0.0; //for GLy
	double csHA3 = 0.0;
	double csHB = 0.0;
	double csHB2 = 0.0;
	double csHB3 = 0.0;
	double csC13 = 0.0;
	double csCA = 0.0;
	double csCB = 0.0;
	HnNoe hncc2 = new HnNoe();
	double cs = 0.0;
	double inten = 0.0;
	HnNoe hnNoe1 = new HnNoe();
	int i, j;
	int no = 0;
	String residue ="";
	String nucleus ="";
	Map aMap = new TreeMap();
	Vector pkVec = new Vector();
	Vector oneNoeVec = new Vector();
	Map noeMap = new TreeMap();
	Residue resid = new Residue();
	Peak pk = new Peak();
	HcchTocsy hcch = new HcchTocsy();
	Vector allVecN = new Vector();
	for(i=0; i<allVec.size(); i++){
	    resid = (Residue)allVec.elementAt(i);
	    no =  resid.getResidueNo();
	    residue = (resid.getResidueType()).toUpperCase();;
	    pkVec = resid.getPeakVec();

	    if (!residue.equals("GLY")){
		csHA = -999.99;  //for control
		csCA = -999.99;
		csHB2 = -999.99;  //for control
		csHB3 = -999.99;  //for control
		csCB = -999.99;
		for (j=0; j<pkVec.size(); j++){
		    pk = (Peak) pkVec.elementAt(j);
		    aMap.put(pk.getNucleus(), new Double (pk.getCS()) ) ;
		    nucleus = pk.getNucleus();
		    if (nucleus.equals("HA"))
			csHA = pk.getCS();
		    else if (nucleus.equals("CA"))
			csCA = pk.getCS();
		    else if(nucleus.equals("CB"))
			csCB = pk.getCS();
		    else if(nucleus.equals("HB"))
			csHB = pk.getCS();
		    else if (nucleus.equals("HB2"))
			csHB2 = pk.getCS();
		    else if (nucleus.equals("HB3"))
			csHB3 = pk.getCS();
		}
		for (j=0; j<hcchVec.size(); j++){
		    hcch = (HcchTocsy)hcchVec.elementAt(j);
		    csHa = hcch.getHA();
		    csC13 = hcch.getC13();
		    if ( Math.abs(csHa - csHA) < haErr && Math.abs(csC13-csCA) < c13Err){
			oneVec.add(hcch);
		    }else if ( Math.abs(csHa - csHB) < haErr && Math.abs(csC13-csCB) < c13Err){
			oneVec.add(hcch);
		    }else if ( (Math.abs(csHa - csHB2) < haErr || Math.abs(csHa - csHB3) < haErr)
			 && Math.abs(csC13-csCB) < c13Err){
			oneVec.add(hcch);
		    }
		}
	    }else if (residue.equals("GLY")){
		csHA2 = -999.99;  //for control
		csHA3 = -999.99;  //for control
		csCA = -999.99;
		for (j=0; j<pkVec.size(); j++){
		    pk = (Peak) pkVec.elementAt(j);
		    aMap.put(pk.getNucleus(),  new Double (pk.getCS()) ) ;
		    nucleus = pk.getNucleus();
		    if (nucleus.equals("HA2"))
			csHA2 = pk.getCS();
		    else if (nucleus.equals("HA3"))
			csHA3 = pk.getCS();
		    else if (nucleus.equals("CA"))
			csCA = pk.getCS();
		}
		for (j=0; j<hcchVec.size(); j++){
		    hcch = (HcchTocsy)hcchVec.elementAt(j);
		    csHa = hcch.getHA();
		    csC13 = hcch.getC13();
		    if ( (Math.abs(csHa - csHA2) < haErr || Math.abs(csHa - csHA3) < haErr )
			 && Math.abs(csC13 - csCA) < c13Err){
			oneVec.add(hcch);
		    }
		}
	    }
	    resid.setHcchVec(oneVec);
	    oneVec = new Vector();
	    allVecN.add(resid);
	}
	return allVecN;
    }

   /**
     * delete repeated back noes, mainly from methyl groups.
     * 
     * @param vecBackNoe  back computed NOE patterns.
     * 
     * @return new back noes
     */  
    public Vector BackNoeDeleteRepeat(Vector vecBackNoe)
    {
    	Vector vecBackNoeNew=new Vector();
    	int i,j;
    	BackNoe backNoe,backNoe_temp;
    	double csH1, csHeavy, csH2;
    	double csH1_temp, csHeavy_temp,csH2_temp;
    	boolean isRepeat=false;
    	
    	for (i=0;i<vecBackNoe.size(); i++)
    	{
    		isRepeat=false;
    		backNoe=(BackNoe)vecBackNoe.elementAt(i);
    		csH1=backNoe.getCSH1();
    		csHeavy=backNoe.getCSHeavy();
    		csH2=backNoe.getCSH2();
    		
    		for (j=i+1; j<vecBackNoe.size();j++)
    		{
    			backNoe_temp=(BackNoe)vecBackNoe.elementAt(j);
    			csH1_temp=backNoe_temp.getCSH1();
    			csHeavy_temp=backNoe_temp.getCSHeavy();
    			csH2_temp=backNoe_temp.getCSH2();
    			if (csH1==csH1_temp && csHeavy==csHeavy_temp && csH2==csH2_temp)
    				isRepeat=true;    			
    		}//for (j=i; j<vecBackNoe.size();j++)
    		
    		if (isRepeat==false)
    			vecBackNoeNew.add(backNoe);
    		
    		
    	}//for (i=0;i<vecBackNoe.size(); i++)
    	
    	return vecBackNoeNew;    	
    }
    
    /**
     * recursively compute the max matching score.
     * This is an old function. Need to be careful if this is used again.
     * 
     * @param vecBackNoe  back computed NOE patterns
     * @param preMisPenalty the pre mis penalty
     * @param preMatchScore the pre match score
     * @param depth the depth
     * @param finalDepth the final depth
     * @param missingIndex the missing index
     * 
     * @return void
     */  
    public void CompMaxMatchScore_old(Vector vecBackNoe,double preMisPenalty, 
    		double preMatchScore,int depth, int finalDepth,int [] missingIndex)
    {
    	int [] msIndex= (int [])missingIndex.clone();
    	double sigma=2.0;
    	
    	if (depth==finalDepth)
    	{
    		double dbMax=getMaxValue();
    		double sc=(Math.exp(-preMisPenalty)+ preMatchScore);
    		if (dbMax< sc ) 
    		{
    			setMaxValue(sc);
    			setMissingIndex(missingIndex);
    		}
    		return;/////
    		
    	}//if (depth==finalDepth-1)
    	
    	//compute the score if current depth is missing
    	msIndex[depth]=1;
    	double curPenalty=((BackNoe)vecBackNoe.elementAt(depth)).getMissingPenalty();
    	
    	CompMaxMatchScore_old(vecBackNoe, curPenalty,preMatchScore,depth+1,finalDepth,msIndex);
    	    	
    	//compute the score if current depth is not missing    	
    	Gaussian gs=new Gaussian();
    	msIndex[depth]=0;
    	double rmsd=((BackNoe)vecBackNoe.elementAt(depth)).getRmsd();
    	double curMatchScore=preMatchScore*(gs.Phi(rmsd,0.0,sigma)-gs.Phi(-rmsd,0.0,sigma) );
    	CompMaxMatchScore_old(vecBackNoe, preMisPenalty,curMatchScore,depth+1,finalDepth,msIndex);
    	 
    }
    
    /**
     * This is an old function, and can be deleted.
     * Compute the matching score between backcomputed NOE pattern and original NOE list
     * 
     * @param csErrH the cs err h
     * @param csErrN the cs err n
     * @param numPeaks number of non-missing peaks
     * @param csErrCA the cs err ca
     * @param vecBackNoe the vec back noe
     * @param vecNoesy the vec noesy
     * 
     * @return matching score
     */  
    public double NoePatternMatchScore_old2(double csErrH, double csErrN, double csErrCA,
    		Vector vecBackNoe, Vector vecNoesy, int [] numPeaks)
    {    		
    	double csErrHeavy=0.0;
    	int i,j;
    	BackNoe backNoe;
    	String heavyName;
    	double score=0.0;
    	
    	Noesy noesy;
    	
    	double csBackH1=0.0;
    	double csBackHeavy=0.0;
    	double csBackH2=0.0;
    	double csExpH1=0.0;
    	double csExpHeavy=0.0;
    	double csExpH2=0.0;    	
    	
    	int counter=0;
    	    	
    	double sigmaH=csErrH/4;
    	
    	double sigmaHN=csErrH/4;
    	double sigmaN=csErrN/4; 
    	double sigmaHA=csErrH/4; 
    	double sigmaCA=csErrCA/4;
    	double sigmaHeavy=0.0;
    	
    	double prob_score=0.0;
    	double tempp=0.0;
    	for (i=0; i<vecBackNoe.size();i++)
    	{
    		backNoe=(BackNoe)vecBackNoe.elementAt(i);
    		csBackH1=backNoe.getCSH1();
    		csBackHeavy=backNoe.getCSHeavy();
    		csBackH2=backNoe.getCSH2();
    		    		
    		heavyName=backNoe.getHeavyName();
    		tempp=0.0;   		
    		if (heavyName.substring(0,1).equalsIgnoreCase("N"))
    		{
    			sigmaHeavy=sigmaN;
    			csErrHeavy=csErrN;
    			sigmaH=sigmaHN;    			
    		}
    		else
    		{
    			sigmaHeavy=sigmaCA;
    			csErrHeavy=csErrCA;
    			sigmaH=sigmaHA;
    		}
    		
    		for (j=0;j<vecNoesy.size();j++)
    		{    				
    			noesy=(Noesy)vecNoesy.elementAt(j);
    			csExpH1=noesy.getH1();  
    			csExpHeavy=noesy.getHeavy(); 
    			csExpH2=noesy.getH2();     				
    			
    			if (Math.abs(csExpH1-csBackH1)<csErrH && Math.abs(csExpH2-csBackH2)<csErrH &&
    					Math.abs(csExpHeavy-csBackHeavy)<csErrHeavy)
    			{
    				counter =counter +1;
    				double cs_dist=((csExpH1-csBackH1)*(csExpH1-csBackH1))/(sigmaH*sigmaH)+
        				( (csExpHeavy-csBackHeavy)*(csExpHeavy-csBackHeavy) )/(sigmaHeavy*sigmaHeavy)+
        				((csExpH2-csBackH2)*(csExpH2-csBackH2))/(sigmaH*sigmaH);
        			    
    				cs_dist=cs_dist/3;    					
    				
    				tempp=tempp+Math.exp(-0.5*cs_dist);
    			}
    				
    		}//for (j=0;j<vecNoesy.size();j++)	
    		
    		if(tempp>=1)
				tempp=1.0;
    		
    		prob_score=prob_score+tempp;    		
    		
    	}//for (i=0; i<vecBackNoe.size();i++)
    	
    	score =prob_score/(vecBackNoe.size()) ;
    	
    	numPeaks[0]=counter;
    	return score;
    }
    
   
    /**
     * This is an old function with adding the calibration lines in the code.
     * Compute the matching score between backcomputed NOE pattern and original NOE list
     * 
     * @param csErrH  error window of the first proton attached to the heavy atom (indirect proton)
     * @param csErrN the cs err n
     * @param numPeaks number of non-missing peaks
     * @param csErrCA the cs err ca
     * @param vecBackNoe the vec back noe
     * @param vecNoesy the vec noesy
     * @param isSkipIntensity the is skip intensity
     * @param constant the constant
     * 
     * @return matching score
     */  
   public double NoePatternMatchScoreWCali(double csErrH, double csErrN, double csErrCA,
    		Vector vecBackNoe, Vector vecNoesy, int [] numPeaks,boolean isSkipIntensity,double constant)
    {    	
    	double csErrHeavy=0.0;
    	int i,j;
    	BackNoe backNoe;
    	String heavyName;
    	double score=0.0;    	
    	Noesy noesy;    	
    	double csBackH1=0.0;
    	double csBackHeavy=0.0;
    	double csBackH2=0.0;
    	double csExpH1=0.0;
    	double csExpHeavy=0.0;
    	double csExpH2=0.0;     	
    	int counter=0;    	    	
    	double sigmaH=0.04;//    	
    	double sigmaHN=0.04;
    	double sigmaN=0.1;
    	double sigmaHA=0.04;
    	double sigmaCA=0.1;
    	double sigmaHeavy=0.1;
    	double sigmaInten=3.0;
    	
    	double prob_score=0.0;
    	double tempp=0.0;
    	double bkCompDist=0.0;
    	double distSigma=1.0;
    	int bkBinIndex=0;
    	int expBinIndex=0;
    	double distExp=0.0;
    	Noesy nesy=new Noesy();
  
    	for (i=0; i<vecBackNoe.size();i++)
    	{
    		backNoe=(BackNoe)vecBackNoe.elementAt(i);
    		csBackH1=backNoe.getCSH1();
    		csBackHeavy=backNoe.getCSHeavy();
    		csBackH2=backNoe.getCSH2();    		
    		
    		int no1=backNoe.getFirstResNo();
    		int no2=backNoe.getSecondResNo();
    		String res1=backNoe.getFirstResName();
    		String res2=backNoe.getSecondResName();
    		String atom1=backNoe.getH1Name();
    		String atom2=backNoe.getH2Name();
    		double distanceT=backNoe.getDistance();    		
    		
    		
    		//we calculate the bin index of the back-computed noe peak:
    		bkCompDist=backNoe.getDistance();
    		if(bkCompDist<=2.7)
    			bkBinIndex=1;
    		else if(bkCompDist<=3.3)
    			bkBinIndex=2;
    		else if(bkCompDist<5.0)
    			bkBinIndex=3;
    		else if(bkCompDist<6.0)
    			bkBinIndex=4;      		
    		    		
    		heavyName=backNoe.getHeavyName();
    		tempp=0.0;   
    		double dist_min=999.9;
    		if (heavyName.substring(0,1).equalsIgnoreCase("N"))
    		{
    			sigmaHeavy=sigmaN;
    			csErrHeavy=csErrN;
    			sigmaH=sigmaHN;    			
    		}
    		else
    		{
    			sigmaHeavy=sigmaCA;
    			csErrHeavy=csErrCA;
    			sigmaH=sigmaHA;
    		}
    		
    		for (j=0;j<vecNoesy.size();j++)
    		{    				
    			noesy=(Noesy)vecNoesy.elementAt(j);
    			csExpH1=noesy.getH1();  
    			csExpHeavy=noesy.getHeavy(); 
    			csExpH2=noesy.getH2();  
    			expBinIndex=noesy.getBinIndex();
    			
    			
    			double intensity=noesy.getIntensity();
    			if (Math.abs(csExpH1-csBackH1)<0.04 && (Math.abs(csExpH2-csBackH2)<0.03) &&
        			Math.abs(csExpHeavy-csBackHeavy)<0.1 )//0.04, 0.03,o.2 for gb1
    			{
    				distExp=nesy.SetCalibrationOneCyana(backNoe,noesy,constant);
    				
    			    if(bkCompDist > (distExp+0.0) )
        				continue;
    				
        			if(Math.abs(bkCompDist - distExp) > 2.5)//2.5
        				continue;
    						
    			
    				counter =counter +1;
    				double inten_dist=((distExp-bkCompDist)*(distExp-bkCompDist))/(distSigma*distSigma);
    				
    				
    				double cs_dist=((csExpH1-csBackH1)*(csExpH1-csBackH1))/(sigmaH*sigmaH)+
        				( (csExpHeavy-csBackHeavy)*(csExpHeavy-csBackHeavy) )/(sigmaHeavy*sigmaHeavy)+
        				((csExpH2-csBackH2)*(csExpH2-csBackH2))/(sigmaH*sigmaH);
        			    
    				cs_dist=cs_dist/3;
    				
    					if(isSkipIntensity)
    					inten_dist=0.0;
    				
    				cs_dist=0.5*(cs_dist+inten_dist);   				
    				
    				if (cs_dist<dist_min)
    					dist_min=cs_dist;
    				
    			}
    				
    		}//for (j=0;j<vecNoesy.size();j++)	
    		
    		tempp=tempp+Math.exp(-0.5*dist_min);
    		
    		prob_score=prob_score+tempp;    		
    		
    	}//for (i=0; i<vecBackNoe.size();i++)
    	
    	score =prob_score/(vecBackNoe.size()) ;
    	    	
    	numPeaks[0]=counter;
    	return score;
    }
   
   /**
    * This is a test in order to address the comments from the second reviewer (point 4)
    * from the JBNMR09 submission. Basically change the current Hausdorff-based measure to
    * Bayesian metric and RMSD-based metric.
    * This function is to test the Bayesian metric.
    * 
    * This is an old function with adding the calibration lines in the code.
    * Compute the matching score between backcomputed NOE pattern and original NOE list
    * 
    * @param csErrH  error window of the first proton attached to the heavy atom (indirect proton)
    * @param csErrN the cs err n
    * @param numPeaks number of non-missing peaks
    * @param csErrCA the cs err ca
    * @param vecBackNoe the vec back noe
    * @param vecNoesy the vec noesy
    * @param isSkipIntensity the is skip intensity
    * @param constant the constant
    * 
    * @return matching score
    */  
  public double NoePatternMatchScoreWCaliTestJBNMR1(double csErrH, double csErrN, double csErrCA,
   		Vector vecBackNoe, Vector vecNoesy, int [] numPeaks,boolean isSkipIntensity,double constant)
   {    	
   	double csErrHeavy=0.0;
   	int i,j;
   	BackNoe backNoe;
   	String heavyName;
   	double score=0.0;    	
   	Noesy noesy;    	
   	double csBackH1=0.0;
   	double csBackHeavy=0.0;
   	double csBackH2=0.0;
   	double csExpH1=0.0;
   	double csExpHeavy=0.0;
   	double csExpH2=0.0;     	
   	int counter=0;    	    	
   	double sigmaH=0.04;//    	
   	double sigmaHN=0.04;
   	double sigmaN=0.1;
   	double sigmaHA=0.04;
   	double sigmaCA=0.1;
   	double sigmaHeavy=0.1;
   	double sigmaInten=3.0;
   	
   	double prob_score=0.0;
   	double tempp=0.0;
   	double bkCompDist=0.0;
   	double distSigma=1.0;
   	int bkBinIndex=0;
   	int expBinIndex=0;
   	double distExp=0.0;
   	Noesy nesy=new Noesy();
 
   	for (i=0; i<vecBackNoe.size();i++)
   	{
   		backNoe=(BackNoe)vecBackNoe.elementAt(i);
   		csBackH1=backNoe.getCSH1();
   		csBackHeavy=backNoe.getCSHeavy();
   		csBackH2=backNoe.getCSH2();    		
   		
   		int no1=backNoe.getFirstResNo();
   		int no2=backNoe.getSecondResNo();
   		String res1=backNoe.getFirstResName();
   		String res2=backNoe.getSecondResName();
   		String atom1=backNoe.getH1Name();
   		String atom2=backNoe.getH2Name();
   		double distanceT=backNoe.getDistance();    		
   		
   		
   		//we calculate the bin index of the back-computed noe peak:
   		bkCompDist=backNoe.getDistance();
   		if(bkCompDist<=2.7)
   			bkBinIndex=1;
   		else if(bkCompDist<=3.3)
   			bkBinIndex=2;
   		else if(bkCompDist<5.0)
   			bkBinIndex=3;
   		else if(bkCompDist<6.0)
   			bkBinIndex=4;      		
   		    		
   		heavyName=backNoe.getHeavyName();
   		tempp=0.0;   
   		double dist_min=0.0;//999.9;
   		if (heavyName.substring(0,1).equalsIgnoreCase("N"))
   		{
   			sigmaHeavy=sigmaN;
   			csErrHeavy=csErrN;
   			sigmaH=sigmaHN;    			
   		}
   		else
   		{
   			sigmaHeavy=sigmaCA;
   			csErrHeavy=csErrCA;
   			sigmaH=sigmaHA;
   		}
   		
   		//System.out.println(i+"-th back-computed NOE (csH1, csH2, csHeavy, csDist): "+ csBackH1 + ", "+ csBackH2+"," +csBackHeavy+", "+bkCompDist);
   		
   		for (j=0;j<vecNoesy.size();j++)
   		{    				
   			noesy=(Noesy)vecNoesy.elementAt(j);
   			csExpH1=noesy.getH1();  
   			csExpHeavy=noesy.getHeavy(); 
   			csExpH2=noesy.getH2();  
   			expBinIndex=noesy.getBinIndex();
   			
   			
   			double intensity=noesy.getIntensity();
   			if (Math.abs(csExpH1-csBackH1)<0.04 && (Math.abs(csExpH2-csBackH2)<0.04) &&
       			Math.abs(csExpHeavy-csBackHeavy)<0.4 )//0.04, 0.03,0.2 for gb1
   			{
   				//System.out.println("         "+csExpH1+", "+ csExpH2+", "+csExpHeavy+",  "+intensity);
   				
   				distExp=nesy.SetCalibrationOneCyana(backNoe,noesy,constant);
   				
   			    if(bkCompDist > (distExp+0.0) )
       				continue;
   				
       			if(Math.abs(bkCompDist - distExp) > 2.5)//2.5
       				continue;
   						
   			
   				counter =counter +1;
   				double inten_dist=((distExp-bkCompDist)*(distExp-bkCompDist))/(distSigma*distSigma);
   				
   				
   				double cs_dist=((csExpH1-csBackH1)*(csExpH1-csBackH1))/(sigmaH*sigmaH)+
       				( (csExpHeavy-csBackHeavy)*(csExpHeavy-csBackHeavy) )/(sigmaHeavy*sigmaHeavy)+
       				((csExpH2-csBackH2)*(csExpH2-csBackH2))/(sigmaH*sigmaH);
       			    
   				cs_dist=cs_dist/3;
   				
   					if(isSkipIntensity)
   					inten_dist=0.0;
   				
   				cs_dist=0.5*(cs_dist+inten_dist);   				
   				
   				//if (cs_dist<dist_min)
   				//	dist_min=cs_dist;
   				dist_min=dist_min+cs_dist;//"Bayesian" metric
   				
   			}
   				
   		}//for (j=0;j<vecNoesy.size();j++)	
   		
   		if(dist_min>0.0000001)
   			tempp=tempp+Math.exp(-0.5*dist_min);
   		else
   			tempp=0.0;
   		
   		prob_score=prob_score+tempp;    		
   		
   	}//for (i=0; i<vecBackNoe.size();i++)
   	
   	score =prob_score/(vecBackNoe.size()) ;
   	    	
   	numPeaks[0]=counter;
   	return score;
   }
 
  /**
   * This is a test in order to address the comments from the second reviewer (point 4)
   * from the JBNMR09 submission. Basically change the current Hausdorff-based measure to
   * Bayesian metric and RMSD-based metric.
   * This function is to test the RMSD-based metric.
   * 
   * This is an old function with adding the calibration lines in the code.
   * Compute the matching score between backcomputed NOE pattern and original NOE list
   * 
   * @param csErrH  error window of the first proton attached to the heavy atom (indirect proton)
   * @param csErrN the cs err n
   * @param numPeaks number of non-missing peaks
   * @param csErrCA the cs err ca
   * @param vecBackNoe the vec back noe
   * @param vecNoesy the vec noesy
   * @param isSkipIntensity the is skip intensity
   * @param constant the constant
   * 
   * @return matching score
   */  
 public double NoePatternMatchScoreWCaliTestJBNMR2(double csErrH, double csErrN, double csErrCA,
  		Vector vecBackNoe, Vector vecNoesy, int [] numPeaks,boolean isSkipIntensity,double constant)
  {    	
  	double csErrHeavy=0.0;
  	int i,j;
  	BackNoe backNoe;
  	String heavyName;
  	double score=0.0;    	
  	Noesy noesy;    	
  	double csBackH1=0.0;
  	double csBackHeavy=0.0;
  	double csBackH2=0.0;
  	double csExpH1=0.0;
  	double csExpHeavy=0.0;
  	double csExpH2=0.0;     	
  	int counter=0;    	    	
  	double sigmaH=0.04;//    	
  	double sigmaHN=0.04;
  	double sigmaN=0.1;
  	double sigmaHA=0.04;
  	double sigmaCA=0.1;
  	double sigmaHeavy=0.1;
  	double sigmaInten=3.0;
  	
  	double prob_score=0.0;
  	double tempp=0.0;
  	double bkCompDist=0.0;
  	double distSigma=1.0;
  	int bkBinIndex=0;
  	int expBinIndex=0;
  	double distExp=0.0;
  	Noesy nesy=new Noesy();

  	for (i=0; i<vecBackNoe.size();i++)
  	{
  		backNoe=(BackNoe)vecBackNoe.elementAt(i);
  		csBackH1=backNoe.getCSH1();
  		csBackHeavy=backNoe.getCSHeavy();
  		csBackH2=backNoe.getCSH2();    		
  		
  		int no1=backNoe.getFirstResNo();
  		int no2=backNoe.getSecondResNo();
  		String res1=backNoe.getFirstResName();
  		String res2=backNoe.getSecondResName();
  		String atom1=backNoe.getH1Name();
  		String atom2=backNoe.getH2Name();
  		double distanceT=backNoe.getDistance();    		
  		
  		
  		
  		heavyName=backNoe.getHeavyName();
  		tempp=0.0;   
  		double weighting=1.0;
  		double dist_min=9999.9;
  		if (heavyName.substring(0,1).equalsIgnoreCase("N"))
  		{
  			sigmaHeavy=sigmaN;
  			csErrHeavy=csErrN;
  			sigmaH=sigmaHN;    	
  			weighting=0.1;
  		}
  		else
  		{
  			sigmaHeavy=sigmaCA;
  			csErrHeavy=csErrCA;
  			sigmaH=sigmaHA;
  			weighting=0.2;
  		}
  		
  		//System.out.println(i+"-th back-computed NOE (csH1, csH2, csHeavy, csDist): "+ csBackH1 + ", "+ csBackH2+"," +csBackHeavy+", "+bkCompDist);
  		
  		for (j=0;j<vecNoesy.size();j++)
  		{    				
  			noesy=(Noesy)vecNoesy.elementAt(j);
  			csExpH1=noesy.getH1();  
  			csExpHeavy=noesy.getHeavy(); 
  			csExpH2=noesy.getH2();  
  			expBinIndex=noesy.getBinIndex();
  			  			
  				//System.out.println("         "+csExpH1+", "+ csExpH2+", "+csExpHeavy+",  "+intensity);
  				  				
  		    double cs_dist=((csExpH1-csBackH1)*(csExpH1-csBackH1))+
      				(weighting*weighting* (csExpHeavy-csBackHeavy)*(csExpHeavy-csBackHeavy) )+
      				((csExpH2-csBackH2)*(csExpH2-csBackH2));
      			    
  				cs_dist=cs_dist/3;
  				 						
  				
  				if (cs_dist<dist_min)
  					dist_min=cs_dist;
  				
  		
  				
  		}//for (j=0;j<vecNoesy.size();j++)	
  			
  		
  		prob_score=prob_score+dist_min;    		
  		
  	}//for (i=0; i<vecBackNoe.size();i++)
  	
  	score =Math.sqrt( prob_score/(vecBackNoe.size()) ) ;
  	    	
  	numPeaks[0]=counter;
  	return score;
  }

    /**
     * This is an old function, for hydrophilic residues.
     * 
     * @param csErrH  error window of the first proton attached to the heavy atom (indirect proton)
     * @param csErrN the cs err n
     * @param numPeaks number of non-missing peaks
     * @param csErrCA the cs err ca
     * @param vecBackNoe the vec back noe
     * @param vecNoesy the vec noesy
     * @param isSkipIntensity the is skip intensity
     * @param constant the constant
     * 
     * @return matching score
     */  
    public double NoePatternMatchScoreWCaliHdphic(double csErrH, double csErrN, double csErrCA,
    		Vector vecBackNoe, Vector vecNoesy, int [] numPeaks,boolean isSkipIntensity,double constant)
    {   	
    	double csErrHeavy=0.0;
    	int i,j;
    	BackNoe backNoe;
    	String heavyName;
    	double score=0.0;    	
    	Noesy noesy;    	
    	double csBackH1=0.0;
    	double csBackHeavy=0.0;
    	double csBackH2=0.0;
    	double csExpH1=0.0;
    	double csExpHeavy=0.0;
    	double csExpH2=0.0;     	
    	int counter=0;    	    	
    	double sigmaH=0.01; 	
    	double sigmaHN=0.01;
    	double sigmaN=0.1;
    	double sigmaHA=0.01;
    	double sigmaCA=0.1;
    	double sigmaHeavy=0.1;
    	double sigmaInten=3.0;
    	
    	double prob_score=0.0;
    	double tempp=0.0;
    	double bkCompDist=0.0;
    	double distSigma=0.3;
    	int bkBinIndex=0;
    	int expBinIndex=0;
    	double distExp=0.0;
    	Noesy nesy=new Noesy();
    	
    	for (i=0; i<vecBackNoe.size();i++)
    	{
    		backNoe=(BackNoe)vecBackNoe.elementAt(i);
    		csBackH1=backNoe.getCSH1();
    		csBackHeavy=backNoe.getCSHeavy();
    		csBackH2=backNoe.getCSH2();
    		
    		int no1=backNoe.getFirstResNo();
    		int no2=backNoe.getSecondResNo();
    		String res1=backNoe.getFirstResName();
    		String res2=backNoe.getSecondResName();
    		String atom1=backNoe.getH1Name();
    		String atom2=backNoe.getH2Name();
    		double distanceT=backNoe.getDistance();    		
    		    		
    		//we calculate the bin index of the back-computed noe peak:
    		bkCompDist=backNoe.getDistance();
    		if(bkCompDist<=2.7)
    			bkBinIndex=1;
    		else if(bkCompDist<=3.3)
    			bkBinIndex=2;
    		else if(bkCompDist<5.0)
    			bkBinIndex=3;
    		else if(bkCompDist<6.0)
    			bkBinIndex=4;      		
    		    		
    		heavyName=backNoe.getHeavyName();
    		tempp=0.0;   
    		double dist_min=999.9;
    		if (heavyName.substring(0,1).equalsIgnoreCase("N"))
    		{
    			sigmaHeavy=sigmaN;
    			csErrHeavy=csErrN;
    			sigmaH=sigmaHN;    			
    		}
    		else
    		{
    			sigmaHeavy=sigmaCA;
    			csErrHeavy=csErrCA;
    			sigmaH=sigmaHA;
    		}
    		
    		for (j=0;j<vecNoesy.size();j++)
    		{    				
    			noesy=(Noesy)vecNoesy.elementAt(j);
    			csExpH1=noesy.getH1();  
    			csExpHeavy=noesy.getHeavy(); 
    			csExpH2=noesy.getH2();  
    			expBinIndex=noesy.getBinIndex();
    			
    			double intensity=noesy.getIntensity();
    			if (Math.abs(csExpH1-csBackH1)<0.04 && (Math.abs(csExpH2-csBackH2)<0.04) &&
        			Math.abs(csExpHeavy-csBackHeavy)<0.2 )//0.04, 0.03,o.2 for gb1
    			{
    				distExp=nesy.SetCalibrationOneCyana(backNoe,noesy,constant);
    			 
    				counter =counter +1;
    				double inten_dist=((distExp-bkCompDist)*(distExp-bkCompDist))/(distSigma*distSigma);
    				
    				
    				double cs_dist=((csExpH1-csBackH1)*(csExpH1-csBackH1))/(sigmaH*sigmaH)+
        				( (csExpHeavy-csBackHeavy)*(csExpHeavy-csBackHeavy) )/(sigmaHeavy*sigmaHeavy)+
        				((csExpH2-csBackH2)*(csExpH2-csBackH2))/(sigmaH*sigmaH);        			    
    			
    				cs_dist=cs_dist/3;    				
    				  
    				if(isSkipIntensity)
    					inten_dist=0.0;
    				
    				cs_dist=0.5*(cs_dist+inten_dist);
    				    			
    				if (cs_dist<dist_min)
    					dist_min=cs_dist;
    			
    			}
    				
    		}//for (j=0;j<vecNoesy.size();j++)	
    		
    		tempp=tempp+Math.exp(-0.5*dist_min);
    		
    		prob_score=prob_score+tempp;    		
    		
    	}//for (i=0; i<vecBackNoe.size();i++)
    	
    	score =prob_score/(vecBackNoe.size()) ;
        	
    	numPeaks[0]=counter;
    	return score;
    }
    
 
    /**
     * Refined function for the general case:
     * Compute the matching score between back-computed NOE pattern and original NOE list.
     * 
     * @param csErrH  error window of the first proton attached to the heavy atom (indirect proton)
     * @param csErrN the cs err n
     * @param numPeaks number of non-missing peaks
     * @param csErrCA the cs err ca
     * @param vecBackNoe the vec back noe
     * @param vecNoesy the vec noesy
     * @param isSkipIntensity the is skip intensity
     * 
     * @return matching score
     */  
    public double NoePatternMatchScore(double csErrH, double csErrN, double csErrCA,
    		Vector vecBackNoe, Vector vecNoesy, int [] numPeaks,boolean isSkipIntensity)
    {    	
    	double wtIntra=6;//8 or 9 is the best 
       	double csErrHeavy=0.0;
    	int i,j;
    	BackNoe backNoe;
    	String heavyName;
    	double score=0.0;
    	
    	Noesy noesy;
    	
    	double csBackH1=0.0;
    	double csBackHeavy=0.0;
    	double csBackH2=0.0;
    	double csExpH1=0.0;
    	double csExpHeavy=0.0;
    	double csExpH2=0.0;    	
    	
    	int counter=0;
    	    	
    	double sigmaH=0.01;
    	
    	double sigmaHN=0.01;
    	double sigmaN=0.05;
    	double sigmaHA=0.01;
    	double sigmaCA=0.05;
    	double sigmaHeavy=0.05;
    	double sigmaInten=1.0;
    	
    	double prob_score=0.0;
    	double tempp=0.0;
    	double bkCompDist=0.0;
    	double distSigma=1.2;//1.0;
    	int bkBinIndex=0;
    	int expBinIndex=0;
    	double distExp=0.0;
    	
    	for (i=0; i<vecBackNoe.size();i++)
    	{
    		backNoe=(BackNoe)vecBackNoe.elementAt(i);
    		csBackH1=backNoe.getCSH1();
    		csBackHeavy=backNoe.getCSHeavy();
    		csBackH2=backNoe.getCSH2();
    		double wt=1.0;
    		
    		int no1=backNoe.getFirstResNo();
    		int no2=backNoe.getSecondResNo();
    		String res1=backNoe.getFirstResName();
    		String res2=backNoe.getSecondResName();
    		String atom1=backNoe.getH1Name();
    		String atom2=backNoe.getH2Name();
    		double distanceT=backNoe.getDistance();
    		double minExpDist=6.0;
    		
    		//we calculate the bin index of the back-computed noe peak:
    		bkCompDist=backNoe.getDistance();
    		if(bkCompDist<=2.7)
    			bkBinIndex=1;
    		else if(bkCompDist<=3.3)
    			bkBinIndex=2;
    		else if(bkCompDist<5.0)
    			bkBinIndex=3;
    		else if(bkCompDist<6.0)
    			bkBinIndex=4;      		
    		    		
    		heavyName=backNoe.getHeavyName();
    		tempp=0.0;   
    		double dist_min=999.9;
    		if (heavyName.substring(0,1).equalsIgnoreCase("N"))
    		{
    			sigmaHeavy=sigmaN;
    			csErrHeavy=csErrN;
    			sigmaH=sigmaHN;    			
    		}
    		else
    		{
    			sigmaHeavy=sigmaCA;
    			csErrHeavy=csErrCA;
    			sigmaH=sigmaHA;
    		}
    		
    		boolean isFound=false;
    		for (j=0;j<vecNoesy.size();j++)
    		{    				
    			noesy=(Noesy)vecNoesy.elementAt(j);
    			csExpH1=noesy.getH1();  
    			csExpHeavy=noesy.getHeavy(); 
    			csExpH2=noesy.getH2();  
    			expBinIndex=noesy.getBinIndex();
    			distExp=noesy.getUpperDist();
    			double intensity=noesy.getIntensity();
    	
    			if (Math.abs(csExpH1-csBackH1)<0.03 && (Math.abs(csExpH2-csBackH2)<0.03) &&
        			Math.abs(csExpHeavy-csBackHeavy)<0.1 )
    			{
    				
    				double inten_dist=((expBinIndex-bkBinIndex)*(expBinIndex-bkBinIndex))/(sigmaInten*sigmaInten);
    			
    				isFound=true;
    		
    				double cs_dist=((csExpH1-csBackH1)*(csExpH1-csBackH1))/(sigmaH*sigmaH)+
        				( (csExpHeavy-csBackHeavy)*(csExpHeavy-csBackHeavy) )/(sigmaHeavy*sigmaHeavy)+
        				((csExpH2-csBackH2)*(csExpH2-csBackH2))/(sigmaH*sigmaH);        			    
    			
    				cs_dist=cs_dist/3;
    				
    				if(isSkipIntensity)
    					inten_dist=0.0;
    				
    				cs_dist=0.5*(cs_dist+inten_dist);
    			
    				if (cs_dist<dist_min)
    				{
    					dist_min=cs_dist;
    					minExpDist=distExp;
    				}    	    		
    			}
    				
    		}//for (j=0;j<vecNoesy.size();j++)	
    		    		
    		if(isFound)
    			counter =counter +1;    		
    		
    		if(Math.abs(no1-no2)==0)
    			wt=wtIntra;
    		else
    			wt=1.0;
    	
    		tempp=tempp+wt*Math.exp(-0.5*dist_min);
    		   		
    		
    		prob_score=prob_score+tempp;    		
    		
    	}//for (i=0; i<vecBackNoe.size();i++)    	
    	
    	score =prob_score/(vecBackNoe.size()) ;
    	
    	numPeaks[0]=counter;
    	return score;
    }
 
    /**
     * Compute the matching score between backcomputed NOE pattern and original NOE list.
     * 
     * @param vecBackNoe  back computed NOE patterns
     * @param vecHnNoe   HnNOE peak list
     * @param vecCaNoe  CA NOE peak list
     * @param numPeaks number of non-missing peaks
     * 
     * @return matching score
     */  
    public double NoePatternMatchScore(final Vector vecBackNoe, final Vector vecHnNoe, final Vector vecCaNoe, int [] numPeaks)
    {
    	double csErrH=0.04;
    	double csErrN=0.2;
    	double csErrCA=0.12;
    	
    	int i,j;
    	BackNoe backNoe;
    	String heavyName;
    	double score=0.0;
    	double temp=0.0;
    	HnNoe hnNoe;
    	HcchTocsy caNoe;
    	
    	double csBackH1=0.0;
    	double csBackHeavy=0.0;
    	double csBackH2=0.0;
    	double csExpH1=0.0;
    	double csExpHeavy=0.0;
    	double csExpH2=0.0;
    	
    	//the weight of convert N or C cs to the same scale of proton
    	double wtCSfromNtoH=0.04; 
    	double wtCSfromCtoH=0.09; 
    
    	//find the nearest point in original peak list for each backcomputed NOE
    	double rmsd=0.0;
    	double sumH1=0.0;
    	double sumHeavy=0.0;
    	double sumH2=0.0;
    	int index=0;
    	int count=0;
    	
    	double mu=0.0;
    	double sigmaH=csErrH/4;
    	double sigmaHN=csErrH/4;
    	double sigmaN=csErrN/4; 
    	double sigmaHA=csErrH/4; 
    	double sigmaCA=csErrCA/4;
    	double sigmaHeavy=0.0;
    	double winHeavy=0.0;
    	double sigmaDist=0.9;
    	double wtC2=1.0;
    	double prob_score=0.0;
    	double tempp=0.0;
    	for (i=0; i<vecBackNoe.size();i++)
    	{
    		backNoe=(BackNoe)vecBackNoe.elementAt(i);
    		csBackH1=backNoe.getCSH1();
    		csBackHeavy=backNoe.getCSHeavy();
    		csBackH2=backNoe.getCSH2();
    		rmsd=9999.9;
    		index=0;
    	    count=0;
    		heavyName=backNoe.getHeavyName();
    		tempp=0.0;
    		if (heavyName.substring(0,1).equalsIgnoreCase("N"))
    		{
    			sumH1=0.0; 
    			sumHeavy=0.0;
    			sumH2=0.0;  count=0;  	
    			sigmaHeavy=sigmaN;
				sigmaH=sigmaHN;
    			for (j=0;j<vecHnNoe.size();j++)
    			{    				
    				hnNoe=(HnNoe)vecHnNoe.elementAt(j);
    				csExpH1=hnNoe.getHN(); //be careful of the order of H1 and H2
    				csExpHeavy=hnNoe.getN15();
    				csExpH2=hnNoe.getH1();    				
    			
    				if (Math.abs(csExpH1-csBackH1)<csErrH && Math.abs(csExpH2-csBackH2)<csErrH &&
    						Math.abs(csExpHeavy-csBackHeavy)<csErrN)
    				{
    					backNoe.addHnNoe(hnNoe);    	
    					sumH1=sumH1+Math.abs(csExpH1-csBackH1);
    					sumHeavy=sumHeavy+Math.abs(csExpHeavy-csBackHeavy);
    					sumH2=sumH2+Math.abs(csExpH2-csBackH2);
    					
    					//compute the probability score
    					double cs_dist=((csExpH1-csBackH1)*(csExpH1-csBackH1))/(sigmaH*sigmaH)+
        				( (csExpHeavy-csBackHeavy)*(csExpHeavy-csBackHeavy) )/(sigmaHeavy*sigmaHeavy)+
        				((csExpH2-csBackH2)*(csExpH2-csBackH2))/(sigmaH*sigmaH);
        			    
    					cs_dist=cs_dist/3;    					
    					
    					tempp=tempp+Math.exp(-0.5*cs_dist);
    				}
    				
    			}//for (j=0;j<vecHnNoe.size();j++)
    			count=backNoe.getHnNoeVector().size();
    			if(count>0)
    				backNoe.setRmsdAll(sumH1/count,sumHeavy/count,sumH2/count);
    			else
    				backNoe.setIsMissing(true);
    				
    			
    		}//if (heavyName.subSequence(0,1)).equalsIgnoreCase("N"))
    		else
    		{
    			sumH1=0.0; 
    			sumHeavy=0.0;
    			sumH2=0.0;  count=0; 
    			sigmaHeavy=sigmaCA;
				sigmaH=sigmaHA;
    			for (j=0;j<vecCaNoe.size();j++)
    			{
    				caNoe=(HcchTocsy)vecCaNoe.elementAt(j);
    				csExpH1=caNoe.getHA();  //be careful of the order of H1 and H2
    				csExpHeavy=caNoe.getC13();
    				csExpH2=caNoe.getH1();
    				if (Math.abs(csExpH1-csBackH1)<csErrH && Math.abs(csExpH2-csBackH2)<csErrH &&
    						Math.abs(csExpHeavy-csBackHeavy)<csErrCA)
    				{
    					backNoe.addCaNoe(caNoe);     	
    					sumH1=sumH1+Math.abs(csExpH1-csBackH1);
    					sumHeavy=sumHeavy+Math.abs(csExpHeavy-csBackHeavy);
    					sumH2=sumH2+Math.abs(csExpH2-csBackH2);
    					
                        //compute the probability score
    					double cs_dist=((csExpH1-csBackH1)*(csExpH1-csBackH1))/(sigmaH*sigmaH)+
        				( (csExpHeavy-csBackHeavy)*(csExpHeavy-csBackHeavy) )/(sigmaHeavy*sigmaHeavy)+
        				((csExpH2-csBackH2)*(csExpH2-csBackH2))/(sigmaH*sigmaH);
    					cs_dist=cs_dist/3;        					
    					
    					tempp=tempp+Math.exp(-0.5*cs_dist);
    				}
    				
    			}//for (j=0;j<vecHnNoe.size();j++)
    			count=backNoe.getCaNoeVector().size();
    			if (count>0)
    				backNoe.setRmsdAll(sumH1/count,sumHeavy/count,sumH2/count);
    			else 
    				backNoe.setIsMissing(true);
    			
    			
    		}//else
    		if(tempp>=1)
				tempp=1.0;
    		
    		prob_score=prob_score+tempp;
    		
    	}//for (i=0; i<vecBackNoe.size();i++)
    	
    	/////////////////////////////
    	//compute the matching score
    	/////////////////////////////
    	    	
    	//penalty for missing back-computed noe peaks
    	double msPenalty4LargeDist=0.5;
    	double msPenalty4SmallDist=3.0;
    	double distThreshold=4.5;
    	double distUpBound=6.0;
    	
    	double penaltySC=0.0;
    	double matchSC=0.0;
    	Gaussian gs=new Gaussian();
    	double msPr=0.0;
    	double prScore=0.0;
    	double dbPenalty=0;
    	double score_temp=0.0;
    	int counter=0;
    	
    	for (i=0;i<vecBackNoe.size();i++)
    	{
    		backNoe=(BackNoe)vecBackNoe.elementAt(i);
    		if (backNoe.getDistance()==0.0)
    			continue;
    		if (backNoe.getCSH1()==-999.9 || backNoe.getCSH2()==-999.9 || backNoe.getCSHeavy()==-999.9)
    			continue;  		
    	
    		msPr=gs.phiExtend(Math.abs(distUpBound-backNoe.getDistance()),mu,sigmaDist );
    		
    		backNoe.setMissingProb(msPr);
    		
    		if(backNoe.getIsMissing()==true)
    		{
    			backNoe.setMissingPenalty(distThreshold,msPenalty4SmallDist,msPenalty4LargeDist);		
    			penaltySC=penaltySC+backNoe.getMissingPenalty();
    			prScore=prScore+dbPenalty;
    			
    			if(backNoe.getHeavyName().substring(0,1).equalsIgnoreCase("N"))
    			{
    				sigmaHeavy=sigmaN;
    				winHeavy=csErrN;
    			}
    			else
    			{
    				sigmaHeavy=sigmaCA;
    				winHeavy=csErrCA;
    			}
    			double temp8=1-gs.Phi(csErrH,mu,sigmaH);
    			double temp9=1-gs.Phi(winHeavy,mu,sigmaHeavy);
    			double temp4=Math.log(1-gs.Phi(csErrH,mu,sigmaH));
    			double temp5=Math.log(1-gs.Phi(winHeavy,mu,sigmaHeavy));
    			
    			score_temp=score_temp+Math.log(1-gs.Phi(csErrH,mu,sigmaH))+Math.log(1-gs.Phi(winHeavy,mu,sigmaHeavy))+Math.log(1-gs.Phi(csErrH,mu,sigmaH));
    		
    		}
    		else
    		{if(backNoe.getHeavyName().substring(0,1).equalsIgnoreCase("N"))
    	    	{
    				sigmaHeavy=sigmaN;
    				sigmaH=sigmaHN;
    	    	}
    			else
    			{
    				sigmaHeavy=sigmaCA;
    				sigmaH=sigmaHA;
    			}
    			double temp1=((backNoe.getRmsdH1()-mu)*(backNoe.getRmsdH1()-mu))/(sigmaH*sigmaH);
    			double temp2= ((backNoe.getRmsdHeavy()-mu)*(backNoe.getRmsdHeavy()-mu))/(sigmaHeavy*sigmaHeavy);
    			double temp3= ((backNoe.getRmsdH2()-mu)*(backNoe.getRmsdH2()-mu))/(sigmaH*sigmaH);
    			double temp_dist=((backNoe.getRmsdH1()-mu)*(backNoe.getRmsdH1()-mu))/(sigmaH*sigmaH)+
    				((backNoe.getRmsdHeavy()-mu)*(backNoe.getRmsdHeavy()-mu))/(sigmaHeavy*sigmaHeavy)+
    				((backNoe.getRmsdH2()-mu)*(backNoe.getRmsdH2()-mu))/(sigmaH*sigmaH);
    			temp_dist=temp_dist/3;
    			
    			matchSC=matchSC+Math.exp(-0.5*temp_dist);
    			counter =counter +1;
    		
    		}
    	
    	}//for (i=0;i<vecBackNoe.size();i++)       
    
    	double score2 =matchSC/vecBackNoe.size() ;    	  	
    	
    	score =prob_score/(vecBackNoe.size()) ;
    	
    	numPeaks[0]=counter;
    	return score;
    }

    /**
     * swap the stereospecific assignment for a given residue
     * we assume the atom naming scheme is PDB-New.
     * 
     * @param vecAsg  vector of  resonance assignments
     * @param resNo the res no
     * 
     * @return new resonance assignments
     */  
    public Vector SwapStereoAsg(Vector vecAsg, int resNo)
    {
    	Vector vecNewAsg=new Vector();
    	for(int i=0;i<vecAsg.size();i++)
    	{
    		H1CS h1cs=(H1CS)vecAsg.elementAt(i);
    		String atomOld=h1cs.getAtomName();
    		String atom=new String(atomOld);
    		
    		double csH=h1cs.getH1CS();
    		String subAtom="";
    		if(atom.length()>2)
    			subAtom=atom.substring(0,2);
    		else
    			subAtom=atom;
    		String res=h1cs.getResidueType();
    		int no=h1cs.getResidueNo();
    		boolean isChanged=false;
    		if(no==resNo && subAtom.equalsIgnoreCase("HB"))
    		{
    			if(res.equalsIgnoreCase("CYS")||res.equalsIgnoreCase("ASP")||res.equalsIgnoreCase("GLU")||
    					res.equalsIgnoreCase("PHE")||res.equalsIgnoreCase("HIS")||res.equalsIgnoreCase("LYS")||
    					res.equalsIgnoreCase("LEU")||res.equalsIgnoreCase("MET")||res.equalsIgnoreCase("ASN")||
    					res.equalsIgnoreCase("PRO")||res.equalsIgnoreCase("GLN")||res.equalsIgnoreCase("ARG")||
    					res.equalsIgnoreCase("SER")||res.equalsIgnoreCase("TRP")||res.equalsIgnoreCase("TYR"))
    			{
    				if(atom.equalsIgnoreCase("HB1"))
    					atom="HB2";
    				else if(atom.equalsIgnoreCase("HB2"))
    					atom="HB1";
    			}   			
    		}    		
    		if(no==resNo && subAtom.equalsIgnoreCase("HA")&& res.equalsIgnoreCase("GLY") )
    		{
    			if(atom.equalsIgnoreCase("HA1"))
    				atom="HA2";
				else if(atom.equalsIgnoreCase("HA2"))
					atom="HA1";
    		}
    		if(no==resNo && (subAtom.equalsIgnoreCase("HG")||subAtom.equalsIgnoreCase("CG")  )&& res.equalsIgnoreCase("VAL") )
    		{
    			if(atom.equalsIgnoreCase("HG1"))
    				atom="HG2";
				else if(atom.equalsIgnoreCase("HG2"))
					atom="HG1";
				else if(atom.equalsIgnoreCase("CG1"))
    				atom="CG2";
				else if(atom.equalsIgnoreCase("CG2"))
					atom="CG1";
    		}
    		if(no==resNo && (subAtom.equalsIgnoreCase("HD")||subAtom.equalsIgnoreCase("CD"))&& res.equalsIgnoreCase("LEU") )
    		{
    			if(atom.equalsIgnoreCase("HD1"))
    				atom="HD2";
				else if(atom.equalsIgnoreCase("HD2"))
					atom="HD1";
				else if(atom.equalsIgnoreCase("CD1"))
    				atom="CD2";
				else if(atom.equalsIgnoreCase("CD2"))
    				atom="CD1";
    		}
    		vecNewAsg.add(new H1CS(no,res,atom,csH));    		
    	}//for(int i=0;i<vecAsg.size();i++)
    	return vecNewAsg;
    }
    
    /**
     * consider the stereospecific assignments.
     * 
     * @param vecNoe  vector of  original noe assignments
     * 
     * @return new noe assignments
     */  
    public Vector StereospecificNoeAsg(Vector vecNoe)
    {
    	int i,j;
    	Noe noe=new Noe();
    	int firstResNo,secondResNo;
    	String firstResName, secondResName;
    	String firstAtomName, secondAtomName; 
    	Peak pk=new Peak();
    	
    	boolean isFound=false;
    	
    	Vector vecRefinedNoe=new Vector();   
    	for(i=0;i<vecNoe.size();i++)
    	{  			
    		noe=(Noe)vecNoe.elementAt(i);
    		firstResNo=noe.getResidueNoA();
    		secondResNo=noe.getResidueNoB();
    		firstResName=noe.getResidueA();
    		secondResName=noe.getResidueB();
    		firstAtomName=noe.getAtomA();///
    		secondAtomName=noe.getAtomB();
    		double dist=noe.getUpper();
    		boolean isStereo=false;
    		if(firstAtomName.length()>2)
    		{
    			if(firstAtomName.substring(0,2).equalsIgnoreCase("HB"))
    				firstAtomName="HB";
    			if(firstResName.equalsIgnoreCase("LEU") && firstAtomName.substring(0,2).equalsIgnoreCase("HD"))
    				firstAtomName="HD";
    			if(firstResName.equalsIgnoreCase("VAL") && firstAtomName.substring(0,2).equalsIgnoreCase("HG"))
    				firstAtomName="HG";
    			isStereo=true;
    		}
    		if(secondAtomName.length()>2)
    		{
    			if(secondAtomName.substring(0,2).equalsIgnoreCase("HB"))
    				secondAtomName="HB";
    			if(secondAtomName.equalsIgnoreCase("LEU") && secondAtomName.substring(0,2).equalsIgnoreCase("HD"))
    				secondAtomName="HD";
    			if(secondAtomName.equalsIgnoreCase("VAL") && secondAtomName.substring(0,2).equalsIgnoreCase("HG"))
    				secondAtomName="HG";
    			isStereo=true;
    		}
    		noe.setAtom1Name(firstAtomName);
    		noe.setAtom2Name(secondAtomName);
    		if(isStereo)
    			dist=dist+2.0;
    		noe.setDistUpper(dist);    		
    		vecRefinedNoe.add(noe);     			
    	}
    	
    	return vecRefinedNoe;   	
    }
       
    /**
     * Prune ambiguous NOE assignment using given structural information.
     * 
     * @param pdbVec the pdb structure after combining selected rotamers and backbone
     * @param vecNoe  vector of ambiguous noe assignments
     * @param noeLimit noe upper limit to prune ambiguous noes
     * @param methylCorrection  always 0 in new verstion.
     * @param nIsoriginalUp whether keep the original noe upper bound, 1 for yes, 0 for using distance information
     * 
     * @return noe assignments after filtration
     */  
    public Vector<Noe> PrunAmbgNoeAsg(Vector<Pdb> pdbVec, Vector<Noe> vecNoe,double noeLimit, double methylCorrection,int nIsoriginalUp )
    {
    	int i,j;
    	Noe noe=new Noe();
    	int firstResNo,secondResNo;
    	String firstResName, secondResName;
    	String firstAtomName, secondAtomName; 
    	Peak pk=new Peak();
    	double dist_from_inten=0.0;
    	
    	boolean isFound=false;
    	double [] noeDist = new double[2];
    	Vector vecRefinedNoe=new Vector();    	
    	
    	int numPrunedPeaks=0;
    	
    	for(i=0;i<vecNoe.size();i++)
    	{  			
    		noe=(Noe)vecNoe.elementAt(i);
    		firstResNo=noe.getResidueNoA();
    		secondResNo=noe.getResidueNoB();
    		firstResName=noe.getResidueA();
    		secondResName=noe.getResidueB();
    		firstAtomName=noe.getAtomA();///
    		secondAtomName=noe.getAtomB();
    		dist_from_inten=noe.getUpper();
    		double noeUpperPrune=0.0;
    		if (noeLimit> 0) 
    			noeUpperPrune=noeLimit;
    		else
    			noeUpperPrune=dist_from_inten;
    		noeUpperPrune=noeUpperPrune+0.0;    	
    		
    		//check whether the NOE is conistent with the pdb
    		isFound=pk.checkNoeNew(pdbVec,firstResNo,firstAtomName,secondResNo,secondAtomName,
    				noeUpperPrune,0.0,noeDist);	    		    		
    	
    		if(isFound )//remove the case of noe to itself
    		{
    			if(nIsoriginalUp==0)
    				noe.setDistUpper(noeDist[0]+0.5);
    			else if (nIsoriginalUp==2)
    				noe.setDistUpper(6.0);    			
    			
    			vecRefinedNoe.add(noe);     			
    		}    			
    	}
    	
    	return vecRefinedNoe;   	
    }
     
    /**
     * The same as InitAmbiAssignment, but only output those unique noe assignments.
     * 
     * @param nErr  error tolerance in the N dimension
     * @param cErr   error tolerance in the c dimension
     * @param hnErr   error tolerance in the hn dimension
     * @param haErr   error tolerance in the ha dimension
     * @param vecAsg assigned resonance list
     * @param vecAllH1 all sorted protons
     * @param vecNoesy  NOESY peaks, including both c13 and n15 noesy peaks
     * @param h2Err the h2 err
     * 
     * @return ambiguous noe assignments (vector of Noe)
     */  
    public Vector<Noe> InitAmbiAssignmentUnique(double nErr, double cErr, double hnErr, double haErr,
    		double h2Err,Vector vecAsg, Vector vecAllH1,Vector vecNoesy)
    {
    	
    	int debugCounter=0;
    	
    	double h1Err=0.0, heavyErr=0.0;
    	h1Err=Math.max(hnErr, haErr);
    	Vector vecAmbgNoe=new Vector();
    	int i,j;
    	Noesy noesy=new Noesy();
    	H1CS h1CS=new H1CS();
    	double csH1, csH2,csHeavy;
    	String heavyName;
    	
    	Peak pk=new Peak();
    	Vector vecH1Temp=new Vector();
    	Vector vecH2Temp=new Vector();  
    	
    	Peak peakA=new Peak();
    	String resIdA, resA,nucleusA;
    	int indexA, resNoA; 
    	
    	Peak peakB=new Peak();
    	String resIdB, resB,nucleusB;
    	int indexB, resNoB; 
    	double distUpper=0.0;
    	for (i=0;i<vecNoesy.size();i++)
    	{    	
    		boolean isWait=false;
    		boolean isRemoved=false;
    		
    		noesy=(Noesy)vecNoesy.elementAt(i);
    		csHeavy=noesy.getHeavy();
    		csH1=noesy.getH1();
    		csH2=noesy.getH2();
    		distUpper=noesy.getUpperDist();
    		//if(Math.abs(csH1-7.941)<0.05 && Math.abs(csHeavy-121.263)<0.4 && Math.abs(csH2-5.257)<0.05 )
    			//System.out.println("debugging...");
    		
    		//h1Err is maximum of haerr and hnerr
    		vecH1Temp = pk.rangeSearch(vecAllH1, csH1, h1Err);
    		vecH2Temp=pk.rangeSearch(vecAllH1,csH2,h2Err);
    		
    		for (j=0;j<vecH1Temp.size();j++)
    		{
    			peakA = (Peak)vecH1Temp.elementAt(j);
	    		resIdA = peakA.getNucleus();
	    		indexA = resIdA.indexOf("-");
	    		resNoA = Integer.parseInt(resIdA.substring(3, indexA)); //"3" see the method rangeSearch
	    		resA    = resIdA.substring(0, 3);
	    		nucleusA = resIdA.substring(indexA+1);
	    		if(nucleusA.equalsIgnoreCase("HN"))
	    			nucleusA="H";
	    		double csResA=peakA.getCS();
	    		
	    		heavyName=pk.GetHeavyAtomFromProton(resA,nucleusA);
	    		if ( !(heavyName.substring(0,1).equalsIgnoreCase("N")||heavyName.substring(0,1).equalsIgnoreCase("C")) )
	    			continue;
    			if(heavyName.substring(0,1).equalsIgnoreCase("N"))
    			{
    				h1Err=hnErr;
    				heavyErr=nErr;
    			}
    			else
    			{
    				h1Err=haErr;
    				heavyErr=cErr;
    			}
    		
    			if (Math.abs(csResA-csH1)>h1Err)
    				continue;
    			double csHeavyTemp=0.0;
    			
    			for (int k=0;k<vecAsg.size();k++)
    			{
    				H1CS h1cs=(H1CS)vecAsg.elementAt(k);
    				int asgResNo=h1cs.getResidueNo();
    				String asgRes=h1cs.getResidueType();
    				String asgAtom=h1cs.getAtomName();
    				double asgCS=h1cs.getH1CS();
    				if( (asgResNo==resNoA) && (asgAtom.equalsIgnoreCase(heavyName)))
    				{
    					csHeavyTemp=asgCS;    			
    					break;
    				}
    			}//for (int k=0;k<vecAsg.size();k++)
    			if(Math.abs(csHeavyTemp-csHeavy)>heavyErr)
    				continue;   			
    			
    			for (int k=0; k<vecH2Temp.size();k++ )
    			{
    				peakB = (Peak)vecH2Temp.elementAt(k);
    				
    	    		resIdB = peakB.getNucleus();
    	    		indexB = resIdB.indexOf("-");
    	    		resNoB = Integer.parseInt(resIdB.substring(3, indexB)); //"3" see the method rangeSearch
    	    		resB    = resIdB.substring(0, 3);
    	    		nucleusB = resIdB.substring(indexB+1);
    	    		if(nucleusB.equalsIgnoreCase("HN"))
    	    			nucleusB="H";    	    		
    	   
    	    		if(!isWait)
    	    		{
    	    			vecAmbgNoe.add(new Noe(i,resNoA,resNoB, resA, resB,nucleusA, nucleusB,1.8,distUpper  ));
    	    			isWait=true;
    	    			isRemoved=false;
    	    		}
    	    		else
    	    		{
    	    			if(!isRemoved)
    	    			{
    	    				vecAmbgNoe.remove(vecAmbgNoe.size()-1);
    	    				isRemoved=true;
    	    			}
    	    		}
    	    		
    			}//for (int k=0; k<vecH2Temp.size();k++ )
    			
    		}//for (j=0;j<vecH1Temp.size();j++)
    		
    		
    	}//for (i=0;i<vecNoesy.size();i++)
    
    	return vecAmbgNoe;
    }

    /**
     * Compute the intiial NOE assignments using only chemcial shift information.
     * 
     * @param nErr  error tolerance in the N dimension
     * @param cErr   error tolerance in the c dimension
     * @param hnErr   error tolerance in the hn dimension
     * @param haErr   error tolerance in the ha dimension
     * @param vecAsg assigned resonance list
     * @param vecAllH1 all sorted protons
     * @param vecNoesy  NOESY peaks, including both c13 and n15 noesy peaks
     * @param h2Err the h2 err
     * 
     * @return ambiguous noe assignments (vector of Noe)
     */  
    public Vector<Noe> InitAmbiAssignment(double nErr, double cErr, double hnErr, double haErr,
    		double h2Err,Vector<H1CS> vecAsg, Vector<Peak> vecAllH1,Vector<Noesy> vecNoesy)
    {    
    	double h1Err=0.0, heavyErr=0.0;
    	h1Err=Math.max(hnErr, haErr);
    	Vector vecAmbgNoe=new Vector();
    	int i,j;
    	Noesy noesy=new Noesy();
    	H1CS h1CS=new H1CS();
    	double csH1, csH2,csHeavy;
    	String heavyName;
    	
    	Peak pk=new Peak();
    	Vector vecH1Temp=new Vector();
    	Vector vecH2Temp=new Vector();  
    	
    	Peak peakA=new Peak();
    	String resIdA, resA,nucleusA;
    	int indexA, resNoA; 
    	
    	Peak peakB=new Peak();
    	String resIdB, resB,nucleusB;
    	int indexB, resNoB; 
    	double distUpper=0.0;
    	for (i=0;i<vecNoesy.size();i++)
    	{    		
    		noesy=(Noesy)vecNoesy.elementAt(i);
    		csHeavy=noesy.getHeavy();
    		csH1=noesy.getH1();
    		csH2=noesy.getH2();
    		distUpper=noesy.getUpperDist();
    		if(Math.abs(csH1-csH2) < 0.05)
    			continue;
    		//h1Err is maximum of haerr and hnerr
    		vecH1Temp = pk.rangeSearch(vecAllH1, csH1, h1Err);
    		vecH2Temp=pk.rangeSearch(vecAllH1,csH2,h2Err);
    		
    		for (j=0;j<vecH1Temp.size();j++)
    		{
    			peakA = (Peak)vecH1Temp.elementAt(j);
	    		resIdA = peakA.getNucleus();
	    		indexA = resIdA.indexOf("-");
	    		resNoA = Integer.parseInt(resIdA.substring(3, indexA)); //"3" see the method rangeSearch
	    		resA    = resIdA.substring(0, 3);
	    		nucleusA = resIdA.substring(indexA+1);
	    		if(nucleusA.equalsIgnoreCase("HN"))
	    			nucleusA="H";
	    		double csResA=peakA.getCS();
	    		
	    		heavyName=pk.GetHeavyAtomFromProton(resA,nucleusA);
	    		if ( !(heavyName.substring(0,1).equalsIgnoreCase("N")||heavyName.substring(0,1).equalsIgnoreCase("C")) )
	    			continue;
    			if(heavyName.substring(0,1).equalsIgnoreCase("N"))
    			{
    				h1Err=hnErr;
    				heavyErr=nErr;
    			}
    			else
    			{
    				h1Err=haErr;
    				heavyErr=cErr;
    			}
    		
    			if (Math.abs(csResA-csH1)>h1Err)
    				continue;
    			double csHeavyTemp=0.0;
    			
    			for (int k=0;k<vecAsg.size();k++)
    			{
    				H1CS h1cs=(H1CS)vecAsg.elementAt(k);
    				int asgResNo=h1cs.getResidueNo();
    				String asgRes=h1cs.getResidueType();
    				String asgAtom=h1cs.getAtomName();
    				double asgCS=h1cs.getH1CS();
    				if( (asgResNo==resNoA) && (asgAtom.equalsIgnoreCase(heavyName)))
    				{
    					csHeavyTemp=asgCS;    			
    					break;
    				}
    			}//for (int k=0;k<vecAsg.size();k++)
    			if(Math.abs(csHeavyTemp-csHeavy)>heavyErr)
    				continue;   			
    			
    			
    			for (int k=0; k<vecH2Temp.size();k++ )
    			{
    				peakB = (Peak)vecH2Temp.elementAt(k);
    				
    	    		resIdB = peakB.getNucleus();
    	    		indexB = resIdB.indexOf("-");
    	    		resNoB = Integer.parseInt(resIdB.substring(3, indexB)); //"3" see the method rangeSearch
    	    		resB    = resIdB.substring(0, 3);
    	    		nucleusB = resIdB.substring(indexB+1);
    	    		if(nucleusB.equalsIgnoreCase("HN"))
    	    			nucleusB="H";
    	    		
    	    		vecAmbgNoe.add(new Noe(i,resNoA,resNoB, resA, resB,nucleusA, nucleusB,1.8,distUpper  ));
    	    		
    			}//for (int k=0; k<vecH2Temp.size();k++ )
    			
    		}//for (j=0;j<vecH1Temp.size();j++)
    		
    		
    	}//for (i=0;i<vecNoesy.size();i++)    	
    	
    	return vecAmbgNoe;
    }
  
    
    
    /**
     * Initial NOE assignment for 4D NOESY.
     * 
     * @param nErr  error tolerance in the N dimension
     * @param cErr   error tolerance in the c dimension
     * @param hnErr   error tolerance in the hn dimension
     * @param haErr   error tolerance in the ha dimension
     * @param vecAsg assigned resonance list
     * @param vecAllH1 all sorted protons
     * @param vecNoesy  4d NOESY peaks, including both c13 and n15 noesy peaks
     * @param h2Err the h2 err
     * 
     * @return ambiguous noe assignments (vector of Noe)
     */  
    public Vector InitAmbiAssignment4D(double nErr, double cErr, double hnErr, double haErr,
    		double h2Err,Vector vecAsg, Vector vecAllH1,Vector vecNoesy)
    {        
    	double h1Err=0.0, heavyErr=0.0;
    	h1Err=Math.max(hnErr, haErr);
    	Vector vecAmbgNoe=new Vector();
    	int i,j;
    	Noesy noesy=new Noesy();
    	H1CS h1CS=new H1CS();
    	double csH1, csH2,csHeavy1,csHeavy2;
    	String heavyName;
    	
    	Peak pk=new Peak();
    	Vector vecH1Temp=new Vector();
    	Vector vecH2Temp=new Vector();  
    	
    	Peak peakA=new Peak();
    	String resIdA, resA,nucleusA;
    	int indexA, resNoA; 
    	
    	Peak peakB=new Peak();
    	String resIdB, resB,nucleusB;
    	int indexB, resNoB; 
    	double distUpper=0.0;
    	for (i=0;i<vecNoesy.size();i++)
    	{    		
    		noesy=(Noesy)vecNoesy.elementAt(i);
    		csHeavy1=noesy.getHeavy1();
    		csHeavy2=noesy.getHeavy2();
    		csH1=noesy.getH1();
    		csH2=noesy.getH2();
    		distUpper=noesy.getUpperDist();
    		
    		//h1Err is maximum of haerr and hnerr
    		vecH1Temp = pk.rangeSearch(vecAllH1, csH1, h1Err);
    		vecH2Temp=pk.rangeSearch(vecAllH1,csH2,h2Err);
    		
    		for (j=0;j<vecH1Temp.size();j++)
    		{
    			peakA = (Peak)vecH1Temp.elementAt(j);
	    		resIdA = peakA.getNucleus();
	    		indexA = resIdA.indexOf("-");
	    		resNoA = Integer.parseInt(resIdA.substring(3, indexA)); //"3" see the method rangeSearch
	    		resA    = resIdA.substring(0, 3);
	    		nucleusA = resIdA.substring(indexA+1);
	    		if(nucleusA.equalsIgnoreCase("HN"))
	    			nucleusA="H";
	    		double csResA=peakA.getCS();
	    		
	    		heavyName=pk.GetHeavyAtomFromProton(resA,nucleusA);
	    		if ( !(heavyName.substring(0,1).equalsIgnoreCase("N")||heavyName.substring(0,1).equalsIgnoreCase("C")) )
	    			continue;
    			if(heavyName.substring(0,1).equalsIgnoreCase("N"))
    			{
    				h1Err=hnErr;
    				heavyErr=nErr;
    			}
    			else
    			{
    				h1Err=haErr;
    				heavyErr=cErr;
    			}
    		
    			if (Math.abs(csResA-csH1)>h1Err)
    				continue;
    			double csHeavyTemp=0.0;
    			
    			for (int k=0;k<vecAsg.size();k++)
    			{
    				H1CS h1cs=(H1CS)vecAsg.elementAt(k);
    				int asgResNo=h1cs.getResidueNo();
    				String asgRes=h1cs.getResidueType();
    				String asgAtom=h1cs.getAtomName();
    				double asgCS=h1cs.getH1CS();
    				if( (asgResNo==resNoA) && (asgAtom.equalsIgnoreCase(heavyName)))
    				{
    					csHeavyTemp=asgCS;    			
    					break;
    				}
    			}//for (int k=0;k<vecAsg.size();k++)
    			if(Math.abs(csHeavyTemp-csHeavy1)>heavyErr)
    				continue;   			   			
    			
    			for (int k=0; k<vecH2Temp.size();k++ )
    			{
    				peakB = (Peak)vecH2Temp.elementAt(k);
    				
    	    		resIdB = peakB.getNucleus();
    	    		indexB = resIdB.indexOf("-");
    	    		resNoB = Integer.parseInt(resIdB.substring(3, indexB)); //"3" see the method rangeSearch
    	    		resB    = resIdB.substring(0, 3);
    	    		nucleusB = resIdB.substring(indexB+1);
    	    		if(nucleusB.equalsIgnoreCase("HN"))
    	    			nucleusB="H";
    	    		
    	    		double csResB=peakB.getCS();
    	    		heavyName=pk.GetHeavyAtomFromProton(resB,nucleusB);
    	    		if ( !(heavyName.substring(0,1).equalsIgnoreCase("N")||heavyName.substring(0,1).equalsIgnoreCase("C")) )
    	    			continue;
        			if(heavyName.substring(0,1).equalsIgnoreCase("N"))
        			{        				
        				heavyErr=nErr;
        			}
        			else
        			{        				
        				heavyErr=cErr;
        			}
        			if (Math.abs(csResB-csH2)>h2Err)
        				continue;
        			csHeavyTemp=0.0;
        			
        			for (int t=0;t<vecAsg.size();t++)
        			{
        				H1CS h1cs=(H1CS)vecAsg.elementAt(t);
        				int asgResNo=h1cs.getResidueNo();
        				String asgRes=h1cs.getResidueType();
        				String asgAtom=h1cs.getAtomName();
        				double asgCS=h1cs.getH1CS();
        				if( (asgResNo==resNoA) && (asgAtom.equalsIgnoreCase(heavyName)))
        				{
        					csHeavyTemp=asgCS;    			
        					break;
        				}
        			}//for (int k=0;k<vecAsg.size();k++)
        			if(Math.abs(csHeavyTemp-csHeavy2)>heavyErr)
        				continue;   			   			
    	    		
    	    		vecAmbgNoe.add(new Noe(i,resNoA,resNoB, resA, resB,nucleusA, nucleusB,1.8,distUpper  ));
    	    		
    			}//for (int k=0; k<vecH2Temp.size();k++ )
    			
    		}//for (j=0;j<vecH1Temp.size();j++)    		
    	
    	}//for (i=0;i<vecNoesy.size();i++)    	
    	
    	return vecAmbgNoe;
    }
   
   /**
    * written by LW.
    * Please note that in order to sort the map structure we
    * first converted it into a vector
    * IT's a stupid ans slow but I am in a hurry
    * 
    * @param allVec the all vec
    * @param hcchVec the hcch vec
    * 
    * @return the vector
    */
    public Vector mergeAll3(Vector allVec, Vector hcchVec){
	Vector oneVec = new Vector();
	HnNoe hncc2 = new HnNoe();
	HnNoe hnNoe1 = new HnNoe();
	int i, j;
	int no = 0,  no2 = 0;
	Map aMap = new TreeMap();
	Vector pkVec = new Vector();
	Vector oneNoeVec = new Vector();
	Residue resid = new Residue();
	Peak pk = new Peak();
	HcchTocsy hcch = new HcchTocsy();
	Vector allVecN = new Vector();
	for(i=0; i<allVec.size(); i++){
	    resid = (Residue)allVec.elementAt(i);
	    no =  resid.getResidueNo();
	    pkVec = resid.getPeakVec();
	    for (j=0; j<pkVec.size(); j++){
		pk = (Peak) pkVec.elementAt(j);
		aMap.put(pk.getNucleus(), new Double (pk.getCS()) ) ;
	    }
	    for (j=0; j<hcchVec.size(); j++){
		hcch = (HcchTocsy)hcchVec.elementAt(j);
		no2 = hcch.getResidueNo();
		if (no2 == no)
		    oneVec.add(hcch);
	    }
	    resid.setHcchVec(oneVec);
	    oneVec = new Vector();
	    allVecN.add(resid);
	}
	return allVecN;
    }

    /**
     * written by LW.
     * Please note that in order to sort the map structure we
     * first converted it into a vector
     * IT's a stupid ans slow but I am in a hurry
     * 
     * @param allHcchVec the all hcch vec
     * 
     * @return the vector
     */
    public Vector mergeAll4(Vector allHcchVec){
	double h1Err = 0.01;
	Residue res = new Residue();
	Peak pk = new Peak();
	HcchTocsy hcch = new HcchTocsy();
	Vector pkVec = new Vector();
	Vector hcchVec2 = new Vector();
	int    residueNo; 
	String residue;
	double h1 = 0.0;
	double h2 = 0.0;
	int i = 0, j = 0, k = 0;
	Vector pkVecN = new Vector();
	Vector allVec = new Vector();
	String nucleus1;
	String nucleus2;
	for(i=0; i<allHcchVec.size(); i++){
	    res = (Residue)allHcchVec.elementAt(i);
	    residueNo = res.getResidueNo();
	    residue   = res.getResidueType();
  	    System.out.println(residueNo +" "+residue);
	    pkVec = res.getPeakVec();
	    Collections.sort(pkVec, new Peak.csComparator());
	    hcchVec2 = res.getHcchVec();
	    Collections.sort(hcchVec2, new HcchTocsy.haComparator());
 	    System.out.println();
	    for(j=0; j<pkVec.size(); j++){
		pk = (Peak)pkVec.elementAt(j);
		nucleus1 = pk.getNucleus();
		h1 = pk.getCS();
		System.out.println(pk);
	    }
	    for(k=0; k<hcchVec2.size(); k++){
		hcch = (HcchTocsy) hcchVec2.elementAt(k);
		System.out.println(hcch);

	    }
	    allVec.add(new Residue(residueNo, residue, pkVecN, hcchVec2));
	    pkVecN = new Vector();
	}
	return allVec;
    }
    
    /**
     * NOE assignment only based on the chemical shift information,
     * and output the NOE table.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     */       
    public void NOEAsgFromCS(String src, String strOut, String strInput)
    {	    	 
    	boolean isDebug=true;
    	Hsqc hqc = new Hsqc();
    	Peak pk = new Peak();
    	
    	ModelRdc mdc = new ModelRdc();    	
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////
    	int i, j;    	
    	Pdb  pp = new Pdb();    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double haErr  = 0.0;
    	double h1Err  = 0.0;
    	double c13Err = 0.0;
    	double hnErr  = 0.0;
    	double nErr   = 0.0;
    	
    	String strReson="", strSeq="", strNoesy2D="", strHnNoesy3D="", strResNameScheme="", 
    		strHaNoesy3D="", strResFormat="", strIsUnique="",strIsInterSSE="",stroutName="",
    		strNoesyFormat="",strIsLongNoeCheck="",strIsCheckSym="",strIsDeltWeak="",
    		strIsDeltBBNoe="",strOutOrFormat="",strSSEBB="";
    	double SSE1_first=0, SSE1_second=0, SSE2_first=0, SSE2_second=0;
    	
    	for (i=0;i<paraVec.size();i++)
    	{
    		//get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("HAERR"))
    			 haErr  =  Double.parseDouble((String)paraMap.get("HAERR"));
    		if(paraMap.containsKey("H1ERR"))
   			 	h1Err  =  Double.parseDouble((String)paraMap.get("H1ERR"));
    		if(paraMap.containsKey("C13ERR"))
    			c13Err  =  Double.parseDouble((String)paraMap.get("C13ERR"));
    		if(paraMap.containsKey("HNERR"))
    			hnErr  =  Double.parseDouble((String)paraMap.get("HNERR"));
    		if(paraMap.containsKey("NERR"))
    			nErr  =  Double.parseDouble((String)paraMap.get("NERR"));
    		//get the input file name   
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("RESFORMAT"))
    			strResFormat  =  (String)paraMap.get("RESFORMAT");
    		if(paraMap.containsKey("SSEBB"))
    			strSSEBB  =  (String)paraMap.get("SSEBB");
    		if(paraMap.containsKey("RESNAMESCHEME"))
    			strResNameScheme  =  (String)paraMap.get("RESNAMESCHEME");
    		if(paraMap.containsKey("NOESY-FORMAT"))
    			strNoesyFormat  =  (String)paraMap.get("NOESY-FORMAT");   		
    		
    		if(paraMap.containsKey("RESONANCE"))
    			strReson  =  (String)paraMap.get("RESONANCE");
    		strReson=strReson.toLowerCase();
    		if(paraMap.containsKey("2D-NOESY"))
    			strNoesy2D  =  (String)paraMap.get("2D-NOESY");
    		
    		if(paraMap.containsKey("3D-N15-NOESY"))
    			strHnNoesy3D  =  (String)paraMap.get("3D-N15-NOESY");
    		
    		if(paraMap.containsKey("3D-C13-NOESY"))
    			strHaNoesy3D  =  (String)paraMap.get("3D-C13-NOESY");    		
    		if(paraMap.containsKey("ISLONGNOECHECK"))
    			strIsLongNoeCheck  =  (String)paraMap.get("ISLONGNOECHECK");    		
    		if(paraMap.containsKey("ISSYMCHECK"))
    			strIsCheckSym  =  (String)paraMap.get("ISSYMCHECK");    		
    		if(paraMap.containsKey("ISDELETEWEAK"))
    			strIsDeltWeak  =  (String)paraMap.get("ISDELETEWEAK");    		
    		if(paraMap.containsKey("ISDELETEBBNOE"))
    			strIsDeltBBNoe  =  (String)paraMap.get("ISDELETEBBNOE");  		
    		
    		if(paraMap.containsKey("OUTNOENAME"))
    			stroutName  =  (String)paraMap.get("OUTNOENAME");
    		if(paraMap.containsKey("ISUNIQUE"))
    			strIsUnique  =  (String)paraMap.get("ISUNIQUE");
    		if(paraMap.containsKey("ISINTERSSES"))
    			strIsInterSSE  =  (String)paraMap.get("ISINTERSSES");    		
    		if(paraMap.containsKey("SSE1START"))
    			SSE1_first = Double.parseDouble((String)paraMap.get("SSE1START"));
    		if(paraMap.containsKey("SSE1END"))
    			SSE1_second = Double.parseDouble((String)paraMap.get("SSE1END"));
    		if(paraMap.containsKey("SSE2START"))
    			SSE2_first = Double.parseDouble((String)paraMap.get("SSE2START"));
    		if(paraMap.containsKey("SSE2END"))
    			SSE2_second = Double.parseDouble((String)paraMap.get("SSE2END"));    		
    		if(paraMap.containsKey("ISOUTORFORMAT"))
    			strOutOrFormat  =  (String)paraMap.get("ISOUTORFORMAT");  	 		
    		
    	}       	   	
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------
    	
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	//-------------------------------------
    	// (1.2) Read the resonance list 
    	// 
    	// in the "resonance.prot" format
    	//-------------------------------------    	   	   
    	
    	//following is for the general case
    	H1CS h1CS=new H1CS();
    
    	String assignFile = src+ strReson;
    	//this is the vector that stores resonance assignments;
    	//each one includes resNo, resName, atomName, csH1(chemical shift value)
    	Vector assignVec=new Vector();
    	if(strResFormat.equalsIgnoreCase("CYANA"))
    		assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	else if(strResFormat.equalsIgnoreCase("BMRB"))
    		assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	else 
    	{
    		System.out.println("unknown format of the resonance file...");
    		System.exit(0);
    	}
    	
    	//all sorted proton resonances
    	Vector allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator());
    	    	
    	//-------------------------------------
    	// (1.3) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	
    	String strNoeFile = ""; 
    	Vector hnNoeVec=new Vector();//for storing 3d n15 NOESY peaks
    	Vector cnoeVec=new Vector();//for storing 3d c13 noesy peaks
    	if (!strNoesy2D.equalsIgnoreCase("NULL"))
    	{
    		//to do: extension to 2d case...
    	}
    	if (!strHnNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHnNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			hnNoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			hnNoeVec = noesy.NoesyReaderNMRView(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	if (!strHaNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHaNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			cnoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			cnoeVec=noesy.NoesyReaderNMRView(strNoeFile);
    	}
    	
    	Vector vecNoesy=new Vector();
    	vecNoesy.addAll(hnNoeVec);
    	vecNoesy.addAll(cnoeVec);

    	///////////////////////////////
    	//distance calibration
    	String pdbSSEFile = src+strSSEBB;
    	Vector vecSSE = pp.readPdb(pdbSSEFile);
    	double[] constant=new double[1];
    	noesy.SetCalibrationN15(hnNoeVec,vecSSE,assignVec,constant);
    	//Vector vecC13Noesy=noesy.SetCalibrationC13Sec(cnoeVec,vecSeq,assignVec);
    	///////////////////////////////
    	
    	////////////////////////////////////
    	//remove the diagonal NOESY peaks:
    	Vector vecNoesyNoDia=new Vector();
    	for (i=0;i< vecNoesy.size();i++)
    	{
    		Noesy noesy2=(Noesy)vecNoesy.elementAt(i);
    		double cs_h1=noesy2.getH1();
    		double cs_h2=noesy2.getH2();
    		    		    		
    		if (Math.abs(cs_h1-cs_h2)>0.01)
    			vecNoesyNoDia.add(noesy2);    		
    		
    	}//for (i=0;i< vecNoesy.size();i++)
    	vecNoesy=new Vector();
    	vecNoesy.addAll(vecNoesyNoDia);    	
    	////////////////////////////////////
    	//intensity calibration
    	//using the xplor format...
    	//Vector vecNewNoesy=noesy.SetCalibration(vecNoesy); 
    	Vector vecNewNoesy=new Vector();
    	vecNewNoesy.addAll(vecNoesy);
    	
    	/////////////////////////////////////////////////
    	// 2. NOE assignment Part
    	//
    	/////////////////////////////////////////////////
    	
    	// -------------------------------------
    	// (2.1) Initial ambiguous NOE assignment 
    	// for each noe peak, assign noes within 
    	// error tolerance in each dimension
    	//-------------------------------------
    	//Vector vecAmbgAsg=asg.InitAmbiAssignmentHSRI(nErr,c13Err,hnErr,haErr,h1Err,assignVec,allH1Vec,vecNewNoesy );
    	Vector vecAmbgAsg=new Vector();
    	Vector vecUniqueAsg=asg.InitAmbiAssignmentUnique(nErr,c13Err,hnErr,haErr,h1Err,assignVec,allH1Vec,vecNewNoesy );
    	if(strIsUnique.equalsIgnoreCase("1"))
    		vecAmbgAsg.addAll(vecUniqueAsg);
    	else
    		vecAmbgAsg=asg.InitAmbiAssignment(nErr,c13Err,hnErr,haErr,h1Err,assignVec,allH1Vec,vecNewNoesy );
    	
    	Collections.sort(vecAmbgAsg, new Noe.NoeComparator());     	
    	
    	int firstResNo,secondResNo;    	
    
    	Vector vecNewNoes=new Vector();
    	vecNewNoes.addAll(vecAmbgAsg);
    
    	//change the naming scheme from bmrb-new to pdb-new:
    	if(strResNameScheme.equalsIgnoreCase("PDB-NEW"))
    	{
    		Vector vecTemp=new Vector();
    		Noe ee=new Noe();
    		vecTemp=ee.NameConvertFromPDBNewToPDBNew(vecNewNoes);
    		vecNewNoes=new Vector();
    		vecNewNoes.addAll(vecTemp);    		
    	}
    	if(strResNameScheme.equalsIgnoreCase("BMRB-NEW"))
    	{
    		Vector vecTemp=new Vector();
    		Noe ee=new Noe();
    		vecTemp=ee.NameConvertFromBMRBNewToPDBNew(vecNewNoes);
    		vecNewNoes=new Vector();
    		vecNewNoes.addAll(vecTemp);
    		
    		Vector vecAsgNew=new Vector();
    		H1CS h1cs=new H1CS();
    		vecAsgNew=h1cs.CSNameConvertFromBMRBNewToPDBNew(assignVec);
    		assignVec=new Vector();
    		assignVec.addAll(vecAsgNew);
    	}//if(strResNameScheme.equalsIgnoreCase("BMRB-NEW"))    	

    	
    	////////////////////////////////////
    	//calibration for all initial NOE assignments
    	Vector vecAmbgAsgTemp=noesy.SetCalibrationCyana(vecNewNoes,vecNewNoesy,constant[0]);
    	
    	vecNewNoes=new Vector();
    	vecNewNoes.addAll(vecAmbgAsgTemp);
     	///////////////////////////////////////
    	///////////////////////////////////////
    	//delete bb-to-bb NOEs with distance <2.7A
    	if(strIsDeltBBNoe.equalsIgnoreCase("1"))
    	{
    		Vector vecNoeTemp=new Vector();
    	
			for(i=0;i<vecNewNoes.size();i++)
	     	{				
				Noe noe=(Noe)vecNewNoes.elementAt(i);
				int noA=noe.getResidueNoA();
				int noB=noe.getResidueNoB();
				if(Math.abs(noA-noB)<=4)
				{
					vecNoeTemp.add(noe);
					continue;
				}
				String atomA=noe.getAtomA();
	     		String atomB=noe.getAtomB();
	     		double distUp=noe.getUpper();
	     		String atomA_new="";
	     		String atomB_new="";
	     		if(atomA.length()>=2)
	     			atomA_new=atomA.substring(0,2);
	     		else
	     			atomA_new=atomA;
	     		if(atomB.length()>=2)
	     			atomB_new=atomB.substring(0,2);
	     		else
	     			atomB_new=atomB;
	     		if((atomA_new.equalsIgnoreCase("H")|| atomA_new.equalsIgnoreCase("HN") || atomA_new.equalsIgnoreCase("HA")|| atomA_new.equalsIgnoreCase("HB")  )
	     				&& (atomB_new.equalsIgnoreCase("H")||atomB_new.equalsIgnoreCase("HN")||atomB_new.equalsIgnoreCase("HA") || atomB_new.equalsIgnoreCase("HB")) 
	     				&& (distUp<3.4) )	     		
	     			continue;
	     		vecNoeTemp.add(noe);	     		
	     	}//for(i=0;i<vecNewNoes.size();i++)
			vecNewNoes=new Vector();
			vecNewNoes.addAll(vecNoeTemp);
    	}//if(strIsDeltBBNoe.equalsIgnoreCase("1"))    	
    	//////////////////////////////////////////  
    	
    	//////////////////////////////////////////////
    	//check whether checking symmetric NOEs
    	if(strIsCheckSym.equalsIgnoreCase("1"))
    	{
    		Vector vecTemp=pk.CheckNoeSym(vecNewNoes);
    		vecNewNoes=new Vector();
    		vecNewNoes.addAll(vecTemp);
    	}
    	/////////////////////////////////////////////  	
    	
    	
    	///////////////////////////////////
    	//delete 20% weakest NOEs
    	if(strIsDeltWeak.equalsIgnoreCase("1"))
    	{
    		Vector vecNoeTemp=new Vector();
    		for(i=0;i<vecNewNoes.size();i++)
         	{
    			Noe noe=(Noe)vecNewNoes.elementAt(i);
    			double distUp=noe.getUpper();
    			if(distUp<5.9)
    				vecNoeTemp.add(noe);    			
         	}//for(i=0;i<vecNewNoes.size();i++)
    		vecNewNoes=new Vector();
    		vecNewNoes.addAll(vecNoeTemp);
    	}//if(strIsDeltWeak.equalsIgnoreCase("1"))
    	///////////////////////////////////////
    	

    	///////////////////////////////////////////
    	//keep all unique NOE assignments:
    	//vecNewNoes.addAll(vecUniqueAsg);
    	
    	//////////////////////////////////////////
    	
    	///////////////////////////////////////
     	//delete symmetric and diagonal NOEs (which are redundent):
    	//also delete repeated NOEs.
     	Vector vecNoeRefine2=new Vector();
     	
     	for(i=0;i<vecNewNoes.size();i++)
     	{
     		Noe noe=(Noe)vecNewNoes.elementAt(i);
     		int noA=noe.getResidueNoA();
     		int noB=noe.getResidueNoB();
     		String atomA=noe.getAtomA();
     		String atomB=noe.getAtomB();
     		if(noA==noB && atomA.equalsIgnoreCase(atomB))
     			continue;     		
     		boolean isIn=true;     		
     		if(strOutOrFormat.equalsIgnoreCase("0"))
     		{ //if outputing OR format, we don't delte symmetric NOE assignments
	     		for (j=0;j<vecNoeRefine2.size();j++)
	     		{
	     			Noe noeT=(Noe)vecNoeRefine2.elementAt(j);
	     			int noAT=noeT.getResidueNoA();
	     			int noBT=noeT.getResidueNoB();
	     			String atomAT=noeT.getAtomA();
	     			String atomBT=noeT.getAtomB();
	     			if(noA==noAT && atomA.equalsIgnoreCase(atomAT) && noB==noBT && atomB.equalsIgnoreCase(atomBT))
	     				isIn=false;
	     			if(noA==noBT && atomA.equalsIgnoreCase(atomBT) && noB==noAT && atomB.equalsIgnoreCase(atomAT))
	     				isIn=false;
	     		}//for (j=0;j<vecNoeRefine2.size();j++)
	     		
     		}//if(strOutOrFormat.equalsIgnoreCase("0"))
     		if(	isIn)
     			vecNoeRefine2.add(noe);
     	}
     	
     	vecNewNoes=new Vector();     	
     	vecNewNoes.addAll(vecNoeRefine2);
     	///////////////////////////////////////    	
    	
    	/////////////////////////////////////////////////////
    	//check long-range NOEs by neighboring information:    
     	if(strIsLongNoeCheck.equalsIgnoreCase("1"))
    	{
     		Vector  vecRefineLongNOE=pk.CheckLongRangNOE(vecNewNoes,10);   //thershold >=3 seems reasonable 	
	    	//Vector vecRefineLongNOE2=pk.CheckLongRangNOE(vecRefineLongNOE,3);
     		Vector vecRefineLongNOE2=new Vector();
     		vecRefineLongNOE2.addAll(vecRefineLongNOE);
     		
     		
	    	vecNewNoes=new Vector();
	    	vecNewNoes.addAll(vecRefineLongNOE2);    	
	    	Collections.sort(vecNewNoes, new Noe.NoeComparator());   
    	}
    	/////////////////////////////////////////////////////
     	
     	////////////////////////////////////
     	//check the inter-sse NOEs
    	Vector vecTemp=new Vector();
    	if(strIsInterSSE.equalsIgnoreCase("1"))
    	{
    		for(i=0;i<vecNewNoes.size();i++)
        	{
    			boolean isIn=false;
    			Noe noe=new Noe();
        		noe=(Noe)vecNewNoes.elementAt(i);
        		firstResNo=noe.getResidueNoA();
        		secondResNo=noe.getResidueNoB();
        		
        		if ((firstResNo<=SSE1_second && firstResNo>=SSE1_first) && 
        				(secondResNo<=SSE2_second && secondResNo>=SSE2_first )  )
        			isIn=true;
        		
        		if ((secondResNo<=SSE1_second && secondResNo>=SSE1_first) && 
        				(firstResNo<=SSE2_second && firstResNo>=SSE2_first )  )
        			isIn=true;
        		if(isIn)
        			vecTemp.add(noe);
        		
        	}//for(i=0;i<vecAmbgAsg.size();i++)
    		
    	} //if(strIsInterSSE.equalsIgnoreCase("1"))   		
    	else
    		vecTemp.addAll(vecNewNoes);
    	
    	vecNewNoes=new Vector();
    	vecNewNoes.addAll(vecTemp);
     	
    	//	-------------------------------------
    	// (2.4) Output the final NOE assignments in standard xplor format
    	// 
    	//
    	//-------------------------------------
    	String userDirTemp = System.getProperty("user.dir");////
    	if(strOutOrFormat.equalsIgnoreCase("0"))
    	{
    		String fileName=userDirTemp+stroutName;    	
    	
	    	String xplorNoeStr="";
	    	double[] disUpper=new double[1];
	    	try{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		out.println("!REMARK: Total number of NOEs is "+vecNewNoes.size());
	    		out.println("");
	    		out.println("");
	    		
		       	for (i=0;i<vecNewNoes.size();i++)
		    	{
		    		Noe noe=(Noe)vecNewNoes.elementAt(i);
		    		int resNo1=noe.getResidueNoA();
		    		int resNo2=noe.getResidueNoB();
		    		String res1=noe.getResidueA();
		    		String res2=noe.getResidueB();
		    		String atom1=noe.getAtomA();
		    		String atom2=noe.getAtomB();
		    		double distUpper=noe.getUpper();	    		
		    		
		    		//xplor format:	    		
					xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
					if(isDebug)
						System.out.println(xplorNoeStr); 
					out.println(xplorNoeStr);
		    	}//for (i=0;i<vecRefinedNoes.size();i++)
		    	out.close();
		    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
	    	}catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + fileName);
			}catch (IOException e)
			{
			   System.out.println("IOException: the stack trace is:");
			   e.printStackTrace();
			}
			
    	}//if(strOutOrFormat.equalsIgnoreCase("0"))
    	else if(strOutOrFormat.equalsIgnoreCase("1")) //we output the NOEs in OR format:
    	{
    		Noe noeTemp=new Noe();    		
    		Vector vecMultiAsg=noeTemp.ConvertSingleToMultiAsg(vecNewNoes);
    		String fileName=userDirTemp+stroutName;  
    		noeTemp.OutputMultiNOEAsg(vecMultiAsg,fileName);    		
    	}
    	
       	long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Total time for NOE assignment:  "+ totalTime +" minutes");
    	
    }

    /**
     * HANA algorithm for the NOE assignment
     * and output the NOE table.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     * 
     * @throws JampackException the jampack exception
     */       
    public void doHANANOEAsg(String src, String strOut, String strInput)throws JampackException
    {	    	 
    	boolean isDebug=true;
    	Hsqc hqc = new Hsqc();
    	Peak pk = new Peak();
    	
    	ModelRdc mdc = new ModelRdc();    	
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////
    	int i, j;    	
    	Pdb  pp = new Pdb();    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double haErr  = 0.0;
    	double h1Err  = 0.0;
    	double c13Err = 0.0;
    	double hnErr  = 0.0;
    	double nErr   = 0.0;
    	
    	int nIs4DNoesy=0;
    	
    	String strReson="", strSeq="", strNoesy2D="", strHnNoesy3D="", strResNameScheme="",strIsCheckLongAA="",
    		strHaNoesy3D="", strResFormat="",strBB="",stroutName="",strNoesyFormat="",strOutOrFormat="",strSSEBB="",
    		strIsCheck="",strRefPdb="",strRefNameSchem="",strIsWholeStr="",strPdbNameScheme="",str4DNoesy="";
    	int nIsoriginalUp=0; 
    	double noeLimit=0.0,metCor=0.0;
    	
    	for (i=0;i<paraVec.size();i++)
    	{
    		//get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("HAERR"))
    			 haErr  =  Double.parseDouble((String)paraMap.get("HAERR"));
    		if(paraMap.containsKey("H1ERR"))
   			 	h1Err  =  Double.parseDouble((String)paraMap.get("H1ERR"));
    		if(paraMap.containsKey("C13ERR"))
    			c13Err  =  Double.parseDouble((String)paraMap.get("C13ERR"));
    		if(paraMap.containsKey("HNERR"))
    			hnErr  =  Double.parseDouble((String)paraMap.get("HNERR"));
    		if(paraMap.containsKey("NERR"))
    			nErr  =  Double.parseDouble((String)paraMap.get("NERR"));
    		//get the input file name   
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("RESFORMAT"))
    			strResFormat  =  (String)paraMap.get("RESFORMAT");
    		if(paraMap.containsKey("RESONANCE"))
    			strReson  =  (String)paraMap.get("RESONANCE");
    		strReson=strReson.toLowerCase();
    		if(paraMap.containsKey("2D-NOESY"))
    			strNoesy2D  =  (String)paraMap.get("2D-NOESY");
    		
    		if(paraMap.containsKey("3D-N15-NOESY"))
    			strHnNoesy3D  =  (String)paraMap.get("3D-N15-NOESY");
    	
    		if(paraMap.containsKey("3D-C13-NOESY"))
    			strHaNoesy3D  =  (String)paraMap.get("3D-C13-NOESY");
    		
    		if(paraMap.containsKey("PDBNAMESCHEME"))
    			strPdbNameScheme  =  (String)paraMap.get("PDBNAMESCHEME");	
    		if(paraMap.containsKey("OUTNOENAME"))
    			stroutName  =  (String)paraMap.get("OUTNOENAME");   	
    		if(paraMap.containsKey("BACKBONE"))
    			strBB  =  (String)paraMap.get("BACKBONE");
    		if(paraMap.containsKey("SSEBB"))
    			strSSEBB  =  (String)paraMap.get("SSEBB");
    		
    		if(paraMap.containsKey("NOELIMIT"))
    			noeLimit  =  Double.parseDouble((String)paraMap.get("NOELIMIT"));
    		if(paraMap.containsKey("IS4DNOESY"))
    			nIs4DNoesy  =   Integer.parseInt((String)paraMap.get("IS4DNOESY"));
    		
    		if(paraMap.containsKey("4D-NOESY"))
    			str4DNoesy  =  (String)paraMap.get("4D-NOESY");
    		
    		if(paraMap.containsKey("ISORIGINALUP"))
    			nIsoriginalUp  =   Integer.parseInt((String)paraMap.get("ISORIGINALUP"));	
    		if(paraMap.containsKey("RESNAMESCHEME"))
    			strResNameScheme  =  (String)paraMap.get("RESNAMESCHEME");	
    		if(paraMap.containsKey("NOESY-FORMAT"))
    			strNoesyFormat  =  (String)paraMap.get("NOESY-FORMAT");   	
    		if(paraMap.containsKey("ISCHECKLONGAA"))
    			strIsCheckLongAA =  (String)paraMap.get("ISCHECKLONGAA"); 	
    		if(paraMap.containsKey("ISCHECK"))
    			strIsCheck  =  (String)paraMap.get("ISCHECK");   	
    		if(paraMap.containsKey("REFPDB"))
    			strRefPdb  =  (String)paraMap.get("REFPDB");  
    		if(paraMap.containsKey("METHYL-CORRECTION"))
    			metCor  =  Double.parseDouble((String)paraMap.get("METHYL-CORRECTION"));    		
    		if(paraMap.containsKey("REFNAMESCHEME"))
    			strRefNameSchem  =  (String)paraMap.get("REFNAMESCHEME");  
    		if(paraMap.containsKey("ISWHOLESTRUCTURE"))
    			strIsWholeStr  =  (String)paraMap.get("ISWHOLESTRUCTURE");  
    		if(paraMap.containsKey("ISOUTORFORMAT"))
    			strOutOrFormat  =  (String)paraMap.get("ISOUTORFORMAT");  	 	
    	}       	   	
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------
    	//String seqFile = src+"cbf.seq";
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	//-------------------------------------
    	// (1.2) Read the resonance list 
    	// 
    	// in the "resonance.prot" format
    	//-------------------------------------
        	
    	//following is for the general case
    	H1CS h1CS=new H1CS();
    	   
    	String assignFile = src+ strReson; 
    	//this is the vector that stores resonance assignments;
    	//each one includes resNo, resName, atomName, csH1(chemical shift value)
    	Vector assignVec=new Vector();
    	if(strResFormat.equalsIgnoreCase("CYANA"))
    		assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	else if(strResFormat.equalsIgnoreCase("BMRB"))
    		assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	else 
    	{
    		System.out.println("unknown format of the resonance file...");
    		System.exit(0);
    	}
    	
    	//all sorted proton resonances
    	Vector allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator()); 	    	
    	
    	
    	//-------------------------------------
    	// (1.3) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	
    	String strNoeFile = ""; 
    	Vector hnNoeVec=new Vector();//for storing 3d n15 NOESY peaks
    	Vector cnoeVec=new Vector();//for storing 3d c13 noesy peaks
    	Vector Noe4DVec=new Vector();//for storing 4d NOESY peaks
    	if (!strNoesy2D.equalsIgnoreCase("NULL"))
    	{
    		//to do: extension to 2d case...
    	}
    	if (!strHnNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHnNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			hnNoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			hnNoeVec = noesy.NoesyReaderNMRView(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	if (!strHaNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHaNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			cnoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			cnoeVec=noesy.NoesyReaderNMRView(strNoeFile);
    	}
    	
    	//4D NOESY data
    	if (!str4DNoesy.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+str4DNoesy;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			Noe4DVec = noesy.NoesyReader(strNoeFile);// to be changed....
    		else
    			Noe4DVec = noesy.NoesyReaderNMRView4D(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	Vector vecNoesy=new Vector();
    	vecNoesy.addAll(hnNoeVec);
    	vecNoesy.addAll(cnoeVec);
    	
    	Vector vecNoesy4D=new Vector();
    	vecNoesy4D.addAll(Noe4DVec);//for 4D noesy   
    	
    	///////////////////////////////
    	//distance calibration
    	String pdbSSEFile = src+strSSEBB;
    	Vector vecSSE = pp.readPdb(pdbSSEFile);
    	double[] constant=new double[1];
    	//noesy.SetCalibrationN15(vecNoesy,vecSSE,assignVec,constant);
    	noesy.SetCalibrationN15(hnNoeVec,vecSSE,assignVec,constant);
    	///////////////////////////////    		
    	
    	////////////////////////////////////
    	//remove the diagonal NOESY peaks:
    	Vector vecNoesyNoDia=new Vector();
    	for (i=0;i< vecNoesy.size();i++)
    	{
    		Noesy noesy2=(Noesy)vecNoesy.elementAt(i);
    		double cs_h1=noesy2.getH1();
    		double cs_h2=noesy2.getH2();
    		    		    		
    		if (Math.abs(cs_h1-cs_h2)>0.01)//range from 0.01  to 0.05
    			vecNoesyNoDia.add(noesy2);    		
    		
    	}//for (i=0;i< vecNoesy.size();i++)
    	vecNoesy=new Vector();
    	vecNoesy.addAll(vecNoesyNoDia);    	
    	
    	//for 4D
    	vecNoesyNoDia=new Vector();
    	for (i=0;i< vecNoesy4D.size();i++)
    	{
    		Noesy noesy2=(Noesy)vecNoesy4D.elementAt(i);
    		double cs_h1=noesy2.getH1();
    		double cs_h2=noesy2.getH2();
    		double cs_heavy1=noesy2.getHeavy1();
    		double cs_heavy2=noesy2.getHeavy2();   		    		
    		if (Math.abs(cs_h1-cs_h2)>0.04 && Math.abs(cs_heavy1-cs_heavy2)>0.06)//range from 0.01  to 
    			vecNoesyNoDia.add(noesy2);    		
    		
    	}//for (i=0;i< vecNoesy.size();i++)
    	vecNoesy4D=new Vector();
    	vecNoesy4D.addAll(vecNoesyNoDia);    	
    	////////////////////////////////////
    	    	
    	//intensity calibration    	
    	Vector vecNewNoesy=new Vector();    	
    	Vector vecNewNoesy4D=new Vector();	
    	
    	vecNewNoesy.addAll(vecNoesy);
    	vecNewNoesy4D.addAll(vecNoesy4D);
    	// -------------------------------------
    	// (1.2) Read the backbone structure 
    	// (input file name: backbone.pdb)
    	// Note: the backbone must be connected. 
    	//-------------------------------------    	
    	//String pdbFile = src+strBB;
    	String userDirTemp = System.getProperty("user.dir");////
    	String pdbFile = userDirTemp+strBB;
    	
    	
    	Vector pdbVecBB = pp.readPdb(pdbFile);
    	Vector pdbVecNew=new Vector();    	
    	
    	//update backbone atom names according to protein sequence input
    	//residue name in previous backbone may be all in "ALA".
    	if(strIsWholeStr.equalsIgnoreCase("1"))    	
    		pdbVecNew= pp.residueNameUpdateNoStr(vecSeq, pdbVecBB); 
    	else
    		pdbVecNew = pp.residueNameUpdate(vecSeq, pdbVecBB);
    	
    	Vector pdbVecNewSec=new Vector();
		if(strPdbNameScheme.equalsIgnoreCase("PDB-OLD"))
			pdbVecNewSec=pp.nameConvertOrder(pdbVecNew); 
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-NEW"))
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecNew);//need to double check
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-OLD"))
		{
			Vector pdbVecSSE22=pp.nameConvertOrder(pdbVecNew); 
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecSSE22);//need to double check
		}
		else
			pdbVecNewSec.addAll(pdbVecNew);  
    	
		Vector vecStructureAll=new Vector();
		vecStructureAll.addAll(pdbVecNewSec);
		
		Vector vecBBTemp=pp.OutputBackbone(pdbVecNewSec);
		pdbVecNewSec=new Vector();
		pdbVecNewSec.addAll(vecBBTemp);
		
		
		
    	/////////////////////////////////////////////////
    	// 2. NOE assignment Part
    	//
    	/////////////////////////////////////////////////
    	
    	// -------------------------------------
    	// (2.1) Initial ambiguous NOE assignment 
    	// for each noe peak, assign noes within 
    	// error tolerance in each dimension
    	//-------------------------------------
    	Vector vecAmbgAsg=new Vector();    	
    	vecAmbgAsg=asg.InitAmbiAssignment(nErr,c13Err,hnErr,haErr,h1Err,assignVec,allH1Vec,vecNewNoesy );
    	    	
    	Vector vecAmbgAsg4D=new Vector();    	
    	vecAmbgAsg4D=asg.InitAmbiAssignment4D(nErr,c13Err,hnErr,haErr,h1Err,assignVec,allH1Vec,vecNewNoesy4D );
    	
    	vecAmbgAsg.addAll(vecAmbgAsg4D);//////////////////
    	
    	
    	///////////////////////////////////////
    	//consider the stereospecific asssignment
    	//Vector vecAmbgAsgTT=asg.StereospecificNoeAsg(vecAmbgAsg);
    	//vecAmbgAsg=new Vector();
    	//vecAmbgAsg.addAll(vecAmbgAsgTT);
    	///////////////////////////////////////
    	
    	Collections.sort(vecAmbgAsg, new Noe.NoeComparator());        	
    
    	Vector vecNewNoes=new Vector();
    	vecNewNoes.addAll(vecAmbgAsg);
   		// change the naming scheme from bmrb-new to pdb-new:
    	if(strResNameScheme.equalsIgnoreCase("PDB-NEW"))
    	{
    		Vector vecTemp=new Vector();
    		Noe ee=new Noe();
    		vecTemp=ee.NameConvertFromPDBNewToPDBNew(vecNewNoes);
    		vecNewNoes=new Vector();
    		vecNewNoes.addAll(vecTemp);    		
    	}
    	if(strResNameScheme.equalsIgnoreCase("BMRB-NEW"))
    	{
    		Vector vecTemp=new Vector();
    		Noe ee=new Noe();
    		vecTemp=ee.NameConvertFromBMRBNewToPDBNew(vecNewNoes);
    		vecNewNoes=new Vector();
    		vecNewNoes.addAll(vecTemp);
    		
    		Vector vecAsgNew=new Vector();
    		H1CS h1cs=new H1CS();
    		vecAsgNew=h1cs.CSNameConvertFromBMRBNewToPDBNew(assignVec);
    		assignVec=new Vector();
    		assignVec.addAll(vecAsgNew);
    	}//if(strResNameScheme.equalsIgnoreCase("BMRB-NEW"))
    	vecAmbgAsg=new Vector();
    	vecAmbgAsg.addAll(vecNewNoes);
    	
    	////////////////////////////////////
    	//    	calibration for all initial NOE assignments
    	Vector vecAmbgAsgTemp=noesy.SetCalibrationCyana(vecAmbgAsg,vecNewNoesy,constant[0]);
    	
    	vecAmbgAsg=new Vector();
    	vecAmbgAsg.addAll(vecAmbgAsgTemp);
     	///////////////////////////////////////
     	//delete symmetric NOEs (which are redundent):
     	Vector vecNoeRefine2=new Vector();
     	
     	for(i=0;i<vecAmbgAsg.size();i++)
     	{
     		Noe noe=(Noe)vecAmbgAsg.elementAt(i);
     		int noA=noe.getResidueNoA();
     		int noB=noe.getResidueNoB();
     		String atomA=noe.getAtomA();
     		String atomB=noe.getAtomB();
     		if(noA==noB && atomA.equalsIgnoreCase(atomB))
     			continue;
     		
     		boolean isIn=true;
     		if(strOutOrFormat.equalsIgnoreCase("0"))
     		{
	     		for (j=0;j<vecNoeRefine2.size();j++)
	     		{
	     			Noe noeT=(Noe)vecNoeRefine2.elementAt(j);
	     			int noAT=noeT.getResidueNoA();
	     			int noBT=noeT.getResidueNoB();
	     			String atomAT=noeT.getAtomA();
	     			String atomBT=noeT.getAtomB();
	     			if(noA==noAT && atomA.equalsIgnoreCase(atomAT) && noB==noBT && atomB.equalsIgnoreCase(atomBT))
	     				isIn=false;
	     			if(noA==noBT && atomA.equalsIgnoreCase(atomBT) && noB==noAT && atomB.equalsIgnoreCase(atomAT))
	     				isIn=false;
	     		}//for (j=0;j<vecNoeRefine2.size();j++)
     		}//if(strOutOrFormat.equalsIgnoreCase("0"))
     		if(	isIn)
     			vecNoeRefine2.add(noe);
     	}
     	
     	vecAmbgAsg=new Vector();     	
     	vecAmbgAsg.addAll(vecNoeRefine2);
     	///////////////////////////////////////
     	
       	//////////////////////////////////////
    	//sort the NOE table
    	if(!strOutOrFormat.equalsIgnoreCase("1"))
    	{
	    	for(i=0;i<vecAmbgAsg.size();i++)
	    	{
	    		Noe noe=(Noe)vecAmbgAsg.elementAt(i);
	    		int firstResNo=noe.getResidueNoA();
	    		int secondResNo=noe.getResidueNoB();
	    		String firstResName=noe.getResidueA();
	    		String secondResName=noe.getResidueB();
	    		String firstAtomName=noe.getAtomA();///
	    		String secondAtomName=noe.getAtomB();     		
	    		
	    		
	    		if(secondResNo<firstResNo)
	    		{
	    			noe.setResNoA(secondResNo);
	    			noe.setResNoB(firstResNo);
	    			noe.setAtom1Name(secondAtomName);
	    			noe.setAtom2Name(firstAtomName);
	    			noe.setResNameA(secondResName);
	    			noe.setResNameB(firstResName);
	    		}
	    	}//for(i=0;i<vecManAsgNew.size();i++)
    	
	    	Collections.sort(vecAmbgAsg, new Noe.noeComparatorC());
    	}
    	//////////////////////////////////////////////////////   	

     	
    	//-------------------------------------
    	// Read the rotamer library path 
    	// 
    	// use the small Xtal' library from Richardson's lab
    	//-------------------------------------
    	String userDir = System.getProperty("user.dir");////
    	String rotSrc= userDir+"/system/rot-lib/";// src+ "rotasamp-small/";    	
    	
    	//-------------------------------------
    	// (2.2) Rotamer selection and combination of low resolution structure 
    	// select optimal rotamer at each residue 
    	// and place optimal rotamers into backbone
    	//-------------------------------------

    	//methyl correction:
    	Vector vecNoeAfterMetCor=new Vector();
    	Noe ee=new Noe();
    	vecNoeAfterMetCor=ee.NoeMethylCorrection(vecAmbgAsg,2.5);
    	vecAmbgAsg=new Vector();
    	vecAmbgAsg.addAll(vecNoeAfterMetCor);
 	    	
    	Vector vecRotStructure=pp.RotamSelectAndStructure(hnErr, nErr, c13Err,pdbVecNewSec,assignVec,rotSrc,vecNewNoesy,
    			4.5, 1,constant[0]);
    	
    	
    	////////////////////////////
    	if(strIsWholeStr.equalsIgnoreCase("1"))
    	{
    		vecRotStructure=new Vector();
    		vecRotStructure.addAll(vecStructureAll);
    	}
    	
       	//-------------------------------------
    	// (2.3) Filtration of ambiguous NOE assignments
    	// select optimal rotamer at each residue 
    	// and place optimal rotamers into backbone
    	//-------------------------------------
    	Vector vecRefinedNoes=new Vector();
    	
    	
    	vecRefinedNoes=asg.PrunAmbgNoeAsg(vecRotStructure,vecAmbgAsg,noeLimit,0.0,nIsoriginalUp);
    	
     	Collections.sort(vecRefinedNoes, new Noe.NoeComparator()); 
     	
     	
     	
     	
     	
    	////////////////////////////////////////////////
    	if(strIsCheckLongAA.equalsIgnoreCase("1"))
    	{
    		//Peak pk=new Peak();
    		Vector vecTemp=new Vector();
    		for(i=0;i<vecRefinedNoes.size();i++)
        	{
        		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
        		int firstResNo=noe.getResidueNoA();
        		int secondResNo=noe.getResidueNoB();
        		String firstResName=noe.getResidueA();
        		String secondResName=noe.getResidueB();
        		String atomA=noe.getAtomA();
        		String atomB=noe.getAtomB();        		
        		String subAtomA=atomA;
        		String subAtomB=atomB;
        		if (atomA.length()>=2)
        			subAtomA=atomA.substring(0,2);
        		if(atomB.length()>=2) 
        			subAtomB=atomB.substring(0,2);
        			
        		if(Math.abs(firstResNo-secondResNo)<=4 )
        		{
        			vecTemp.add(noe);
        			continue;
        		}
        		        		
        		boolean isKeep=true;
        		if ( pk.isCharged(firstResName) && pk.isHydrophobic(secondResName) )
        			isKeep=false;
        		if ( pk.isCharged(secondResName) && pk.isHydrophobic(firstResName) )
        			isKeep=false;
        		if( (subAtomB.equalsIgnoreCase("HN") || subAtomB.equalsIgnoreCase("H")||subAtomB.equalsIgnoreCase("HA")||subAtomB.equalsIgnoreCase("HB")) &&
        				(subAtomA.equalsIgnoreCase("HN")||subAtomA.equalsIgnoreCase("H")||subAtomA.equalsIgnoreCase("HA")|| subAtomA.equalsIgnoreCase("HB")   ))
        			isKeep=true;
        		
        		if(isKeep)
        			vecTemp.add(noe);
        	}//for(i=0;i<vecManAsgNew.size();i++)
    		
    		vecRefinedNoes=new Vector();
    		vecRefinedNoes.addAll(vecTemp);
    	}//if(strIsCheckLongAA.equalsIgnoreCase("1"))
    	//////////////////////////////////////////////////
       	
    	//sort the NOE restraints, and updated the atom name for metyl groups:
    	Collections.sort(vecRefinedNoes, new Noe.NoeComparator()); 
        Noe ne=new Noe();
        Vector vecNoeNewT=ne.NameConvertFromPDBNewToPDBNew(vecRefinedNoes);
        vecRefinedNoes=new Vector();
        vecRefinedNoes.addAll(vecNoeNewT);
        
     	
     	///////////////////////////////////////
     	//compute the numbers of unique and multiple assignments:
     	vecNoeRefine2=new Vector();
     	Vector vecNoeUniq=new Vector();
     	for(i=0;i<vecRefinedNoes.size();i++)
     	{
     		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
     		int pkID=noe.getPeakID();
     		boolean isUniq=true;
     		for(j=0;j<vecRefinedNoes.size();j++)
     		{
     			if (i==j)
     				continue;
     			Noe noeT=(Noe)vecRefinedNoes.elementAt(j);
     			int pkIDT=noeT.getPeakID();
     			if(pkID==pkIDT)
     				isUniq=false;
     		}
     		if(isUniq)
     			vecNoeUniq.add(noe);
     	}
     	///////////////////////////////////////////
 
     	
     	//////////////////////////////////////////
     	double[] number=new double[4];
     	if(strIsCheck.equalsIgnoreCase("1"))
     	{
     		Noe noe_temp=new Noe();
     		
     		String pdbFileXray = src+strRefPdb;  
        	Vector vecTempPdbBB=pp.readPdb(pdbFileXray);
        	Vector vecTempPdbBB2=new Vector();
        	if(strRefNameSchem.equalsIgnoreCase("PDB-OLD")) 	
        		vecTempPdbBB2 = pp.nameConvertOrder(vecTempPdbBB);
        	else if(strRefNameSchem.equalsIgnoreCase("BMRB-NEW"))
        		vecTempPdbBB2=pp.nameConvert2PDBNew(vecTempPdbBB);
        	else
        		vecTempPdbBB2.addAll(vecTempPdbBB);
        	System.out.println("============================================ ");
        	System.out.println("====Comparisons with reference structure===== ");
        	noe_temp.CompareAsgToRefStr(vecRefinedNoes,vecTempPdbBB2,6.0,2.5,number);
        	
        	System.out.println("Number of correct assignments in reference structure=  "+ number[0] );
        	System.out.println("Number of wrong assignments in reference structure=  "+ number[1] );    	
        	System.out.println("Number of correct Long-Rang NOE assignments in reference structure=  "+ number[2] );
        	System.out.println("Number of wrong Long-Rang NOE assignments in reference structure=  "+ number[3] );
        	
        	System.out.println("============================================ ");
     	}//if(strIsCheck.equalsIgnoreCase("1"))
     	/////////////////////////////////////////////
     	
     	
    	//	-------------------------------------
    	// (2.4) Output the final NOE assignments in standard xplor format
    	// 
    	//
    	//-------------------------------------
         	
     	if(strOutOrFormat.equalsIgnoreCase("0"))
    	{
	     	//String fileName=strOut+stroutName;//old path    	
	     	String fileName=userDirTemp+stroutName; 
	     	
	    	String xplorNoeStr="";
	    	double[] disUpper=new double[1];
	    	try{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		out.println("!REMARK: Total number of NOESY cross peaks is "+ (vecNoesy.size()+vecNoesy4D.size()));    		
	    		out.println("!REMARK: Total number of NOEs is "+vecRefinedNoes.size());
	    		out.println("!REMARK: the number of unique NOE assignments is "+vecNoeUniq.size());
	    		out.println("!REMARK: the number of multiple NOE assignments is "+(vecRefinedNoes.size()-vecNoeUniq.size()) );
	    		    		
	    		out.println("!REMARK: Number of correct assignments in reference structure=  "+ number[0] );
	        	out.println("!REMARK: Number of wrong assignments in reference structure=  "+ number[1] );    	
	        	out.println("!REMARK: Number of correct Long-Rang NOE assignments in reference structure=  "+ number[2] );
	        	out.println("!REMARK: Number of wrong Long-Rang NOE assignments in reference structure=  "+ number[3] );
	    		out.println("");
	    		out.println("");
	    		
		        for (i=0;i<vecRefinedNoes.size();i++)		       	
		    	{
		    		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
		    		int resNo1=noe.getResidueNoA();
		    		int resNo2=noe.getResidueNoB();
		    		String res1=noe.getResidueA();
		    		String res2=noe.getResidueB();
		    		String atom1=noe.getAtomA();
		    		String atom2=noe.getAtomB();
		    		double distUpper=noe.getUpper();
		    		
		    		//xplor format:	    		
					xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
					if(isDebug)
						System.out.println(xplorNoeStr); 
					out.println(xplorNoeStr);
		    	}//for (i=0;i<vecRefinedNoes.size();i++)
		    	out.close();
		    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
	    	}catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + fileName);
			}catch (IOException e)
			{
			   System.out.println("IOException: the stack trace is:");
			   e.printStackTrace();
			}
    	}
     	else if(strOutOrFormat.equalsIgnoreCase("1")) //we output the NOEs in OR format:
    	{
    		Noe noeTemp=new Noe();    		
    		Vector vecMultiAsg=noeTemp.ConvertSingleToMultiAsg(vecRefinedNoes);
    		String fileName=strOut+stroutName;  
    		noeTemp.OutputMultiNOEAsg(vecMultiAsg,fileName);    		
    	}
     	
       	long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Total time for NOE assignment:  "+ totalTime +" minutes");
    	
    }
    
    /**
     * compare two NOE table and output the summary.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     */       
    public void doCompareTwoNOETables(String src, String strOut, String strInput)
    {	  
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();
    	
    	String strSeq="",strNoeFormat="", strNOETable="",strRefNOETalbe="",strRefNoeFormat="";
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	for (int i=0;i<paraVec.size();i++)
    	{
    		// get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");
    		if(paraMap.containsKey("REFNOEFORMAT"))
    			strRefNoeFormat  =  (String)paraMap.get("REFNOEFORMAT");
    		if(paraMap.containsKey("NOE-TABLE"))
    			strNOETable  =  (String)paraMap.get("NOE-TABLE");   
    		if(paraMap.containsKey("REF-NOE-TABLE"))
    			strRefNOETalbe  =  (String)paraMap.get("REF-NOE-TABLE");
    	}
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------    	
    	String seqFile = src + strSeq;
    	
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	//read the NOE assignment table from cyana
    	String manualAsgFile = src+ strNOETable;
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();
    	
    	// read from cyana (namely upl) format:
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,"PDB-NEW");//////////////////0.0
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	//read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		vecManAsg=noe_temp.noeReader2(manualAsgFile,0.0); 
    		vecManAsgNew.addAll(vecManAsg);    	
    	}
    	
    	Vector vecRefManAsg=new Vector();
    	Vector vecRefManAsgNew=new Vector ();
    	String manualRefAsgFile = src+ strRefNOETalbe;//"
    	// read from cyana (namely upl) format:
    	if (strRefNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecRefManAsg=noe_temp.LongRangeNoeReader(manualRefAsgFile,0.0,"PDB-NEW");//////////////////0.0
    		vecRefManAsgNew.addAll(vecRefManAsg);
    	}
    	//read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		vecRefManAsg=noe_temp.noeReader2(manualRefAsgFile,0.0); 
    		vecRefManAsgNew.addAll(vecManAsg);    	 	
    	}
    	//-------------------------------------
    	// Comparison with manual assignments
    	// 
    	//
    	//-------------------------------------
    	double[] number=new double[4];
    	System.out.println("============================================ ");
    	System.out.println("============Comparisons with reference NOE assignment result================== ");
    	//System.out.println("wrong assingments=========================== ");
    	noe_temp.CompareAsgToManualAsg(vecManAsgNew,vecRefManAsgNew,number);
    	double numCorrect=number[0];
    	double numWrong=number[1];
    	System.out.println("Total number of reference NOEs = "+ vecRefManAsgNew.size());
    	System.out.println("Number of correct assignments=  "+ numCorrect );
    	System.out.println("Number of wrong assignments=  "+ numWrong );
    	System.out.println("Number of correct Long-Rang NOE assignments=  "+ number[2] );
    	System.out.println("Number of wrong Long-Rang NOE assignments=  "+ number[3] );
    	System.out.println("============================================ ");
    	
    }      
    
    /**
     * extract NOEs in certain regions.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     */       
    public void doExtractNOEs(String src, String strOut, String strInput)
    {	  
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();
    	double SSE1_first=0, SSE1_second=0, SSE2_first=0, SSE2_second=0;
    	int nIsoriginalUp=0;
    	String strSeq="",strNoeFormat="", strNOETable="",stroutName="",strIsLongRange="",strIsInMultiAsg="",
    	strIsOutMultiAsg="";
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	for (int i=0;i<paraVec.size();i++)
    	{
    		// get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");
    		if(paraMap.containsKey("OUTNOENAME"))
    			stroutName  =  (String)paraMap.get("OUTNOENAME");
    		if(paraMap.containsKey("NOE-TABLE"))
    			strNOETable  =  (String)paraMap.get("NOE-TABLE");       		   		
    		
    		if(paraMap.containsKey("ISLONGRANGE"))
    			strIsLongRange  =  (String)paraMap.get("ISLONGRANGE"); 
    		if(paraMap.containsKey("SSE1START"))
    			SSE1_first = Double.parseDouble((String)paraMap.get("SSE1START"));
    		if(paraMap.containsKey("SSE1END"))
    			SSE1_second = Double.parseDouble((String)paraMap.get("SSE1END"));
    		if(paraMap.containsKey("ISORIGINALUP"))
    			nIsoriginalUp  =   Integer.parseInt((String)paraMap.get("ISORIGINALUP"));
    		if(paraMap.containsKey("ISINMULTIASSIGNMENT"))
    			strIsInMultiAsg =  (String)paraMap.get("ISINMULTIASSIGNMENT"); 
    		if(paraMap.containsKey("ISOUTMULTIASGFORMAT"))
    			strIsOutMultiAsg =  (String)paraMap.get("ISOUTMULTIASGFORMAT");    		
    	}
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------    
    	String seqFile = src + strSeq;    	
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	//read the NOE assignment table from cyana
    	String manualAsgFile = src+ strNOETable;
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();
    	
    	// read from cyana (namely upl) format:
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,"PDB-NEW");//////////////////0.0
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	//read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		if(strIsInMultiAsg.equalsIgnoreCase("1"))
    		{
    			vecManAsg=noe_temp.noeReaderMultiAsg(manualAsgFile,vecSeq);
    			Vector vecTemp=noe_temp.ConvertMultiToSingleAsg(vecManAsg);
    			vecManAsg=new Vector();
    			vecManAsg.addAll(vecTemp);
    		}
    		else
    			vecManAsg=noe_temp.noeReader(manualAsgFile,0.0,vecSeq);
    		
    		vecManAsgNew.addAll(vecManAsg);
    		//vecManAsgNew= noe_temp.ConvertXplorAsgToUplFormat(vecManAsg,vecSeq);    	
    	}
    	int firstResNo=0,secondResNo=0; 
    	Vector vecNewNoes=new Vector();
    	for(int i=0;i<vecManAsgNew.size();i++)
    	{
			boolean isIn=false;
			Noe noe=new Noe();
    		noe=(Noe)vecManAsgNew.elementAt(i);
    		firstResNo=noe.getResidueNoA();
    		secondResNo=noe.getResidueNoB();   		
    		
    		if ((firstResNo<=SSE1_second && firstResNo>=SSE1_first)   )
    			isIn=true;
    		
    		if ((secondResNo<=SSE1_second && secondResNo>=SSE1_first)   )
    			isIn=true;
    		if(strIsLongRange.equalsIgnoreCase("1"))
    		{
    			if(Math.abs(firstResNo-secondResNo)<=3)
    				isIn=false;
    		}
    		if(strIsLongRange.equalsIgnoreCase("2"))
    		{
    			if(Math.abs(firstResNo-secondResNo)>=4)
    				isIn=false;
    		}
    		if(isIn)
    			vecNewNoes.add(noe);
    		
    	}//for(i=0;i<vecAmbgAsg.size();i++)
    	//	-------------------------------------
    	// (2.4) Output the final NOE assignments in standard xplor format
    	// 
    	//
    	//-------------------------------------
    	
    	String fileName=strOut+stroutName;    	
    	Peak pk=new Peak();
    	if(strIsOutMultiAsg.equalsIgnoreCase("1"))
    	{
    		Noe noeTemp=new Noe();    		
    		Vector vecMultiAsg=noeTemp.ConvertSingleToMultiAsg(vecNewNoes);
    		//String fileName=strOut+stroutName;  
    		noeTemp.OutputMultiNOEAsg(vecMultiAsg,fileName); 
    	}
    	else
    	{
	    	String xplorNoeStr="";
	    	double[] disUpper=new double[1];
	    	try{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		out.println("!REMARK: Total number of NOEs is "+vecNewNoes.size());
	    		out.println("");
	    		out.println("");
	    		
		       	for (int i=0;i<vecNewNoes.size();i++)
		    	{
		    		Noe noe=(Noe)vecNewNoes.elementAt(i);
		    		int resNo1=noe.getResidueNoA();
		    		int resNo2=noe.getResidueNoB();
		    		String res1=noe.getResidueA();
		    		String res2=noe.getResidueB();
		    		String atom1=noe.getAtomA();
		    		String atom2=noe.getAtomB();
		    		double distUpper=noe.getUpper();
		    		if(nIsoriginalUp==0)
		    			distUpper=6.0;
		    		//xplor format:	    		
					xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
					if(Const.isDebug)
						System.out.println(xplorNoeStr); 
					out.println(xplorNoeStr);
		    	}//for (i=0;i<vecRefinedNoes.size();i++)
		    	out.close();
		    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
	    	}catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + fileName);
			}catch (IOException e)
			{
			   System.out.println("IOException: the stack trace is:");
			   e.printStackTrace();
			}
    	}
    	
    	long endTime = System.currentTimeMillis() ;
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
  	   	if(Const.isDebug)
  		   System.out.println("The total time for ExtractNOEs is:  "+ totalTime +" minutes");  	
    
    }

    
    /**
     * refine the long-range NOE assignment without structure information,
     * better version of doLongNOEsAsgWOStructure.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     */       
    public void doLongNOEsAsgWOStructureNew(String src, String strOut, String strInput)
    {	  
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();
    	double SSE1_first=0, SSE1_second=0;
    	int nIsoriginalUp=0;
    	String strSeq="",strNoeFormat="", strNOETable="",stroutName="",strIsFragement="";
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	for (int i=0;i<paraVec.size();i++)
    	{
    		// get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");
    		if(paraMap.containsKey("OUTNOENAME"))
    			stroutName  =  (String)paraMap.get("OUTNOENAME");
    		if(paraMap.containsKey("NOE-TABLE"))
    			strNOETable  =  (String)paraMap.get("NOE-TABLE");       		   		
    		if(paraMap.containsKey("ISFRAGMENT"))
    			strIsFragement  =  (String)paraMap.get("ISFRAGMENT");       
    		if(paraMap.containsKey("SSE1START"))
    			SSE1_first = Double.parseDouble((String)paraMap.get("SSE1START"));
    		if(paraMap.containsKey("SSE1END"))
    			SSE1_second = Double.parseDouble((String)paraMap.get("SSE1END"));
    		   
    		if(paraMap.containsKey("ISORIGINALUP"))
    			nIsoriginalUp  =   Integer.parseInt((String)paraMap.get("ISORIGINALUP"));
    		
    	}
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------
    	//String seqFile = src+"cbf.seq";
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	//read the NOE assignment table from cyana
    	String manualAsgFile = src+ strNOETable;//"noe-H123.txt";//strManualAsg; //"resonance.prot";  
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();
    	
    	// read from cyana (namely upl) format:
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,"PDB-NEW");//////////////////0.0
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	//read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		vecManAsg=noe_temp.noeReader(manualAsgFile,0.0,vecSeq);  	//xplor format   
    		vecManAsgNew.addAll(vecManAsg);   		
    		
    	}
    	int firstResNo=0,secondResNo=0; 
    	Vector vecNewNoes=new Vector();    	
    	//extract long-range NOEs
    	for(int i=0;i<vecManAsgNew.size();i++)
    	{
			boolean isIn=true;
			Noe noe=new Noe();
    		noe=(Noe)vecManAsgNew.elementAt(i);
    		firstResNo=noe.getResidueNoA();
    		secondResNo=noe.getResidueNoB(); 	    	
    		if(Math.abs(firstResNo-secondResNo)<=3)
    				isIn=false;    		
    		if(isIn)
    			vecNewNoes.add(noe);   		
    	}//for(i=0;i<vecAmbgAsg.size();i++)   	
    
    	//extract reference NOEs
    	Vector vecRefNoes=new Vector(); 
    	for(int i=0;i<vecNewNoes.size();i++)
    	{
    		boolean isIn=true;
    		boolean isInFragment=false;
    		boolean isLegal=true;
			Noe noe=new Noe();
    		noe=(Noe)vecNewNoes.elementAt(i);
    		firstResNo=noe.getResidueNoA();
    		secondResNo=noe.getResidueNoB();   
    		String resA=noe.getResidueA();
    		String resB=noe.getResidueB();
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		double upper_dist=noe.getUpper();
    		if (atom1.equalsIgnoreCase("H"))
    			atom1="HN";
    		if(atom2.equalsIgnoreCase("H"))
    			atom2="HN";
    		String atom1_new="";
    		String atom2_new="";
    		if(atom1.length()>=2)
    			atom1_new=atom1.substring(0,2);
    		else 
    			atom1_new=atom1;
    		if(atom2.length()>=2)
    			atom2_new=atom2.substring(0,2);
    		else
    			atom2_new=atom2;
    		
    		if((atom1_new.equalsIgnoreCase("HN") || atom1_new.equalsIgnoreCase("HA")||atom1_new.equalsIgnoreCase("HB") ) &&
    				(atom2_new.equalsIgnoreCase("HN") || atom2_new.equalsIgnoreCase("HA") ||atom2_new.equalsIgnoreCase("HB") ) &&
    				(upper_dist<3.5)  )    			
    			isIn=false;
    		   		
    		if(strIsFragement.equalsIgnoreCase("1"))
    		{
	    		if ((firstResNo<=SSE1_second && firstResNo>=SSE1_first)   )
	    			isInFragment=true;
	    		
	    		if ((secondResNo<=SSE1_second && secondResNo>=SSE1_first)   )
	    			isInFragment=true;    		
    		}
    		else 
    			isInFragment=true;
    		
    		boolean isHydrophobicA=false; 
    		boolean isHydrophobicB=false; 
    		Peak pk=new Peak();
    		if( !(atom1_new.equalsIgnoreCase("HN") || atom1_new.equalsIgnoreCase("HA")||atom1_new.equalsIgnoreCase("HB") ) &&
    				( pk.isHydrophobic(resA) )  )    			
    			isHydrophobicA=true;
    		if( !(atom2_new.equalsIgnoreCase("HN") || atom2_new.equalsIgnoreCase("HA")||atom2_new.equalsIgnoreCase("HB") ) &&
    				( pk.isHydrophobic(resB) )  )    			
    			isHydrophobicB=true;
    		
    		boolean isChargedA=false;
    		boolean isChargedB=false;
    		if( !(atom1_new.equalsIgnoreCase("HN") || atom1_new.equalsIgnoreCase("HA")||atom1_new.equalsIgnoreCase("HB") ) &&
    				( pk.isCharged(resA) )  )    			
    			isChargedA=true;
    		if( !(atom2_new.equalsIgnoreCase("HN") || atom2_new.equalsIgnoreCase("HA")||atom2_new.equalsIgnoreCase("HB") ) &&
    				( pk.isCharged(resB) )  )    			
    			isChargedB=true;
    		
    		if(isHydrophobicA && isChargedB)
    			isLegal=false;
    		if(isChargedA && isHydrophobicB)
    			isLegal=false;
    		
    		if(isIn && isInFragment && isLegal)
    			vecRefNoes.add(noe); 
    	}//for(int i=0;i<vecNewNoes.size();i++)
    	
    	
    	int ResNo1Ref=0,ResNo2Ref=0;
    	//calcualte the long range noe score:  
    	//the long range nOe is pruned if the number of intereactions between different protons of residue1 and different
    	//protons of residues is <=2; see Y. Huang, R. Tejero, R. Powers, and G. Montelione, proteins, 06/05.
    	Vector vecFinalNoes=new Vector();
    	for(int i=0;i<vecRefNoes.size();i++)
    	{
    		boolean isSym=false;
    		int counter=0;
			Noe noe=new Noe();
    		noe=(Noe)vecRefNoes.elementAt(i);
    		firstResNo=noe.getResidueNoA();
    		secondResNo=noe.getResidueNoB(); 
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		for(int j=0;j<vecRefNoes.size();j++)
    		{
    			if (i==j)
    				continue;
    			Noe noeRef=new Noe();
    			noeRef=(Noe)vecRefNoes.elementAt(j);
    			ResNo1Ref=noeRef.getResidueNoA();
        		ResNo2Ref=noeRef.getResidueNoB(); 
        		
        		if(ResNo1Ref==secondResNo && ResNo2Ref==firstResNo)
        			counter++;
        		else if(firstResNo==ResNo1Ref && secondResNo==ResNo2Ref)
        			counter++;    			
    		}//for(int j=0;j<vecRefNoes.size();j++)
    		
    		if (counter>=3)     		
    			vecFinalNoes.add(noe); 		
    			
    	}//for(int i=0;i<vecNewNoes.size();i++)
    	
    
    	//	-------------------------------------
    	// (2.4) Output the final NOE assignments in standard xplor format
    	// 
    	//
    	//-------------------------------------
    	String fileName=strOut+stroutName;    	
    	Peak pk=new Peak();
    	String xplorNoeStr="";
    	double[] disUpper=new double[1];
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    		out.println("!REMARK: Total number of NOEs is "+vecFinalNoes.size());
    		out.println("");
    		out.println("");
    		
	       	for (int i=0;i<vecFinalNoes.size();i++)
	    	{
	    		Noe noe=(Noe)vecFinalNoes.elementAt(i);
	    		int resNo1=noe.getResidueNoA();
	    		int resNo2=noe.getResidueNoB();
	    		String res1=noe.getResidueA();
	    		String res2=noe.getResidueB();
	    		String atom1=noe.getAtomA();
	    		String atom2=noe.getAtomB();
	    		double distUpper=noe.getUpper();
	    		if(nIsoriginalUp==0)
	    			distUpper=6.0;
	    		//xplor format:	    		
				xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
				if(Const.isDebug)
					System.out.println(xplorNoeStr); 
				out.println(xplorNoeStr);
	    	}//for (i=0;i<vecRefinedNoes.size();i++)
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		
    
    }
     
    
    /**
     * refine the long-range NOE assignment without structure information.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     */       
    public void doLongNOEsAsgWOStructure(String src, String strOut, String strInput)
    {	  
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();
    	double SSE1_first=0, SSE1_second=0;
    	int nIsoriginalUp=0;
    	String strSeq="",strNoeFormat="", strNOETable="",stroutName="",strIsFragement="",strSurfResidues="",strNotLoopResidues="";
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	for (int i=0;i<paraVec.size();i++)
    	{
    		// get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");
    		if(paraMap.containsKey("OUTNOENAME"))
    			stroutName  =  (String)paraMap.get("OUTNOENAME");
    		if(paraMap.containsKey("NOE-TABLE"))
    			strNOETable  =  (String)paraMap.get("NOE-TABLE");       		   		
    		if(paraMap.containsKey("ISFRAGMENT"))
    			strIsFragement  =  (String)paraMap.get("ISFRAGMENT");       
    		if(paraMap.containsKey("SSE1START"))
    			SSE1_first = Double.parseDouble((String)paraMap.get("SSE1START"));
    		if(paraMap.containsKey("SSE1END"))
    			SSE1_second = Double.parseDouble((String)paraMap.get("SSE1END"));
    		if(paraMap.containsKey("SURFRESIDUES"))
    			strSurfResidues  =  (String)paraMap.get("SURFRESIDUES");     
    		if(paraMap.containsKey("ISORIGINALUP"))
    			nIsoriginalUp  =   Integer.parseInt((String)paraMap.get("ISORIGINALUP"));
    		if(paraMap.containsKey("HYDROPHOBICLOOPRESIDUES"))
    			strNotLoopResidues  =  (String)paraMap.get("HYDROPHOBICLOOPRESIDUES");   
    	
    	}
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------
    	//String seqFile = src+"cbf.seq";
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	//read the NOE assignment table from cyana
    	String manualAsgFile = src+ strNOETable;//"noe-H123.txt";//strManualAsg; //"resonance.prot";  
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();
    	
    	// read from cyana (namely upl) format:
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,"PDB-NEW");//////////////////0.0
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	//read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		vecManAsg=noe_temp.noeReader(manualAsgFile,0.0,vecSeq);  	//xplor format   
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	int firstResNo=0,secondResNo=0; 
    	Vector vecNewNoes=new Vector();    	
    	//extract long-range NOEs
    	for(int i=0;i<vecManAsgNew.size();i++)
    	{
			boolean isIn=true;
			Noe noe=new Noe();
    		noe=(Noe)vecManAsgNew.elementAt(i);
    		firstResNo=noe.getResidueNoA();
    		secondResNo=noe.getResidueNoB(); 	    	
    		if(Math.abs(firstResNo-secondResNo)<=3)
    				isIn=false;    		
    		if(isIn)
    			vecNewNoes.add(noe);   		
    	}//for(i=0;i<vecAmbgAsg.size();i++)
    	
    	// read the surface residues
    	Vector vecSurfResidues=asg.getOrder(strSurfResidues);   	
    	
    	//extract reference NOEs
    	Vector vecRefNoes=new Vector(); 
    	for(int i=0;i<vecNewNoes.size();i++)
    	{
    		boolean isIn=true;
    		boolean isInFragment=false;
    		boolean isInSurface=false;
			Noe noe=new Noe();
    		noe=(Noe)vecNewNoes.elementAt(i);
    		firstResNo=noe.getResidueNoA();
    		secondResNo=noe.getResidueNoB();     		
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		double upper_dist=noe.getUpper();
    		//if(upper_dist>5.5)
    			//isIn=false;
    		if( ( atom1.equalsIgnoreCase("HN") || atom1.equalsIgnoreCase("H") || atom1.equalsIgnoreCase("HA")   ) &&
    				( atom2.equalsIgnoreCase("HN") || atom2.equalsIgnoreCase("H") || atom2.equalsIgnoreCase("HA") ) )
    				isIn=false;
    		
    		if( ( atom1.equalsIgnoreCase("HN") || atom1.equalsIgnoreCase("H") || atom1.equalsIgnoreCase("HA")|| atom1.substring(0,2).equalsIgnoreCase("HB")   ) &&
    				( atom2.equalsIgnoreCase("HN") || atom2.equalsIgnoreCase("H") || atom2.equalsIgnoreCase("HA") ||atom2.substring(0,2).equalsIgnoreCase("HB")) )
    			if (upper_dist<3.5)
    				isIn=false;
    		
    		
    		if(strIsFragement.equalsIgnoreCase("1"))
    		{
	    		if ((firstResNo<=SSE1_second && firstResNo>=SSE1_first)   )
	    			isInFragment=true;
	    		
	    		if ((secondResNo<=SSE1_second && secondResNo>=SSE1_first)   )
	    			isInFragment=true;
    		}
    		else 
    			isInFragment=true;
    		
    		if ((firstResNo<=SSE1_second && firstResNo>=SSE1_first) )
    		{
    			if ((secondResNo<=SSE1_second && secondResNo>=SSE1_first)   )
	    			isInSurface=true;
    			
    			for (int t=0;t<vecSurfResidues.size();t++)
    			{
    				String strSurResNo=(String) vecSurfResidues.elementAt(t);
    				int ResNo= Integer.parseInt(strSurResNo);
    				if( ResNo==secondResNo)
    				{
    					isInSurface=true;
    					break;
    				}
    			}    			
    		}//if ((firstResNo<=SSE1_second && firstResNo>=SSE1_first) )
    		if ((secondResNo<=SSE1_second && secondResNo>=SSE1_first))
    		{
    			if ((firstResNo<=SSE1_second && firstResNo>=SSE1_first)   )
    				isInSurface=true;
    			for (int t=0;t<vecSurfResidues.size();t++)
    			{
    				String strSurResNo=(String) vecSurfResidues.elementAt(t);
    				int ResNo= Integer.parseInt(strSurResNo);
    				if( ResNo==firstResNo)
    				{
    					isInSurface=true;
    					break;
    				}
    			}    			
    		}//if ((secondResNo<=SSE1_second && secondResNo>=SSE1_first))
    		
    		if(isIn && isInFragment && isInSurface)
    			vecRefNoes.add(noe); 
    	}//for(int i=0;i<vecNewNoes.size();i++)
    	
    	Vector vecFinalNoes=new Vector();
    	int ResNo1Ref=0,ResNo2Ref=0;
    	//calcualte the long range noe score:    	
    	for(int i=0;i<vecRefNoes.size();i++)
    	{
    		boolean isSym=false;
    		double sc=0.0;
			Noe noe=new Noe();
    		noe=(Noe)vecRefNoes.elementAt(i);
    		firstResNo=noe.getResidueNoA();
    		secondResNo=noe.getResidueNoB(); 
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		for(int j=0;j<vecRefNoes.size();j++)
    		{
    			Noe noeRef=new Noe();
    			noeRef=(Noe)vecRefNoes.elementAt(j);
    			ResNo1Ref=noeRef.getResidueNoA();
        		ResNo2Ref=noeRef.getResidueNoB(); 
        		
        		if(ResNo1Ref==secondResNo && ResNo2Ref==firstResNo)
        		{
        			isSym=true;
        			sc=sc+1.0;
        		}
        		else if(firstResNo==ResNo1Ref && secondResNo==ResNo2Ref)
    				sc=sc+1.0;
    			
    		
    		}//for(int j=0;j<vecRefNoes.size();j++)
    	
    		if (sc>=1.0 && isSym)     		
    			vecFinalNoes.add(noe);
    		if(Const.isDebug)
    			System.out.println("the score in "+ i +"th NOE is: "+ sc);
    		
    			
    	}//for(int i=0;i<vecNewNoes.size();i++)
    	
    	//use hydrophobic loop residues for pruning ambiguous NOEs: 
    	Vector vecNotLoopResidues=asg.getOrder(strNotLoopResidues);
    	
    	Vector vecFinalNew=new Vector();
    	for (int i=0;i<vecFinalNoes.size();i++)
    	{
    		Noe noe=(Noe)vecFinalNoes.elementAt(i);
    		int resNo1=noe.getResidueNoA();
    		int resNo2=noe.getResidueNoB();
    		String res1=noe.getResidueA();
    		String res2=noe.getResidueB();
    		String atom1=noe.getAtomA();
    		String atom2=noe.getAtomB();
    		boolean isInLoop=true;
    		for (int t=0;t<vecNotLoopResidues.size();t++)
			{
				String strSurResNo=(String) vecNotLoopResidues.elementAt(t);
				int ResNo= Integer.parseInt(strSurResNo);
				if( ResNo==resNo1)
				{
					if(atom1.equalsIgnoreCase("HN")|| atom1.equalsIgnoreCase("HA") ||atom1.equalsIgnoreCase("HB1")||atom1.equalsIgnoreCase("HB2")||atom1.equalsIgnoreCase("HB3"))
						continue;
					else
					{
						isInLoop=false;
						break;
					}					
				}//if( ResNo==resNo1)
				
				if( ResNo==resNo2)
				{
					if(atom2.equalsIgnoreCase("HN")|| atom2.equalsIgnoreCase("HA") ||atom2.equalsIgnoreCase("HB1")||atom2.equalsIgnoreCase("HB2")||atom2.equalsIgnoreCase("HB3"))
						continue;
					else
					{
						isInLoop=false;
						break;
					}					
				}//if( ResNo==resNo2)
				
			}    //for (int t=0;t<vecNotLoopResidues.size();t++)
    		
    		if(isInLoop)
    			vecFinalNew.add(noe);
    		
    	}//for (int i=0;i<vecFinalNoes.size();i++)
    	
    	
    	
    	
    	//	-------------------------------------
    	// (2.4) Output the final NOE assignments in standard xplor format
    	// 
    	//
    	//-------------------------------------
    	String fileName=strOut+stroutName;    	
    	Peak pk=new Peak();
    	String xplorNoeStr="";
    	double[] disUpper=new double[1];
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    		out.println("!REMARK: Total number of NOEs is "+vecFinalNew.size());
    		out.println("");
    		out.println("");
    		
	       	for (int i=0;i<vecFinalNew.size();i++)
	    	{
	    		Noe noe=(Noe)vecFinalNew.elementAt(i);
	    		int resNo1=noe.getResidueNoA();
	    		int resNo2=noe.getResidueNoB();
	    		String res1=noe.getResidueA();
	    		String res2=noe.getResidueB();
	    		String atom1=noe.getAtomA();
	    		String atom2=noe.getAtomB();
	    		double distUpper=noe.getUpper();
	    		if(nIsoriginalUp==0)
	    			distUpper=6.0;
	    		//xplor format:	    		
				xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
				if(Const.isDebug)
					System.out.println(xplorNoeStr); 
				out.println(xplorNoeStr);
	    	}//for (i=0;i<vecRefinedNoes.size();i++)
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		
    
    }
    
    /**
     * note: randomStructureSlect..need to be fixed,
     * refine the NOE assignment using given the backbone
     * and output the refine NOE table.
     * 
     * @param src location of the input file
     * @param strOut location of the output file
     * @param strInput  input file name
     * 
     * @return void
     */       
    public void doRefineNOEsWBB(String src, String strOut, String strInput)
    {	    	 
    	boolean isDebug=true;
    	Hsqc hqc = new Hsqc();
    	Peak pk = new Peak();
    	
    	ModelRdc mdc = new ModelRdc();    	
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();
    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////
    	int i, j;    	
    	Pdb  pp = new Pdb();    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double haErr  = 0.0;
    	double h1Err  = 0.0;
    	double c13Err = 0.0;
    	double hnErr  = 0.0;
    	double nErr   = 0.0;
    	double noe_upper =6.0;
    	
    	String strReson="", strSeq="", strNoesy2D="", strHnNoesy3D="", 
    		strHaNoesy3D="", strResFormat="",strBB="",stroutName="",
    	strNoeFormat="",strNOETable="";
    	
    	for (i=0;i<paraVec.size();i++)
    	{
    		//get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("HAERR"))
    			 haErr  =  Double.parseDouble((String)paraMap.get("HAERR"));
    		if(paraMap.containsKey("H1ERR"))
   			 	h1Err  =  Double.parseDouble((String)paraMap.get("H1ERR"));
    		if(paraMap.containsKey("C13ERR"))
    			c13Err  =  Double.parseDouble((String)paraMap.get("C13ERR"));
    		if(paraMap.containsKey("HNERR"))
    			hnErr  =  Double.parseDouble((String)paraMap.get("HNERR"));
    		if(paraMap.containsKey("NERR"))
    			nErr  =  Double.parseDouble((String)paraMap.get("NERR"));
    		//get the input file name   
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("RESFORMAT"))
    			strResFormat  =  (String)paraMap.get("RESFORMAT");
    		if(paraMap.containsKey("RESONANCE"))
    			strReson  =  (String)paraMap.get("RESONANCE");
    		strReson=strReson.toLowerCase();
    		if(paraMap.containsKey("2D-NOESY"))
    			strNoesy2D  =  (String)paraMap.get("2D-NOESY");
    		//strNoesy2D=strNoesy2D.toLowerCase();
    		if(paraMap.containsKey("3D-N15-NOESY"))
    			strHnNoesy3D  =  (String)paraMap.get("3D-N15-NOESY");
    		//strHnNoesy3D=strHnNoesy3D.toLowerCase();
    		if(paraMap.containsKey("3D-C13-NOESY"))
    			strHaNoesy3D  =  (String)paraMap.get("3D-C13-NOESY");
    		//strHaNoesy3D=strHaNoesy3D.toLowerCase();    		
    	
    		if(paraMap.containsKey("OUTNOENAME"))
    			stroutName  =  (String)paraMap.get("OUTNOENAME");   	
    		if(paraMap.containsKey("BACKBONE"))
    			strBB  =  (String)paraMap.get("BACKBONE");
    		if(paraMap.containsKey("NOEFORMAT"))
    			strNoeFormat  =  (String)paraMap.get("NOEFORMAT");
    		if(paraMap.containsKey("NOE-TABLE"))
    			strNOETable  =  (String)paraMap.get("NOE-TABLE");    		
    		
    		if(paraMap.containsKey("NOE-UPPER"))
    			noe_upper  =  Double.parseDouble((String)paraMap.get("NOE-UPPER"));
    	}       	   	
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------
    	//String seqFile = src+"cbf.seq";
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	//-------------------------------------
    	// (1.2) Read the resonance list 
    	// 
    	// in the "resonance.prot" format
    	//-------------------------------------
    	   	
    	//**this is for ubq assignment table assigned by Lincong**//
    	 /*	String assignFileUbq = src+"assignByMe.txt";
    	Vector assignVecUbq  = asg.asgReader3(assignFileUbq); */
    	
    	//following is for the general case
    	H1CS h1CS=new H1CS();
    	//String assignFile = src+ "resonance.prot";    
    	String assignFile = src+ strReson; //"resonance.prot"; 
    	//this is the vector that stores resonance assignments;
    	//each one includes resNo, resName, atomName, csH1(chemical shift value)
    	Vector assignVec=new Vector();
    	if(strResFormat.equalsIgnoreCase("CYANA"))
    		assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	else if(strResFormat.equalsIgnoreCase("BMRB"))
    		assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	else 
    	{
    		System.out.println("unknown format of the resonance file...");
    		System.exit(0);
    	}
    	
    	//all sorted proton resonances
    	Vector allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator());
    	    	
    	//-------------------------------------
    	// (1.3) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	
    	String strNoeFile = ""; 
    	Vector hnNoeVec=new Vector();//for storing 3d n15 NOESY peaks
    	Vector cnoeVec=new Vector();//for storing 3d c13 noesy peaks
    	if (!strNoesy2D.equalsIgnoreCase("NULL"))
    	{
    		//to do: extension to 2d case...
    	}
    	if (!strHnNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHnNoesy3D;
    		hnNoeVec = noesy.NoesyReaderNMRView(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	if (!strHaNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHaNoesy3D;
    		cnoeVec=noesy.NoesyReaderNMRView(strNoeFile);
    	}
    	
    	Vector vecNoesy=new Vector();
    	vecNoesy.addAll(hnNoeVec);
    	vecNoesy.addAll(cnoeVec);
    	
    	//intensity calibration
    	//using the xplor format...
    	Vector vecNewNoesy=noesy.SetCalibration(vecNoesy);   	
    	
    	// -------------------------------------
    	// (1.2) Read the backbone structure 
    	// (input file name: backbone.pdb)
    	// Note: the backbone must be connected. 
    	//-------------------------------------
    	//String pdbFile = src+"backbone.pdb"; 
    	String pdbFile = src+strBB;
    	Vector pdbVecBB = pp.readPdb(pdbFile);

    	//update backbone atom names according to protein sequence input
    	//residue name in previous backbone may be all in "ALA".     		
    	Vector pdbVecNew = pp.residueNameUpdate(vecSeq, pdbVecBB);     	
    	
    	
    	/////////////////////////////////////////////////
    	// 2. NOE assignment Part
    	//
    	/////////////////////////////////////////////////
    	
    	// -------------------------------------
    	// 
    	// read the NOE assignment table from cyana
    	String manualAsgFile = src+ strNOETable;
    	Noe noe_temp=new Noe();
    	
    	Vector vecManAsg=new Vector();
    	Vector vecManAsgNew=new Vector ();
    	// read from cyana (namely upl) format:
    	if (strNoeFormat.equalsIgnoreCase("CYANA"))
    	{
    		vecManAsg=noe_temp.LongRangeNoeReader(manualAsgFile,0.0,"PDB-NEW");//////////////////0.0
    		vecManAsgNew.addAll(vecManAsg);
    	}
    	//read noes from xplor format: (note: does not include residue names...)
    	else
    	{
    		vecManAsg=noe_temp.noeReader2(manualAsgFile,0.0);
    		vecManAsgNew.addAll(vecManAsg);
    		//vecManAsgNew= noe_temp.ConvertXplorAsgToUplFormat(vecManAsg,vecSeq);    	
    	}
    	  	
    	
    	//-------------------------------------
    	// Read the rotamer library path 
    	// 
    	// use the small Xtal' library from Richardson's lab
    	//-------------------------------------
    	String rotSrc=  src+ "rotasamp-small/";    	
    	
    	//-------------------------------------
    	// (2.2) Rotamer selection and combination of low resolution structure 
    	// select optimal rotamer at each residue 
    	// and place optimal rotamers into backbone
    	//-------------------------------------
    	//this structure is in BMRB-NEW format:    	    	
    	Vector vecRotStructure=pp.RotamSelectAndStructure(2*h1Err, 2*nErr, 2*c13Err,pdbVecNew,assignVec,rotSrc,vecNewNoesy,
    			Const.noeUpBound4PruneAmBNOEs, 1,0.0);
    	
    	
       	//-------------------------------------
    	// (2.3) Filtration of ambiguous NOE assignments
    	// select optimal rotamer at each residue 
    	// and place optimal rotamers into backbone
    	//-------------------------------------
    	Vector vecRefinedNoes=new Vector();
    	
    	//note: we have changed something in this function specifially for hsri...
    	//vecRefinedNoes=asg.PrunAmbgNoeAsg(vecRotStructure,vecAmbgAsg,noeBound,0.5);
    	vecRefinedNoes=asg.PrunAmbgNoeAsg(vecRotStructure,vecManAsgNew,noe_upper, Const.methylCorrection,0);
    	
     	Collections.sort(vecRefinedNoes, new Noe.NoeComparator()); 
    	
    	//	-------------------------------------
    	// (2.4) Output the final NOE assignments in standard xplor format
    	// 
    	//
    	//-------------------------------------
    	String fileName=strOut+stroutName;    	
    	
    	String xplorNoeStr="";
    	double[] disUpper=new double[1];
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
    		out.println("!REMARK: The number of original NOEs is "+vecManAsgNew.size());
    		out.println("!REMARK: Total number of NOEs is "+vecRefinedNoes.size());
    		out.println("");
    		out.println("");
    		
	       	for (i=0;i<vecRefinedNoes.size();i++)
	    	{
	    		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
	    		int resNo1=noe.getResidueNoA();
	    		int resNo2=noe.getResidueNoB();
	    		String res1=noe.getResidueA();
	    		String res2=noe.getResidueB();
	    		String atom1=noe.getAtomA();
	    		String atom2=noe.getAtomB();
	    		double distUpper=noe.getUpper();
	    		
	    		//xplor format:	    		
				xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
				if(isDebug)
					System.out.println(xplorNoeStr); 
				out.println(xplorNoeStr);
	    	}//for (i=0;i<vecRefinedNoes.size();i++)
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + fileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}
		
       	long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Total time for NOE assignment:  "+ totalTime +" minutes");
    	
    }
       

 
}
