package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	ModelRdc.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////


/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

     
import java.io. *;
import java.util. *;
import java.text.NumberFormat;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.FieldPosition;

import Jampack.JampackException;

// TODO: Auto-generated Javadoc
/** * 
 *  
*  A program for computing all the backbond dihedral ($\phi$, $\psi$)
 * angles of an $\alpha$-helix or a $\beta$-strand based on exact
 * solution and a systematic search. The computed dihedral angles will
 * (a) satisfy the experimental CH and NH RDCs as well as possible and
 * (b) simultaneously have the ($\phi$, $\psi$) angles as close as
 * possible to their average values in the PDB.  However, the minimum
 * is defined with respect to the all the backbone angles and all the
 * RDCs for a secondary element, not just one residue. Please see the
 * paper for the detail. In other words, the target function consists
 * of terms representing, respectively, the deviation from the average
 * helix and the deviation from the measured RDCs.  Also included are
 * programs for building models from a sequence of ($\phi, \psi$)
 * angles and eight backbone angles with values beings set to be their
 * averages extracted from 23 ultra-high resolution PDBS with protons
 * built in.
 * 
 * Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
 * */
public class ModelRdc implements Cloneable{
    
    /** The residue no. */
    private int residueNo;  //Though Phi and Psi are between two residues 
    
    /** The residue name. */
    private String residue; //We count it as belonging to the first of the two
    
    /** The phi angle. */
    private double phi;
    
    /** The psi angle. */
    private double psi;

    /**
     * Instantiates a new model rdc.
     */
    public ModelRdc(){
	residueNo = 0;
	residue = null;
        phi = 0.0;
        psi = 0.0;
    }	
    
    /**
     * Instantiates a new model rdc.
     * 
     * @param No the no
     */
    public ModelRdc(int No){
	residueNo = No;
	residue = ""; 
	phi = 0.0;
        psi = 0.0;
    }
    
    /**
     * Instantiates a new model rdc.
     * 
     * @param no the no
     * @param re the re
     * @param x the x
     * @param y the y
     */
    public ModelRdc(int no, String re, double x, double y){
	residueNo = no;
	residue = re;
        phi = x;
        psi = y;
    }
    //getting the values	
    /**
     * Gets the residue no.
     * 
     * @return the residue no
     */
    public int getResidueNo(){
	return residueNo;
    }	
    
    /**
     * Gets the residue.
     * 
     * @return the residue
     */
    public String getResidue(){
	return residue;
    }	
    
    /**
     * Gets the phi.
     * 
     * @return the phi
     */
    public double getPhi(){
	return phi;
    }
    
    /**
     * Gets the psi.
     * 
     * @return the psi
     */
    public double getPsi(){
	return psi;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString(){
  	String desc = String.valueOf(residueNo) + "  " + residue
  	    +"  " + String.valueOf(phi*180/Math.PI) + "  " 
	    + String.valueOf(psi*180.0/Math.PI);
	return desc;	
    }

    /**
     * The Class mdComparator.
     */
    public static class mdComparator implements Comparator{
	
	/* (non-Javadoc)
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Object o1, Object o2){
	    ModelRdc n1 = (ModelRdc)o1;
	    ModelRdc n2 = (ModelRdc)o2;
	    int d1 = n1.getResidueNo();
	    int d2 = n2.getResidueNo();
	    if (d1 < d2)
		return -1;
	    else if (d1 > d2)
		return 1;
	    else return 0;
	}
    }

    /**
     * compute the length of a  vector v1.
     * 
     * @param v1 the v1
     * 
     * @return the double
     */
    public double length(double[] v1){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	return v1Len;
    }
    
    /**
     * calculate the angles between two vectors v1 and v2
     * The returned angle is in the [0,Pi] range.
     * 
     * @param v1 the v1
     * @param v2 the v2
     * 
     * @return the double
     */
    public double interAngle(double[] v1, double[] v2){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	double v2Len = Math.sqrt(v2[0]*v2[0] + v2[1]*v2[1] + v2[2]*v2[2]);
	double c = (v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2]) / (v1Len * v2Len);
	return Math.acos(c);  //acos [0, PI] so should be OK
    }

    /**
     * cal the directioal cosines of the vector vec.
     * 
     * @param vec the vec
     * 
     * @return the double[]
     */
    public double[] dirCos(double [] vec) {
	double len = 0.0;
	double [] dirs = new double[vec.length];
	for (int i=0; i<vec.length; i++)
	    len += vec[i] * vec[i];
	for (int j=0; j<vec.length; j++)
	    dirs[j] = vec[j] / Math.sqrt(len);
	return dirs;
    }

    /**
     * Compute the internuclear vector between two atoms.
     * 
     * @param n1 the coordinate for atom 1
     * @param n2 the coordinate for atom 2
     * 
     * @return a vector from n1->n2
     */
    public double[] internuclearVec(double[] n1, double[] n2){
        return new double[]{n2[0]-n1[0], n2[1]-n1[1], n2[2]-n1[2]};
    }

    /**
     * adding two vectors.
     * 
     * @param n1 the coordinate (direction cosine) for vector n1
     * @param n2 the coordinate (direction cosine) for vector n2
     * 
     * @return a vector from n1+n2
     */
    public double[] addCoords(double[] n1, double[] n2){
        return new double[]{n2[0]+n1[0], n2[1]+n1[1], n2[2]+n1[2]};
    }

  /**
   * A method for computing coordinates of backbone atoms.
   * 
   * @param phi     The backbone \phi angle
   * @param n1      the direction cosine for amide nitrogen of the original residue
   * @param nh1     the direction cosine for amide proton
   * @param ca1     the direction cosine for Ca
   * 
   * @return the coordinates for HA and CO in this order
   * Please note that the coordinates of N, NH and CA of residue (i+1) are returned by
   * overwriting the original array for them: This may be BUG prone.
   */    
    public double[][] haCoCoords(double phi, double[] n1, double[] nh1, double[] ca1){
	Matrix rg = Matrix.identity(3,3);
	Matrix rgInv = new Matrix(3,3);
  	Matrix r2yInv = rg.rotationMat(phi, "-y"); //transpose ==inverse ==rotate in opposite direction
	PhiPsi ff = new PhiPsi();
	double [] nToNHVec =  internuclearVec(n1,  nh1);
	double [] nToCAVec = internuclearVec(n1, ca1);
	rg = ff.RgCal(nToNHVec, nToCAVec);
	rgInv = rg.transpose();
 	Matrix matT = rgInv.times(Const.r1x9yInv.times(r2yInv));

	//Compute the HA coordinate of residue (i)
	double [] coordHA = new double[]{0.0,  0.0, Const.dCA2HA};
	double [] caToHAVec = matT.times(Const.rHA2HA1Inv.times(coordHA));
	coordHA =  addCoords(caToHAVec,  ca1);

	//Compute the CO coordinate residue (i)
	matT = matT.times(Const.r3xInv);
	double [] coordCO   = new double[]{0.0, 0.0, Const.dCA2CO};
	double [] caToCOVec = matT.times(coordCO);
	coordCO   = addCoords(caToCOVec,  ca1); 
	double[][] haco = new double[2][3];
	haco[0] = coordHA;
	haco[1] = coordCO;
	return haco;
    }
 
 /**
  * A method for computing coordinates of backbone atoms.
  * 
  * @param phi     The backbone \phi angle
  * @param n1      the direction cosine for amide nitrogen of the original residue
  * @param nh1     the direction cosine for amide proton
  * @param ca1     the direction cosine for Ca
  * @param first   whether the residue is the first residue of the fragment, if it is the first the
  * rotational matrix should be identity
  * 
  * @return the coordinates for HA and CO in this order
  * Please note that the coordinates of N, NH and CA of residue (i+1) are returned by
  * overwriting the original array for them: This may be BUG prone.
  */    
    public double[][] haCoCoords(double phi, double[] n1, double[] nh1, double[] ca1, boolean first){
	Matrix rg = Matrix.identity(3,3);
	Matrix rgInv = new Matrix(3,3);
  	Matrix r2yInv = rg.rotationMat(phi, "-y"); //transpose ==inverse ==rotate in opposite direction
	if (!first){ //compute the rotational matrix to the peptide plane i
	    PhiPsi ff = new PhiPsi();
	    double [] nToNHVec =  internuclearVec(n1,  nh1);
	    double [] nToCAVec = internuclearVec(n1, ca1);
	    rg = ff.RgCal(nToNHVec, nToCAVec);
	}
	rgInv = rg.transpose();
 	Matrix matT = rgInv.times(Const.r1x9yInv.times(r2yInv));

	//Compute the HA coordinate of residue (i)
	double [] coordHA = new double[]{0.0,  0.0, Const.dCA2HA};
	double [] caToHAVec = matT.times(Const.rHA2HA1Inv.times(coordHA));
	coordHA =  addCoords(caToHAVec,  ca1);

	//Compute the CO coordinate residue (i)
	matT = matT.times(Const.r3xInv);
	double [] coordCO   = new double[]{0.0, 0.0, Const.dCA2CO};
	double [] caToCOVec = matT.times(coordCO);
	coordCO   = addCoords(caToCOVec,  ca1); 
	double[][] haco = new double[2][3];
	haco[0] = coordHA;
	haco[1] = coordCO;
	return haco;
    }
    
    
    /* Print all possible orientations of structures whose rdc rmsd is within threshold*/
    /**
     * Prints the all by grid search.
     * 
     * @param vecHelixBB the vec helix bb
     * @param rdc1Vec the rdc1 vec
     * @param rdc2Vec the rdc2 vec
     * @param Syy the syy
     * @param Szz the szz
     * @param resolution the resolution
     * @param rdcRmsdThreshold the rdc rmsd threshold
     */
    public void PrintAllByGridSearch(Vector vecHelixBB, final Vector<Dipolar> rdc1Vec, final Vector<Dipolar> rdc2Vec, double Syy, double Szz,double resolution,double rdcRmsdThreshold )
	{    	
    	PdbRdc pdr = new PdbRdc();
    	Pdb pp=new Pdb();
    	
		double [] rmsds = new double[2];
		boolean debugEulerFit = true;
		Matrix[] mm = new Matrix[4];
		double rdc1Rms = 0.0, rdc2Rms = 0.0; 
		
		rdc1Rms =rmsds[0] ;
		rdc2Rms =rmsds[1];
		
		Vector helixVecN=new Vector();
		Matrix[] mm_temp = new Matrix[4];		
		
		int i=3;
		{
			
			helixVecN = pp.newPdb(vecHelixBB, Const.mat4ThreeDirs[i - 1]);
			mm_temp[0] = pdr.eulerFitPrintEnsemble(helixVecN, rdc1Vec, rdc2Vec, Syy, Szz, rmsds, resolution, debugEulerFit,rdcRmsdThreshold);
			rdc1Rms = rmsds[0]; 
			rdc2Rms = rmsds[1]; 
			
		}//for (int i = 1; i < 4; i++)
		Vector vecPdbOptimal=pp.newPdb(vecHelixBB,mm_temp[0] );
	
	}
    

    /**
     * A method for computing coordinates of backbone atoms.
     * 
     * @param phi     The backbone \phi angle
     * @param psi     The backbone \psi angle
     * @param n the n
     * @param nh the nh
     * @param ca the ca
     * 
     * @return Pdb object consisting of atoms: NH, CA, CO, HA and O of residue (i).
     * Please note that the coordinates of N, NH and CA of residue (i+1) are returned by
     * overwriting the original array for them: This may be BUG prone.
     */    
    public double[][] nNhCaCal(double phi, double psi, double[] n, double[] nh, double[] ca){
	Matrix rg = Matrix.identity(3,3);
  	Matrix r2yInv = rg.rotationMat(phi, "-y"); //transpose == inverse == rotate in opposite direction
	Matrix r4zInv = rg.rotationMat(psi+Math.PI, "-z"); 
	//intraResidue N(i)->NH(i) of the 1st residue
	double [] nToNHVec =  internuclearVec(n,  nh);
	double [] nToCAVec = internuclearVec(n, ca);
	PhiPsi ff = new PhiPsi();
	rg = ff.RgCal(nToNHVec, nToCAVec);
	Matrix rgInv = rg.transpose();
 	Matrix mat = rgInv.times(Const.r1x9yInv.times(r2yInv.times(Const.r3xInv)));

	//Compute the CO coordinate residue (i)
	double [] coordCO   = new double[]{0.0, 0.0, Const.dCA2CO};
	double [] caToCOVec = mat.times(coordCO);
	coordCO   = addCoords(caToCOVec,  ca); 

	//Compute the N coordinate residue (i+1)
	double [] coordN  = new double[]{0.0, Const.dCO2N, 0.0};
	mat = mat.times(r4zInv.times(Const.r5xInv));
	double [] co1ToNVec = mat.times(coordN);
	coordN = addCoords(coordCO, co1ToNVec);

	//Compute the CA coordinate residue (i+1)
	nToCAVec = mat.times(Const.dirCAcnt);
	double[] coordCA = addCoords(coordN, nToCAVec);

	//Compute the NH coordinate residue (i+1)
	nToNHVec = mat.times(Const.dirNHcnt);
	double[] coordNH = addCoords(coordN, nToNHVec);
	double [][] nHnCaCoords = new double[3][3];
	nHnCaCoords[0] = coordN; 
	nHnCaCoords[1] = coordCA; 
	nHnCaCoords[2] = coordNH; 
	return 	nHnCaCoords;
    }
  
  /**
   * A method for computing coordinates of backbone atoms.
   * 
   * @param phi     The backbone \phi angle
   * @param psi     The backbone \psi angle
   * @param first    whether the residue is the first residue of the fragment
   * @param n the n
   * @param nh the nh
   * @param ca the ca
   * 
   * @return Pdb object consisting of atoms: NH, CA, CO, HA and O of residue (i).
   * Please note that the coordinates of N, NH and CA of residue (i+1) are returned by
   * overwriting the original array for them: This may be BUG prone.
   */    
    public double[][] nNhCaCal(double phi, double psi, double[] n, double[] nh, double[] ca, boolean first){
	Matrix rg = Matrix.identity(3,3);
  	Matrix r2yInv = rg.rotationMat(phi, "-y"); //transpose == inverse == rotate in opposite direction
	Matrix r4zInv = rg.rotationMat(psi+Math.PI, "-z"); 
	//intraResidue N(i)->NH(i) of the 1st residue
	double [] nToNHVec =  internuclearVec(n,  nh);
	double [] nToCAVec = internuclearVec(n, ca);
	PhiPsi ff = new PhiPsi();
	if (!first)
	    rg = ff.RgCal(nToNHVec, nToCAVec);
	Matrix rgInv = rg.transpose();
 	Matrix mat = rgInv.times(Const.r1x9yInv.times(r2yInv.times(Const.r3xInv)));

	//Compute the CO coordinate residue (i)
	double [] coordCO   = new double[]{0.0, 0.0, Const.dCA2CO};
	double [] caToCOVec = mat.times(coordCO);
	coordCO   = addCoords(caToCOVec,  ca); 

	//Compute the N coordinate residue (i+1)
	double [] coordN  = new double[]{0.0, Const.dCO2N, 0.0};
	mat = mat.times(r4zInv.times(Const.r5xInv));
	double [] co1ToNVec = mat.times(coordN);
	coordN = addCoords(coordCO, co1ToNVec);

	//Compute the CA coordinate residue (i+1)
	nToCAVec = mat.times(Const.dirCAcnt);
	double[] coordCA = addCoords(coordN, nToCAVec);

	//Compute the NH coordinate residue (i+1)
	nToNHVec = mat.times(Const.dirNHcnt);
	double[] coordNH = addCoords(coordN, nToNHVec);
	double [][] nHnCaCoords = new double[3][3];
	nHnCaCoords[0] = coordN; 
	nHnCaCoords[1] = coordCA; 
	nHnCaCoords[2] = coordNH; 
	return 	nHnCaCoords;
    }

  /**
   * A method for computing coordinates of backbone atoms backward: with direasing residue no.
   * 
   * @param phi     The backbone \phi_i angle
   * @param psi     The backbone \psi_i angle
   * @param n1      the coordinate of amide nitrogen of the residue (i+1)
   * @param nh1     the coordinate of amide proton atom of residue (i+1)
   * @param ca1     the coordinates of Ca atom of residue (i+1)
   * 
   * @return the coordinates for atom N, NH, CA, HA.
   * Please note that the coordinates of N, NH and CA of residue i are returned by
   * overwriting the original array for them: This may be BUG prone.
   */    
    public double[][] nNhCaHaByBackward(double phi, double psi, double[] n1, double[] nh1, double[] ca1){
	//Compute the Matrix Rg
	double[] nToNHVec = internuclearVec(n1, nh1);
	double[] nToCAVec = internuclearVec(n1, ca1);
	PhiPsi ff = new PhiPsi();
	Matrix rg = ff.RgCal(nToNHVec, nToCAVec);
	Matrix rgInv = rg.transpose();

	//compute the CO coordinate
	double [] n2coCnt = {0.0, 0.0, Const.dCO2N};
	Matrix mat = rgInv.times(Const.r9z1xAlphaInv);
	double [] n2co = mat.times(n2coCnt);
	double [] coCal = addCoords(n1, n2co);

	//compute the CA coordinate
	double [] co2caCnt = {0.0, 0.0, Const.dCA2CO};
	mat = mat.times(Const.r6zAlphaInv.times(Const.r2xAlphaInv));
	double[] co2ca = mat.times(co2caCnt);
	double[] caCal = addCoords(coCal, co2ca);

	//compute the N coordinate
	Matrix rzPsiInv = mat.rotationMat(psi+Math.PI, "-z");
	mat = mat.times(rzPsiInv.times(Const.r3xAlphaInv));
	double[] ca2n = mat.times(Const.coordCA);
	double [] nCal = addCoords(caCal, ca2n);
	
	//compute the HA coordinate
	Matrix matHA = mat.times(Const.matHAInv);
	double[] haCal = addCoords(caCal, matHA.times(Const.coordHA));

	//compute the NH coordinate
	Matrix ryPhiInv = mat.rotationMat(phi, "-y");
	Matrix ra10Inv = rg.rotationMat(Math.PI, "-y");
	mat = mat.times(Const.r8yAlphaInv.times(ryPhiInv.times(ra10Inv.times(Const.r5xAlphaInv))));
	double[] n2nh = mat.times(Const.coordNH);
	double [] nhCal = addCoords(nCal, n2nh);
	double [][] nNhCaHaCoords = new double[4][3];
	nNhCaHaCoords[0] = nCal; 
	nNhCaHaCoords[1] = nhCal; 
	nNhCaHaCoords[2] = caCal; 
	nNhCaHaCoords[3] = haCal; 
	return nNhCaHaCoords;
    }

  /**
   * A method for computing coordinates of backbone atoms backward: with direasing residue no.
   * 
   * @param phi     The backbone \phi_i angle
   * @param psi     The backbone \psi_i angle
   * @param n1      the coordinate of amide nitrogen of the residue (i+1)
   * @param nh1     the coordinate of amide proton atom of residue (i+1)
   * @param ca1     the coordinates of Ca atom of residue (i+1)
   * @param residueNo  the residue number of residue i
   * @param resid1     the name of the residue i
   * 
   * @return Pdb object consisting of atoms: NH, CA, CO, HA and O of residue (i).
   * Please note that the coordinates of N, NH and CA of residue i are returned by
   * overwriting the original array for them: This may be BUG prone.
   */    
    public Pdb coordByBackward(double phi, double psi, double[] n1, double[] nh1, double[] ca1, 
			       int residueNo, String resid1)
    {
		//Compute the Matrix Rg
		double[] nToNHVec = internuclearVec(n1, nh1);
		double[] nToCAVec = internuclearVec(n1, ca1);
		PhiPsi ff = new PhiPsi();
		Matrix rg = ff.RgCal(nToNHVec, nToCAVec);
		Matrix rgInv = rg.transpose();
		Vector atomVec  = new Vector();
	
		//compute the CO coordinate
		double [] n2coCnt = {0.0, 0.0, Const.dCO2N};
		double [] n2co = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(n2coCnt)));
		double [] coCal = addCoords(n1, n2co);

		//compute the O coordinate
		double [] co2oCnt = {0.0, Const.dCO2O, 0.0};
		double [] co2o  = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.matOInv.times(co2oCnt))));
		double [] o2Cal = addCoords(coCal, co2o);
	
	
		//compute the CA coordinate
		double [] co2caCnt = {0.0, 0.0, Const.dCA2CO};
		Matrix mat =  rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.r6zAlphaInv.times(Const.r2xAlphaInv))));
		double[] co2ca = mat.times(co2caCnt);
		double[] caCal = addCoords(coCal, co2ca);

		//compute the N coordinate
		Matrix rzPsiInv = mat.rotationMat(psi+Math.PI, "-z");
		mat = mat.times(rzPsiInv.times(Const.r3xAlphaInv));
		double[] ca2nCnt = {0.0, Const.dN2CA, 0.0};
		double[] ca2n = mat.times(ca2nCnt);
		double [] nCal = addCoords(caCal, ca2n);
	
		double[] haCnt = {0.0, 0.0, Const.dCA2HA};
		Matrix matHA = mat.times(Const.matHAInv);
		double[] haCal = addCoords(caCal, matHA.times(haCnt));
	
		double[] cbCnt = {0.0, 0.0, Const.dCA2CB};
		Matrix matCB = mat.times(Const.matCBInv);
		double[] cbCal = addCoords(caCal, matCB.times(cbCnt));
	
		Matrix ryPhiInv = mat.rotationMat(phi, "-y");
		Matrix ra10Inv = rg.rotationMat(Math.PI, "-y");
		mat = mat.times(Const.r8yAlphaInv.times(ryPhiInv.times(ra10Inv.times(Const.r5xAlphaInv))));
		double[] n2nhCnt = {0.0, 0.0, -Const.dN2H};
		double[] n2nh = mat.times(n2nhCnt);
		double [] nhCal = addCoords(nCal, n2nh);
		atomVec.add(new Cartesian("N", nCal));
		atomVec.add(new Cartesian("HN", nhCal));
		atomVec.add(new Cartesian("CA", caCal));
		atomVec.add(new Cartesian("HA", haCal));
		atomVec.add(new Cartesian("CB", cbCal));
		atomVec.add(new Cartesian("C", coCal));
		atomVec.add(new Cartesian("O", o2Cal));
	
		return (new Pdb(residueNo, resid1, atomVec));
    }
    
    /**
     * By Michael Zeng.
     * A method for computing coordinates of backbone atoms backward: with direasing residue no.
     * The same as coordByBackward, but also adding the c, ca,co atoms of residue i-1.
     * 
     * @param phi     The backbone \phi_i angle
     * @param psi     The backbone \psi_i angle
     * @param n1      the coordinate of amide nitrogen of the residue (i+1)
     * @param nh1     the coordinate of amide proton atom of residue (i+1)
     * @param ca1     the coordinates of Ca atom of residue (i+1)
     * @param residueNo  the residue number of residue i
     * 
     * @return Pdb object consisting of atoms: NH, CA, CO, HA and O of residue (i).
     * Please note that the coordinates of N, NH and CA of residue i are returned by
     * overwriting the original array for them: This may be BUG prone.
     */    
    public Vector coordByBackwardNew(double phi, double psi, double[] n1, double[] nh1, double[] ca1, 
			       int residueNo)
    {
		//Compute the Matrix Rg
		double[] nToNHVec = internuclearVec(n1, nh1);
		double[] nToCAVec = internuclearVec(n1, ca1);
		PhiPsi ff = new PhiPsi();
		Matrix rg = ff.RgCal(nToNHVec, nToCAVec);
		Matrix rgInv = rg.transpose();
		Vector atomVec  = new Vector();
	
		//compute the CO coordinate
		double [] n2coCnt = {0.0, 0.0, Const.dCO2N};
		double [] n2co = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(n2coCnt)));
		double [] coCal = addCoords(n1, n2co);

		//compute the O coordinate
		double [] co2oCnt = {0.0, Const.dCO2O, 0.0};
		double [] co2o  = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.matOInv.times(co2oCnt))));
		double [] o2Cal = addCoords(coCal, co2o);
	
	
		//compute the CA coordinate
		double [] co2caCnt = {0.0, 0.0, Const.dCA2CO};
		Matrix mat =  rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.r6zAlphaInv.times(Const.r2xAlphaInv))));
		double[] co2ca = mat.times(co2caCnt);
		double[] caCal = addCoords(coCal, co2ca);

		//compute the N coordinate
		Matrix rzPsiInv = mat.rotationMat(psi+Math.PI, "-z");
		mat = mat.times(rzPsiInv.times(Const.r3xAlphaInv));
		double[] ca2nCnt = {0.0, Const.dN2CA, 0.0};
		double[] ca2n = mat.times(ca2nCnt);
		double [] nCal = addCoords(caCal, ca2n);
	
		double[] haCnt = {0.0, 0.0, Const.dCA2HA};
		Matrix matHA = mat.times(Const.matHAInv);
		double[] haCal = addCoords(caCal, matHA.times(haCnt));
	
		double[] cbCnt = {0.0, 0.0, Const.dCA2CB};
		Matrix matCB = mat.times(Const.matCBInv);
		double[] cbCal = addCoords(caCal, matCB.times(cbCnt));
	
		Matrix ryPhiInv = mat.rotationMat(phi, "-y");
		Matrix ra10Inv = rg.rotationMat(Math.PI, "-y");
		mat = mat.times(Const.r8yAlphaInv.times(ryPhiInv.times(ra10Inv.times(Const.r5xAlphaInv))));
		double[] n2nhCnt = {0.0, 0.0, -Const.dN2H};
		double[] n2nh = mat.times(n2nhCnt);
		double [] nhCal = addCoords(nCal, n2nh);
		atomVec.add(new Cartesian("N", nCal));
		atomVec.add(new Cartesian("HN", nhCal));
		atomVec.add(new Cartesian("CA", caCal));
		atomVec.add(new Cartesian("HA", haCal));
		atomVec.add(new Cartesian("CB", cbCal));
		atomVec.add(new Cartesian("C", coCal));
		atomVec.add(new Cartesian("O", o2Cal));
	
		Vector vecPdb=new Vector();	
		//add PDB of residue i
		vecPdb.add(new Pdb(residueNo, "ALA", atomVec));
		
		//add pdb of residue i-1:
		//Compute the Matrix Rg
		nToNHVec = internuclearVec(nCal, nhCal);
		nToCAVec = internuclearVec(nCal, caCal);
		
		rg = ff.RgCal(nToNHVec, nToCAVec);
		rgInv = rg.transpose();
		atomVec  = new Vector();
		
		//compute the CO coordinate of residue i-1
		//n2coCnt = {0.0, 0.0, Const.dCO2N};
		n2co = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(n2coCnt)));
		coCal = addCoords(nCal, n2co);
		
		//compute the O coordinate of residue i-1
		//double [] co2oCnt = {0.0, Const.dCO2O, 0.0};
		co2o  = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.matOInv.times(co2oCnt))));
		o2Cal = addCoords(coCal, co2o);	
	
		//compute the CA coordinate of residue i-1
		//double [] co2caCnt = {0.0, 0.0, Const.dCA2CO};
		mat =  rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.r6zAlphaInv.times(Const.r2xAlphaInv))));
		co2ca = mat.times(co2caCnt);
		caCal = addCoords(coCal, co2ca);
		atomVec.add(new Cartesian("CA", caCal));	
		atomVec.add(new Cartesian("C", coCal));
		atomVec.add(new Cartesian("O", o2Cal));
		vecPdb.add(new Pdb(residueNo-1, "ALA", atomVec));
		Collections.sort(vecPdb, new Pdb.PdbComparator());
		return vecPdb;
    }

  /**
   * A method for computing coordinates of backbone atoms.
   * 
   * @param phi     The backbone \phi angle
   * @param psi     The backbone \psi angle
   * @param n1      the direction cosine for amide nitrogen of the original residue
   * @param nh1     the direction cosine for amide proton
   * @param ca1     the direction cosine for Ca
   * @param residueNo  the residue number
   * @param resid1     the name of the residue
   * @param first    whether the residue is the first residue of the fragment
   * 
   * @return Pdb object consisting of atoms: NH, CA, CO, HA and O of residue (i).
   * Please note that the coordinates of N, NH and CA of residue (i+1) are returned by
   * overwriting the original array for them: This may be BUG prone.
   */    
    public Pdb coordByResidue(double phi, double psi, double[] n1, double[] nh1, double[] ca1, 
			      int residueNo, String resid1, boolean first)
    {
		Matrix rg = Matrix.identity(3,3);
		Matrix rgInv = new Matrix(3,3);
	  	Matrix r2yInv = rg.rotationMat(phi, "-y"); //transpose == inverse == rotate in opposite direction
		Matrix r4zInv = rg.rotationMat(psi+Math.PI, "-z"); 
		double [] nToCO1Vec = new double[3];
		double [] coToCAVec = new double[3];
		double [] caToNVec  = new double[3];
		Vector atomVec    = new Vector();
	
		//intraResidue N(i)->NH(i) of the 1st residue
		double [] nToNHVec =  internuclearVec(n1,  nh1);
		//N(i)->CA(i) vector
		double [] nToCAVec = internuclearVec(n1, ca1);
		PhiPsi ff = new PhiPsi();
		if (!first)
		    rg = ff.RgCal(nToNHVec, nToCAVec);
		rgInv = rg.transpose();
	 	Matrix matT = rgInv.times(Const.r1x9yInv.times(r2yInv));

		//Compute the HA coordinate of residue (i)
		double [] coordHA = new double[]{0.0,  0.0, Const.dCA2HA};
		double [] caToHAVec = matT.times(Const.rHA2HA1Inv.times(coordHA));
		coordHA =  addCoords(caToHAVec,  ca1);
		atomVec.add(new Cartesian("HA", coordHA[0], coordHA[1], coordHA[2]));

		//Compute the CB coordinate of residue (i)
		double [] coordCb = new double[]{0.0,  0.0, Const.dCA2CB};
		double[] caToCbVec = matT.times(Const.rCb2Cb1Inv.times(coordCb));
		coordCb =  addCoords(caToCbVec,  ca1);
		atomVec.add(new Cartesian("CB", coordCb[0], coordCb[1], coordCb[2]));

		//Compute the CO coordinate residue (i)
		matT = matT.times(Const.r3xInv);
		double [] coordCO   = new double[]{0.0, 0.0, Const.dCA2CO};
		double [] caToCOVec = matT.times(coordCO);
		coordCO   = addCoords(caToCOVec,  ca1); 
		atomVec.add(new Cartesian("C", coordCO[0], coordCO[1], coordCO[2]));

		//Compute the O coordinate residue (i)
		double[] coordO = new double[]{0.0, 0.0, Const.dCO2O};
		double[] coToOVec = matT.times(r4zInv.times(Const.rOInv.times(coordO)));
		coordO  = addCoords(coToOVec, coordCO);
		atomVec.add(new Cartesian("O", coordO[0], coordO[1], coordO[2]));

		//Compute the N coordinate residue (i+1)
		double [] coordN  = new double[]{0.0, Const.dCO2N, 0.0};
		matT = matT.times(r4zInv.times(Const.r5xInv));
		double [] co1ToNVec = matT.times(coordN);
		coordN = addCoords(coordCO, co1ToNVec);
	
		//updated N coordinate
		n1[0] = coordN[0];
		n1[1] = coordN[1];
		n1[2] = coordN[2];

		//Compute the CA coordinate residue (i+1)
		double [] coordCA = new double[]{0.0, Const.dN2CA, 0.0};
	  	matT = matT.times(Const.r6yInv.times(Const.r7xInv));
		nToCAVec = matT.times(Const.r8zInv.times(Const.r1xInv.times(coordCA)));
		coordCA = addCoords(coordN, nToCAVec);

		//updated CA coordinate
		ca1[0] = coordCA[0];
		ca1[1] = coordCA[1];
		ca1[2] = coordCA[2];
	
		//Compute the NH coordinate residue (i+1)
		double [] coordNH = new double[]{0.0, 0.0,  -Const.dN2H};
		double [] nToNHVec2 = matT.times(coordNH);
		coordNH = addCoords(coordN, nToNHVec2);
		//updated NH coordinate
		nh1[0] = coordNH[0];
		nh1[1] = coordNH[1];
		nh1[2] = coordNH[2];
		return (new Pdb(residueNo, resid1, atomVec));
    }
    
    /**
     * *A method for computing coordinates of backbone atoms,
     * including all backbone atoms, ha, ca, n, hn, cb,c o.
     * 
     * @param phi     The backbone \phi angle
     * @param psi     The backbone \psi angle
     * @param n1      the direction cosine for amide nitrogen of the original residue
     * @param nh1     the direction cosine for amide proton
     * @param ca1     the direction cosine for Ca
     * @param residueNo  the residue number
     * @param resid1     the name of the residue
     * @param first    whether the residue is the first residue of the fragment
     * 
     * @return Pdb object consisting of atoms: NH, CA, CO, HA and O of residue (i).
     * Please note that the coordinates of N, NH and CA of residue (i+1) are returned by
     * overwriting the original array for them: This may be BUG prone.
     */    
    public Pdb coordByResidueFull(double phi, double psi,  final double[] n1, final double[] nh1, final double[] ca1, 
			      int residueNo, String resid1, boolean first){
	Matrix rg = Matrix.identity(3,3);
	Matrix rgInv = new Matrix(3,3);
  	Matrix r2yInv = rg.rotationMat(phi, "-y"); //transpose == inverse == rotate in opposite direction
	Matrix r4zInv = rg.rotationMat(psi+Math.PI, "-z"); 
	double [] nToCO1Vec = new double[3];
	double [] coToCAVec = new double[3];
	double [] caToNVec  = new double[3];
	Vector atomVec    = new Vector();
	atomVec.add(new Cartesian("N", n1[0], n1[1], n1[2]));
	atomVec.add(new Cartesian("HN", nh1[0], nh1[1], nh1[2]));
	atomVec.add(new Cartesian("CA", ca1[0], ca1[1], ca1[2]));
	
	
	//intraResidue N(i)->NH(i) of the 1st residue
	double [] nToNHVec =  internuclearVec(n1,  nh1);
	//N(i)->CA(i) vector
	double [] nToCAVec = internuclearVec(n1, ca1);
	PhiPsi ff = new PhiPsi();
	if (!first)
	    rg = ff.RgCal(nToNHVec, nToCAVec);
	rgInv = rg.transpose();
 	Matrix matT = rgInv.times(Const.r1x9yInv.times(r2yInv));

	//Compute the HA coordinate of residue (i)
	double [] coordHA = new double[]{0.0,  0.0, Const.dCA2HA};
	double [] caToHAVec = matT.times(Const.rHA2HA1Inv.times(coordHA));
	coordHA =  addCoords(caToHAVec,  ca1);
	atomVec.add(new Cartesian("HA", coordHA[0], coordHA[1], coordHA[2]));

	//Compute the CB coordinate of residue (i)
	double [] coordCb = new double[]{0.0,  0.0, Const.dCA2CB};
	double[] caToCbVec = matT.times(Const.rCb2Cb1Inv.times(coordCb));
	coordCb =  addCoords(caToCbVec,  ca1);
	atomVec.add(new Cartesian("CB", coordCb[0], coordCb[1], coordCb[2]));

	//Compute the CO coordinate residue (i)
	matT = matT.times(Const.r3xInv);
	double [] coordCO   = new double[]{0.0, 0.0, Const.dCA2CO};
	double [] caToCOVec = matT.times(coordCO);
	coordCO   = addCoords(caToCOVec,  ca1); 
	atomVec.add(new Cartesian("C", coordCO[0], coordCO[1], coordCO[2]));

	//Compute the O coordinate residue (i)
	double[] coordO = new double[]{0.0, 0.0, Const.dCO2O};
	double[] coToOVec = matT.times(r4zInv.times(Const.rOInv.times(coordO)));
	coordO  = addCoords(coToOVec, coordCO);
	atomVec.add(new Cartesian("O", coordO[0], coordO[1], coordO[2]));

	//Compute the N coordinate residue (i+1)
	double [] coordN  = new double[]{0.0, Const.dCO2N, 0.0};
	matT = matT.times(r4zInv.times(Const.r5xInv));
	double [] co1ToNVec = matT.times(coordN);
	coordN = addCoords(coordCO, co1ToNVec);
	

	//Compute the CA coordinate residue (i+1)
	double [] coordCA = new double[]{0.0, Const.dN2CA, 0.0};
  	matT = matT.times(Const.r6yInv.times(Const.r7xInv));
	nToCAVec = matT.times(Const.r8zInv.times(Const.r1xInv.times(coordCA)));
	coordCA = addCoords(coordN, nToCAVec);


	//Compute the NH coordinate residue (i+1)
	double [] coordNH = new double[]{0.0, 0.0,  -Const.dN2H};
	double [] nToNHVec2 = matT.times(coordNH);
	coordNH = addCoords(coordN, nToNHVec2);
	
	return (new Pdb(residueNo, resid1, atomVec));
    }


    /**
     * Build model (compute the coordinates of backbone atoms (N, NH, CA, CO, O and HA) from
     * the standard geometry and Phi/Psi angle.
     * 
     * @param phiPsiVec  an array of Phi/Psi angles for the fragment.
     * @param n1    the direction cosine for amide nitrogen of the first residue
     * @param nh1   the direction cosine for amide proton
     * @param ca1   the direction cosine for Ca
     * 
     * @return a vector of Pdb object
     */
    public Vector modelBuild(Vector phiPsiVec, double [] n1, double[] nh1, double[] ca1){
	Vector tmpVec = new Vector();
	int i = 0, j = 0;
	double [] coordN  = new double[3]; 
	System.arraycopy(n1, 0, coordN, 0, 3);
	double [] coordNH = new double[3];
	System.arraycopy(nh1, 0, coordNH, 0, 3);
	double [] coordCA = new double[3]; 
	System.arraycopy(ca1, 0, coordCA, 0, 3);

	double [] nhVec1 = new double[3];
	double [] n2CA = new double[3];
	Vector atomVec = new Vector();
	Vector pdbVec = new Vector();
	Cartesian cc = new Cartesian();
 	atomVec.add(new Cartesian("N", n1));
	atomVec.add(new Cartesian("H", nh1));
 	atomVec.add(new Cartesian("CA", ca1));
	nhVec1 = internuclearVec(n1, nh1);
	Matrix rg = Matrix.identity(3,3);
	PhiPsi ff = (PhiPsi)phiPsiVec.elementAt(0);
	int no = ff.getResidueNo();
	String	resid1 = ff.getResidue();
 	double phi = ff.getPhi(); 
 	double psi = ff.getPsi(); 
	//The first residue
	Pdb pp = coordByResidue(phi, psi, coordN, coordNH, coordCA, no, resid1, true);
	atomVec.addAll(pp.getAtomVec());
	pdbVec.add(new Pdb(no, resid1, atomVec));
	atomVec = new Vector();
	atomVec.add(new Cartesian("N", coordN));
	atomVec.add(new Cartesian("H", coordNH));
	atomVec.add(new Cartesian("CA", coordCA));
	//Compute New NH, N2CA and 
	nhVec1 = internuclearVec(coordN, coordNH); //update new NH1
	n2CA =   internuclearVec(coordN, coordCA);
	rg = ff.RgCal(nhVec1, n2CA);
	double[] co1ToN = new double[3];
	double[] co1    = new double[3];
	String atom = "";
	for (i=1; i<phiPsiVec.size(); i++){
	    ff = (PhiPsi)phiPsiVec.elementAt(i);
	    no = ff.getResidueNo();
	    resid1 = ff.getResidue();
   	    phi = ff.getPhi(); 
   	    psi = ff.getPsi(); 

	    pp = coordByResidue(phi, psi, coordN, coordNH, coordCA, no, resid1, false);
	    atomVec.addAll(pp.getAtomVec());
	    for (j=0; j<atomVec.size(); j++){//get CO coordinates BACK in case O and HA are there 
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		if (atom.equals("C"))
		    co1 = cc.getXYZ();
	    }
	    pdbVec.add(new Pdb(no, resid1, atomVec));
	    atomVec = new Vector();
	    atomVec.add(new Cartesian("N",  coordN));
	    atomVec.add(new Cartesian("H",  coordNH));
	    atomVec.add(new Cartesian("CA", coordCA));
	    //Update NEW NH, N2CA and CO1ToN vectors
 	    nhVec1 = internuclearVec(coordN, coordNH); //update new NH1
	    co1ToN = internuclearVec(coordN, co1);
	    n2CA   = internuclearVec(coordN, coordCA);
	    if (i == phiPsiVec.size() - 1)
		pdbVec.add(new Pdb(no+1, resid1, atomVec));
	    rg = ff.RgCal(nhVec1, n2CA);
	}
	
	Vector vecPdbNew=new Vector();
	for (i=0;i< pdbVec.size()-1;i++)
	{
		Pdb ppT=(Pdb)pdbVec.elementAt(i);
		vecPdbNew.add(ppT);
	}
	return vecPdbNew;
    }


    /*
     * Build model (compute the coordinates of backbone atoms (N, NH, CA, CO, O and HA) from 
     * the standard geometry and Phi/Psi angle.
     @param phiPsiVec  an array of Phi/Psi angles for the fragment.
     @param n1    the direction cosine for amide nitrogen of the first residue
     @param nh1   the direction cosine for amide proton
     @param ca1   the direction cosine for Ca
     @return a vector of Pdb object
     */
    /**
     * Model build.
     * 
     * @param phiPsiVec the phi psi vec
     * @param n1 the n1
     * @param nh1 the nh1
     * @param ca1 the ca1
     * @param isLink the is link
     * 
     * @return the vector
     */
    public Vector modelBuild(Vector phiPsiVec, double [] n1, double[] nh1, double[] ca1, boolean isLink)
    {
		Vector tmpVec = new Vector();
		int i = 0, j = 0;
		double [] coordN  = new double[3]; 
		System.arraycopy(n1, 0, coordN, 0, 3);
		double [] coordNH = new double[3];
		System.arraycopy(nh1, 0, coordNH, 0, 3);
		double [] coordCA = new double[3]; 
		System.arraycopy(ca1, 0, coordCA, 0, 3);
	
		double [] nhVec1 = new double[3];
		double [] n2CA = new double[3];
		Vector atomVec = new Vector();
		Vector pdbVec = new Vector();
		Cartesian cc = new Cartesian();
	 	atomVec.add(new Cartesian("N", n1));
		atomVec.add(new Cartesian("H", nh1));
	 	atomVec.add(new Cartesian("CA", ca1));
		nhVec1 = internuclearVec(n1, nh1);
		Matrix rg = Matrix.identity(3,3);
		PhiPsi ff = (PhiPsi)phiPsiVec.elementAt(0);
		int no = ff.getResidueNo();
		String	resid1 = ff.getResidue();
	 	double phi = ff.getPhi(); 
	 	double psi = ff.getPsi(); 
		//The first residue
		Pdb pp = coordByResidue(phi, psi, coordN, coordNH, coordCA, no, resid1, !isLink);
		atomVec.addAll(pp.getAtomVec());
		pdbVec.add(new Pdb(no, resid1, atomVec));
		atomVec = new Vector();
		atomVec.add(new Cartesian("N", coordN));
		atomVec.add(new Cartesian("H", coordNH));
		atomVec.add(new Cartesian("CA", coordCA));
		//Compute New NH, N2CA and 
		nhVec1 = internuclearVec(coordN, coordNH); //update new NH1
		n2CA =   internuclearVec(coordN, coordCA);
		rg = ff.RgCal(nhVec1, n2CA);
		double[] co1ToN = new double[3];
		double[] co1    = new double[3];
		String atom = "";
		for (i=1; i<phiPsiVec.size(); i++)
		{
		    ff = (PhiPsi)phiPsiVec.elementAt(i);
		    no = ff.getResidueNo();
		    resid1 = ff.getResidue();
	   	    phi = ff.getPhi(); 
	   	    psi = ff.getPsi(); 
		    pp = coordByResidue(phi, psi, coordN, coordNH, coordCA, no, resid1, false);
		    atomVec.addAll(pp.getAtomVec());
		    for (j=0; j<atomVec.size(); j++)
		    {//get CO coordinates BACK in case O and HA are there 
				cc = (Cartesian)atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equals("C"))
				    co1 = cc.getXYZ();
		    }
		    pdbVec.add(new Pdb(no, resid1, atomVec));
		    atomVec = new Vector();
		    atomVec.add(new Cartesian("N",  coordN));
		    atomVec.add(new Cartesian("H",  coordNH));
		    atomVec.add(new Cartesian("CA", coordCA));
		    //Update NEW NH, N2CA and CO1ToN vectors
	 	    nhVec1 = internuclearVec(coordN, coordNH); //update new NH1
		    co1ToN = internuclearVec(coordN, co1);
		    n2CA   = internuclearVec(coordN, coordCA);
		    if (i == phiPsiVec.size() - 1)
			pdbVec.add(new Pdb(no+1, resid1, atomVec));
		    rg = ff.RgCal(nhVec1, n2CA);
		}
		Vector vecPdbNew=new Vector();
		for (i=0;i< pdbVec.size()-1;i++)
		{
			Pdb ppT=(Pdb)pdbVec.elementAt(i);
			vecPdbNew.add(ppT);
		}
		return vecPdbNew;
    }
    
    /**
     * By Michael Zeng.
     * A method for computing the loop backbone atoms backward: with direasing residue no.
     * 
     * @param phiPsiVec the vector for storing phi/psi angles
     * @param startNo the first residue number, which is the end of backward computation.
     * @param endNo  the residue number of the last residue for starting the backward computation.
     * @param nCoord the n coord
     * @param nhCoord the nh coord
     * @param caCoord the ca coord
     * 
     * @return a vector Pdb objects, which is the loop backbone fragment.
     * Note: the pdb at residue startNo-1 contains ca, c, co atoms, and will be also returned.
     */    
    public Vector modelBuildLoopBackward(Vector phiPsiVec, final double [] nCoord, final double[] nhCoord, final double[] caCoord,
    		int startNo,int endNo)
    {
    	double [] n1=new double[3];
    	double [] nh1=new double[3];
    	double [] ca1=new double[3];
    	n1[0]=nCoord[0];n1[1]=nCoord[1];n1[2]=nCoord[2];
    	nh1[0]=nhCoord[0];nh1[1]=nhCoord[1];nh1[2]=nhCoord[2];
    	ca1[0]=caCoord[0];ca1[1]=caCoord[1];ca1[2]=caCoord[2];
    	
    	PhiPsi ff=new PhiPsi();
    	Vector vecPdb=new Vector();	//for storing the new computed pdbs
    	
    	double[] nToNHVec = internuclearVec(n1, nh1);
		double[] nToCAVec = internuclearVec(n1, ca1);   
		Matrix rg = ff.RgCal(nToNHVec, nToCAVec);
		
    	Matrix rgInv = rg.transpose();
    	Matrix mat =  rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.r6zAlphaInv.times(Const.r2xAlphaInv))));
		
    	double [] n2coCnt = {0.0, 0.0, Const.dCO2N};
    	double [] co2oCnt = {0.0, Const.dCO2O, 0.0};
    	double [] co2caCnt = {0.0, 0.0, Const.dCA2CO};
    	double[] co2ca=new double[3];
    	double [] n2co=new double[3];
    	double [] nCal=new double[3];
    	double [] co2o  =new double[3];
    	
    	//atom coodinate of the current residue:
    	double[] caCal =new double[3];
    	double[] coCal =new double[3];
    	double[] o2Cal =new double[3];    	
    			
    	for (int curNo=endNo;curNo>=startNo;curNo--)
    	{
    		int ind = Collections.binarySearch(phiPsiVec, new PhiPsi(curNo), new PhiPsi.PPComparator());
    	    ff = (PhiPsi)phiPsiVec.elementAt(ind);
    	    int no = ff.getResidueNo();
    	    String resid1 = ff.getResidue();
       	    double phiValue = ff.getPhi(); 
       	    double psiValue = ff.getPsi(); 
    		
       	    //Compute the Matrix Rg
    		nToNHVec = internuclearVec(n1, nh1);
    		nToCAVec = internuclearVec(n1, ca1);    		
    		rg = ff.RgCal(nToNHVec, nToCAVec);
    		rgInv = rg.transpose();
    		Vector atomVec  = new Vector();    		
    		
    		//compute the CO coordinate of residue (curNo)    		
    		n2co = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(n2coCnt)));
    		coCal = addCoords(n1, n2co);

    		//compute the O coordinate of residue (curNo)    		
    		co2o  = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.matOInv.times(co2oCnt))));
    		o2Cal = addCoords(coCal, co2o);    	
    	
    		//compute the CA coordinate of residue (curNo)    		
    		mat =  rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.r6zAlphaInv.times(Const.r2xAlphaInv))));
    		co2ca = mat.times(co2caCnt);
    		caCal = addCoords(coCal, co2ca);

    		//compute the N coordinate of residue (curNo)
    		Matrix rzPsiInv = mat.rotationMat(psiValue+Math.PI, "-z");
    		mat = mat.times(rzPsiInv.times(Const.r3xAlphaInv));
    		double[] ca2nCnt = {0.0, Const.dN2CA, 0.0};
    		double[] ca2n = mat.times(ca2nCnt);
    		nCal = addCoords(caCal, ca2n);

    		//compute the ha coordinate of residue (curNo)
    		double[] haCnt = {0.0, 0.0, Const.dCA2HA};
    		Matrix matHA = mat.times(Const.matHAInv);
    		double[] haCal = addCoords(caCal, matHA.times(haCnt));
    	
    		//compute the cb coordinate of residue (curNo)
    		double[] cbCnt = {0.0, 0.0, Const.dCA2CB};
    		Matrix matCB = mat.times(Const.matCBInv);
    		double[] cbCal = addCoords(caCal, matCB.times(cbCnt));

    		//compute the hn coordinate of residue (curNo)
    		Matrix ryPhiInv = mat.rotationMat(phiValue, "-y");
    		Matrix ra10Inv = rg.rotationMat(Math.PI, "-y");
    		mat = mat.times(Const.r8yAlphaInv.times(ryPhiInv.times(ra10Inv.times(Const.r5xAlphaInv))));
    		double[] n2nhCnt = {0.0, 0.0, -Const.dN2H};
    		double[] n2nh = mat.times(n2nhCnt);
    		double [] nhCal = addCoords(nCal, n2nh);
    		
    		//add the coordinates of backbonea atoms
    		atomVec.add(new Cartesian("N", nCal));
    		atomVec.add(new Cartesian("HN", nhCal));
    		atomVec.add(new Cartesian("CA", caCal));
    		atomVec.add(new Cartesian("HA", haCal));
    		atomVec.add(new Cartesian("CB", cbCal));
    		atomVec.add(new Cartesian("C", coCal));
    		atomVec.add(new Cartesian("O", o2Cal));    		
    		
    		//add PDB of residue (curNo)    	
    		vecPdb.add(new Pdb(curNo, "ALA", atomVec));

    		//compute the coordinate of residue curNo-1:
    		//Compute the Matrix Rg
    		nToNHVec = internuclearVec(nCal, nhCal);
    		nToCAVec = internuclearVec(nCal, caCal);
    		
    		rg = ff.RgCal(nToNHVec, nToCAVec);
    		rgInv = rg.transpose();
    		
    		//Update the coordinate of n, hn and ca for the next computational  
    		n1[0]=nCal[0];
    		n1[1]=nCal[1];
    		n1[2]=nCal[2];
    		
    		nh1[0]=nhCal[0];
    		nh1[1]=nhCal[1];
    		nh1[2]=nhCal[2];
    		
    		ca1[0]=caCal[0];
    		ca1[1]=caCal[1];
    		ca1[2]=caCal[2];    		
    	}//for (int i=1; i<phiPsiVec.size(); i++)

    	//compute the CO coordinate of residue startNo-1:
		//n2coCnt = {0.0, 0.0, Const.dCO2N};
		n2co = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(n2coCnt)));
		coCal = addCoords(nCal, n2co);
		
		//compute the O coordinate of residue startNo-1:
		//double [] co2oCnt = {0.0, Const.dCO2O, 0.0};
		co2o  = rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.matOInv.times(co2oCnt))));
		o2Cal = addCoords(coCal, co2o);	
	
		//compute the CA coordinate of residue startNo-1:
		//double [] co2caCnt = {0.0, 0.0, Const.dCA2CO};
		mat =  rgInv.times(Const.r9zAlphaInv.times(Const.r1xAlphaInv.times(Const.r6zAlphaInv.times(Const.r2xAlphaInv))));
		co2ca = mat.times(co2caCnt);
		caCal = addCoords(coCal, co2ca);
		
    	//add the atom coordinates of residue startNo-1:
    	Vector atomVec  = new Vector();
    	atomVec.add(new Cartesian("CA", caCal));	
		atomVec.add(new Cartesian("C", coCal));
		atomVec.add(new Cartesian("O", o2Cal));
		vecPdb.add(new Pdb(startNo-1, "ALA", atomVec));
		Collections.sort(vecPdb, new Pdb.PdbComparator());
		
		return vecPdb;
    }
    
    /**
     * Build model (compute the coordinates of backbone atoms (N, NH, CA, CO, O and HA) from
     * the standard geometry and Phi/Psi angle.
     * 
     * @param phiArr an array of Phi angles for the fragment.
     * @param psiArr an array of Psi angles for the fragment.
     * @param n1  the direction cosine for amide nitrogen of the first residue
     * @param nh1 the direction cosine for amide proton
     * @param ca1 the direction cosine for Ca
     * @param no1 the residue number of the first residue of a fragment
     * @param first  whether the residue is the first residue of the fragment
     * 
     * @return a vector of Pdb object
     */
    public Vector modelBuild(double[] phiArr, double[] psiArr, double[] n1, double[] nh1, double[] ca1, 
			     int no1, boolean first)
    {
		Vector tmpVec = new Vector();
		int i = 0, j = 0;
		double [] coordN  = new double[3]; 
		System.arraycopy(n1, 0, coordN, 0, 3);
		double [] coordNH = new double[3];
		System.arraycopy(nh1, 0, coordNH, 0, 3);
		double [] coordCA = new double[3]; 
		System.arraycopy(ca1, 0, coordCA, 0, 3);
		int no = no1; 
		String resid ="ALA";
		PhiPsi ff = new PhiPsi();
		double [] n2CA = new double[3];
		Vector atomVec = new Vector();
		Vector pdbVec = new Vector();
		Cartesian cc = new Cartesian();
	 	atomVec.add(new Cartesian("N", n1));
		atomVec.add(new Cartesian("H", nh1));
	 	atomVec.add(new Cartesian("CA", ca1));
		double [] nhVec1 = internuclearVec(n1, nh1);
		Matrix rg = Matrix.identity(3,3);
		//The first residue
		Pdb pp = coordByResidue(phiArr[0], psiArr[0], coordN, coordNH, coordCA, no, resid, first);
		atomVec.addAll(pp.getAtomVec());
		pdbVec.add(new Pdb(no, resid, atomVec));
		atomVec = new Vector();
		atomVec.add(new Cartesian("N",  coordN));
		atomVec.add(new Cartesian("H",  coordNH));
		atomVec.add(new Cartesian("CA", coordCA));
		//Compute New NH, N2CA and 
		nhVec1 = internuclearVec(coordN, coordNH); //update new NH1
		n2CA =   internuclearVec(coordN, coordCA);
		rg = ff.RgCal(nhVec1, n2CA);
		double[] co1ToN = new double[3];
		double[] co1    = new double[3];
		String atom = "";
		for (i=1; i<phiArr.length; i++)
		{
		    no++;
		    pp = coordByResidue(phiArr[i], psiArr[i], coordN, coordNH, coordCA, no, resid, false);
		    atomVec.addAll(pp.getAtomVec());
		    for (j=0; j<atomVec.size(); j++)
		    { //get CO coordinates BACK in case O and HA are there 
				cc = (Cartesian)atomVec.elementAt(j);
				atom = cc.getAtom();
				if (atom.equals("C"))
				    co1 = cc.getXYZ();
		    }
		    pdbVec.add(new Pdb(no, resid, atomVec));
		    atomVec = new Vector();
		    atomVec.add(new Cartesian("N",  coordN));
		    atomVec.add(new Cartesian("H",  coordNH));
		    atomVec.add(new Cartesian("CA", coordCA));
		    //Update NEW NH, N2CA and CO1ToN vectors
	 	    nhVec1 = internuclearVec(coordN, coordNH);
		    co1ToN = internuclearVec(coordN, co1);
		    n2CA   = internuclearVec(coordN, coordCA);
		   // if (i == phiArr.length - 1)
		    //	pdbVec.add(new Pdb(no+1, resid, atomVec));
		    rg = ff.RgCal(nhVec1, n2CA);
		}
		pdbVec.add(new Pdb(no+1, resid, atomVec));
		
		Vector vecPdbNew=new Vector();
		for (i=0;i< pdbVec.size();i++)
		{
			Pdb ppT=(Pdb)pdbVec.elementAt(i);
			vecPdbNew.add(ppT);
		}
		return vecPdbNew;
    }

    /**
     * Refine helix w4 rd cs.
     * 
     * @param vecBB the vec bb
     * @param rdc1Vec the rdc1 vec
     * @param rdc2Vec the rdc2 vec
     * @param helixRdcCaCoVec the helix rdc ca co vec
     * @param helixRdcCoNVec the helix rdc co n vec
     * @param Syy the syy
     * @param Szz the szz
     * @param ramaFilter the rama filter
     * @param phiAve the phi ave
     * @param psiAve the psi ave
     * @param refineCycle the refine cycle
     * @param initialCycle the initial cycle
     * @param w4Angles the w4 angles
     * @param resolution the resolution
     * @param debugDFS the debug dfs
     * @param printResults the print results
     * @param vecTalos the vec talos
     * 
     * @return the vector< pdb>
     * 
     * @throws JampackException the jampack exception
     */
    public Vector<Pdb> refineHelixW4RDCs(Vector vecBB, final Vector<Dipolar> rdc1Vec, final Vector<Dipolar> rdc2Vec, 
    		final Vector<Dipolar> helixRdcCaCoVec, final Vector<Dipolar> helixRdcCoNVec,double Syy, double Szz,
		       double[] ramaFilter, double phiAve, double psiAve, int refineCycle, int initialCycle, double w4Angles,
		       double resolution, boolean debugDFS, boolean printResults, Vector vecTalos)throws JampackException
	{
    	int i = 0;
		double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
		double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	
		//build an initial model
		Pdb pp = new Pdb();
		
		Dipolar dd1 = (Dipolar)rdc1Vec.elementAt(0);
		Dipolar dd2 = (Dipolar)rdc2Vec.elementAt(0);
		dd1 = (Dipolar)rdc1Vec.elementAt(rdc1Vec.size() - 1);
		dd2 = (Dipolar)rdc2Vec.elementAt(rdc2Vec.size() - 1);
		
		Pdb pp1=(Pdb)vecBB.elementAt(0);
		int firstResidueNo = pp1.getResidueNo();	
		Pdb pp2=(Pdb)vecBB.elementAt(vecBB.size()-1);
		int lastResidueNo  = pp2.getResidueNo();
			
		
		
		Vector<Pdb> pdbStrand=new Vector();
		pdbStrand.addAll(vecBB);
		
	
		//Compute the orientation of the first peptide by grid-search
		PdbRdc pdr = new PdbRdc();
		double [] rmsds = new double[4];
		Vector<Pdb> pdbStrandN = new Vector<Pdb>();
		Matrix[] mm = new Matrix[4];
		double[] rdc1Rmsd = new double[4];
		double[] rdc2Rmsd = new double[4];
		int N = lastResidueNo - firstResidueNo;
		double[] rdc1Cal = new double[N];
		double[] rdc2Cal = new double[N];
		double[] rdcCaCoCal = new double[N];
		double[] rdcCoNCal = new double[N];
		double rdc1Rms = 0.0, rdc2Rms = 0.0, rdcRms_caco=0.0, rdcRms_con=0.0; 
		Matrix[] mm_temp = new Matrix[4];
		double[] temp1=new double[1];
		int[] temp2=new int[1];		
		
		double [][] ss    = {{1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {0.0, 0.0, 1.0}}; //invert Z direction
	
		Matrix ssMat  = new Matrix(ss);
	 
		
		//only consider ch and nh rdcs during the grid search
		boolean debugEulerFit = false; //true; //rdc1Vec is NH and rdc2Vec is CH
		mm[0] = pdr.eulerFit(pdbStrand, rdc1Vec, rdc2Vec, Syy, Szz, rmsds, resolution, debugEulerFit);
		rdc1Rms =rmsds[0] ;// Math.sqrt(rmsds[0] / N);
		rdc2Rms =rmsds[1];// Math.sqrt(rmsds[1] / N);
		
	
		mm[0].print(14,14);
		if (printResults)
		{
		 System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2");
		 System.out.println(rdc1Rms+"  "+rdc2Rms);
		 mm[0].print(14, 14);
		}
		
		rdc1Rmsd[0] = pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H", mm[0], -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
		rdc2Rmsd[0] = pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",mm[0], -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
		rdc1Rms = rdc1Rmsd[0];
		rdc2Rms = rdc2Rmsd[0];
		double rms_save=rdc1Rms+rdc2Rms;
		int i_save=0;
			
		Vector<Pdb> helixVecN = new Vector<Pdb>();
		for (i = 1; i < 4; i++){
			mm[i] = Const.mat4ThreeDirs[i - 1].times(mm[0]);
			rdc1Rmsd[i] =pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H", mm[i], -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
			rdc2Rmsd[i] =pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",mm[i], -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
			
			System.out.println("i="+i);
			helixVecN = pp.newPdb(pdbStrand, Const.mat4ThreeDirs[i - 1]);
			mm_temp[0] = pdr.eulerFit(helixVecN, rdc1Vec, rdc2Vec, Syy, Szz, rmsds, resolution, debugEulerFit);
			rdc1Rms = rmsds[0]; 
			rdc2Rms = rmsds[1]; 
			
			if (rdc1Rms+rdc2Rms<rms_save)
			{
				
				rms_save=rdc1Rms+rdc2Rms;
				i_save=i;			
			}
			
			mm_temp[0].print(14,14);
			
			 System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2");
			 System.out.println(rdc1Rms+"  "+rdc2Rms);		
		}
		System.out.println("i_save=" +i_save);
		pdbStrandN = pp.newPdb(pdbStrand, mm[i_save]);//be careful here... almost the same, so we chose anyone
	
		Matrix mm2 = new Matrix(3, 3);
		int nCycle = 1; 
		debugDFS = true;
		printResults = true;
		boolean isHelix = true;
		Matrix iMat = Matrix.identity(3, 3);
		double[] saupe = new double[5];
		Vector<Pdb> pdbS2 = new Vector<Pdb>();
		for(i = 1; i< (refineCycle + 1); i++){
		  nCycle = initialCycle; //i * initialCycle;
		 pdbS2 = minHelix4RDCs(rdc2Vec, rdc1Vec,helixRdcCaCoVec,helixRdcCoNVec,rdc2Cal, rdc1Cal, pdbStrandN, Syy, Szz,
				     rdc2Rms, rdc1Rms, nCycle, w4Angles, debugDFS, printResults, isHelix,vecTalos,1.0,1.0);
		 
		 
		 mm2 = pdr.bestFit(pdbS2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);//only for printing the fit data, so do not use all four rdcs
		
		 rdc1Rms=saupe[2];
		 rdc2Rms=saupe[3];
		
		 pdbStrandN=new Vector();
		 pdbStrandN.addAll(pdbS2);
		 
		 System.out.println("refinecycle="+ i + ": " );
			
		 pp.print(pdbStrandN);
		 
	}
		
		System.out.println("====================================");
		
		return pdbStrandN;
	}
    
    /**
     * refine the Helix with the weights for CaCo NCO RDCs.
     * 
     * @param rdc1Vec NH RDCs
     * @param rdc2Vec CH RDCs
     * @param Syy the Saupe element
     * @param Szz the Saupe element
     * @param refineCycle the number of cycles for iterating through the refinement
     * and computation of Saupe matrices.
     * @param initialCycle  the number of initial cycles for the DFS-based systematic search.
     * @param w4Angles  weight for the \phi and \psi angles in the score funtion
     * @param resolution the resolution for the grid-search for the first peptide plane
     * @param debugDFS  for monitoring the progress in systematic search-based minimization
     * @param printResults for printing the results for each iteration
     * @param wtCoCa the weight for CoCa RDC
     * @param wtCoN the weight for NCo RDC
     * @param vecPreBB previous strcture
     * @param vecBB the vec bb
     * @param helixRdcCaCoVec the helix rdc ca co vec
     * @param helixRdcCoNVec the helix rdc co n vec
     * @param ramaFilter the rama filter
     * @param phiAve the phi ave
     * @param psiAve the psi ave
     * @param vecTalos the vec talos
     * @param starNo the star no
     * @param endNo the end no
     * @param vecSeq the vec seq
     * 
     * @return the refined Pdb vector
     * 
     * @throws JampackException the jampack exception
     */
    public Vector<Pdb> refineHelixW4RDCs(Vector vecPreBB,Vector vecBB, final Vector<Dipolar> rdc1Vec, final Vector<Dipolar> rdc2Vec, 
    		final Vector<Dipolar> helixRdcCaCoVec, final Vector<Dipolar> helixRdcCoNVec,double Syy, double Szz,
		       double[] ramaFilter, double phiAve, double psiAve, int refineCycle, int initialCycle,
		       double w4Angles, double resolution, boolean debugDFS, boolean printResults, Vector vecTalos,
		       double wtCoCa, double wtCoN, int starNo, int endNo, Vector vecSeq  )throws JampackException
	{
    	int i = 0;
		double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
		double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	
	    Pdb pp = new Pdb();	
		Vector<Pdb> pdbStrand=new Vector();
		pdbStrand.addAll(vecBB);
		
		//Compute the orientation of the first peptide by grid-search
		PdbRdc pdr = new PdbRdc();
		double [] rmsds = new double[4];
		Vector<Pdb> pdbStrandN = new Vector<Pdb>();
		Matrix[] mm = new Matrix[4];
		double[] rdc1Rmsd = new double[4];
		double[] rdc2Rmsd = new double[4];
		int N = vecBB.size();//endNo - starNo+1;
		double[] rdc1Cal = new double[N];
		double[] rdc2Cal = new double[N];
		double[] rdcCaCoCal = new double[N];
		double[] rdcCoNCal = new double[N];
		double rdc1Rms = 0.0, rdc2Rms = 0.0, rdcRms_caco=0.0, rdcRms_con=0.0; 
		Matrix[] mm_temp = new Matrix[4];
		double[] temp1=new double[1];
		int[] temp2=new int[1];		
		
		double [][] ss    = {{1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {0.0, 0.0, 1.0}}; 
	
		Matrix ssMat  = new Matrix(ss);	
		
		//only consider ch and nh rdcs during the grid search
		boolean debugEulerFit = false; //true; //rdc1Vec is NH and rdc2Vec is CH
		mm[0] = pdr.eulerFit(pdbStrand, rdc1Vec, rdc2Vec, Syy, Szz, rmsds, resolution, debugEulerFit);
		rdc1Rms =rmsds[0] ;
		rdc2Rms =rmsds[1];	
	
		mm[0].print(14,14);
		if (printResults)
		{
			 System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2");
			 System.out.println(rdc1Rms+"  "+rdc2Rms);
			 mm[0].print(14, 14);
		}
		
		rdc1Rmsd[0] = pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H", mm[0], -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
		rdc2Rmsd[0] = pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",mm[0], -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
		rdc1Rms = rdc1Rmsd[0];
		rdc2Rms = rdc2Rmsd[0];
		double rms_save=rdc1Rms+rdc2Rms;
		int i_save=0;
			
		Vector<Pdb> helixVecN = new Vector<Pdb>();
		
		System.out.println("i_save=" +i_save);
		pdbStrandN = pp.newPdb(pdbStrand, mm[i_save]);//be careful here... almost the same, so we chose anyone
		
		
		Matrix mm2 = new Matrix(3, 3);
		int nCycle = 1; 
		debugDFS = true;
		printResults = true;
		boolean isHelix = true;
		Matrix iMat = Matrix.identity(3, 3);
		double[] saupe = new double[5];
		Vector<Pdb> pdbS2 = new Vector<Pdb>();
		for(i = 1; i< (refineCycle + 1); i++){
		  nCycle = initialCycle; //i * initialCycle;
		 pdbS2 = minHelix4RDCs(vecPreBB,rdc2Vec, rdc1Vec,helixRdcCaCoVec,helixRdcCoNVec,rdc2Cal, rdc1Cal, pdbStrandN, Syy, Szz,
				     rdc2Rms, rdc1Rms, nCycle, w4Angles, debugDFS, printResults, isHelix,vecTalos,wtCoCa,wtCoN, starNo, endNo,vecSeq );
		 
		 
		 mm2 = pdr.bestFit(pdbS2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);//only for printing the fit data, so do not use all four rdcs
	
		 rdc1Rms=saupe[2];
		 rdc2Rms=saupe[3];
		 pdbStrandN=new Vector();
		 pdbStrandN.addAll(pdbS2);
		 
		 System.out.println("refinecycle="+ i + ": " );
			
		 pp.print(pdbStrandN);
		 
	}
		
		System.out.println("====================================");
		
		return pdbStrandN;
	}
       
    
    /**
     * Refine helix22.
     * 
     * @param vecBB the vec bb
     * @param rdc1Vec the rdc1 vec
     * @param rdc2Vec the rdc2 vec
     * @param Syy the syy
     * @param Szz the szz
     * @param ramaFilter the rama filter
     * @param phiAve the phi ave
     * @param psiAve the psi ave
     * @param refineCycle the refine cycle
     * @param initialCycle the initial cycle
     * @param w4Angles the w4 angles
     * @param resolution the resolution
     * @param debugDFS the debug dfs
     * @param printResults the print results
     * @param vecTalos the vec talos
     * 
     * @return the vector< pdb>
     * 
     * @throws JampackException the jampack exception
     */
    public Vector<Pdb> refineHelix22(Vector vecBB, final Vector<Dipolar> rdc1Vec, final Vector<Dipolar> rdc2Vec, double Syy, double Szz,
		       double[] ramaFilter, double phiAve, double psiAve, int refineCycle, int initialCycle,
		       double w4Angles, double resolution, boolean debugDFS, boolean printResults, Vector vecTalos)throws JampackException
	{
    	int i = 0;
		double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
		double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	
		//build an initial model
		Pdb pp = new Pdb();
		Vector<PhiPsi> phiPsiVec = new Vector<PhiPsi>();
		Dipolar dd1 = (Dipolar)rdc1Vec.elementAt(0);
		Dipolar dd2 = (Dipolar)rdc2Vec.elementAt(0);
		int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());		
		
		dd1 = (Dipolar)rdc1Vec.elementAt(rdc1Vec.size() - 1);
		dd2 = (Dipolar)rdc2Vec.elementAt(rdc2Vec.size() - 1);
		int lastResidueNo  =  Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;
		for (i=firstResidueNo; i<lastResidueNo; i++)  
			    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
		//Vector<Pdb> pdbStrand = modelBuild(phiPsiVec, n1, nh1, ca1);
		Vector<Pdb> pdbStrand=new Vector();
		pdbStrand.addAll(vecBB);
		
		//Compute the orientation of the first peptide by grid-search
		PdbRdc pdr = new PdbRdc();
		double [] rmsds = new double[2];
		Vector<Pdb> pdbStrandN = new Vector<Pdb>();
		Matrix[] mm = new Matrix[4];
		double[] rdc1Rmsd = new double[4];
		double[] rdc2Rmsd = new double[4];
		int N = lastResidueNo - firstResidueNo;
		double[] rdc1Cal = new double[N];
		double[] rdc2Cal = new double[N];
		double rdc1Rms = 0.0, rdc2Rms = 0.0; 
		Matrix[] mm_temp = new Matrix[4];
		double[] temp1=new double[1];
		int[] temp2=new int[1];		
		
		double [][] ss    = {{1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {0.0, 0.0, 1.0}}; //invert Z direction
	
		Matrix ssMat  = new Matrix(ss);
	 
		rdc1Rmsd[0] = pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H",ssMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
		rdc2Rmsd[0] = pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",ssMat, -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
		rdc1Rms = rdc1Rmsd[0];
		rdc2Rms = rdc2Rmsd[0];
	
	
		boolean debugEulerFit = false; 
		mm[0] = pdr.eulerFit(pdbStrand, rdc1Vec, rdc2Vec, Syy, Szz, rmsds, resolution, debugEulerFit);
		rdc1Rms =rmsds[0] ;
		rdc2Rms =rmsds[1];
		mm[0].print(14,14);
		if (printResults){
		 System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2");
		 System.out.println(rdc1Rms+"  "+rdc2Rms);
		 mm[0].print(14, 14);
		}
		
		rdc1Rmsd[0] = pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H", mm[0], -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
		rdc2Rmsd[0] = pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",mm[0], -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
		rdc1Rms = rdc1Rmsd[0];
		rdc2Rms = rdc2Rmsd[0];
		double rms_save=rdc1Rms+rdc2Rms;
		int i_save=0;
			
		Vector<Pdb> helixVecN = new Vector<Pdb>();
		for (i = 1; i < 4; i++){
			mm[i] = Const.mat4ThreeDirs[i - 1].times(mm[0]);
			rdc1Rmsd[i] =pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H", mm[i], -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
			rdc2Rmsd[i] =pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",mm[i], -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
			
			System.out.println("i="+i);
			helixVecN = pp.newPdb(pdbStrand, Const.mat4ThreeDirs[i - 1]);
			mm_temp[0] = pdr.eulerFit(helixVecN, rdc1Vec, rdc2Vec, Syy, Szz, rmsds, resolution, debugEulerFit);
			rdc1Rms = rmsds[0]; 
			rdc2Rms = rmsds[1]; 
			
			if (rdc1Rms+rdc2Rms<rms_save)
			{
				rms_save=rdc1Rms+rdc2Rms;
				i_save=i;			
			}
			
			mm_temp[0].print(14,14);
			
			 System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2");
			 System.out.println(rdc1Rms+"  "+rdc2Rms);
			 
			
		}
		System.out.println("i_save=" +i_save);
		pdbStrandN = pp.newPdb(pdbStrand, mm[i_save]);//be careful here... almost the same, so we chose anyone
		pp.print(pdbStrandN);
		
		Matrix mm2 = new Matrix(3, 3);
		int nCycle = 1; 
		debugDFS = true;
		printResults = true;
		boolean isHelix = true;
		Matrix iMat = Matrix.identity(3, 3);
		double[] saupe = new double[5];
		Vector<Pdb> pdbS2 = new Vector<Pdb>();
		for(i = 1; i< (refineCycle + 1); i++){
		  nCycle = initialCycle; //i * initialCycle;
		 pdbS2 = minHelix(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbStrandN, Syy, Szz,
				     rdc2Rms, rdc1Rms, nCycle, w4Angles, debugDFS, printResults, isHelix,vecTalos);
		
		 mm2 = pdr.bestFit(pdbS2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
		
		
		 rdc1Rms=saupe[2];
		 rdc2Rms=saupe[3];
		 pdbStrandN=new Vector();
		 pdbStrandN.addAll(pdbS2);
		 
		 System.out.println("refinecycle="+ i + ": " );
			
		 pp.print(pdbStrandN);
		 
	}
		
		System.out.println("====================================");
	
		return pdbStrandN;
	}
    
    /**
     * A recusive function to compute all the backbone Phi/Psi for an
     * n-residue helix or the first strand.
     * 
     * @param rdcVec1  an array of CH RDCs for the fragment
     * @param rdcVec2  an array of NH RDCs for the fragment
     * @param rdc1Cal  an array for storing the back-computed RDcs
     * @param rdc2Cal  an array for storing the back-computed RDcs
     * @param pdbVec the pdb vector previously computed or build from scratch.
     * @param debugDFS if true, print out the information for debugging
     * @param printResults if true, print out the information for debugging
     * @param isHelix if true, the last \psi angle is the average value for helix, otherwise, for beta
     * @param Syy the syy
     * @param Szz the szz
     * @param rmsd1 the rmsd1
     * @param rmsd2 the rmsd2
     * @param nCycle the n cycle
     * @param weightAngles the weight angles
     * 
     * @return the computed structure for the fragment
     */
    public Vector<Pdb> minHelix (Vector<Dipolar> rdcVec1, Vector<Dipolar> rdcVec2, double[] rdc1Cal, double[] rdc2Cal, Vector<Pdb> pdbVec,
			    double Syy, double Szz, double rmsd1, double rmsd2, int nCycle,
			    double weightAngles, boolean debugDFS, boolean printResults, boolean isHelix)
   {  
    
    	
    int i, j, k;
	final int N = pdbVec.size() - 1;     //the number of CH RDC
	Dipolar dd1 = (Dipolar)rdcVec1.elementAt(0);
	Dipolar dd2 = (Dipolar)rdcVec2.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());
	
	double [] rdcArr1 = new double[N]; //CH RDCs, sampled based on Gaussina Dis, use array for efficient 
	double [] rdcArr2 = new double[N]; //NH RDCs, use array for efficient 
	double [] rdcExp1 = new double[N]; //store the experimental values in the array
	double [] rdcExp2 = new double[N];
	double [] rdcErr1 = new double[N]; //store the experimental errs in the array
	double [] rdcErr2 = new double[N];
	int [] seqNos1 = new int[N];
	int [] seqNos2 = new int[N];
	int no = 0;    
	int index = -1;
	//Filled in missing RDCs
	for (j = 0; j < N; j++)
	{
	    no = firstResidueNo+j;
	    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1)
	    {
	    	rdcExp1[j] = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
	    	rdcErr1[j]=((Dipolar)rdcVec1.elementAt(index)).getErr();
	    	seqNos1[j] = 1;
	    }else 
	    {
	    	rdcExp1[j] =  rdc1Cal[j];
	    	seqNos1[j] = 0;
	    	rdcErr1[j]=rmsd1;
	    }
	    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1)
	    {
	    	rdcExp2[j] = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
	    	seqNos2[j] = 1;
	    	rdcErr2[j]=((Dipolar)rdcVec2.elementAt(index)).getErr();
	    }else
	    {
	    	rdcExp2[j] =  rdc2Cal[j];
	    	seqNos2[j] = 0;
	    	rdcErr2[j]=rmsd2;
	    }
	}//for (j = 0; j < N; j++)
	
	Cartesian cc = new Cartesian();
	String atom = "";
	double [] amide = new double[3];
	double [] ca = new double[3];
	double [] nh = new double[3];
	Pdb pp = pdbVec.elementAt(0);
	Vector<Cartesian> atomVec = pp.getAtomVec();
        for (j=0; j<atomVec.size(); j++){
            cc = atomVec.elementAt(j);
            atom = cc.getAtom();
            if (atom.equals("N"))
                amide = cc.getXYZ();
            else if (atom.equals("H"))
                nh = cc.getXYZ();
            else if (atom.equals("CA"))
                ca = cc.getXYZ();
        }
	double[] nToNHVec = internuclearVec(amide, nh);
        double[] nToCAVec = internuclearVec(amide, ca);

        ppGlobal ppg = new ppGlobal();
        Matrix mat = ppg.RgCal(nToNHVec, nToCAVec);

	boolean flag1 = false;
	double rRdc11 = 10000.0, rRdc12 = 10000.0, rT = 100000.0, rTotal = 100000.0;

	double rms1 = 0.0, rms2 = 0.0;
	double u, v; //for CH and NH RDC respectively
	int depth0 = 0;
	Vector<Double> ppVec = new Vector<Double>();
	Vector depthVec = new Vector();
	double[] ppS  = new double[2*N]; //N1 = 5
	double[] phiS = new double[N]; 
	double[] psiS = new double[N]; 
	double[] phiSave = new double[N]; 
	double[] psiSave = new double[N]; 
	double[] rdc1Save = new double[N];   //CH Rdcs
	double[] rdc2Save = new double[N]; 

	int noOfSln = 0;
	int n = 0;	
	int N11 =  rdcVec1.size();
	int N22 =  rdcVec2.size();
	double phiRmsd = 0.0;
	double psiRmsd = 0.0;
	double rRdcFinal1 = rmsd1 * rmsd1 * N11; // that's the total for all residues, we want 2 fold better
	double rRdcFinal2 = rmsd2 * rmsd2 * N22; // the pairs should has quite close values

	//Sample the RDC to select a set satisfy certain conditions. specified later
	long startTime = System.currentTimeMillis();
	boolean depthFlag = true;

	boolean rightHand = true;
	PhiPsi ff = new PhiPsi();
	double phiAve = 0.0;
	double psiAve = 0.0;
	if(isHelix){
	    phiAve = Const.phiAveHelix;
	    psiAve = Const.psiAveHelix;
	}else{
	    phiAve = Const.phiAveBeta;
	    psiAve = Const.psiAveBeta;
	}
	boolean isChanged =false;
	long seed =10051; 
		Random rr = new Random(seed);
  	for (int mmm =0; mmm < nCycle; mmm++){  	   
  		
	    flag1 = false;
	    while (!flag1){
		flag1 = true;
		rRdc11 = 0.0;
		rRdc12 = 0.0;
		for (j = 0; j < N; j++){   //For the 1st beta strand of the pair
			
		    rdcArr1[j] = rdcExp1[j] + rmsd1 * rr.nextGaussian();// rmsd1 * rr.nextGaussian(); 
		    
		    rdcArr2[j] = rdcExp2[j] +rmsd2 * rr.nextGaussian() ;//rmsd2 * rr.nextGaussian();
		    rms1 = Math.abs(rdcArr1[j] - rdcExp1[j]);
		    rRdc11 += rms1 * rms1 * seqNos1[j] ;
		    rms2 = Math.abs(rdcArr2[j] - rdcExp2[j]);
		    rRdc12 += rms2 * rms2 * seqNos2[j] ;
		}
		if ( (rRdc11 > rRdcFinal1) || (rRdc12 > rRdcFinal2) )
		    flag1 = false;
		else flag1 = true;
	    }//end of while
	    ppVec = new Vector<Double>();
	    depthFlag = false;
    	depthFlag = ff.phiPsiChain(rdcArr1, rdcArr2, mat, Syy, Szz, depth0, N - 1, ppS, ppVec, rightHand, isHelix);
	    if (ppVec.size() > 0 ){

		noOfSln = (int)ppVec.size() / (2*N);  //the total number of solutions for this rdc set
		
		for(k=0; k < noOfSln; k++)
		{   //please note the index
		    phiRmsd = 0.0;
		    psiRmsd = 0.0;
		    for (n = k* 2 * N; n < (k+1)* 2 *N; n += 2){  //original "N"
			phi = ppVec.elementAt(n).doubleValue();
			psi = ppVec.elementAt(n+1).doubleValue();
			phiS[(int)(n - k* 2 * N) / 2] = phi;
			psiS[(int)(n - k* 2 * N) / 2] = psi;
			phiRmsd += (phi - phiAve) * (phi - phiAve);
			psiRmsd += (psi - psiAve) * (psi - psiAve);
		    }
		    rT = Math.sqrt(rRdc11 / N11 ) +  (Math.sqrt(rRdc12 / (N22 - 1) )) + 
			weightAngles*(Math.sqrt(phiRmsd / N ) + Math.sqrt(psiRmsd / N));
		    if (rT < rTotal){
		    isChanged=true;//added by zeng
			rTotal = rT;
			System.out.println( mmm +": "+Math.sqrt(rRdc11 / N11 )+"  "
					    + Math.sqrt(rRdc12 / (N22 - 1) )+";  "+(Math.sqrt(phiRmsd / N ) / Const.cst) 
					    + "  "+(Math.sqrt(psiRmsd / N) / Const.cst)+", RT = "+rTotal);
			System.arraycopy(phiS, 0, phiSave, 0, N);
			System.arraycopy(psiS, 0, psiSave, 0, N);
			System.arraycopy(rdcArr1, 0, rdc1Save, 0, N);
			System.arraycopy(rdcArr2, 0, rdc2Save, 0, N);
		    }
		}
	    }
	}
	long endTime = System.currentTimeMillis() ;
	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes

 	Vector<PhiPsi> phiPsiVec  = new Vector<PhiPsi>();
  	for (i=0; i<N; i++)  //build the model
       	    phiPsiVec.add(new PhiPsi(i, "ALA", phiSave[i], psiSave[i]));
   	Vector<Pdb> pdbHelixN = new Vector();
   	if (isChanged)
   		pdbHelixN= 	modelBuild(phiSave, psiSave, amide, nh, ca, firstResidueNo, false);
   	else
   	{
   		pdbHelixN.addAll(pdbVec);
   		System.out.println("no better solution is found....");
   	}
   	
	if(printResults){
            System.out.println(" CH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc1Save[k] +"  "+rdcExp1[k]+"  "+Math.abs(rdc1Save[k]-rdcExp1[k]));
            System.out.println(" NH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc2Save[k] +"  "+rdcExp2[k]+"  "+ Math.abs(rdc2Save[k]-rdcExp2[k]));

            System.out.println(" Phi/Psi: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+"   "+(phiSave[k]/Const.cst)+"  "+(psiSave[k]/Const.cst));
        }
	return pdbHelixN;
    }
    
    /**
     * A recusive function to compute all the backbone Phi/Psi for an
     * n-residue helix or the first strand.
     * 
     * @param rdcVec1  an array of CH RDCs for the fragment
     * @param rdcVec2  an array of NH RDCs for the fragment
     * @param rdc1Cal  an array for storing the back-computed RDcs
     * @param rdc2Cal  an array for storing the back-computed RDcs
     * @param pdbVec the pdb vector previously computed or build from scratch.
     * @param debugDFS if true, print out the information for debugging
     * @param printResults if true, print out the information for debugging
     * @param isHelix if true, the last \psi angle is the average value for helix, otherwise, for beta
     * @param Syy the syy
     * @param Szz the szz
     * @param rmsd1 the rmsd1
     * @param rmsd2 the rmsd2
     * @param nCycle the n cycle
     * @param weightAngles the weight angles
     * @param vecTalos the vec talos
     * 
     * @return the computed structure for the fragment
     * 
     * @throws JampackException the jampack exception
     */
    public Vector<Pdb> minHelix (Vector<Dipolar> rdcVec1, Vector<Dipolar> rdcVec2, double[] rdc1Cal, double[] rdc2Cal, Vector<Pdb> pdbVec,
			    double Syy, double Szz, double rmsd1, double rmsd2, int nCycle,
			    double weightAngles, boolean debugDFS, boolean printResults, boolean isHelix,Vector vecTalos)throws JampackException
    {     
    int i, j, k;
	final int N = pdbVec.size() - 1;     //the number of CH RDC
	Dipolar dd1 = (Dipolar)rdcVec1.elementAt(0);
	Dipolar dd2 = (Dipolar)rdcVec2.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());
	
	double [] rdcArr1 = new double[N]; //CH RDCs, sampled based on Gaussina Dis, use array for efficient 
	double [] rdcArr2 = new double[N]; //NH RDCs, use array for efficient 
	double [] rdcExp1 = new double[N]; //store the experimental values in the array
	double [] rdcExp2 = new double[N];
	double [] rdcErr1 = new double[N]; //store the experimental errs in the array
	double [] rdcErr2 = new double[N];
	int [] seqNos1 = new int[N];
	int [] seqNos2 = new int[N];
	int no = 0;    
	int index = -1;
	//Filled in missing RDCs
	for (j = 0; j < N; j++)
	{
	    no = firstResidueNo+j;
	    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1)
	    {
	    	rdcExp1[j] = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
	    	rdcErr1[j]=((Dipolar)rdcVec1.elementAt(index)).getErr();
	    	seqNos1[j] = 1;
	    }else 
	    {
	    	rdcExp1[j] =  rdc1Cal[j];
	    	seqNos1[j] = 0;
	    	rdcErr1[j]=rmsd1;
	    }
	    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1)
	    {
	    	rdcExp2[j] = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
	    	seqNos2[j] = 1;
	    	rdcErr2[j]=((Dipolar)rdcVec2.elementAt(index)).getErr();
	    }else
	    {
	    	rdcExp2[j] =  rdc2Cal[j];
	    	seqNos2[j] = 0;
	    	rdcErr2[j]=rmsd2;
	    }
	}//for (j = 0; j < N; j++)
	
	Cartesian cc = new Cartesian();
	String atom = "";
	double [] amide = new double[3];
	double [] ca = new double[3];
	double [] nh = new double[3];
	Pdb pp = pdbVec.elementAt(0);
	Vector<Cartesian> atomVec = pp.getAtomVec();
        for (j=0; j<atomVec.size(); j++){
            cc = atomVec.elementAt(j);
            atom = cc.getAtom();
            if (atom.equals("N"))
                amide = cc.getXYZ();
            else if (atom.equals("H"))
                nh = cc.getXYZ();
            else if (atom.equals("CA"))
                ca = cc.getXYZ();
        }
	double[] nToNHVec = internuclearVec(amide, nh);
        double[] nToCAVec = internuclearVec(amide, ca);

        ppGlobal ppg = new ppGlobal();
        Matrix mat = ppg.RgCal(nToNHVec, nToCAVec);

	boolean flag1 = false;
	double rRdc11 = 10000.0, rRdc12 = 10000.0, rT = 100000.0, rTotal = 100000.0;
	
	double rms1 = 0.0, rms2 = 0.0;
	double u, v; //for CH and NH RDC respectively
	int depth0 = 0;
	Vector<Double> ppVec = new Vector<Double>();
	Vector depthVec = new Vector();
	double[] ppS  = new double[2*N]; //N1 = 5
	double[] phiS = new double[N]; 
	double[] psiS = new double[N]; 
	double[] phiSave = new double[N]; 
	double[] psiSave = new double[N]; 
	double[] rdc1Save = new double[N];   //CH Rdcs
	double[] rdc2Save = new double[N]; 

	int noOfSln = 0;
	int n = 0;	
	int N11 =  rdcVec1.size();
	int N22 =  rdcVec2.size();
	double phiRmsd = 0.0;
	double psiRmsd = 0.0;
	double rRdcFinal1 = rmsd1 * rmsd1 * N11; // that's the total for all residues, we want 2 fold better
	double rRdcFinal2 = rmsd2 * rmsd2 * N22; // the pairs should has quite close values

	//Sample the RDC to select a set satisfy certain conditions. specified later
	long startTime = System.currentTimeMillis();
	boolean depthFlag = true;

	boolean rightHand = true;
	PhiPsi ff = new PhiPsi();
	double phiAve = 0.0;
	double psiAve = 0.0;
	if(isHelix){
	    phiAve = Const.phiAveHelix;
	    psiAve = Const.psiAveHelix;
	}else{
	    phiAve = Const.phiAveBeta;
	    psiAve = Const.psiAveBeta;
	}
	boolean isChanged =false;
	long seed =99933;//88091;
	Random rr = new Random(seed);
	String userDir = System.getProperty("user.dir");////
    String src=userDir+"/inputFiles/";
    	
	vdw vander = new vdw();
	Vector vdwVec = new Vector();
  	for (int mmm =0; mmm < nCycle; mmm++){
  		
	    flag1 = false;
	    while (!flag1){
		flag1 = true;
		rRdc11 = 0.0;
		rRdc12 = 0.0;
		for (j = 0; j < N; j++){   //For the 1st beta strand of the pair
			
		    rdcArr1[j] = rdcExp1[j] + rmsd1 * rr.nextGaussian();
		    
		    rdcArr2[j] = rdcExp2[j] +rmsd2 * rr.nextGaussian() ;
		    rms1 = Math.abs(rdcArr1[j] - rdcExp1[j]);
		    rRdc11 += rms1 * rms1 * seqNos1[j] ;
		    rms2 = Math.abs(rdcArr2[j] - rdcExp2[j]);
		    rRdc12 += rms2 * rms2 * seqNos2[j] ;
		}
		if ( (rRdc11 > rRdcFinal1) || (rRdc12 > rRdcFinal2) )
		    flag1 = false;
		else flag1 = true;
	    }//end of while
	    ppVec = new Vector<Double>();
	    depthFlag = false;
    	depthFlag = ff.phiPsiChainTalos(rdcArr1, rdcArr2, mat, Syy, Szz, depth0, N - 1, ppS, ppVec, rightHand, isHelix,firstResidueNo,vecTalos);
	    if (ppVec.size() > 0 )
	    {

		noOfSln = (int)ppVec.size() / (2*N);  //the total number of solutions for this rdc set
		
		for(k=0; k < noOfSln; k++)
		{   //please note the index
		    phiRmsd = 0.0;
		    psiRmsd = 0.0;
		    for (n = k* 2 * N; n < (k+1)* 2 *N; n += 2)
		    {  //original "N"
			phi = ppVec.elementAt(n).doubleValue();
			psi = ppVec.elementAt(n+1).doubleValue();
			phiS[(int)(n - k* 2 * N) / 2] = phi;
			psiS[(int)(n - k* 2 * N) / 2] = psi;
			phiRmsd += (phi - phiAve) * (phi - phiAve);
			psiRmsd += (psi - psiAve) * (psi - psiAve);
		    }
		    
		    rT = Math.sqrt(rRdc11 / N11 ) +  (Math.sqrt(rRdc12 / N22 )) + 
			weightAngles*(Math.sqrt(phiRmsd / N ) + Math.sqrt(psiRmsd / N));
		    if (rT < rTotal)
		    {
		    	//check the steric clashes
		    	Vector<PhiPsi> phiPsiVec_temp  = new Vector<PhiPsi>();
		      	for (i=0; i<N; i++)  //build the model
		      		phiPsiVec_temp.add(new PhiPsi(i, "ALA", phiS[i], psiS[i]));
		       	Vector<Pdb> pdbHelixTemp = new Vector();
		       	
		       	pdbHelixTemp= 	modelBuild(phiS, psiS, amide, nh, ca, firstResidueNo, false);

		       	boolean[] resIndex=new boolean[pdbHelixTemp.size()];
	    		
	    		for (int kk=0;kk<resIndex.length;kk++)
	    			resIndex[kk]=false;
	    		Vector vecPdbSseRot2=new Vector();
	    		String rotSrc=  src+ "rotasamp-small/"; 
		       	vecPdbSseRot2=pp.AlaninizeStructure(pdbHelixTemp,resIndex,rotSrc);
	    		

		       	double[] vdwValue = new double[1];
		       	double vdwLevel = 0.05;
		       	boolean printVDWViolation = false;
		       	int[] clashResNo=new int[2];
		    	vdwVec = vander.convert2VDW(vecPdbSseRot2);
		    	boolean isStericClash=vander.checkStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, 
						true, 1.0, 1, clashResNo); 
		    	
		    	if(isStericClash)
		    	{
		    		System.out.println("here we have one steric clash...");
		    		continue;
		    	}
		    	//to add vdw checking here...
			    isChanged=true;//added by zeng
				rTotal = rT;
				System.out.println( mmm +": "+Math.sqrt(rRdc11 / N11 )+"  "
						    + Math.sqrt(rRdc12 / (N22 - 1) )+";  "+(Math.sqrt(phiRmsd / N ) / Const.cst) 
						    + "  "+(Math.sqrt(psiRmsd / N) / Const.cst)+", RT = "+rTotal);
				System.arraycopy(phiS, 0, phiSave, 0, N);
				System.arraycopy(psiS, 0, psiSave, 0, N);
				System.arraycopy(rdcArr1, 0, rdc1Save, 0, N);
				System.arraycopy(rdcArr2, 0, rdc2Save, 0, N);
		    }
		}
	    }
	}
	long endTime = System.currentTimeMillis() ;
	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes

 	Vector<PhiPsi> phiPsiVec  = new Vector<PhiPsi>();
  	for (i=0; i<N; i++)  //build the model
       	    phiPsiVec.add(new PhiPsi(i, "ALA", phiSave[i], psiSave[i]));
   	Vector<Pdb> pdbHelixN = new Vector();
   	if (isChanged)
   		pdbHelixN= 	modelBuild(phiSave, psiSave, amide, nh, ca, firstResidueNo, false);
   	else
   	{
   		pdbHelixN.addAll(pdbVec);
   		System.out.println("no better solution is found....");
   	}
   	
	if(printResults){
            System.out.println(" CH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc1Save[k] +"  "+rdcExp1[k]+"  "+Math.abs(rdc1Save[k]-rdcExp1[k]));
            System.out.println(" NH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc2Save[k] +"  "+rdcExp2[k]+"  "+ Math.abs(rdc2Save[k]-rdcExp2[k]));

            System.out.println(" Phi/Psi: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+"   "+(phiSave[k]/Const.cst)+"  "+(psiSave[k]/Const.cst));
        }
	return pdbHelixN;
    }
    
    /**
     * A recusive function to compute all the backbone Phi/Psi for an
     * n-residue helix or the first strand.
     * 
     * @param rdcVec1  an array of CH RDCs for the fragment
     * @param rdcVec2  an array of NH RDCs for the fragment
     * @param rdc1Cal  an array for storing the back-computed RDcs
     * @param rdc2Cal  an array for storing the back-computed RDcs
     * @param pdbVec the pdb vector previously computed or build from scratch.
     * @param debugDFS if true, print out the information for debugging
     * @param printResults if true, print out the information for debugging
     * @param isHelix if true, the last \psi angle is the average value for helix, otherwise, for beta
     * @param vecTalos talos angle table
     * @param wtCoCa weight for CaCo rdc
     * @param wtCoN weight for CoN RDC
     * @param helixRdcCaCoVec the helix rdc ca co vec
     * @param helixRdcCoNVec the helix rdc co n vec
     * @param Syy the syy
     * @param Szz the szz
     * @param rmsd1 the rmsd1
     * @param rmsd2 the rmsd2
     * @param nCycle the n cycle
     * @param weightAngles the weight angles
     * 
     * @return the computed structure for the fragment
     * 
     * @throws JampackException the jampack exception
     */
    public Vector<Pdb> minHelix4RDCs (Vector<Dipolar> rdcVec1, Vector<Dipolar> rdcVec2,
    		final Vector<Dipolar> helixRdcCaCoVec, final Vector<Dipolar> helixRdcCoNVec, double[] rdc1Cal,
    		double[] rdc2Cal, Vector<Pdb> pdbVec, double Syy, double Szz, double rmsd1, double rmsd2, 
    		int nCycle,double weightAngles, boolean debugDFS, boolean printResults, boolean isHelix,
    		 Vector vecTalos, double wtCoCa, double wtCoN)throws JampackException
    {     
	    int i, j, k;
		int N = pdbVec.size()-1;// - 1;     //the number of CH RDC
		Dipolar dd1 = (Dipolar)rdcVec1.elementAt(0);
		Dipolar dd2 = (Dipolar)rdcVec2.elementAt(0);
		Pdb pdbTemp=pdbVec.elementAt(0);
		int firstResidueNo=pdbTemp.getResidueNo();
		
		
		PdbRdc pdr =new PdbRdc();
		double [] rdcArr1 = new double[N]; //CH RDCs, sampled based on Gaussina Dis, use array for efficient 
		double [] rdcArr2 = new double[N]; //NH RDCs, use array for efficient 
		double [] rdcExp1 = new double[N]; //store the experimental values in the array
		double [] rdcExp2 = new double[N];
		double [] rdcExpCaCo = new double[N];
		double [] rdcExpCoN = new double[N];
		
		double [] rdcErr1 = new double[N]; //store the experimental errs in the array
		double [] rdcErr2 = new double[N];
		double [] rdcErrCaCo = new double[N];
		double [] rdcErrCoN = new double[N];
		int [] seqNos1 = new int[N];
		int [] seqNos2 = new int[N];
		int [] seqNosCaCo = new int[N];
		int [] seqNosCoN = new int[N];
		int no = 0;    
		int index = -1;
		//Filled in missing RDCs
		for (j = 0; j < N; j++)
		{
		    no = firstResidueNo+j;
		    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
		    if (index > - 1)
		    {
		    	rdcExp1[j] = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
		    	rdcErr1[j]=((Dipolar)rdcVec1.elementAt(index)).getErr();
		    	seqNos1[j] = 1;
		    }else 
		    {
		    	rdcExp1[j] =  rdc1Cal[j];
		    	seqNos1[j] = 0;
		    	rdcErr1[j]=rmsd1;
		    }
		    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
		    if (index > - 1)
		    {
		    	rdcExp2[j] = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
		    	seqNos2[j] = 1;
		    	rdcErr2[j]=((Dipolar)rdcVec2.elementAt(index)).getErr();
		    }else
		    {
		    	rdcExp2[j] =  rdc2Cal[j];
		    	seqNos2[j] = 0;
		    	rdcErr2[j]=rmsd2;
		    }
		    
		    index = Collections.binarySearch(helixRdcCaCoVec, new Dipolar(no), new Dipolar.rdcComparator());
		    if (index > - 1)
		    {
		    	rdcExpCaCo[j] = ((Dipolar)helixRdcCaCoVec.elementAt(index)).getRdc();
		    	seqNosCaCo[j] = 1;
		    	rdcErrCaCo[j]=((Dipolar)helixRdcCaCoVec.elementAt(index)).getErr();
		    }else
		    {
		    	rdcExpCaCo[j] = 0.0;//not used actually
		    	seqNosCaCo[j] = 0;
		    	rdcErrCaCo[j]=0.0;//haven't been used yet
		    }
		    
		    index = Collections.binarySearch(helixRdcCoNVec, new Dipolar(no), new Dipolar.rdcComparator());
		    if (index > - 1)
		    {
		    	rdcExpCoN[j] = ((Dipolar)helixRdcCoNVec.elementAt(index)).getRdc();
		    	seqNosCoN[j] = 1;
		    	rdcErrCoN[j]=((Dipolar)helixRdcCoNVec.elementAt(index)).getErr();
		    }else
		    {
		    	rdcExpCoN[j] = 0.0;//not used actually
		    	seqNosCoN[j] = 0;
		    	rdcErrCoN[j]=0.0;//haven't been used yet
		    }
		}//for (j = 0; j < N; j++)
		
		Cartesian cc = new Cartesian();
		String atom = "";
		double [] amide = new double[3];
		double [] ca = new double[3];
		double [] nh = new double[3];
		Pdb pp = pdbVec.elementAt(0);
		
		//debugg:
		String userDir2 = System.getProperty("user.dir");////
		String src2=userDir2+"/inputFiles/";    		
		String strH1File=src2+"H1.pdb";
		Vector<Pdb> ppVecH1 = pp.readPdb(strH1File);
		pp=new Pdb();
		pp=ppVecH1.elementAt(12);
		/////////////////
		
		
		Vector<Cartesian> atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++)
	    {
            cc = atomVec.elementAt(j);
            atom = cc.getAtom();
            if (atom.equals("N"))
                amide = cc.getXYZ();
            else if (atom.equals("H"))
                nh = cc.getXYZ();
            else if (atom.equals("CA"))
                ca = cc.getXYZ();
        }
	    double[] nToNHVec = internuclearVec(amide, nh);
        double[] nToCAVec = internuclearVec(amide, ca);
        //      rg = RgCal(nToNHVec, nToCAVec);
        ppGlobal ppg = new ppGlobal();
        Matrix mat = ppg.RgCal(nToNHVec, nToCAVec);

		boolean flag1 = false;
		double rRdc11 = 10000.0, rRdc12 = 10000.0, rT = 100000.0, rTotal = 100000.0;

		double rms1 = 0.0, rms2 = 0.0;
		double u, v; //for CH and NH RDC respectively
		int depth0 = 0;
		Vector<Double> ppVec = new Vector<Double>();
		Vector depthVec = new Vector();
		double[] ppS  = new double[2*N]; //N1 = 5
		double[] phiS = new double[N]; 
		double[] psiS = new double[N]; 
		double[] phiSave = new double[N]; 
		double[] psiSave = new double[N]; 
		double[] rdc1Save = new double[N];   //CH Rdcs
		double[] rdc2Save = new double[N]; 
		double[] rdcCaCoSave = new double[N]; 
		double[] rdcCoNSave = new double[N]; 

		int noOfSln = 0;
		int n = 0;	
		int N11 =  rdcVec1.size();
		int N22 =  rdcVec2.size();
		double phiRmsd = 0.0;
		double psiRmsd = 0.0;
		double rRdcFinal1 = rmsd1 * rmsd1 * N11; // that's the total for all residues, we want 2 fold better
		double rRdcFinal2 = rmsd2 * rmsd2 * N22; // the pairs should has quite close values
	
		long startTime = System.currentTimeMillis();
		boolean depthFlag = true;

		boolean rightHand = true;
		PhiPsi ff = new PhiPsi();
		double phiAve = 0.0;
		double psiAve = 0.0;
		if(isHelix)
		{
		    phiAve = Const.phiAveHelix;
		    psiAve = Const.psiAveHelix;
		}
		else
		{
		    phiAve = Const.phiAveBeta;
		    psiAve = Const.psiAveBeta;
		}
		boolean isChanged =false;
		long seed =999991;//88091;
		Random rr = new Random(seed);
		String userDir = System.getProperty("user.dir");////
	    String src=userDir+"/inputFiles/";
	    	
		vdw vander = new vdw();
		Vector vdwVec = new Vector();
	  	for (int mmm =0; mmm < nCycle; mmm++)
	  	{
	  		
		    flag1 = false;
		    while (!flag1){
			flag1 = true;
			rRdc11 = 0.0;
			rRdc12 = 0.0;
			for (j = 0; j < N; j++)
			{   //For the 1st beta strand of the pair			
			    rdcArr1[j] = rdcExp1[j] + rmsd1 * rr.nextGaussian();// rmsd1 * rr.nextGaussian(); 		    
			    rdcArr2[j] = rdcExp2[j] +rmsd2 * rr.nextGaussian() ;//rmsd2 * rr.nextGaussian();
			    rms1 = Math.abs(rdcArr1[j] - rdcExp1[j]);
			    rRdc11 += rms1 * rms1 * seqNos1[j] ;
			    rms2 = Math.abs(rdcArr2[j] - rdcExp2[j]);
			    rRdc12 += rms2 * rms2 * seqNos2[j] ;
			}
			if ( (rRdc11 > rRdcFinal1) || (rRdc12 > rRdcFinal2) )
			    flag1 = false;
			else flag1 = true;
		    }//end of while
		    ppVec = new Vector<Double>();
		    depthFlag = false;
	    	depthFlag = ff.phiPsiChainTalos(rdcArr1, rdcArr2, mat, Syy, Szz, depth0, N - 1, ppS, ppVec, rightHand, isHelix,firstResidueNo,vecTalos);
		    if (ppVec.size() > 0 )
		    {
		    	
				noOfSln = (int)ppVec.size() / (2*N);  //the total number of solutions for this rdc set
					
				for(k=0; k < noOfSln; k++)
				{   //please note the index
				    phiRmsd = 0.0;
				    psiRmsd = 0.0;
				    for (n = k* 2 * N; n < (k+1)* 2 *N; n += 2)
				    {  //original "N"
						phi = ppVec.elementAt(n).doubleValue();
						psi = ppVec.elementAt(n+1).doubleValue();
						phiS[(int)(n - k* 2 * N) / 2] = phi;
						psiS[(int)(n - k* 2 * N) / 2] = psi;
						phiRmsd += (phi - phiAve) * (phi - phiAve);
						psiRmsd += (psi - psiAve) * (psi - psiAve);
				    }
				    		
				    Vector<Pdb> pdbHelixTemp = new Vector();			       	
			       	pdbHelixTemp= 	modelBuild(phiS, psiS, amide, nh, ca, firstResidueNo, false);
			       	double [][] Inv_temp = {{1.0, 0.0, 0.0},{0.0, 1.0, 0.0},{0.0, 0.0, 1.0}};
			       	Matrix Mat  = new Matrix(Inv_temp);
			       	double[] sRmsd= new double[1];
					int[] sizeRdc=new int[1];
					double[] rdcCoNCal  = new double[pdbHelixTemp.size() ];
					double[] rdcCaCoCal  = new double[pdbHelixTemp.size() ];
			       
					double r_temp=pdr.BackCal(pdbHelixTemp, rdcVec1, "CA", "HA", Mat, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdcCaCoCal,sRmsd, sizeRdc);
				    					
					double r_caco=0.0;
					double r_con=0.0;
					if(helixRdcCaCoVec.size()>0)
						r_caco=pdr.BackCalCACO(pdbHelixTemp, helixRdcCaCoVec, "C", "CA", Mat, -Syy-Szz, Syy, Szz, Const.cacoRatio, rdcCaCoCal,sRmsd, sizeRdc,false);
					if(helixRdcCoNVec.size()>0)
						r_con=pdr.BackCalCON(pdbHelixTemp, helixRdcCoNVec, "C", "N",  Mat, -Syy-Szz, Syy, Szz, Const.conRatio, rdcCoNCal,sRmsd, sizeRdc,false);
					
				  
				    rT = Math.sqrt((rRdc11/Const.cahaRatio) / N11 ) +  (Math.sqrt((rRdc12/Const.nhRatio) / N22 )) + wtCoCa* (r_caco/Const.cacoRatio) +
				    wtCoN*(r_con/Const.conRatio)+ weightAngles*(Math.sqrt(phiRmsd / N ) + Math.sqrt(psiRmsd / N));
				    if (rT < rTotal)
				    {
				    	//check the steric clashes
				    	Vector<PhiPsi> phiPsiVec_temp  = new Vector<PhiPsi>();
				      	for (i=0; i<N; i++)  //build the model
				      		phiPsiVec_temp.add(new PhiPsi(i, "ALA", phiS[i], psiS[i]));				       	
		
				       	boolean[] resIndex=new boolean[pdbHelixTemp.size()];
			    		
			    		for (int kk=0;kk<resIndex.length;kk++)
			    			resIndex[kk]=false;
			    		
			    		
			    		
			    		Vector vecPdbSseRot2=new Vector();
			    		String rotSrc=  src+ "rotasamp-small/"; 
				       	vecPdbSseRot2=pp.AlaninizeStructure(pdbHelixTemp,resIndex,rotSrc);		    		
		
				       	double[] vdwValue = new double[1];
				       	double vdwLevel = 0.05;
				       	boolean printVDWViolation = false;
				       	int[] clashResNo=new int[2];
				    	vdwVec = vander.convert2VDW(vecPdbSseRot2);
				    	boolean isStericClash=vander.checkStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, 
								true, 0.4, 1, clashResNo); 
				    	
				    	if(isStericClash)
				    	{
				    		System.out.println("here we have one steric clash...");
				    		continue;
				    	}
				    	//to add vdw checking here...
					    isChanged=true;//added by zeng
						rTotal = rT;
						System.out.println( mmm +": "+Math.sqrt(rRdc11 / N11 )+"  "
								    + Math.sqrt(rRdc12 / (N22 - 1) )+";  "+(Math.sqrt(phiRmsd / N ) / Const.cst) 
								    + "  "+(Math.sqrt(psiRmsd / N) / Const.cst)+", RT = "+rTotal);
						System.arraycopy(phiS, 0, phiSave, 0, N);
						System.arraycopy(psiS, 0, psiSave, 0, N);
						System.arraycopy(rdcArr1, 0, rdc1Save, 0, N);
						System.arraycopy(rdcArr2, 0, rdc2Save, 0, N);
						System.arraycopy(rdcCaCoCal, 0, rdcCaCoSave, 0, N);//ca-co rdc 
						System.arraycopy(rdcCoNCal, 0, rdcCoNSave, 0, N);
				    }
				}
		    }
		}
		long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
	
	 	Vector<PhiPsi> phiPsiVec  = new Vector<PhiPsi>();
	  	for (i=0; i<N; i++)  //build the model
	       	    phiPsiVec.add(new PhiPsi(i, "ALA", phiSave[i], psiSave[i]));
	   	Vector<Pdb> pdbHelixN = new Vector();
	   	if (isChanged)
	   		pdbHelixN= 	modelBuild(phiSave, psiSave, amide, nh, ca, firstResidueNo, false);
	   	else
	   	{
	   		pdbHelixN.addAll(pdbVec);
	   		System.out.println("no better solution is found....");
	   	}
	   	
		if(printResults)
		{
	        System.out.println(" CH RDC: ");
	        for (k=0; k < N; k++)
	            System.out.println((firstResidueNo+k)+": "+rdc1Save[k] +"  "+rdcExp1[k]+"  "+Math.abs(rdc1Save[k]-rdcExp1[k]));
	        System.out.println(" NH RDC: ");
	        for (k=0; k < N; k++)
	            System.out.println((firstResidueNo+k)+": "+rdc2Save[k] +"  "+rdcExp2[k]+"  "+ Math.abs(rdc2Save[k]-rdcExp2[k]));
	       
	        System.out.println(" CA-CO RDC: ");
	        for (k=0; k < N; k++)
	            System.out.println((firstResidueNo+k)+": "+rdcCaCoSave[k] +"  "+rdcExpCaCo[k]+"  "+ Math.abs(rdcCaCoSave[k]-rdcExpCaCo[k]));
	        
	        System.out.println(" CO-N RDC: ");
	        for (k=0; k < N; k++)
	            System.out.println((firstResidueNo+k)+": "+rdcCoNSave[k] +"  "+rdcExpCoN[k]+"  "+ Math.abs(rdcCoNSave[k]-rdcExpCoN[k]));
	       
	        System.out.println(" Phi/Psi: ");
	        for (k=0; k < N; k++)
	            System.out.println((firstResidueNo+k)+"   "+(phiSave[k]/Const.cst)+"  "+(psiSave[k]/Const.cst));
		}
		return pdbHelixN;
    }
   
    /**
     * A recusive function to compute all the backbone Phi/Psi for an
     * n-residue helix or the first strand.
     * 
     * @param vecPrePdb previous structure
     * @param rdcVec1  an array of CH RDCs for the fragment
     * @param rdcVec2  an array of NH RDCs for the fragment
     * @param rdc1Cal  an array for storing the back-computed RDcs
     * @param rdc2Cal  an array for storing the back-computed RDcs
     * @param pdbVec the pdb vector previously computed or build from scratch.
     * @param debugDFS if true, print out the information for debugging
     * @param printResults if true, print out the information for debugging
     * @param isHelix if true, the last \psi angle is the average value for helix, otherwise, for beta
     * @param vecTalos talos angle table
     * @param wtCoCa weight for CaCo rdc
     * @param wtCoN weight for CoN RDC
     * @param helixRdcCaCoVec the helix rdc ca co vec
     * @param helixRdcCoNVec the helix rdc co n vec
     * @param Syy the syy
     * @param Szz the szz
     * @param rmsd1 the rmsd1
     * @param rmsd2 the rmsd2
     * @param nCycle the n cycle
     * @param weightAngles the weight angles
     * @param startNo the start no
     * @param endNo the end no
     * @param vecSeq the vec seq
     * 
     * @return the computed structure for the fragment
     * 
     * @throws JampackException the jampack exception
     */
    public Vector<Pdb> minHelix4RDCs (Vector<Pdb> vecPrePdb, Vector<Dipolar> rdcVec1, Vector<Dipolar> rdcVec2,
    		final Vector<Dipolar> helixRdcCaCoVec, final Vector<Dipolar> helixRdcCoNVec, double[] rdc1Cal,
    		double[] rdc2Cal, Vector<Pdb> pdbVec, double Syy, double Szz, double rmsd1, double rmsd2, 
    		int nCycle,double weightAngles, boolean debugDFS, boolean printResults, boolean isHelix,
    		 Vector vecTalos, double wtCoCa, double wtCoN,int startNo, int endNo,Vector vecSeq)throws JampackException
    {     
	    int i, j, k;
		int N = pdbVec.size();//     //the number of CH RDC		
		
		PdbRdc pdr =new PdbRdc();
		double [] rdcArr1 = new double[N]; //CH RDCs, sampled based on Gaussina Dis, use array for efficient 
		double [] rdcArr2 = new double[N]; //NH RDCs, use array for efficient 
		double [] rdcExp1 = new double[N]; //store the experimental values in the array
		double [] rdcExp2 = new double[N];
		double [] rdcExpCaCo = new double[N];
		double [] rdcExpCoN = new double[N];
		
		double [] rdcErr1 = new double[N]; //store the experimental errs in the array
		double [] rdcErr2 = new double[N];
		double [] rdcErrCaCo = new double[N];
		double [] rdcErrCoN = new double[N];
		int [] seqNos1 = new int[N];
		int [] seqNos2 = new int[N];
		int [] seqNosCaCo = new int[N];
		int [] seqNosCoN = new int[N];
		int no = 0;    
		int index = -1;
		int N11 = 0;// rdcVec1.size();
		int N22 =0;//  rdcVec2.size();
		//Filled in missing RDCs
		for (j = 0; j < N; j++)
		{
		    no = startNo+j;
		    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
		    if (index > - 1)
		    {
		    	rdcExp1[j] = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
		    	rdcErr1[j]=((Dipolar)rdcVec1.elementAt(index)).getErr();
		    	seqNos1[j] = 1;
		    	N11++;
		    }else 
		    {
		    	rdcExp1[j] =  rdc1Cal[j];
		    	seqNos1[j] = 0;
		    	rdcErr1[j]=rmsd1;
		    }
		    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
		    if (index > - 1)
		    {
		    	rdcExp2[j] = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
		    	seqNos2[j] = 1;
		    	rdcErr2[j]=((Dipolar)rdcVec2.elementAt(index)).getErr();
		    	N22++;
		    }else
		    {
		    	rdcExp2[j] =  rdc2Cal[j];
		    	seqNos2[j] = 0;
		    	rdcErr2[j]=rmsd2;
		    }
		    
		    index = Collections.binarySearch(helixRdcCaCoVec, new Dipolar(no), new Dipolar.rdcComparator());
		    if (index > - 1)
		    {
		    	rdcExpCaCo[j] = ((Dipolar)helixRdcCaCoVec.elementAt(index)).getRdc();
		    	seqNosCaCo[j] = 1;
		    	rdcErrCaCo[j]=((Dipolar)helixRdcCaCoVec.elementAt(index)).getErr();
		    }else
		    {
		    	rdcExpCaCo[j] = 0.0;//not used actually
		    	seqNosCaCo[j] = 0;
		    	rdcErrCaCo[j]=0.0;//haven't been used yet
		    }
		    
		    index = Collections.binarySearch(helixRdcCoNVec, new Dipolar(no), new Dipolar.rdcComparator());
		    if (index > - 1)
		    {
		    	rdcExpCoN[j] = ((Dipolar)helixRdcCoNVec.elementAt(index)).getRdc();
		    	seqNosCoN[j] = 1;
		    	rdcErrCoN[j]=((Dipolar)helixRdcCoNVec.elementAt(index)).getErr();
		    }else
		    {
		    	rdcExpCoN[j] = 0.0;//not used actually
		    	seqNosCoN[j] = 0;
		    	rdcErrCoN[j]=0.0;//haven't been used yet
		    }
		}//for (j = 0; j < N; j++)
		//we should ignore the first NH RDC deviation in rmsd
		seqNos2[0]=0;
		N22--;
		
		Cartesian cc = new Cartesian();
		String atom = "";
		double [] amide = new double[3];
		double [] ca = new double[3];
		double [] nh = new double[3];
		
		int indA = Collections.binarySearch(vecPrePdb, new Pdb(startNo), new Pdb.PdbComparator() );
		Pdb pp = new Pdb();
		if(indA>-1)
			pp=vecPrePdb.elementAt(indA);
		else 
			pp=pdbVec.elementAt(0);
		
		Vector<Cartesian> atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++)
	    {
            cc = atomVec.elementAt(j);
            atom = cc.getAtom();
            if (atom.equals("N"))
                amide = cc.getXYZ();
            else if (atom.equals("H"))
                nh = cc.getXYZ();
            else if (atom.equals("CA"))
                ca = cc.getXYZ();
        }
	    double[] nToNHVec = internuclearVec(amide, nh);
        double[] nToCAVec = internuclearVec(amide, ca);
      
        ppGlobal ppg = new ppGlobal();
        Matrix mat = ppg.RgCal(nToNHVec, nToCAVec);

		boolean flag1 = false;
		double rRdc11 = 10000.0, rRdc12 = 10000.0, rT = 100000.0, rTotal = 100000.0;

		
		double rms1 = 0.0, rms2 = 0.0;
		double u, v; //for CH and NH RDC respectively
		int depth0 = 0;
		Vector<Double> ppVec = new Vector<Double>();
		Vector depthVec = new Vector();
		double[] ppS  = new double[2*N]; //N1 = 5
		double[] phiS = new double[N]; 
		double[] psiS = new double[N]; 
		double[] phiSave = new double[N]; 
		double[] psiSave = new double[N]; 
		double[] rdc1Save = new double[N];   //CH Rdcs
		double[] rdc2Save = new double[N]; 
		double[] rdcCaCoSave = new double[N]; 
		double[] rdcCoNSave = new double[N]; 
	    // 	Matrix mat = firstRg.transpose();
		int noOfSln = 0;
		int n = 0;	
		
		double phiRmsd = 0.0;
		double psiRmsd = 0.0;
		double rRdcFinal1 = rmsd1 * rmsd1 * N11; // that's the total for all residues, we want 2 fold better
		double rRdcFinal2 = rmsd2 * rmsd2 * N22; // the pairs should has quite close values
	
		//Sample the RDC to select a set satisfy certain conditions. specified later
		long startTime = System.currentTimeMillis();
		boolean depthFlag = true;
	
		boolean rightHand = true;
		PhiPsi ff = new PhiPsi();
		double phiAve = 0.0;
		double psiAve = 0.0;
		if(isHelix)
		{
		    phiAve = Const.phiAveHelix;
		    psiAve = Const.psiAveHelix;
		}
		else
		{
		    phiAve = Const.phiAveBeta;
		    psiAve = Const.psiAveBeta;
		}
		boolean isChanged =false;
		long seed =888;//88091;
		Random rr = new Random(seed);
				
		String userDir = System.getProperty("user.dir");////
	    String src=userDir+"/inputFiles/";
	    	
		vdw vander = new vdw();
		Vector vdwVec = new Vector();
	  	for (int mmm =0; mmm < nCycle; mmm++)
	  	{	  	
		    flag1 = false;
		    while (!flag1){
			flag1 = true;
			rRdc11 = 0.0;
			rRdc12 = 0.0;
			for (j = 0; j < N; j++)
			{   //For the 1st beta strand of the pair			
			    rdcArr1[j] = rdcExp1[j] + rmsd1 * rr.nextGaussian();// rmsd1 * rr.nextGaussian(); 		    
			    rdcArr2[j] = rdcExp2[j] +rmsd2 * rr.nextGaussian() ;//rmsd2 * rr.nextGaussian();
			    rms1 = Math.abs(rdcArr1[j] - rdcExp1[j]);
			    rRdc11 += rms1 * rms1 * seqNos1[j] ;
			    rms2 = Math.abs(rdcArr2[j] - rdcExp2[j]);
			    rRdc12 += rms2 * rms2 * seqNos2[j] ;
			}
			if ( (rRdc11 > rRdcFinal1) || (rRdc12 > rRdcFinal2) )
			    flag1 = false;
			else flag1 = true;
		    }//end of while
		    ppVec = new Vector<Double>();
		    depthFlag = false;
	    	depthFlag = ff.phiPsiChainTalos(rdcArr1, rdcArr2, mat, Syy, Szz, depth0, N - 1, ppS, ppVec, rightHand, isHelix,startNo,vecTalos);
		    if (ppVec.size() > 0 )
		    {
		    	// 		System.out.println("Seccess in Picking");
				noOfSln = (int)ppVec.size() / (2*N);  //the total number of solutions for this rdc set
				//System.out.println( "noOfSln= : " + noOfSln);///////////////////		
				for(k=0; k < noOfSln; k++)
				{   //please note the index
				    phiRmsd = 0.0;
				    psiRmsd = 0.0;
				    for (n = k* 2 * N; n < (k+1)* 2 *N; n += 2)
				    {  //original "N"
						phi = ppVec.elementAt(n).doubleValue();
						psi = ppVec.elementAt(n+1).doubleValue();
						phiS[(int)(n - k* 2 * N) / 2] = phi;
						psiS[(int)(n - k* 2 * N) / 2] = psi;
						phiRmsd += (phi - phiAve) * (phi - phiAve);
						psiRmsd += (psi - psiAve) * (psi - psiAve);
				    }
				    		
				    Vector<Pdb> pdbHelixTemp = new Vector();			       	
			       	pdbHelixTemp= 	modelBuild(phiS, psiS, amide, nh, ca, startNo, false);
			       	
			       	Vector vecPdbTemp=new Vector();
			       	int indTemp = Collections.binarySearch(vecPrePdb, new Pdb(startNo), new Pdb.PdbComparator() );
			       	if(indTemp>-1)
			       	{
			       		for(int tt=0;tt<indTemp;tt++)
			       		{
			       			Pdb ppTemp=(Pdb)vecPrePdb.elementAt(tt);
			       			vecPdbTemp.add(ppTemp);
			       		}
			       	}		       
			       	vecPdbTemp.addAll(pdbHelixTemp);
			       	
			       	double [][] Inv_temp = {{1.0, 0.0, 0.0},{0.0, 1.0, 0.0},{0.0, 0.0, 1.0}};
			       	Matrix Mat  = new Matrix(Inv_temp);
			       	double[] sRmsd= new double[1];
					int[] sizeRdc=new int[1];
					double[] rdcCoNCal  = new double[vecPdbTemp.size() ];
					double[] rdcCaCoCal  = new double[vecPdbTemp.size() ];
			       
			    					
					double r_caco=0.0;
					double r_con=0.0;
					if(helixRdcCaCoVec.size()>0)
						r_caco=pdr.BackCalCACO(vecPdbTemp, helixRdcCaCoVec, "C", "CA", Mat, -Syy-Szz, Syy, Szz, Const.cacoRatio, rdcCaCoCal,sRmsd, sizeRdc,false);
					if(helixRdcCoNVec.size()>0)
						r_con=pdr.BackCalCON(vecPdbTemp, helixRdcCoNVec, "C", "N",  Mat, -Syy-Szz, Syy, Szz, Const.conRatio, rdcCoNCal,sRmsd, sizeRdc,false);
								       	
				    rT = Math.sqrt((rRdc11/Const.cahaRatio) / N11 ) + 1.0* (Math.sqrt((rRdc12/Const.nhRatio) / N22 )) + wtCoCa* (r_caco/Const.cacoRatio) +
				    wtCoN*(r_con/Const.conRatio)+ weightAngles*(Math.sqrt(phiRmsd / N ) + Math.sqrt(psiRmsd / N));
				    if (rT < rTotal)
				    {
				    	//check the steric clashes
				    	Vector<PhiPsi> phiPsiVec_temp  = new Vector<PhiPsi>();
				      	for (i=0; i<N; i++)  //build the model
				      		phiPsiVec_temp.add(new PhiPsi(i, "ALA", phiS[i], psiS[i]));				       	
		
				       	boolean[] resIndex=new boolean[vecPdbTemp.size()];
			    		
			    		for (int kk=0;kk<resIndex.length;kk++)
			    			resIndex[kk]=false;			 
			    		
			    		Vector pdbVecSSE_temp2 = pp.residueNameUpdate(vecSeq, vecPdbTemp);
			    		 
			    		for (int kk=0;kk<pdbVecSSE_temp2.size();kk++)
			    		{
			    			Pdb ppTemp=(Pdb)pdbVecSSE_temp2.elementAt(kk);
			    			String resTemp=ppTemp.getResidue();
			    			if(resTemp.equalsIgnoreCase("PRO") || resTemp.equalsIgnoreCase("GLY"))
			    				resIndex[kk]=true;		
			    		}			    		
			    		
			    		Vector vecPdbSseRot2=new Vector();
			    		String rotSrc=  userDir+"/system/rot-lib/"; 
				       	//vecPdbSseRot2=pp.AlaninizeStructure(vecPdbTemp,resIndex,rotSrc);		    		
				       	vecPdbSseRot2=pp.AlaninizeStructureResName(vecPdbTemp,rotSrc);	
				       	
				       	double[] vdwValue = new double[1];
				       	double vdwLevel = 0.05;
				       	boolean printVDWViolation = false;
				       	int[] clashResNo=new int[2];
				       
					    isChanged=true;
						rTotal = rT;
						System.out.println( mmm +": "+Math.sqrt(rRdc11 / N11 )+"  "
								    + Math.sqrt(rRdc12 / (N22 - 1) )+";  "+(Math.sqrt(phiRmsd / N ) / Const.cst) 
								    + "  "+(Math.sqrt(psiRmsd / N) / Const.cst)+", RT = "+rTotal);
						System.arraycopy(phiS, 0, phiSave, 0, N);
						System.arraycopy(psiS, 0, psiSave, 0, N);
						System.arraycopy(rdcArr1, 0, rdc1Save, 0, N);
						System.arraycopy(rdcArr2, 0, rdc2Save, 0, N);
						System.arraycopy(rdcCaCoCal, 0, rdcCaCoSave, 0, N);//ca-co rdc 
						System.arraycopy(rdcCoNCal, 0, rdcCoNSave, 0, N);
				    }
				}
		    }
		}
		long endTime = System.currentTimeMillis() ;
		double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
	
	 	Vector<PhiPsi> phiPsiVec  = new Vector<PhiPsi>();
	  	for (i=0; i<N; i++)  //build the model
	       	    phiPsiVec.add(new PhiPsi(i, "ALA", phiSave[i], psiSave[i]));
	   	Vector<Pdb> pdbHelixN = new Vector();
	   	if (isChanged)
	   		pdbHelixN= 	modelBuild(phiSave, psiSave, amide, nh, ca, startNo, false);
	   	else
	   	{	   		
	   		pdbHelixN.addAll(pdbVec);
	   		System.out.println("no better solution is found....");
	   	}
	   
	   	
		if(printResults)
		{
	        System.out.println(" CH RDC: ");
	        for (k=0; k < N; k++)
	            System.out.println((startNo+k)+": "+rdc1Save[k] +"  "+rdcExp1[k]+"  "+Math.abs(rdc1Save[k]-rdcExp1[k]));
	        System.out.println(" NH RDC: ");
	        for (k=0; k < N; k++)
	            System.out.println((startNo+k)+": "+rdc2Save[k] +"  "+rdcExp2[k]+"  "+ Math.abs(rdc2Save[k]-rdcExp2[k]));
	       
	        System.out.println(" CA-CO RDC: ");
	        for (k=0; k < N; k++)
	            System.out.println((startNo+k)+": "+rdcCaCoSave[k] +"  "+rdcExpCaCo[k]+"  "+ Math.abs(rdcCaCoSave[k]-rdcExpCaCo[k]));
	        
	        System.out.println(" CO-N RDC: ");
	        for (k=0; k < N; k++)
	            System.out.println((startNo+k)+": "+rdcCoNSave[k] +"  "+rdcExpCoN[k]+"  "+ Math.abs(rdcCoNSave[k]-rdcExpCoN[k]));
	       
	        System.out.println(" Phi/Psi: ");
	        for (k=0; k < N; k++)
	            System.out.println((startNo+k)+"   "+(phiSave[k]/Const.cst)+"  "+(psiSave[k]/Const.cst));
		}
		return pdbHelixN;
    }
   
  
    /**
     * A recusive function to compute all the backbone Phi/Psi for an
     * n-residue helix or the first strand.
     * 
     * @param rdcVec1  an array of CH RDCs for the fragment
     * @param rdcVec2  an array of NH RDCs for the fragment
     * @param rdc1Cal  an array for storing the back-computed RDcs
     * @param rdc2Cal  an array for storing the back-computed RDcs
     * @param pdbVec the pdb vector previously computed or build from scratch.
     * @param debugDFS if true, print out the information for debugging
     * @param printResults if true, print out the information for debugging
     * @param isHelix if true, the last \psi angle is the average value for helix, otherwise, for beta
     * @param Syy the syy
     * @param Szz the szz
     * @param rmsd1 the rmsd1
     * @param rmsd2 the rmsd2
     * @param nCycle the n cycle
     * @param weightAngles the weight angles
     * @param NhRdcRmsd the nh rdc rmsd
     * @param ChRdcRmsd the ch rdc rmsd
     * 
     * @return the computed structure for the fragment
     */
    public Vector<Pdb> minHelix (Vector<Dipolar> rdcVec1, Vector<Dipolar> rdcVec2, double[] rdc1Cal, double[] rdc2Cal, Vector<Pdb> pdbVec,
			    double Syy, double Szz, double rmsd1, double rmsd2, int nCycle,
			    double weightAngles, boolean debugDFS, boolean printResults, boolean isHelix,
			    double[] NhRdcRmsd, double [] ChRdcRmsd)
	{      	
    	NhRdcRmsd[0]=999.0;
    	ChRdcRmsd[0]=999.0;
    
    	long seed =7299;
		Random rr = new Random(seed);	
    
    int i, j, k;
	final int N = pdbVec.size() - 1;     //the number of CH RDC
	Dipolar dd1 = (Dipolar)rdcVec1.elementAt(0);
	Dipolar dd2 = (Dipolar)rdcVec2.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());
	
	double [] rdcArr1 = new double[N]; //CH RDCs, sampled based on Gaussina Dis, use array for efficient 
	double [] rdcArr2 = new double[N]; //NH RDCs, use array for efficient 
	double [] rdcExp1 = new double[N]; //store the experimental values in the array
	double [] rdcExp2 = new double[N];
	double [] rdcErr1 = new double[N]; //store the experimental errs in the array
	double [] rdcErr2 = new double[N];
	int [] seqNos1 = new int[N];
	int [] seqNos2 = new int[N];
	int no = 0;    
	int index = -1;
	//Filled in missing RDCs
	for (j = 0; j < N; j++)
	{
	    no = firstResidueNo+j;
	    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1)
	    {
	    	rdcExp1[j] = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
	    	rdcErr1[j]=((Dipolar)rdcVec1.elementAt(index)).getErr();
	    	seqNos1[j] = 1;
	    }else 
	    {
	    	rdcExp1[j] =  rdc1Cal[j];
	    	seqNos1[j] = 0;
	    	rdcErr1[j]=rmsd1;
	    }
	    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1)
	    {
	    	rdcExp2[j] = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
	    	seqNos2[j] = 1;
	    	rdcErr2[j]=((Dipolar)rdcVec2.elementAt(index)).getErr();
	    }else
	    {
	    	rdcExp2[j] =  rdc2Cal[j];
	    	seqNos2[j] = 0;
	    	rdcErr2[j]=rmsd2;
	    }
	}//for (j = 0; j < N; j++)
	
	Cartesian cc = new Cartesian();
	String atom = "";
	double [] amide = new double[3];
	double [] ca = new double[3];
	double [] nh = new double[3];
	Pdb pp = pdbVec.elementAt(0);
	Vector<Cartesian> atomVec = pp.getAtomVec();
        for (j=0; j<atomVec.size(); j++){
            cc = atomVec.elementAt(j);
            atom = cc.getAtom();
            if (atom.equals("N"))
                amide = cc.getXYZ();
            else if (atom.equals("H"))
                nh = cc.getXYZ();
            else if (atom.equals("CA"))
                ca = cc.getXYZ();
        }
	double[] nToNHVec = internuclearVec(amide, nh);
        double[] nToCAVec = internuclearVec(amide, ca);

        ppGlobal ppg = new ppGlobal();
        Matrix mat = ppg.RgCal(nToNHVec, nToCAVec);

	boolean flag1 = false;
	double rRdc11 = 10000.0, rRdc12 = 10000.0, rT = 100000.0, rTotal = 100000.0;

	
	double rms1 = 0.0, rms2 = 0.0;
	double u, v; //for CH and NH RDC respectively
	int depth0 = 0;
	Vector<Double> ppVec = new Vector<Double>();
	Vector depthVec = new Vector();
	double[] ppS  = new double[2*N]; //N1 = 5
	double[] phiS = new double[N]; 
	double[] psiS = new double[N]; 
	double[] phiSave = new double[N]; 
	double[] psiSave = new double[N]; 
	double[] rdc1Save = new double[N];   //CH Rdcs
	double[] rdc2Save = new double[N]; 

	int noOfSln = 0;
	int n = 0;	
	int N11 =  rdcVec1.size();
	int N22 =  rdcVec2.size();
	double phiRmsd = 0.0;
	double psiRmsd = 0.0;
	double rRdcFinal1 = rmsd1 * rmsd1 * N11; // that's the total for all residues, we want 2 fold better
	double rRdcFinal2 = rmsd2 * rmsd2 * N22; // the pairs should has quite close values

	//Sample the RDC to select a set satisfy certain conditions. specified later
	long startTime = System.currentTimeMillis();
	boolean depthFlag = true;

	boolean rightHand = true;
	PhiPsi ff = new PhiPsi();
	double phiAve = 0.0;
	double psiAve = 0.0;
	if(isHelix){
	    phiAve = Const.phiAveHelix;
	    psiAve = Const.psiAveHelix;
	}else{
	    phiAve = Const.phiAveBeta;
	    psiAve = Const.psiAveBeta;
	}
	boolean isChanged =false;
	
  	for (int mmm =0; mmm < nCycle; mmm++){  		
  	    
	    flag1 = false;
	    while (!flag1){
		flag1 = true;
		rRdc11 = 0.0;
		rRdc12 = 0.0;
		for (j = 0; j < N; j++){   //For the 1st beta strand of the pair
			
		    rdcArr1[j] = rdcExp1[j] + rmsd1 * rr.nextGaussian();// rmsd1 * rr.nextGaussian(); 
		    
		    rdcArr2[j] = rdcExp2[j] +rmsd2 * rr.nextGaussian() ;//rmsd2 * rr.nextGaussian();
		    rms1 = Math.abs(rdcArr1[j] - rdcExp1[j]);
		    rRdc11 += rms1 * rms1 * seqNos1[j] ;
		    rms2 = Math.abs(rdcArr2[j] - rdcExp2[j]);
		    rRdc12 += rms2 * rms2 * seqNos2[j] ;
		}
		if ( (rRdc11 > rRdcFinal1) || (rRdc12 > rRdcFinal2) )
		    flag1 = false;
		else flag1 = true;
	    }//end of while
	    ppVec = new Vector<Double>();
	    depthFlag = false;
    	    depthFlag = ff.phiPsiChain(rdcArr1, rdcArr2, mat, Syy, Szz, depth0, N - 1, ppS, ppVec, rightHand, isHelix);
	    if (ppVec.size() > 0 ){

		noOfSln = (int)ppVec.size() / (2*N);  //the total number of solutions for this rdc set
		
		for(k=0; k < noOfSln; k++)
		{   //please note the index
		    phiRmsd = 0.0;
		    psiRmsd = 0.0;
		    for (n = k* 2 * N; n < (k+1)* 2 *N; n += 2){  //original "N"
			phi = ppVec.elementAt(n).doubleValue();
			psi = ppVec.elementAt(n+1).doubleValue();
			phiS[(int)(n - k* 2 * N) / 2] = phi;
			psiS[(int)(n - k* 2 * N) / 2] = psi;
			phiRmsd += (phi - phiAve) * (phi - phiAve);
			psiRmsd += (psi - psiAve) * (psi - psiAve);
		    }
		    rT = Math.sqrt(rRdc11 / N11 ) +  (Math.sqrt(rRdc12 / (N22 - 1) )) + 
			weightAngles*(Math.sqrt(phiRmsd / N ) + Math.sqrt(psiRmsd / N));
		    if (rT < rTotal)
		    {
		    isChanged=true;//added by zeng
			rTotal = rT;
			System.out.println( mmm +": "+Math.sqrt(rRdc11 / N11 )+"  "
					    + Math.sqrt(rRdc12 / (N22 - 1) )+";  "+(Math.sqrt(phiRmsd / N ) / Const.cst) 
					    + "  "+(Math.sqrt(psiRmsd / N) / Const.cst)+", RT = "+rTotal);
			NhRdcRmsd[0]=  (Math.sqrt(rRdc12 / (N22 - 1) ));
			ChRdcRmsd[0]=Math.sqrt(rRdc11 / N11 );
				
			System.arraycopy(phiS, 0, phiSave, 0, N);
			System.arraycopy(psiS, 0, psiSave, 0, N);
			System.arraycopy(rdcArr1, 0, rdc1Save, 0, N);
			System.arraycopy(rdcArr2, 0, rdc2Save, 0, N);
		    }
		}
	    }
	}
	long endTime = System.currentTimeMillis() ;
	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes

 	Vector<PhiPsi> phiPsiVec  = new Vector<PhiPsi>();
  	for (i=0; i<N; i++)  //build the model
       	    phiPsiVec.add(new PhiPsi(i, "ALA", phiSave[i], psiSave[i]));
   	Vector<Pdb> pdbHelixN = new Vector();
   	if (isChanged)
   		pdbHelixN= 	modelBuild(phiSave, psiSave, amide, nh, ca, firstResidueNo, false);
   	else
   	{
   		NhRdcRmsd[0]=9999.0;
   		ChRdcRmsd[0]=9999.0;
   		pdbHelixN.addAll(pdbVec);
   		System.out.println("no better solution is found....");
   	}
   	
	if(printResults){
            System.out.println(" CH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc1Save[k] +"  "+rdcExp1[k]+"  "+Math.abs(rdc1Save[k]-rdcExp1[k]));
            System.out.println(" NH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc2Save[k] +"  "+rdcExp2[k]+"  "+ Math.abs(rdc2Save[k]-rdcExp2[k]));

            System.out.println(" Phi/Psi: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+"   "+(phiSave[k]/Const.cst)+"  "+(psiSave[k]/Const.cst));
        }
	return pdbHelixN;
    }

    
  /**
   * A recusive function to compute all the backbone Phi/Psi for an
   * n-residue fragment.
   * 
   * @param rdcVec1  an array of CH RDCs for the fragment
   * @param rdcVec2  an array of NH RDCs for the fragment
   * @param rdc1Cal the rdc1 cal
   * @param rdc2Cal the rdc2 cal
   * @param pdbVec the pdb vec
   * @param Syy the syy
   * @param Szz the szz
   * @param rmsd1 the rmsd1
   * @param rmsd2 the rmsd2
   * @param nCycle the n cycle
   * @param weightAngles the weight angles
   * @param debugDFS the debug dfs
   * @param printResults the print results
   * @param isHelix the is helix
   * 
   * @return the vector
   */
    public Vector minHelix_old (Vector rdcVec1, Vector rdcVec2, double[] rdc1Cal, double[] rdc2Cal, Vector pdbVec,
			    double Syy, double Szz, double rmsd1, double rmsd2, int nCycle,
			    double weightAngles, boolean debugDFS, boolean printResults, boolean isHelix){  
	int i, j, k;
	final int N = pdbVec.size() - 1;     //the number of CH RDC
	Dipolar dd1 = (Dipolar)rdcVec1.elementAt(0);
	Dipolar dd2 = (Dipolar)rdcVec2.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());

	double [] rdcArr1 = new double[N]; //CH RDCs, sampled based on Gaussina Dis, use array for efficient 
	double [] rdcArr2 = new double[N]; //NH RDCs, use array for efficient 
	double [] rdcExp1 = new double[N]; //store the experimental values in the array
	double [] rdcExp2 = new double[N];
	int [] seqNos1 = new int[N];
	int [] seqNos2 = new int[N];
	int no = 0;    
	int index = -1;
	//Filled in missing RDCs
	for (j = 0; j < N; j++){
	    no = firstResidueNo+j;
	    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1){
		rdcExp1[j] = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
		seqNos1[j] = 1;
	    }else {
		rdcExp1[j] =  rdc1Cal[j];
		seqNos1[j] = 0;
	    }
	    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1){
		rdcExp2[j] = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
		seqNos2[j] = 1;
	    }else {
		rdcExp2[j] =  rdc2Cal[j];
		seqNos2[j] = 0;
	    }
	}
	Cartesian cc = new Cartesian();
	String atom = "";
	double [] amide = new double[3];
	double [] ca = new double[3];
	double [] nh = new double[3];
	Pdb pp = (Pdb)pdbVec.elementAt(0);
	Vector atomVec = pp.getAtomVec();
        for (j=0; j<atomVec.size(); j++){
            cc = (Cartesian)atomVec.elementAt(j);
            atom = cc.getAtom();
            if (atom.equals("N"))
                amide = cc.getXYZ();
            else if (atom.equals("H"))
                nh = cc.getXYZ();
            else if (atom.equals("CA"))
                ca = cc.getXYZ();
        }
	double[] nToNHVec = internuclearVec(amide, nh);
        double[] nToCAVec = internuclearVec(amide, ca);

        ppGlobal ppg = new ppGlobal();
        Matrix mat = ppg.RgCal(nToNHVec, nToCAVec);

	boolean flag1 = false;
	double rRdc11 = 10000.0, rRdc12 = 10000.0, rT = 100000.0, rTotal = 100000.0;

	long seed = 95738579;
	Random rr = new Random(seed);
	double rms1 = 0.0, rms2 = 0.0;
	double u, v; //for CH and NH RDC respectively
	int depth0 = 0;
	Vector ppVec = new Vector();
	Vector depthVec = new Vector();
	double[] ppS  = new double[2*N]; //N1 = 5
	double[] phiS = new double[N]; 
	double[] psiS = new double[N]; 
	double[] phiSave = new double[N]; 
	double[] psiSave = new double[N]; 
	double[] rdc1Save = new double[N];   //CH Rdcs
	double[] rdc2Save = new double[N]; 

	int noOfSln = 0;
	int n = 0;	
	int N11 =  rdcVec1.size();
	int N22 =  rdcVec2.size();
	double phiRmsd = 0.0;
	double psiRmsd = 0.0;
	double rRdcFinal1 = rmsd1 * rmsd1 * N11; // that's the total for all residues, we want 2 fold better
	double rRdcFinal2 = rmsd2 * rmsd2 * N22; // the pairs should has quite close values

	//Sample the RDC to select a set satisfy certain conditions. specified later
	long startTime = System.currentTimeMillis();
	boolean depthFlag = true;
	int searchDepth = 0;
	boolean rightHand = true;
	PhiPsi ff = new PhiPsi();
	double phiAve = 0.0;
	double psiAve = 0.0;
	if(isHelix){
	    phiAve = Const.phiAveHelix;
	    psiAve = Const.psiAveHelix;
	}else{
	    phiAve = Const.phiAveBeta;
	    psiAve = Const.psiAveBeta;
	}
  	for (int mmm =0; mmm < nCycle; mmm++){
	    flag1 = false;
	    while (!flag1){
		flag1 = true;
		rRdc11 = 0.0;
		rRdc12 = 0.0;
		for (j = 0; j < N; j++){   //For the 1st beta strand of the pair
		    rdcArr1[j] = rdcExp1[j] + rmsd1 * rr.nextGaussian(); 
		    rdcArr2[j] = rdcExp2[j] + rmsd2 * rr.nextGaussian();
		    rms1 = Math.abs(rdcArr1[j] - rdcExp1[j]);
		    rRdc11 += rms1 * rms1 * seqNos1[j] ;
		    rms2 = Math.abs(rdcArr2[j] - rdcExp2[j]);
		    rRdc12 += rms2 * rms2 * seqNos2[j] ;
		}
		if ( (rRdc11 > rRdcFinal1) || (rRdc12 > rRdcFinal2) )
		    flag1 = false;
		else flag1 = true;
	    }
	    ppVec = new Vector();
	    depthFlag = false;
    	    depthFlag = ff.phiPsiChain(rdcArr1, rdcArr2, mat, Syy, Szz, depth0, N - 1, ppS, ppVec, rightHand, isHelix);
	    if (ppVec.size() > 0 ){

		noOfSln = (int)ppVec.size() / (2*N);  //the total number of solutions for this rdc set
		for(k=0; k < noOfSln; k++){   //please note the index
		    phiRmsd = 0.0;
		    psiRmsd = 0.0;
		    for (n = k* 2 * N; n < (k+1)* 2 *N; n += 2){  //original "N"
			phi = ((Double)ppVec.elementAt(n)).doubleValue();
			psi = ((Double)ppVec.elementAt(n+1)).doubleValue();
			phiS[(int)(n - k* 2 * N) / 2] = phi;
			psiS[(int)(n - k* 2 * N) / 2] = psi;
			phiRmsd += (phi - phiAve) * (phi - phiAve);
			psiRmsd += (psi - psiAve) * (psi - psiAve);
		    }
		    rT = Math.sqrt(rRdc11 / N11 ) + Math.sqrt(rRdc12 / (N22 - 1) ) + 
			weightAngles*(Math.sqrt(phiRmsd / N ) + Math.sqrt(psiRmsd / N));
		    if (rT < rTotal){
			rTotal = rT;
			System.out.println( mmm +": "+Math.sqrt(rRdc11 / N11 )+"  "
					    + Math.sqrt(rRdc12 / (N22 - 1) )+";  "+(Math.sqrt(phiRmsd / N ) / Const.cst) 
					    + "  "+(Math.sqrt(psiRmsd / N) / Const.cst)+", RT = "+rTotal);
			System.arraycopy(phiS, 0, phiSave, 0, N);
			System.arraycopy(psiS, 0, psiSave, 0, N);
			System.arraycopy(rdcArr1, 0, rdc1Save, 0, N);
			System.arraycopy(rdcArr2, 0, rdc2Save, 0, N);
		    }
		}
	    }
	}
	long endTime = System.currentTimeMillis() ;
	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes

 	Vector phiPsiVec  = new Vector();
  	for (i=0; i<N; i++)  //build the model
       	    phiPsiVec.add(new PhiPsi(i, "ALA", phiSave[i], psiSave[i]));
   	Vector pdbHelixN = modelBuild(phiSave, psiSave, amide, nh, ca, firstResidueNo, false); 

	if(printResults){
            System.out.println(" CH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc1Save[k] +"  "+rdcExp1[k]+"  "+Math.abs(rdc1Save[k]-rdcExp1[k]));
            System.out.println(" NH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc2Save[k] +"  "+rdcExp2[k]+"  "+ Math.abs(rdc2Save[k]-rdcExp2[k]));

            System.out.println(" Phi/Psi: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+"   "+(phiSave[k]/Const.cst)+"  "+(psiSave[k]/Const.cst));
        }
	return pdbHelixN;
    }
    
    /**
     * Modified by zeng for computing the alignment tensor based on structure and RDCs
     * Update the Saupe elements for the second time.
     * 
     * @param pdbVec the Pdb vector before the update
     * @param saupeSave for saving the computed Saupe elements {Syy, Szz, rmsd4NHRDC, rmsd4CHRDC}
     * @param nhRdc the nh rdc
     * @param cahaRdc the caha rdc
     * 
     * @return  the refined Pdb vector.
     */
    public Vector<Pdb> refineSaupe3 (Vector<Pdb> pdbVec,Vector<Dipolar>  nhRdc, 
    		Vector<Dipolar>  cahaRdc,  double[] saupeSave){
    PdbRdc pdr = new PdbRdc();
       Pdb pp = new Pdb();
    //compute new Saupes
	int N = pdbVec.size();
 	double[] rdc1Cal = new double[N];
	double[] rdc2Cal = new double[N];
	
	System.out.println("================================================================================");	
	System.out.println("The alignment tensor for both CH and NH RDC:  ");
    Matrix mm = pdr.bestFit(pdbVec, nhRdc, cahaRdc, rdc1Cal, rdc2Cal, saupeSave);
 	//Vector<Pdb> pdbVec2 = pp.newPdb(pdbVec, mm);
    
    // 	added by zeng
 	double Syy = saupeSave[0];
 	double Szz = saupeSave[1];
 	double Sxx=-Syy-Szz;
 	double rdc1Rmsd = saupeSave[2];
 	double rdc2Rmsd = saupeSave[3];
 	//we try to compute the alignment tensor in the conventional format,
 	//which is |Sxx|<|Syy| < |Szz|
 	//double SzzNew=Math.max(  Math.abs(Sxx), Math.abs(Syy) );//, Math.abs(Szz) );
 	double SxxNew=0.0, SyyNew=0.0, SzzNew=0.0;
 	if ( (Math.abs(Sxx)> Math.abs(Syy)) &&(Math.abs(Sxx)> Math.abs(Szz)))
 		SzzNew=Sxx;
 	if ( (Math.abs(Syy)> Math.abs(Sxx)) &&(Math.abs(Syy)> Math.abs(Szz)))
 		SzzNew=Syy;
 	if ( (Math.abs(Szz)> Math.abs(Sxx)) &&(Math.abs(Szz)> Math.abs(Syy)))
 		SzzNew=Szz;
 	
 	if( (Math.abs(Sxx) < Math.abs(Szz) ) && (Math.abs(Sxx)< Math.abs(Syy) ) )
 		SxxNew=Sxx;
 	if( (Math.abs(Syy) < Math.abs(Szz) ) && (Math.abs(Syy)< Math.abs(Sxx) ) )
 		SxxNew=Syy;
 	if( (Math.abs(Szz) < Math.abs(Sxx) ) && (Math.abs(Szz)< Math.abs(Syy) ) )
 		SxxNew=Szz;
 	
 	SyyNew=-SxxNew-SzzNew;
 	System.out.println("================================================================================");	
 	System.out.println("The alignment tensor for my own calculation:");
 	
 	System.out.println("Sxx, Syy, Szz =    " +(-Syy-Szz)+"    "   +Syy+"  "+Szz);
	System.out.println("rdc1Rmsd (CH), rdc2Rmsd (NH)=    " +" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
	System.out.println("Second   (Licong's original version) "+Syy+"  "+Szz+" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
 	//System.out.println("Another format of alingment tenosr:  "); //added by zeng
	System.out.println("================================================================================");	
 	
 	
 	
 	System.out.println("================================================================================");	
 	
	System.out.println("The alignment tensor for the whole structure in the conventional format:");
 	System.out.println("----------------------------------------------------------------------------------");
 	System.out.println("A_a, A_r =   "+ (SzzNew/2 ) + "   " + (((-SyyNew-SzzNew)-SyyNew)/3));
 	System.out.println("----------------------------------------------------------------------------------");
 	
  	System.out.println("Sxx, Syy, Szz (in the conventional format)=    " +(-SyyNew-SzzNew)+"    "   +SyyNew+"  "+SzzNew);
	System.out.println("rdc1Rmsd (CH), rdc2Rmsd (NH)=    " +" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
	System.out.println("Second   (Licong's original version) "+Syy+"  "+Szz+" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
 	//System.out.println("Another format of alingment tenosr:  "); //added by zeng
	System.out.println("================================================================================");	
 	
	
	System.out.println("================================================================================");	
	System.out.println("The alignment tensor for only NH RDC:  ");
	//for NH RDC only
 	Vector cahaRdc_temp = new Vector();
 	double[] saupeSave_temp= new double[4];
 	double[] rdc1Cal_temp = new double[N];
	double[] rdc2Cal_temp = new double[N];
 	mm = pdr.bestFit(pdbVec, nhRdc, cahaRdc_temp, rdc1Cal_temp, rdc2Cal_temp, saupeSave_temp); 	
 	Syy = saupeSave_temp[0];
 	Szz = saupeSave_temp[1];
 	rdc1Rmsd = saupeSave_temp[2];
 	rdc2Rmsd = saupeSave_temp[3];
	
 	SxxNew=0.0; SyyNew=0.0; SzzNew=0.0;
 	if ( (Math.abs(Sxx)> Math.abs(Syy)) &&(Math.abs(Sxx)> Math.abs(Szz)))
 		SzzNew=Sxx;
 	if ( (Math.abs(Syy)> Math.abs(Sxx)) &&(Math.abs(Syy)> Math.abs(Szz)))
 		SzzNew=Syy;
 	if ( (Math.abs(Szz)> Math.abs(Sxx)) &&(Math.abs(Szz)> Math.abs(Syy)))
 		SzzNew=Szz;
 	
 	if( (Math.abs(Sxx) < Math.abs(Szz) ) && (Math.abs(Sxx)< Math.abs(Syy) ) )
 		SxxNew=Sxx;
 	if( (Math.abs(Syy) < Math.abs(Szz) ) && (Math.abs(Syy)< Math.abs(Sxx) ) )
 		SxxNew=Syy;
 	if( (Math.abs(Szz) < Math.abs(Sxx) ) && (Math.abs(Szz)< Math.abs(Syy) ) )
 		SxxNew=Szz;
 	
 	SyyNew=-SxxNew-SzzNew;
 	
 	System.out.println("----------------------------------------------------------------------------------");	
 	System.out.println("A_a, A_r =   "+ (Szz/2 ) + "   " + (((-Syy-Szz)-Syy)/3));   
 	System.out.println("----------------------------------------------------------------------------------");
 	
 	System.out.println("Sxx, Syy, Szz=    " +(-Syy-Szz)+"    "   +Syy+"  "+Szz);
	System.out.println("rdc1Rmsd (CH), rdc2Rmsd (NH)=    " +" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
	System.out.println("Second   (Licong's original version) "+Syy+"  "+Szz+" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
 	//System.out.println("Another format of alingment tenosr:  "); //added by zeng
	System.out.println("================================================================================");	
	System.out.println("================================================================================");	
 	
	System.out.println("The alignment tensor for the whole structure in the conventional format:");
 	System.out.println("----------------------------------------------------------------------------------");
 	System.out.println("A_a, A_r =   "+ (SzzNew/2 ) + "   " + (((-SyyNew-SzzNew)-SyyNew)/3));
 	System.out.println("----------------------------------------------------------------------------------");
 	
  	System.out.println("Sxx, Syy, Szz (in the conventional format)=    " +(-SyyNew-SzzNew)+"    "   +SyyNew+"  "+SzzNew);
	System.out.println("rdc1Rmsd (CH), rdc2Rmsd (NH)=    " +" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
	System.out.println("Second   (Licong's original version) "+Syy+"  "+Szz+" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
 	//System.out.println("Another format of alingment tenosr:  "); //added by zeng
	System.out.println("================================================================================");	
 	
	
	
	System.out.println("================================================================================");	
	System.out.println("The alignment tensor for only CH RDC:  ");
 	// 	for CH RDC only
 	Vector nhRdc_temp = new Vector();
 	mm = pdr.bestFit(pdbVec, nhRdc_temp, cahaRdc, rdc1Cal_temp, rdc2Cal_temp, saupeSave_temp); 	
 	Syy = saupeSave_temp[0];
 	Szz = saupeSave_temp[1];
 	rdc1Rmsd = saupeSave_temp[2];
 	rdc2Rmsd = saupeSave_temp[3];
 	
 	SxxNew=0.0; SyyNew=0.0; SzzNew=0.0;
 	if ( (Math.abs(Sxx)> Math.abs(Syy)) &&(Math.abs(Sxx)> Math.abs(Szz)))
 		SzzNew=Sxx;
 	if ( (Math.abs(Syy)> Math.abs(Sxx)) &&(Math.abs(Syy)> Math.abs(Szz)))
 		SzzNew=Syy;
 	if ( (Math.abs(Szz)> Math.abs(Sxx)) &&(Math.abs(Szz)> Math.abs(Syy)))
 		SzzNew=Szz;
 	
 	if( (Math.abs(Sxx) < Math.abs(Szz) ) && (Math.abs(Sxx)< Math.abs(Syy) ) )
 		SxxNew=Sxx;
 	if( (Math.abs(Syy) < Math.abs(Szz) ) && (Math.abs(Syy)< Math.abs(Sxx) ) )
 		SxxNew=Syy;
 	if( (Math.abs(Szz) < Math.abs(Sxx) ) && (Math.abs(Szz)< Math.abs(Syy) ) )
 		SxxNew=Szz;
 	
 	SyyNew=-SxxNew-SzzNew;
 	
 	
 	System.out.println("----------------------------------------------------------------------------------");
 	System.out.println("A_a, A_r =   "+ (Szz/2 ) + "   " + (((-Syy-Szz)-Syy)/3));   
 	System.out.println("----------------------------------------------------------------------------------");
 	System.out.println("Sxx, Syy, Szz=    " +(-Syy-Szz)+"    "   +Syy+"  "+Szz);
	System.out.println("rdc1Rmsd (CH), rdc2Rmsd (NH)=    " +" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
	System.out.println("Second   (Licong's original version) "+Syy+"  "+Szz+" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
 	//System.out.println("Another format of alingment tenosr:  "); //added by zeng
	System.out.println("================================================================================");	
	System.out.println("================================================================================");	
 	
	System.out.println("The alignment tensor for the whole structure in the conventional format:");
 	System.out.println("----------------------------------------------------------------------------------");
 	System.out.println("A_a, A_r =   "+ (SzzNew/2 ) + "   " + (((-SyyNew-SzzNew)-SyyNew)/3));
 	System.out.println("----------------------------------------------------------------------------------");
 	
  	System.out.println("Sxx, Syy, Szz (in the conventional format)=    " +(-SyyNew-SzzNew)+"    "   +SyyNew+"  "+SzzNew);
	System.out.println("rdc1Rmsd (CH), rdc2Rmsd (NH)=    " +" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
	System.out.println("Second   (Licong's original version) "+Syy+"  "+Szz+" :  "+rdc1Rmsd+"  "+rdc2Rmsd);
 	//System.out.println("Another format of alingment tenosr:  "); //added by zeng
	System.out.println("================================================================================");	
 	
	Vector<Pdb> pdbVec2 = pp.newPdb(pdbVec, mm);
	return pdbVec2;   	
    	
    }
    
  /**
   * A recusive function to compute all the backbone Phi/Psi for an
   * n-residue fragment.
   * 
   * @param rdcVec1  an array of CH RDCs for the fragment
   * @param rdcVec2  an array of NH RDCs for the fragment
   * @param rdc1Cal has the back-computed CH RDCs for handling missing data
   * @param rdc2Cal has the back-computed NH RDCs for handling missing data
   * @param Syy  Saupe elements
   * @param Szz  Saupe elements
   * @param rmsd1 the CH RDC rmsd before the refinement
   * @param rmsd2 the NH RDC rmsd before the refinement
   * @param pdbS1 the computed Strands H-bonded to the to be refined Strand
   * @param hbVecOfE12 the vector with H-bond information between the two strands
   * @param pdbVec2 the pdb vec2
   * @param nCycle the n cycle
   * @param weight4Angles the weight4 angles
   * @param hbWeight the hb weight
   * @param debugDFS the debug dfs
   * @param printResults the print results
   * 
   * @return the computed Strand
   */
    public Vector minBeta(Vector rdcVec1, Vector rdcVec2, double[] rdc1Cal, double[] rdc2Cal, 
			  Vector pdbVec2, double Syy, double Szz, double rmsd1, double rmsd2, Vector pdbS1,
			  Vector hbVecOfE12, int nCycle, double weight4Angles, double hbWeight,
			  boolean debugDFS, boolean printResults){  
	int i, j, k;
	final int N = pdbVec2.size() - 1; 
	Dipolar dd1 = (Dipolar)rdcVec1.elementAt(0);
	Dipolar dd2 = (Dipolar)rdcVec2.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());

	double [] rdcArr1 = new double[N]; //CH RDCs, sampled based on Gaussina Dis, use array for efficient 
	double [] rdcArr2 = new double[N]; //NH RDCs, use array for efficient 
	double [] rdcExp1 = new double[N]; //store the experimental values in the array
	double [] rdcExp2 = new double[N];
	int [] seqNos1 = new int[N];
	int [] seqNos2 = new int[N];
	int no = 0;    
	int index = -1;
	//Filled in missing RDCs
	for (j = 0; j < N; j++){
	    no = firstResidueNo+j;
	    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1){
		rdcExp1[j] = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
		seqNos1[j] = 1;
	    }else {
		rdcExp1[j] =  rdc1Cal[j];
		seqNos1[j] = 0;
	    }
	    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1){
		rdcExp2[j] = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
		seqNos2[j] = 1;
	    }else {
		rdcExp2[j] =  rdc2Cal[j];
		seqNos2[j] = 0;
	    }
	}
	//Compute the first rotation matrix to a Coordinate frame defined in the first plane
	Cartesian cc = new Cartesian();
	String atom = "";
	double [] amide = new double[3];
	double [] ca = new double[3];
	double [] nh = new double[3];
	Pdb pp = (Pdb)pdbVec2.elementAt(0);
	Vector atomVec = pp.getAtomVec();
        for (j=0; j<atomVec.size(); j++){
            cc = (Cartesian)atomVec.elementAt(j);
            atom = cc.getAtom();
            if (atom.equals("N"))
                amide = cc.getXYZ();
            else if (atom.equals("H"))
                nh = cc.getXYZ();
            else if (atom.equals("CA"))
                ca = cc.getXYZ();
        }
	double[] nToNHVec = internuclearVec(amide, nh);
        double[] nToCAVec = internuclearVec(amide, ca);

        ppGlobal ppg = new ppGlobal();
        Matrix mat = ppg.RgCal(nToNHVec, nToCAVec);

	boolean flag1 = false;
	double rRdc1 = 10000.0, rRdc2 = 10000.0, rT = 100000.0, rTotal = 100000.0;

	long seed = 95738579;
	Random rr = new Random(seed);
	double rms1 = 0.0, rms2 = 0.0;
	double u, v; //for CH and NH RDC respectively
	int depth0 = 0;
	Vector ppVec = new Vector();
	double[] ppS  = new double[2*N]; 
	double[] phiS = new double[N]; 
	double[] psiS = new double[N]; 
	double[] phiSave = new double[N]; 
	double[] psiSave = new double[N]; 
	double[] rdc1Save = new double[N];   //CH Rdcs
	double[] rdc2Save = new double[N]; 

	int noOfSln = 0;
	int n = 0;	
	int N11 =  rdcVec1.size();
	int N22 =  rdcVec2.size();
	double phiRmsd = 0.0;
	double psiRmsd = 0.0;
	double rRdcFinal1 = rmsd1 * rmsd1 * N11; // that's the total for all residues, we want 2 fold better
	double rRdcFinal2 = rmsd2 * rmsd2 * N22; // the pairs should has quite close values
	System.out.println(rmsd1+"  "+rmsd2+" : "+N11+"  "+N22);
	//Sample the RDC to select a set satisfy certain conditions. specified later
	long startTime = System.currentTimeMillis();
	boolean depthFlag = false;
	boolean rightHand = true;
	PhiPsi ff = new PhiPsi();
	PdbRmsd pR = new PdbRmsd();
	double phiAve = Const.phiAveBeta;
	double psiAve = Const.psiAveBeta;
	double matchScore = 0.0;
  	for (int mmm =0; mmm < nCycle; mmm++){
	    flag1 = false;
	    while (!flag1){
		flag1 = true;
		rRdc1 = 0.0;
		rRdc2 = 0.0;
		for (j = 0; j < N; j++){   //For the 1st beta strand of the pair
		    rdcArr1[j] = rdcExp1[j] + rmsd1 * rr.nextGaussian(); 
		    rdcArr2[j] = rdcExp2[j] + rmsd2 * rr.nextGaussian();
		    rms1 = Math.abs(rdcArr1[j] - rdcExp1[j]);
		    rRdc1 += rms1 * rms1 * seqNos1[j] ;
		    rms2 = Math.abs(rdcArr2[j] - rdcExp2[j]);
		    rRdc2 += rms2 * rms2 * seqNos2[j] ;
		}
		if ( (rRdc1 > rRdcFinal1) || (rRdc2 > rRdcFinal2) )
		    flag1 = false;
		else flag1 = true;
	    }
	    ppVec = new Vector();
    	    depthFlag = ff.phiPsiChain(rdcArr1, rdcArr2, mat, Syy, Szz, depth0, N - 1, ppS, ppVec, rightHand, false);
	    if (ppVec.size() > 0 ){
		noOfSln = (int)ppVec.size() / (2*N);  //the total number of solutions for this rdc set
		for(k=0; k < noOfSln; k++){  //please note the index
		    phiRmsd = 0.0;
		    psiRmsd = 0.0;
		    for (n = k* 2 * N; n < (k+1)* 2 *N; n += 2){  //original "N"
			phi = ((Double)ppVec.elementAt(n)).doubleValue();
			psi = ((Double)ppVec.elementAt(n+1)).doubleValue();
			phiS[(int)(n - k* 2 * N) / 2] = phi;
			psiS[(int)(n - k* 2 * N) / 2] = psi;
			phiRmsd += (phi - phiAve) * (phi - phiAve);
			psiRmsd += (psi - psiAve) * (psi - psiAve);
		    }
		    pdbVec2 = modelBuild(phiS, psiS, amide, nh, ca, firstResidueNo, false);
		    matchScore = pR.centerFit(pdbS1, pdbVec2, hbVecOfE12, false);

		    rT = Math.sqrt(rRdc1 / N11) + Math.sqrt(rRdc2 / (N22 - 1) )
			+ weight4Angles*(Math.sqrt(phiRmsd/N) + Math.sqrt(psiRmsd/N)) + hbWeight * matchScore;

		    if (rT < rTotal){
			rTotal = rT;
			System.out.println( mmm +": "+Math.sqrt(rRdc1 / N11 )+"  "
					    + Math.sqrt(rRdc2 / (N22 - 1) )+";  "+(Math.sqrt(phiRmsd / N ) / Const.cst) 
					    + "  "+(Math.sqrt(psiRmsd / N) / Const.cst)+"  "+matchScore+", RT = "+rTotal);
			System.arraycopy(phiS, 0, phiSave, 0, N);
			System.arraycopy(psiS, 0, psiSave, 0, N);
			System.arraycopy(rdcArr1, 0, rdc1Save, 0, N);
			System.arraycopy(rdcArr2, 0, rdc2Save, 0, N);
		    }
		}
	    }
	}
	long endTime = System.currentTimeMillis() ;
	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes

 	Vector phiPsiVec  = new Vector();
  	for (i=0; i<N; i++)  //build the model
       	    phiPsiVec.add(new PhiPsi(i, "ALA", phiSave[i], psiSave[i]));
   	Vector pdbHelixN = modelBuild(phiSave, psiSave, amide, nh, ca, firstResidueNo, false); 

	if(printResults){
            System.out.println(" CH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc1Save[k] +"  "+rdcExp1[k]+"  "+Math.abs(rdc1Save[k]-rdcExp1[k]));
            System.out.println(" NH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc2Save[k] +"  "+rdcExp2[k]+"  "+Math.abs(rdc2Save[k]-rdcExp2[k]));

            System.out.println(" Phi/Psi: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+"   "+(phiSave[k]/Const.cst)+"  "+(psiSave[k]/Const.cst));
        }
	return pdbHelixN;
    }

 /**
  * A recusive function to compute all the backbone Phi/Psi for an
  * n-residue fragment.
  * 
  * @param rdcVec1  an array of CH RDCs for the fragment
  * @param rdcVec2  an array of NH RDCs for the fragment
  * @param rdc1Cal has the back-computed CH RDCs for handling missing data
  * @param rdc2Cal has the back-computed NH RDCs for handling missing data
  * @param Syy  Saupe elements
  * @param Szz  Saupe elements
  * @param rmsd1 the CH RDC rmsd before the refinement
  * @param rmsd2 the NH RDC rmsd before the refinement
  * @param pdbS1 the computed Strands H-bonded to the to be refined Strand
  * @param hbVecOfE12 the vector with H-bond information between the two strands
  * @param pdbVec2 the pdb vec2
  * @param pdbS2 the pdb s2
  * @param hbVecOfE13 the hb vec of e13
  * @param nCycle the n cycle
  * @param weight4Angles the weight4 angles
  * @param hbWeight the hb weight
  * @param debugDFS the debug dfs
  * @param printResults the print results
  * 
  * @return the computed Strand
  */
    public Vector minBeta2(Vector rdcVec1, Vector rdcVec2, double[] rdc1Cal, double[] rdc2Cal, 
			   Vector pdbVec2, double Syy, double Szz, double rmsd1, double rmsd2, Vector pdbS1,
			   Vector pdbS2, Vector hbVecOfE12, Vector hbVecOfE13, int nCycle, double weight4Angles, 
			   double hbWeight, boolean debugDFS, boolean printResults){  
	int i, j, k;
	final int N = pdbVec2.size() - 1; 
	Dipolar dd1 = (Dipolar)rdcVec1.elementAt(0);
	Dipolar dd2 = (Dipolar)rdcVec2.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());

	double [] rdcArr1 = new double[N]; //CH RDCs, sampled based on Gaussina Dis, use array for efficient 
	double [] rdcArr2 = new double[N]; //NH RDCs, use array for efficient 
	double [] rdcExp1 = new double[N]; //store the experimental values in the array
	double [] rdcExp2 = new double[N];
	int [] seqNos1 = new int[N];
	int [] seqNos2 = new int[N];
	int no = 0;    
	int index = -1;
	//Filled in missing RDCs
	for (j = 0; j < N; j++){
	    no = firstResidueNo+j;
	    index = Collections.binarySearch(rdcVec1, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1){
		rdcExp1[j] = ((Dipolar)rdcVec1.elementAt(index)).getRdc();
		seqNos1[j] = 1;
	    }else {
		rdcExp1[j] =  rdc1Cal[j];
		seqNos1[j] = 0;
	    }
	    index = Collections.binarySearch(rdcVec2, new Dipolar(no), new Dipolar.rdcComparator());
	    if (index > - 1){
		rdcExp2[j] = ((Dipolar)rdcVec2.elementAt(index)).getRdc();
		seqNos2[j] = 1;
	    }else {
		rdcExp2[j] =  rdc2Cal[j];
		seqNos2[j] = 0;
	    }
	}
	//Compute the first rotation matrix to a Coordinate frame defined in the first plane
	Cartesian cc = new Cartesian();
	String atom = "";
	double [] amide = new double[3];
	double [] ca = new double[3];
	double [] nh = new double[3];
	Pdb pp = (Pdb)pdbVec2.elementAt(0);
	Vector atomVec = pp.getAtomVec();
        for (j=0; j<atomVec.size(); j++){
            cc = (Cartesian)atomVec.elementAt(j);
            atom = cc.getAtom();
            if (atom.equals("N"))
                amide = cc.getXYZ();
            else if (atom.equals("H"))
                nh = cc.getXYZ();
            else if (atom.equals("CA"))
                ca = cc.getXYZ();
        }
	double[] nToNHVec = internuclearVec(amide, nh);
        double[] nToCAVec = internuclearVec(amide, ca);

        ppGlobal ppg = new ppGlobal();
        Matrix mat = ppg.RgCal(nToNHVec, nToCAVec);

	boolean flag1 = false;
	double rRdc1 = 10000.0, rRdc2 = 10000.0, rT = 100000.0, rTotal = 100000.0;

	long seed = 95738579;
	Random rr = new Random(seed);
	double rms1 = 0.0, rms2 = 0.0;
	double u, v; //for CH and NH RDC respectively
	int depth0 = 0;
	Vector ppVec = new Vector();
	double[] ppS  = new double[2*N]; 
	double[] phiS = new double[N]; 
	double[] psiS = new double[N]; 
	double[] phiSave = new double[N]; 
	double[] psiSave = new double[N]; 
	double[] rdc1Save = new double[N];   //CH Rdcs
	double[] rdc2Save = new double[N]; 

	int noOfSln = 0;
	int n = 0;	
	int N11 =  rdcVec1.size();
	int N22 =  rdcVec2.size();
	double phiRmsd = 0.0;
	double psiRmsd = 0.0;
	double rRdcFinal1 = rmsd1 * rmsd1 * N11; // that's the total for all residues, we want 2 fold better
	double rRdcFinal2 = rmsd2 * rmsd2 * N22; // the pairs should has quite close values
	System.out.println(rmsd1+"  "+rmsd2+" : "+N11+"  "+N22);
	//Sample the RDC to select a set satisfy certain conditions. specified later
	long startTime = System.currentTimeMillis();
	boolean depthFlag = false;
	boolean rightHand = true;
	PhiPsi ff = new PhiPsi();
	PdbRmsd pR = new PdbRmsd();
	double phiAve = Const.phiAveBeta;
	double psiAve = Const.psiAveBeta;
	double matchScore = 0.0;
  	for (int mmm =0; mmm < nCycle; mmm++){
	    flag1 = false;
	    while (!flag1){
		flag1 = true;
		rRdc1 = 0.0;
		rRdc2 = 0.0;
		for (j = 0; j < N; j++){   //For the 1st beta strand of the pair
		    rdcArr1[j] = rdcExp1[j] + rmsd1 * rr.nextGaussian(); 
		    rdcArr2[j] = rdcExp2[j] + rmsd2 * rr.nextGaussian();
		    rms1 = Math.abs(rdcArr1[j] - rdcExp1[j]);
		    rRdc1 += rms1 * rms1 * seqNos1[j] ;
		    rms2 = Math.abs(rdcArr2[j] - rdcExp2[j]);
		    rRdc2 += rms2 * rms2 * seqNos2[j] ;
		}
		if ( (rRdc1 > rRdcFinal1) || (rRdc2 > rRdcFinal2) )
		    flag1 = false;
		else flag1 = true;
	    }
	    ppVec = new Vector();
    	    depthFlag = ff.phiPsiChain(rdcArr1, rdcArr2, mat, Syy, Szz, depth0, N - 1, ppS, ppVec, rightHand, false);
	    if (ppVec.size() > 0 ){
		noOfSln = (int)ppVec.size() / (2*N);  //the total number of solutions for this rdc set
		for(k=0; k < noOfSln; k++){  //please note the index
		    phiRmsd = 0.0;
		    psiRmsd = 0.0;
		    for (n = k* 2 * N; n < (k+1)* 2 *N; n += 2){  //original "N"
			phi = ((Double)ppVec.elementAt(n)).doubleValue();
			psi = ((Double)ppVec.elementAt(n+1)).doubleValue();
			phiS[(int)(n - k* 2 * N) / 2] = phi;
			psiS[(int)(n - k* 2 * N) / 2] = psi;
			phiRmsd += (phi - phiAve) * (phi - phiAve);
			psiRmsd += (psi - psiAve) * (psi - psiAve);
		    }
		    pdbVec2 = modelBuild(phiS, psiS, amide, nh, ca, firstResidueNo, false);
		    matchScore = pR.centerFit(pdbS1, pdbVec2, hbVecOfE12, false) 
			        + pR.centerFit(pdbVec2, pdbS2, hbVecOfE13, false);

		    rT = Math.sqrt(rRdc1 / N11) + Math.sqrt(rRdc2 / (N22 - 1) )
			+ weight4Angles*(Math.sqrt(phiRmsd/N) + Math.sqrt(psiRmsd/N)) + hbWeight * matchScore;

		    if (rT < rTotal){
			rTotal = rT;
			System.out.println( mmm +": "+Math.sqrt(rRdc1 / N11 )+"  "
					    + Math.sqrt(rRdc2 / (N22 - 1) )+";  "+(Math.sqrt(phiRmsd / N ) / Const.cst) 
					    + "  "+(Math.sqrt(psiRmsd / N) / Const.cst)+"  "+matchScore+", RT = "+rTotal);
			System.arraycopy(phiS, 0, phiSave, 0, N);
			System.arraycopy(psiS, 0, psiSave, 0, N);
			System.arraycopy(rdcArr1, 0, rdc1Save, 0, N);
			System.arraycopy(rdcArr2, 0, rdc2Save, 0, N);
		    }
		}
	    }
	}
	long endTime = System.currentTimeMillis() ;
	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes

 	Vector phiPsiVec  = new Vector();
  	for (i=0; i<N; i++)  //build the model
       	    phiPsiVec.add(new PhiPsi(i, "ALA", phiSave[i], psiSave[i]));
   	Vector pdbHelixN = modelBuild(phiSave, psiSave, amide, nh, ca, firstResidueNo, false); 
	if(printResults){
            System.out.println(" CH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc1Save[k] +"  "+rdcExp1[k]+"  "+Math.abs(rdc1Save[k]-rdcExp1[k]));
            System.out.println(" NH RDC: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+": "+rdc2Save[k] +"  "+rdcExp2[k]+"  "+Math.abs(rdc2Save[k]-rdcExp2[k]));

            System.out.println(" Phi/Psi: ");
            for (k=0; k < N; k++)
                System.out.println((firstResidueNo+k)+"   "+(phiSave[k]/Const.cst)+"  "+(psiSave[k]/Const.cst));
        }
	return pdbHelixN;
    }

    /**
     * Compute the turns based on NOEs and RDCs.
     * 
     * @param no1 the first residue of the fragment (turn and loop)
     * @param no2 the last residue of the fragment (turn and loop)
     * @param h1Vec resonance assignment sorted by H1 CS shift.
     * @param asgVec the merged resonacs assigement and HN and HA NOEs
     * @param rdcVec1 the CH RDC
     * @param rdcVec2 the NH RDC
     * @param a1  the constant used to convert the intensity to distance
     * @param a2  the constant used to convert the intensity to distance
     * @param pdbVec1 the structure for the beginning side of turn or loop
     * @param pdbVec2 the structure for the end side of turn or loop
     * @param Syy the diagnolized Saupe elements
     * @param Szz the diagnolized Saupe elements
     * 
     * @return  the refined Pdb vector
     */
    public Vector minTurns(final Vector h1Vec, final Vector asgVec, Vector pdbVec1, Vector pdbVec2,  
			   final Vector rdcVec1, final Vector rdcVec2, double Syy, double Szz, int no1, int no2,
			   double a1, double a2){
	int i = 0, j = 0;
	PhiPsi ff = new PhiPsi();
	double [] amide1 = new double[3];
	double [] nh1 = new double[3];
	double [] ca1 = new double[3];
	double [] amide2 = new double[3];
	double [] nh2 = new double[3];
	double [] ca2 = new double[3];
	Cartesian cc = new Cartesian();
	String atom = "";
	//Extract the n, nh, ca coordinates of the first residue of turn, which the pdbVec1 have.
	int index = Collections.binarySearch(pdbVec1, new Pdb(no1), new Pdb.PdbComparator());
	if (index < 0){
	    System.out.println("Error in the PDB file 1");
	    return new Vector();
	}
	Pdb pp = (Pdb)pdbVec1.elementAt(index); 
	String resid = pp.getResidue();
	Vector atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide1 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca1 = cc.getXYZ();
	    else if (atom.equals("H"))
		nh1 = cc.getXYZ();
	}
	double[] nToNHVec = internuclearVec(amide1, nh1);
	double[] nToCAVec = internuclearVec(amide1, ca1);
	Matrix rg = ff.RgCal(nToNHVec, nToCAVec);
	//Extract the n, nh, ca coordinates of the first residue of turn, which the pdbVec1 have.
	index = Collections.binarySearch(pdbVec2, new Pdb(no2), new Pdb.PdbComparator());
	if (index < 0){
	    System.out.println("Error in the PDB file 2");
	    return new Vector();
	}
	pp = (Pdb)pdbVec2.elementAt(index); 
	resid  = pp.getResidue();
	atomVec = pp.getAtomVec();
	for (j=0; j<atomVec.size(); j++){
	    cc = (Cartesian)atomVec.elementAt(j);
	    atom = cc.getAtom();
	    if (atom.equals("N"))
		amide2 = cc.getXYZ();
	    else if (atom.equals("CA"))
		ca2= cc.getXYZ();
	    else if (atom.equals("H"))
		nh2 = cc.getXYZ();
	}
	int depth = 0;
	int N = no2 - no1;
	Vector ppVec = new Vector();
	double[] ppS  = new double[2*N];
	double[][] restraints = ff.extractRestraints(no1, no2, h1Vec, asgVec, rdcVec1, rdcVec2, a1, a2);
	int numberOfResidues = 11 - 8 + 1; 
	for (i=0; i<numberOfResidues; i++){
	    for (j=0; j<5; j++)
		System.out.print(restraints[i][j]+"  ");
	    System.out.println();
	}

 	return ppVec;
    }


    /**
     * Refine Saupe elements from an initial model.
     * 
     * @param rdc1Vec NH RDCs
     * @param rdc2Vec CH RDCs
     * @param refineCycle the number of cycles for iterating through the refinement
     * and computation of Saupe matrices.
     * @param nCycle  the number of cycles for the refinement by DFS itself.
     * @param ramaFilter the Ramachandran filter for the favorable phi and psi region
     * @param phiAve  the average phi angle
     * @param psiAve  the average psi angle
     * @param debugDFS  for monitoring the progress in systematic search-based minimization
     * @param printResults for printing the results for each iteration
     * @param rdcCaCoVec the rdc ca co vec
     * @param rdcCoNVec the rdc co n vec
     * @param saupe the saupe
     * @param vecTalos the vec talos
     * 
     * @return  the refined Pdb vector
     * 
     * @throws JampackException the jampack exception
     */
    public Vector refineSaupe4RDCsWOAT (final Vector rdc1Vec, final Vector rdc2Vec, 
    		final Vector rdcCaCoVec, final Vector rdcCoNVec, double[] saupe, 
			       double[] ramaFilter, double phiAve, double psiAve, int refineCycle,
			       int nCycle, boolean debugDFS, boolean printResults, Vector vecTalos)throws JampackException
    {
		int i = 0;
	    double [] n1 = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
	    double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};

		//build an initial  helix model with average \phi and \psi angles.
		Pdb pp = new Pdb();
		Vector phiPsiVec = new Vector();
		Dipolar dd1 = (Dipolar)rdc1Vec.elementAt(0);
		Dipolar dd2 = (Dipolar)rdc2Vec.elementAt(0);
		int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());
	
		dd1 = (Dipolar)rdc1Vec.elementAt(rdc1Vec.size() - 1);
		dd2 = (Dipolar)rdc2Vec.elementAt(rdc2Vec.size() - 1);
		int lastResidueNo  = Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;
		int N = lastResidueNo - firstResidueNo;
	    for (i=firstResidueNo; i<lastResidueNo; i++)
	        phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
	    Vector pdbVec = modelBuild(phiPsiVec, n1, nh1, ca1);
	    
		//Compute by SVD the 7 parameters used for computing other 2ary elements
	 	PdbRdc pdr = new PdbRdc();
		double[] rdc1Cal  = new double[pdbVec.size() - 1];
		double[] rdc2Cal  = new double[pdbVec.size() - 1];
		double weight4Angles = 10.0; //the relative weight for \phi and \psi angles vs. RDCs in target function
		Matrix mm = new Matrix(3, 3);

		Vector pdbVec1 = new Vector();
		Vector pdbVec2 = new Vector();
		boolean isHelix = true; //false;
		mm = pdr.bestFit(pdbVec, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
				
		pdbVec1 = pp.newPdb(pdbVec, mm);
		pdbVec2 = minHelix4RDCs(rdc2Vec, rdc1Vec,rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,1.0,1.0);
				
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec,rdc1Cal, rdc2Cal, saupe);
	  	pdbVec1 = pp.newPdb(pdbVec2, mm);
	  	
		pdbVec2 = minHelix4RDCs(rdc2Vec, rdc1Vec, rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1,  saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,1.0,1.0);
	 	mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec,rdc1Cal, rdc2Cal, saupe);
	  	pdbVec1 = pp.newPdb(pdbVec2, mm);
		pdbVec2 = minHelix4RDCs(rdc2Vec, rdc1Vec,rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,1.0,1.0);
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
		pdbVec1 = pp.newPdb(pdbVec2, mm);
	 	nCycle = 128 * 1024;
		pdbVec2 = minHelix4RDCs(rdc2Vec, rdc1Vec, rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,1.0,1.0);
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec,rdc1Cal, rdc2Cal, saupe);

	
	  	Vector pdbVecN = pp.newPdb(pdbVec2, mm);
	
	 	return pdbVecN;
    }
    
    /**
     * Refine Saupe elements from an initial model.
     * 
     * @param rdc1Vec NH RDCs
     * @param rdc2Vec CH RDCs
     * @param refineCycle the number of cycles for iterating through the refinement
     * and computation of Saupe matrices.
     * @param nCycle  the number of cycles for the refinement by DFS itself.
     * @param ramaFilter the Ramachandran filter for the favorable phi and psi region
     * @param phiAve  the average phi angle
     * @param psiAve  the average psi angle
     * @param debugDFS  for monitoring the progress in systematic search-based minimization
     * @param printResults for printing the results for each iteration
     * @param vecPreBB the vec pre bb
     * @param pdbVec the pdb vec
     * @param rdcCaCoVec the rdc ca co vec
     * @param rdcCoNVec the rdc co n vec
     * @param saupe the saupe
     * @param vecTalos the vec talos
     * @param wtCoCa the wt co ca
     * @param wtCoN the wt co n
     * @param starNo the star no
     * @param endNo the end no
     * @param vecSeq the vec seq
     * 
     * @return  the refined Pdb vector
     * 
     * @throws JampackException the jampack exception
     */
    public Vector refineSaupe4RDCsWOAT (Vector vecPreBB, Vector pdbVec,final Vector rdc1Vec, final Vector rdc2Vec, 
    		final Vector rdcCaCoVec, final Vector rdcCoNVec, double[] saupe, 
			       double[] ramaFilter, double phiAve, double psiAve, int refineCycle,
			       int nCycle, boolean debugDFS, boolean printResults, Vector vecTalos,
			       double wtCoCa, double wtCoN, int starNo, int endNo, Vector vecSeq  )throws JampackException
    {
		int i = 0;
	   
		Pdb pp = new Pdb();
	
		//Compute by SVD the 7 parameters used for computing other 2ary elements
	 	PdbRdc pdr = new PdbRdc();
		double[] rdc1Cal  = new double[pdbVec.size() ];
		double[] rdc2Cal  = new double[pdbVec.size() ];
		double weight4Angles = 10.0; //the relative weight for \phi and \psi angles vs. RDCs in target function
		Matrix mm = new Matrix(3, 3);

		Vector pdbVec1 = new Vector();
		Vector pdbVec2 = new Vector();
		boolean isHelix = true; //false;
		//mm = pdr.bestFit4RDCs(pdbVec, rdc1Vec, rdc2Vec,rdcCaCoVec, rdcCoNVec,rdc1Cal, rdc2Cal, saupe);
		mm = pdr.bestFit(pdbVec, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
				
		pdbVec1 = pp.newPdb(pdbVec, mm);
		pdbVec2 = minHelix4RDCs(vecPreBB,rdc2Vec, rdc1Vec,rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,wtCoCa,wtCoN, starNo, endNo,vecSeq );
				
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec,rdc1Cal, rdc2Cal, saupe);
	  	pdbVec1 = pp.newPdb(pdbVec2, mm);
	  	
		pdbVec2 = minHelix4RDCs(vecPreBB,rdc2Vec, rdc1Vec, rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1,  saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,wtCoCa,wtCoN, starNo, endNo,vecSeq );
	 	mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec,rdc1Cal, rdc2Cal, saupe);
	  	pdbVec1 = pp.newPdb(pdbVec2, mm);
		
	  	pdbVec2 = minHelix4RDCs(vecPreBB,rdc2Vec, rdc1Vec,rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,wtCoCa,wtCoN, starNo, endNo,vecSeq );
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
		pdbVec1 = pp.newPdb(pdbVec2, mm);
	 	
		/////
		pdbVec2 = minHelix4RDCs(vecPreBB,rdc2Vec, rdc1Vec,rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,wtCoCa,wtCoN, starNo, endNo,vecSeq );
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
		pdbVec1 = pp.newPdb(pdbVec2, mm);
		//
		////	/
		pdbVec2 = minHelix4RDCs(vecPreBB,rdc2Vec, rdc1Vec,rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,wtCoCa,wtCoN, starNo, endNo,vecSeq );
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
		pdbVec1 = pp.newPdb(pdbVec2, mm);
		//
		
		////	/
		pdbVec2 = minHelix4RDCs(vecPreBB,rdc2Vec, rdc1Vec,rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,wtCoCa,wtCoN, starNo, endNo,vecSeq );
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
		pdbVec1 = pp.newPdb(pdbVec2, mm);
		//
		////	/
		pdbVec2 = minHelix4RDCs(vecPreBB,rdc2Vec, rdc1Vec,rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,wtCoCa,wtCoN, starNo, endNo,vecSeq );
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
		pdbVec1 = pp.newPdb(pdbVec2, mm);
		//
		nCycle = 128 * 1024;
		pdbVec2 = minHelix4RDCs(vecPreBB,rdc2Vec, rdc1Vec, rdcCaCoVec, rdcCoNVec,rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
				   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos,wtCoCa,wtCoN, starNo, endNo,vecSeq );
		mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec,rdc1Cal, rdc2Cal, saupe);


	  	Vector pdbVecN = pp.newPdb(pdbVec2, mm);
	
	 	return pdbVecN;
    }


    /**
     * Refine Saupe elements from an initial model.
     * 
     * @param rdc1Vec NH RDCs
     * @param rdc2Vec CH RDCs
     * @param refineCycle the number of cycles for iterating through the refinement
     * and computation of Saupe matrices.
     * @param nCycle  the number of cycles for the refinement by DFS itself.
     * @param ramaFilter the Ramachandran filter for the favorable phi and psi region
     * @param phiAve  the average phi angle
     * @param psiAve  the average psi angle
     * @param debugDFS  for monitoring the progress in systematic search-based minimization
     * @param printResults for printing the results for each iteration
     * @param saupe the saupe
     * @param vecTalos the vec talos
     * 
     * @return  the refined Pdb vector
     * 
     * @throws JampackException the jampack exception
     */
    public Vector refineSaupe (final Vector rdc1Vec, final Vector rdc2Vec, double[] saupe, 
			       double[] ramaFilter, double phiAve, double psiAve, int refineCycle,
			       int nCycle, boolean debugDFS, boolean printResults, Vector vecTalos)throws JampackException
    {
	int i = 0;
    	double [] n1 = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};

	//build an initial  helix model with average \phi and \psi angles.
	Pdb pp = new Pdb();
	Vector phiPsiVec = new Vector();
	Dipolar dd1 = (Dipolar)rdc1Vec.elementAt(0);
	Dipolar dd2 = (Dipolar)rdc2Vec.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());

	dd1 = (Dipolar)rdc1Vec.elementAt(rdc1Vec.size() - 1);
	dd2 = (Dipolar)rdc2Vec.elementAt(rdc2Vec.size() - 1);
	int lastResidueNo  = Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;
	int N = lastResidueNo - firstResidueNo;
        for (i=firstResidueNo; i<lastResidueNo; i++)
            phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
        Vector pdbVec = modelBuild(phiPsiVec, n1, nh1, ca1);

        //Compute by SVD the 7 parameters used for computing other 2ary elements
 	PdbRdc pdr = new PdbRdc();
	double[] rdc1Cal  = new double[pdbVec.size() - 1];
	double[] rdc2Cal  = new double[pdbVec.size() - 1];
	double weight4Angles = 10.0; //the relative weight for \phi and \psi angles vs. RDCs in target function
	Matrix mm = new Matrix(3, 3);

	Vector pdbVec1 = new Vector();
	Vector pdbVec2 = new Vector();
	boolean isHelix = true; //false;
	mm = pdr.bestFit(pdbVec, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
	pdbVec1 = pp.newPdb(pdbVec, mm);
	pdbVec2 = minHelix(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
			   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos);
	mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
  	pdbVec1 = pp.newPdb(pdbVec2, mm);
	pdbVec2 = minHelix(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbVec1,  saupe[0], saupe[1], 
			   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos);
 	mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
  	pdbVec1 = pp.newPdb(pdbVec2, mm);
	pdbVec2 = minHelix(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
			   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos);
	mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
	pdbVec1 = pp.newPdb(pdbVec2, mm);
 	nCycle = 128 * 1024;
	pdbVec2 = minHelix(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbVec1, saupe[0], saupe[1], 
			   saupe[2], saupe[3], nCycle, weight4Angles, debugDFS, printResults, isHelix,vecTalos);
	mm = pdr.bestFit(pdbVec2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);

  	Vector pdbVecN = pp.newPdb(pdbVec2, mm);

 	return pdbVecN;
    }

    /**
     * This is an old function. Can be deleted.
     * Update the Saupe elements
     * 
     * @param pdbVec the Pdb vector before the update
     * @param paraVec the set of parameters for computing fragment etc.
     * @param eRdc1Vec RDC data for beta-strands //the following fours are just for convenience
     * @param eRdc2Vec RDC data for beta-strands
     * @param helixRdc1 the helix rdc1
     * @param helixRdc2 the helix rdc2
     * @param saupeSave the saupe save
     * 
     * @return  the refined Pdb vector.
     */
    public Vector refineSaupe2 (Vector pdbVec, Vector paraVec, Vector eRdc1Vec, Vector eRdc2Vec,
				Vector helixRdc1, Vector helixRdc2, double[] saupeSave){
        int i = 0;
        PdbRdc pdr = new PdbRdc();
	Pdb pp = new Pdb();
        Vector pdbVecN = new Vector();

	Vector strand1Rdc1 = (Vector)eRdc1Vec.elementAt(0); //2 - 7 representing individual ones (S1)
	Vector strand1Rdc2 = (Vector)eRdc2Vec.elementAt(0);
	Vector strand2Rdc1 = (Vector)eRdc1Vec.elementAt(1); //11 - 17 representing individual ones (S1)
	Vector strand2Rdc2 = (Vector)eRdc2Vec.elementAt(1);
	Vector strand3Rdc1 = (Vector)eRdc1Vec.elementAt(2); //41 - 45 representing individual ones (S1)
	Vector strand3Rdc2 = (Vector)eRdc2Vec.elementAt(2);
	Vector strand5Rdc1 = (Vector)eRdc1Vec.elementAt(4); //65 - 70 representing individual ones (S1)
	Vector strand5Rdc2 = (Vector)eRdc2Vec.elementAt(4);
	Vector nhRdc = new Vector();
	for(i=1; i< strand1Rdc1.size(); i++)  //remove 1st NH RDC which is NOT used in the computation
	    nhRdc.add(strand1Rdc1.elementAt(i));
	for(i=1; i< strand2Rdc1.size(); i++)
	    nhRdc.add(strand2Rdc1.elementAt(i));
	for(i=1; i< helixRdc1.size(); i++)
	    nhRdc.add(helixRdc1.elementAt(i));
// 	for(i=1; i< strand3Rdc1.size(); i++)
// 	    nhRdc.add(strand3Rdc1.elementAt(i));
	for(i=1; i< strand5Rdc1.size(); i++)
	    nhRdc.add(strand5Rdc1.elementAt(i));
	Vector cahaRdc = new Vector();
	cahaRdc.addAll(strand1Rdc2);
	cahaRdc.addAll(strand2Rdc2);
	cahaRdc.addAll(helixRdc2);

	cahaRdc.addAll(strand5Rdc2);

     //compute new Saupes
	int N = pdbVec.size();
 	double[] rdc1Cal = new double[N];
	double[] rdc2Cal = new double[N];
        Matrix mm = pdr.bestFit(pdbVec, nhRdc, cahaRdc, rdc1Cal, rdc2Cal, saupeSave);
 	Vector pdbVec2 = pp.newPdb(pdbVec, mm);

	return pdbVec2;

    }
    
    /**
     * Refine helix22.
     * 
     * @param vecBB the vec bb
     * @param rdc1Vec the rdc1 vec
     * @param rdc2Vec the rdc2 vec
     * @param Syy the syy
     * @param Szz the szz
     * @param ramaFilter the rama filter
     * @param phiAve the phi ave
     * @param psiAve the psi ave
     * @param refineCycle the refine cycle
     * @param initialCycle the initial cycle
     * @param w4Angles the w4 angles
     * @param resolution the resolution
     * @param debugDFS the debug dfs
     * @param printResults the print results
     * @param nhRdcRmsd the nh rdc rmsd
     * @param chRdcRmsd the ch rdc rmsd
     * 
     * @return the vector< pdb>
     */
    public Vector<Pdb> refineHelix22(Vector vecBB, final Vector<Dipolar> rdc1Vec, final Vector<Dipolar> rdc2Vec, double Syy, double Szz,
		       double[] ramaFilter, double phiAve, double psiAve, int refineCycle, int initialCycle,
		       double w4Angles, double resolution, boolean debugDFS, boolean printResults, double[] nhRdcRmsd, double[] chRdcRmsd)
	{
    	int i = 0;
		double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
		double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	
		//build an initial model
		Pdb pp = new Pdb();
		Vector<PhiPsi> phiPsiVec = new Vector<PhiPsi>();
		Dipolar dd1 = (Dipolar)rdc1Vec.elementAt(0);
		Dipolar dd2 = (Dipolar)rdc2Vec.elementAt(0);
		int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());
		
		
		dd1 = (Dipolar)rdc1Vec.elementAt(rdc1Vec.size() - 1);
		dd2 = (Dipolar)rdc2Vec.elementAt(rdc2Vec.size() - 1);
		int lastResidueNo  =  Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;
		for (i=firstResidueNo; i<lastResidueNo; i++)  
			    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
		
		Vector<Pdb> pdbStrand=new Vector();
		pdbStrand.addAll(vecBB);
	
		//Compute the orientation of the first peptide by grid-search
		PdbRdc pdr = new PdbRdc();
		double [] rmsds = new double[2];
		Vector<Pdb> pdbStrandN = new Vector<Pdb>();
		Matrix[] mm = new Matrix[4];
		double[] rdc1Rmsd = new double[4];
		double[] rdc2Rmsd = new double[4];
		int N = lastResidueNo - firstResidueNo;
		double[] rdc1Cal = new double[N+1];
		double[] rdc2Cal = new double[N+1];
		double rdc1Rms = 0.0, rdc2Rms = 0.0; 
		Matrix[] mm_temp = new Matrix[4];
		double[] temp1=new double[1];
		int[] temp2=new int[1];		
		
		double [][] ss    = {{1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {0.0, 0.0, 1.0}}; //invert Z direction
	
		Matrix ssMat  = new Matrix(ss);
	 
		rdc1Rmsd[0] = pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H",ssMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
		rdc2Rmsd[0] = pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",ssMat, -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
		rdc1Rms = rdc1Rmsd[0];
		rdc2Rms = rdc2Rmsd[0];
	
	
		boolean debugEulerFit = false;
		mm[0] = pdr.eulerFit(pdbStrand, rdc1Vec, rdc2Vec, Syy, Szz, rmsds, resolution, debugEulerFit);
		rdc1Rms =rmsds[0] ;
		rdc2Rms =rmsds[1];
		
		nhRdcRmsd[0]=rdc1Rms;
		chRdcRmsd[0]=rdc2Rms;
		
		mm[0].print(14,14);
		if (printResults){
		 System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2");
		 System.out.println(rdc1Rms+"  "+rdc2Rms);
		 mm[0].print(14, 14);
		}
		
		rdc1Rmsd[0] = pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H", mm[0], -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
		rdc2Rmsd[0] = pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",mm[0], -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
		rdc1Rms = rdc1Rmsd[0];
		rdc2Rms = rdc2Rmsd[0];
		double rms_save=rdc1Rms+rdc2Rms;
		int i_save=0;
			
		Vector<Pdb> helixVecN = new Vector<Pdb>();
		for (i = 1; i < 4; i++){
			mm[i] = Const.mat4ThreeDirs[i - 1].times(mm[0]);
			rdc1Rmsd[i] =pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H", mm[i], -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
			rdc2Rmsd[i] =pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",mm[i], -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
			
			System.out.println("i="+i);
			helixVecN = pp.newPdb(pdbStrand, Const.mat4ThreeDirs[i - 1]);
			mm_temp[0] = pdr.eulerFit(helixVecN, rdc1Vec, rdc2Vec, Syy, Szz, rmsds, resolution, debugEulerFit);
			rdc1Rms = rmsds[0]; 
			rdc2Rms = rmsds[1]; 
			
			if (rdc1Rms+rdc2Rms<rms_save)
			{
				nhRdcRmsd[0]=rdc1Rms;
				chRdcRmsd[0]=rdc2Rms;
				rms_save=rdc1Rms+rdc2Rms;
				i_save=i;			
			}
			
			mm_temp[0].print(14,14);
			
			 System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2");
			 System.out.println(rdc1Rms+"  "+rdc2Rms);		
		}
		System.out.println("i_save=" +i_save);
		pdbStrandN = pp.newPdb(pdbStrand, mm[i_save]);//be careful here... almost the same, so we chose anyone
		//pp.print(pdbStrandN);
		
		Matrix mm2 = new Matrix(3, 3);
		int nCycle = 1; 
		debugDFS = true;
		printResults = true;
		boolean isHelix = true;
		Matrix iMat = Matrix.identity(3, 3);
		double[] saupe = new double[5];
		Vector<Pdb> pdbS2 = new Vector<Pdb>();
		double [] chRdcRmsdTemp=new double[1];
		double [] nhRdcRmsdTemp=new double[1];
		
		for(i = 1; i< (refineCycle + 1); i++)
		{
			nCycle = initialCycle; //i * initialCycle;
			pdbS2 = minHelix(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbStrandN, Syy, Szz,
				     rdc2Rms, rdc1Rms, nCycle, w4Angles, debugDFS, printResults, isHelix,nhRdcRmsdTemp,chRdcRmsdTemp);
			 
			mm2 = pdr.bestFit(pdbS2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
			
			rdc1Rms=saupe[2];
			rdc2Rms=saupe[3];
			pdbStrandN=new Vector();
			pdbStrandN.addAll(pdbS2);
		 
			System.out.println("refinecycle="+ i + ": " );
			
		}
		
		if(nhRdcRmsdTemp[0]<100)
		{
			nhRdcRmsd[0]=nhRdcRmsdTemp[0];
			chRdcRmsd[0]=chRdcRmsdTemp[0];
		}
		
		System.out.println("====================================");
	
		return pdbStrandN;
	}

    /**
     * Refine helix22 no grid search.
     * 
     * @param vecBB the vec bb
     * @param rdc1Vec the rdc1 vec
     * @param rdc2Vec the rdc2 vec
     * @param Syy the syy
     * @param Szz the szz
     * @param ramaFilter the rama filter
     * @param phiAve the phi ave
     * @param psiAve the psi ave
     * @param refineCycle the refine cycle
     * @param initialCycle the initial cycle
     * @param w4Angles the w4 angles
     * @param resolution the resolution
     * @param debugDFS the debug dfs
     * @param printResults the print results
     * @param nhRdcRmsd the nh rdc rmsd
     * @param chRdcRmsd the ch rdc rmsd
     * 
     * @return the vector< pdb>
     */
    public Vector<Pdb> refineHelix22NoGridSearch(Vector vecBB, final Vector<Dipolar> rdc1Vec, final Vector<Dipolar> rdc2Vec, double Syy, double Szz,
		       double[] ramaFilter, double phiAve, double psiAve, int refineCycle, int initialCycle,
		       double w4Angles, double resolution, boolean debugDFS, boolean printResults, double[] nhRdcRmsd, double[] chRdcRmsd)
	{
 	int i = 0;
		double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
		double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
		double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	
		//build an initial model
		Pdb pp = new Pdb();
		Vector<PhiPsi> phiPsiVec = new Vector<PhiPsi>();
		Dipolar dd1 = (Dipolar)rdc1Vec.elementAt(0);
		Dipolar dd2 = (Dipolar)rdc2Vec.elementAt(0);
		int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());
		
		
		dd1 = (Dipolar)rdc1Vec.elementAt(rdc1Vec.size() - 1);
		dd2 = (Dipolar)rdc2Vec.elementAt(rdc2Vec.size() - 1);
		int lastResidueNo  =  Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;
		for (i=firstResidueNo; i<lastResidueNo; i++)  
			    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
		
		Vector<Pdb> pdbStrand=new Vector();
		pdbStrand.addAll(vecBB);
		
		//Compute the orientation of the first peptide by grid-search
		PdbRdc pdr = new PdbRdc();
		double [] rmsds = new double[2];
		Vector<Pdb> pdbStrandN = new Vector<Pdb>();
		pdbStrandN.addAll(vecBB);
		
		Matrix[] mm = new Matrix[4];
		double[] rdc1Rmsd = new double[4];
		double[] rdc2Rmsd = new double[4];
		int N = lastResidueNo - firstResidueNo;
		double[] rdc1Cal = new double[N+1];
		double[] rdc2Cal = new double[N+1];
		double rdc1Rms = 0.0, rdc2Rms = 0.0; 
		Matrix[] mm_temp = new Matrix[4];
		double[] temp1=new double[1];
		int[] temp2=new int[1];		
		
		double [][] ss    = {{1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {0.0, 0.0, 1.0}}; //invert Z direction
	
		Matrix ssMat  = new Matrix(ss);
	 
		rdc1Rmsd[0] = pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H",ssMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal,temp1,temp2);
		rdc2Rmsd[0] = pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",ssMat, -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal,temp1,temp2);
		rdc1Rms = rdc1Rmsd[0];
		rdc2Rms = rdc2Rmsd[0];
	
		
		
		Matrix mm2 = new Matrix(3, 3);
		int nCycle = 1; 
		debugDFS = true;
		printResults = true;
		boolean isHelix = true;
		Matrix iMat = Matrix.identity(3, 3);
		double[] saupe = new double[5];
		Vector<Pdb> pdbS2 = new Vector<Pdb>();
		double [] chRdcRmsdTemp=new double[1];
		double [] nhRdcRmsdTemp=new double[1];
		
		for(i = 1; i< (refineCycle + 1); i++)
		{
			nCycle = initialCycle; //i * initialCycle;
			pdbS2 = minHelix(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbStrandN, Syy, Szz,
				     rdc2Rms, rdc1Rms, nCycle, w4Angles, debugDFS, printResults, isHelix,nhRdcRmsdTemp,chRdcRmsdTemp);
			 
			mm2 = pdr.bestFit(pdbS2, rdc1Vec, rdc2Vec, rdc1Cal, rdc2Cal, saupe);
			
			rdc1Rms=saupe[2];
			rdc2Rms=saupe[3];
			pdbStrandN=new Vector();
			pdbStrandN.addAll(pdbS2);
		 
			System.out.println("refinecycle="+ i + ": " );
		
		}
		
		if(nhRdcRmsdTemp[0]<100)
		{
			nhRdcRmsd[0]=nhRdcRmsdTemp[0];
			chRdcRmsd[0]=chRdcRmsdTemp[0];
		}
		
		System.out.println("====================================");
		
		return pdbStrandN;
	}


   /**
    * For computing the Helix.
    * 
    * @param Syy the Saupe element
    * @param Szz the Saupe element
    * @param rdcVec1 the rdc vec1
    * @param rdcVec2 the rdc vec2
    * @param n2ca the n2ca
    * @param heliceMap the helice map
    * 
    * @return the refined Pdb vector
    */
    public Vector refineHelix(final Vector rdcVec1, final Vector rdcVec2, double[] n2ca, double Syy, 
			      double Szz, Map heliceMap){
	int i = 0;
    	double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};

	//build an initial model
	Pdb pp = new Pdb();
	Vector phiPsiVec = new Vector();
	Dipolar dd1 = (Dipolar)rdcVec1.elementAt(0);
	Dipolar dd2 = (Dipolar)rdcVec2.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());

	dd1 = (Dipolar)rdcVec1.elementAt(rdcVec1.size() - 1);
	dd2 = (Dipolar)rdcVec2.elementAt(rdcVec2.size() - 1);
	int lastResidueNo  = Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;
	int N = lastResidueNo - firstResidueNo + 1;
	double phiAve = Const.phiAveHelix;
	double psiAve = Const.psiAveHelix;

   	for (i=firstResidueNo; i<lastResidueNo; i++)  
    	    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
        Vector pdbVec = modelBuild(phiPsiVec, n1, nh1, ca1);

        int refineCycle = Integer.parseInt((String)heliceMap.get("REFINECYCLE"));
        int dfsCycle    = Integer.parseInt((String)heliceMap.get("DFSCYCLE"));
        boolean debugDFS = new Boolean((String)heliceMap.get("DEBUGDFS")).booleanValue();
        boolean printResults = new Boolean((String)heliceMap.get("PRINTRESULTS")).booleanValue();
        double w4Angles = new Double((String) heliceMap.get("WEIGHT4ANGLES")).doubleValue();
        double resolution =  new Double((String) heliceMap.get("RESOLUTION")).doubleValue();

        //Compute the orientation of the first peptide by grid-search
        PdbRdc pdr = new PdbRdc();
        double [] rmsds = new double[2];
        Vector pdbVec2 = new Vector();
        Matrix iMat = Matrix.identity(3, 3);
        double rdc1Rms = 0.0, rdc2Rms = 0.0; //S11-17 after search through only 180*180*180 for three angles.
        Matrix[] mm = new Matrix[4];
        Matrix[] rr = new Matrix[4];
        double[] rdc1Rmsd = new double[4];
        double[] rdc2Rmsd = new double[4];
        double[] rdc1Cal = new double[N];
        double[] rdc2Cal = new double[N];
	double[] nToCa = new double[3];
        boolean debugEulerFit = true;

    

	double[][] mmArr = {{-0.80382754971551, -0.33178913478673, -0.49373802806329},
			    {0.17818753051321, 0.65759409409609, -0.73199672907708},
			    {0.56754777269227, -0.67637709707486, -0.46947156278589}};
	mm[0] = new Matrix(mmArr);

	nToCa = mm[0].times(ca1);
	pdbVec2 = pp.newPdb(pdbVec, mm[0]);
	rdc1Rmsd[0] =pdr.BackCalNH(pdbVec2, rdcVec1, "N", "H", iMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	rdc2Rmsd[0] =pdr.BackCal(pdbVec2, rdcVec2, "CA","HA",iMat, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdc2Cal);

	double angle = interAngle(nToCa, n2ca);

        for (i = 1; i < 4; i++){
            mm[i] = Const.mat4ThreeDirs[i - 1].times(mm[0]);
	    nToCa = mm[i].times(ca1);
	    angle = interAngle(nToCa, n2ca);

            pdbVec2 = pp.newPdb(pdbVec, mm[i]);
	    rdc1Rmsd[i] = pdr.BackCalNH(pdbVec2, rdcVec1, "N", "H", iMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	    rdc2Rmsd[i] = pdr.BackCal(pdbVec2, rdcVec2, "CA","HA",iMat, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdc2Cal);

        }
	Vector pdbVec1 = pp.newPdb(pdbVec, mm[0]);
	rdc1Rms = rdc1Rmsd[0];
	rdc2Rms = rdc2Rmsd[0];

	int nCycle = 1; 
	debugDFS = true;
	printResults = true;
	boolean isHelix = true;
	Vector pdbS2 = new Vector();
	for(i = 1; i< (refineCycle + 1); i++){
 	    nCycle = i*dfsCycle;
	    pdbS2 = minHelix_old(rdcVec2, rdcVec1, rdc2Cal, rdc1Cal, pdbVec1, Syy, Szz, rdc2Rms, rdc1Rms,
			     nCycle, w4Angles, debugDFS, printResults, isHelix);
	    rdc1Rms = pdr.BackCalNH(pdbS2, rdcVec1, "N", "H", iMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	    rdc2Rms = pdr.BackCal(pdbS2, rdcVec2, "CA","HA",iMat, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdc2Cal);

	    pdbVec1 = pdbS2;
	}

        return pdbS2;
    }


    /**
     * For computing the first Strand which is NOT H-bonded to others.
     * 
     * @param rdc1Vec RDCs in medium 1
     * @param rdc2Vec RDCs in medium 2
     * @param refineCycle the number of cycles for iterating through the refinement
     * and computation of Saupe matrices.
     * @param initialCycle  the number of initial cycles for the DFS-based systematic search.
     * @param ramaFilter the Ramachandran filter for the favorable phi and psi region
     * @param phiAve  the average phi angle
     * @param psiAve  the average psi angle
     * @param w4Angles  weight for the \phi and \psi angles in the score funtion
     * @param resolution the resolution for the grid-search for the first peptide plane
     * @param debugDFS  for monitoring the progress in systematic search-based minimization
     * @param printResults for printing the results for each iteration
     * @param Syy the syy
     * @param Szz the szz
     * 
     * @return the refined Pdb vector
     */
    public Vector refineStrand(final Vector rdc1Vec, final Vector rdc2Vec, double Syy, double Szz,
			       double[] ramaFilter, double phiAve, double psiAve, int refineCycle, int initialCycle,
			       double w4Angles, double resolution, boolean debugDFS, boolean printResults){
	int i = 0;
    	double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};

	//build an initial model
	Pdb pp = new Pdb();
	Vector phiPsiVec = new Vector();
	Dipolar dd1 = (Dipolar)rdc1Vec.elementAt(0);
	Dipolar dd2 = (Dipolar)rdc2Vec.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());

	dd1 = (Dipolar)rdc1Vec.elementAt(rdc1Vec.size() - 1);
	dd2 = (Dipolar)rdc2Vec.elementAt(rdc2Vec.size() - 1);
	int lastResidueNo  =  Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;
   	for (i=firstResidueNo; i<lastResidueNo; i++)  
    	    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
 	Vector pdbStrand = modelBuild(phiPsiVec, n1, nh1, ca1);


	//Compute the orientation of the first peptide by grid-search
 	PdbRdc pdr = new PdbRdc();
	double [] rmsds = new double[2];
 	Vector pdbStrandN = new Vector();
  	Matrix[] mm = new Matrix[4];
   	double[] rdc1Rmsd = new double[4];
   	double[] rdc2Rmsd = new double[4];
	int N = lastResidueNo - firstResidueNo;
	double[] rdc1Cal = new double[N];
	double[] rdc2Cal = new double[N];
  	double rdc1Rms = 0.0, rdc2Rms = 0.0; 

	boolean debugEulerFit = false;
	mm[0] = pdr.eulerFit(pdbStrand, rdc1Vec, rdc2Vec, Syy, Szz, rmsds, resolution, debugEulerFit);
	rdc1Rms = Math.sqrt(rmsds[0] / N);
        rdc2Rms = Math.sqrt(rmsds[1] / N);
	mm[0].print(14,14);
	if (printResults){
	    System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2");
	    System.out.println(rdc1Rms+"  "+rdc2Rms);
	    mm[0].print(14, 14);
	}


	rdc1Rmsd[0] = pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H", mm[0], -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	rdc2Rmsd[0] = pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",mm[0], -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal);
	rdc1Rms = rdc1Rmsd[0];
	rdc2Rms = rdc2Rmsd[0];
	for (i = 1; i < 4; i++){
	    mm[i] = Const.mat4ThreeDirs[i - 1].times(mm[0]);
	    rdc1Rmsd[i] =pdr.BackCalNH(pdbStrand, rdc1Vec, "N", "H", mm[i], -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	    rdc2Rmsd[i] =pdr.BackCal(pdbStrand, rdc2Vec, "CA","HA",mm[i], -Syy-Szz, Syy, Szz, Const.cahaRatio,rdc2Cal);

	}

 	pdbStrandN = pp.newPdb(pdbStrand, mm[2]); //2 is Good for E65-70
	int nCycle = 1; 
	debugDFS = true;
	printResults = true;
	boolean isHelix = false;
	Matrix iMat = Matrix.identity(3, 3);
	Vector pdbS2 = new Vector();
	for(i = 1; i< (refineCycle + 1); i++){
 	    nCycle = i * initialCycle;
	    pdbS2 = minHelix_old(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbStrandN, Syy, Szz,
			     rdc2Rms, rdc1Rms, nCycle, w4Angles, debugDFS, printResults, isHelix);
	    rdc1Rms = pdr.BackCalNH(pdbS2, rdc1Vec, "N", "H", iMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	    rdc2Rms = pdr.BackCal(pdbS2, rdc2Vec, "CA","HA",iMat, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdc2Cal);
	    pdbStrandN = pdbS2;
	}
  	return pdbStrandN;
    }

    /**
     * For computing the others Strands which IS H-bonded to others.
     * 
     * @param rdc1Vec RDCs in medium 1
     * @param rdc2Vec RDCs in medium 2
     * @param refineCycle  the number of cycles for iterating through the refinement
     * and computation of Saupe matrices.
     * @param Syy the syy
     * @param Szz the szz
     * @param phiAve the phi ave
     * @param psiAve the psi ave
     * @param initialCycle the initial cycle
     * @param resolution the resolution
     * @param angleWeight the angle weight
     * @param hbWeight the hb weight
     * @param pdbS1 the pdb s1
     * @param hBVecOfE21 the h b vec of e21
     * @param printEulerSearch the print euler search
     * @param debugDFS the debug dfs
     * @param printResults the print results
     * 
     * @return the refined Pdb vector
     */
    public Vector refineStrand(final Vector rdc1Vec, final Vector rdc2Vec, double Syy, double Szz, double phiAve, 
			       double psiAve, int refineCycle, int initialCycle, double resolution,
			       double angleWeight, double hbWeight, Vector pdbS1, Vector hBVecOfE21, 
			       boolean printEulerSearch, boolean debugDFS, boolean printResults){
	int i = 0;
    	double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	//build an initial ideal helix mode
	Vector phiPsiVec = new Vector();
	Pdb pp = new Pdb();
	Dipolar dd1 = (Dipolar)rdc1Vec.elementAt(0);
	Dipolar dd2 = (Dipolar)rdc2Vec.elementAt(0);
	int firstResidueNo =  Math.min(dd1.getResidueNo(), dd2.getResidueNo());

	dd1 = (Dipolar)rdc1Vec.elementAt(rdc1Vec.size() - 1);
	dd2 = (Dipolar)rdc2Vec.elementAt(rdc2Vec.size() - 1);
	int lastResidueNo  =  Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;
   	for (i=firstResidueNo; i<lastResidueNo; i++) {
	    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
	}
 	Vector pdbStrand = modelBuild(phiPsiVec, n1, nh1, ca1);


 	PdbRdc pdr = new PdbRdc();
 	PdbRmsd pR = new PdbRmsd();
	double [] rmsds = new double[3];
	int N = lastResidueNo - firstResidueNo + 1;
	double[] rdc1Cal = new double[N];
	double[] rdc2Cal = new double[N];
	Vector pdbStrandN = new Vector();
	Matrix iMat = Matrix.identity(3, 3);
 	double w4HBEulerFit = 5.0; 
	printEulerSearch = false;
       
 	Matrix mm = pdr.fit4MissingRdcHB(pdbStrand, rdc1Vec, rdc2Vec, Syy, Szz, pdbS1, rmsds, resolution, 
					 w4HBEulerFit, hBVecOfE21, printEulerSearch);
	double rdc1Rms = Math.sqrt(rmsds[0] / (N-1));
	double rdc2Rms = Math.sqrt(rmsds[1] / (N-1));
	double hbScores = rmsds[2];
	if (printResults){
	    System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2,  hbScore");
	    System.out.println(rdc1Rms+"   "+rdc2Rms+"   "+ hbScores);
	    mm.print(14, 14);
	}

 	Vector pdbVec2 = pp.newPdb(pdbStrand, mm);

	rdc1Rms = pdr.BackCalNH(pdbVec2, rdc1Vec, "N", "H", iMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	rdc2Rms = pdr.BackCal(pdbVec2, rdc2Vec, "CA","HA",iMat, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdc2Cal);


	int nCycle = 1; 
	debugDFS = true;
	printResults = true;
 	Vector pdbS2 = new Vector();
	for(i = 1; i< (refineCycle+1); i++){
	    nCycle = i * initialCycle;
	    pdbS2 = minBeta(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbVec2, Syy, Szz, rdc2Rms, rdc1Rms,
			     pdbS1, hBVecOfE21, nCycle, angleWeight, hbWeight, debugDFS, printResults);
	    rdc1Rms = pdr.BackCalNH(pdbS2, rdc1Vec, "N", "H", iMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	    rdc2Rms = pdr.BackCal(pdbS2, rdc2Vec, "CA","HA",iMat, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdc2Cal);
	    pdbVec2 = pdbS2;

	}
  	Vector pdbVecN = pR.centerFitVec(pdbS1, pdbVec2, hBVecOfE21, true);
   	return pdbVecN;
    }

    /**
     * For computing the others Strands which IS H-bonded to others.
     * 
     * @param rdc1Vec RDCs in medium 1
     * @param rdc2Vec RDCs in medium 2
     * @param refineCycle  the number of cycles for iterating through the refinement
     * and computation of Saupe matrices.
     * @param Syy the syy
     * @param Szz the szz
     * @param phiAve the phi ave
     * @param psiAve the psi ave
     * @param initialCycle the initial cycle
     * @param resolution the resolution
     * @param angleWeight the angle weight
     * @param hbWeight the hb weight
     * @param pdbE1 the pdb e1
     * @param HbondVec the hbond vec
     * @param printEulerSearch the print euler search
     * @param debugDFS the debug dfs
     * @param printResults the print results
     * 
     * @return the refined Pdb vector
     */
    public Vector refineStrand2(final Vector rdc1Vec, final Vector rdc2Vec, double Syy, double Szz, double phiAve, 
				double psiAve, int refineCycle, int initialCycle, double resolution,
				double angleWeight, double hbWeight, Vector pdbE1, Vector HbondVec, 
				boolean printEulerSearch, boolean debugDFS, boolean printResults){
	int i = 0;
    	double [] n1  = {0.0, 0.0, 0.0};         //A coordinate frame in the 1st peptide plane
	double [] nh1 = {0.0, 0.0, -Const.dN2H}; 
    	double [] ca1 = {0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)};
	//build an initial ideal helix mode
	Vector phiPsiVec = new Vector();
	Pdb pp = new Pdb();
	Dipolar dd1 = (Dipolar)rdc1Vec.elementAt(0);
	Dipolar dd2 = (Dipolar)rdc2Vec.elementAt(0);
	int firstResidueNo = Math.min(dd1.getResidueNo(), dd2.getResidueNo());

	dd1 = (Dipolar)rdc1Vec.elementAt(rdc1Vec.size() - 1);
	dd2 = (Dipolar)rdc2Vec.elementAt(rdc2Vec.size() - 1);
	int lastResidueNo  = Math.max(dd1.getResidueNo(), dd2.getResidueNo()) + 1;
   	for (i=firstResidueNo; i<lastResidueNo; i++) {
	    phiPsiVec.add(new PhiPsi(i, "ALA", phiAve, psiAve));
	}
 	Vector pdbStrand = modelBuild(phiPsiVec, n1, nh1, ca1);

	Vector pdbS1 = new Vector();
	int no = 0;
	for (i=0; i<pdbE1.size(); i++){
	    pp = (Pdb)pdbE1.elementAt(i);
	    no = pp.getResidueNo();
	    if ( no < firstResidueNo || no > lastResidueNo){
		pdbS1.add(pp);
	    }
	}

 	PdbRdc pdr = new PdbRdc();
 	PdbRmsd pR = new PdbRmsd();
	Hbond hb = new Hbond();
	double [] rmsds = new double[3];

	int N = lastResidueNo - firstResidueNo + 1;
	double[] rdc1Cal = new double[N];
	double[] rdc2Cal = new double[N];
	Vector pdbStrandN = new Vector();
	Matrix iMat = Matrix.identity(3, 3);
 	double w4HBEulerFit = 1.0;
	printEulerSearch = false;
       
	String id1 = "E5-E3";
	String id2 = "E3-E4";
	Vector hBVecOfE21 = hb.hbOfTwoStrands(HbondVec, id1);
	Vector hBVecOfE23 = hb.hbOfTwoStrands(HbondVec, id2);
 	Matrix mm = pdr.fit4MissingRdcHB2(pdbStrand, rdc1Vec, rdc2Vec, Syy, Szz, pdbS1, pdbS1, rmsds, resolution, 
					  w4HBEulerFit, hBVecOfE21, hBVecOfE23, printEulerSearch);
	double rdc1Rms = Math.sqrt(rmsds[0] / (N-1));
	double rdc2Rms = Math.sqrt(rmsds[1] / (N-1));
	double hbScores = rmsds[2];
	if (printResults){
	    System.out.println("Results from grid-search: rmsdRdc1,  rmsdRdc2,  hbScore");
	    System.out.println(rdc1Rms+"   "+rdc2Rms+"   "+ hbScores);
	    mm.print(14, 14);
	}


 	Vector pdbVec2 = pp.newPdb(pdbStrand, mm);

	rdc1Rms = pdr.BackCalNH(pdbVec2, rdc1Vec, "N", "H", iMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	rdc2Rms = pdr.BackCal(pdbVec2, rdc2Vec, "CA","HA",iMat, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdc2Cal);

	int nCycle = 1; 
	debugDFS = true;
	printResults = true;
 	Vector pdbS2 = new Vector();
	for(i = 1; i< (refineCycle+1); i++){
	    nCycle = i * initialCycle;
	    pdbS2 = minBeta2(rdc2Vec, rdc1Vec, rdc2Cal, rdc1Cal, pdbVec2, Syy, Szz, rdc2Rms, rdc1Rms,
			     pdbS1, pdbS1, hBVecOfE21, hBVecOfE23, nCycle, angleWeight, 
			     hbWeight, debugDFS, printResults);
	    rdc1Rms = pdr.BackCalNH(pdbS2, rdc1Vec, "N", "H", iMat, -Syy-Szz, Syy, Szz, Const.nhRatio, rdc1Cal);
	    rdc2Rms = pdr.BackCal(pdbS2, rdc2Vec, "CA","HA",iMat, -Syy-Szz, Syy, Szz, Const.cahaRatio, rdc2Cal);
	    pdbVec2 = pdbS2;
	}
  	Vector pdbVecN = pR.centerFitVec(pdbS1, pdbVec2, hBVecOfE21, true);
 	Vector sheetPdb = new Vector();
	sheetPdb.addAll(pdbVecN);
	sheetPdb.addAll(pdbS1);
	Collections.sort(sheetPdb, new Pdb.PdbComparator());
   	return sheetPdb;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#clone()
     */
    protected Object clone(){
        try {
	    Object s = super.clone();     
	    return s;                     
        } catch (CloneNotSupportedException e) {
	    throw new InternalError();
        }
    }
}
