package rdcPanda;

 ///////////////////////////////////////////////////////////////////////////////////////////////
//	IsoSubGraph.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////

/**
 * This class implements data structures and functions for NOE assignment
 * without side-chain resonance assignments.
 *  Written by  Jianyang (Michael) Zeng (2005-2009)
* 
*/

/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

import java.io. *;
import java.util. *;
import java.text.NumberFormat;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.lang. *;

import Jampack.JampackException;

public class Nasca 
{
	public class Node
	{
		//for both proton and resonance ndoes:
		private int id=0;
		Vector vecAdj=new Vector();//for storing the adjacent points;
		Vector vecAdjNOESY=new Vector();// for storing interacting NOESY peaks for a resonance node.
				
		//only for proton node
		private int resNo=0;
		private String resName="";
		private String atomName="";
		Vector atomVec = new Vector();//all coordinates corresponding to the same node;
		private double csLower=0.0;//lower bound for chemical shift
		private double csUpper=0.0;
		public int stereoID=-99;//id of the other stereo proton
		public double maxMapScore=-999.9;//for storing the max score of mapped resonance nodes
		public int maxMappedID=-99;//id of resonance node with the max mapping score
		
		//only for resonance node:
		private double CSProton=0.0;//chemical shift of proton
		private double CSHeavy=0.0;//cs of heavy atom attached to proton
		private boolean isAsigned=false;
		private boolean isRestrained=false;
		public int asgedId=-1;//the one-to-one assignment
		private Vector vecMappingSet=new Vector();//store all possible mapping set
		public Vector vecStereoSymSet=new Vector();//store all stereo symmetry resonance nodes
		public Vector vecSymSet=new Vector(); //store symmetry resonance nodes
		public Vector vecCloseAdj=new Vector();
		
		public Node(Node nd)
		{
			id=nd.getID();
			vecAdj=nd.getAdjVec();
			resNo=nd.getResNo();
			resName=nd.getResName();
			atomName=nd.getAtomName();
			atomVec=nd.getAtomVec();
			csLower=nd.csLower;
			csUpper=nd.csUpper;
			CSProton=nd.getProtonCS();
			CSHeavy=nd.getHeavyCS();
			isAsigned=nd.getIsAssigned();
			asgedId=nd.getAssignProtonID();
			vecMappingSet=nd.getMappingSet();
			isRestrained=false;
		}
		public Node()
		{
			id=0;
			resNo=0;
			vecAdj=new Vector();
		}
		public Node(int idT)
		{
			id=idT;
			resNo=0;
			vecAdj=new Vector();
		}
		
		//initialization for proton node
		public Node(int idT, int resIDT, String res, String atom, Vector vecAtom)
		{
			id=idT;
			resNo=resIDT;
			resName=res;
			atomName=atom;
			atomVec.addAll(vecAtom);
			if(atom.equalsIgnoreCase("CA") ||atom.equalsIgnoreCase("N")|| 
					atom.equalsIgnoreCase("HN") ||atom.equalsIgnoreCase("H")||
					atom.equalsIgnoreCase("HA")||atom.equalsIgnoreCase("HA1")|| atom.equalsIgnoreCase("HA2")||atom.equalsIgnoreCase("HA3")||
					atom.equalsIgnoreCase("C")||atom.equalsIgnoreCase("CO")||atom.equalsIgnoreCase("CB"))
				isAsigned=true;
			else
				isAsigned=false;
			isRestrained=false;
			vecAdj=new Vector();
		}
		
		//initialization for resonanance node
		public Node(int idT, double cs_H, double cs_heavy, boolean isAsged)
		{
			id=idT;
			CSProton=cs_H;
			CSHeavy=cs_heavy;
			isAsigned=isAsged;			
			vecAdj=new Vector();
			isRestrained=false;
		}
		
		public int getID(){return id;}
		public int getResNo(){return resNo;}
		public String getResName(){return resName;}
		public String getAtomName(){return atomName;}
		public Vector getAtomVec(){return atomVec;}
		public Vector getAdjVec(){return vecAdj;}
		public double getProtonCS(){return CSProton;}
		public double getHeavyCS(){return CSHeavy;}
		public void setProtonCS(double cs_h){ CSProton=cs_h;}
		public void setIsAssigned(boolean tf){ isAsigned=tf;}
		public void setHeavyCS(double cs_heavy){ CSHeavy=cs_heavy;}
		public void setResNo(int no){ resNo=no;}
		public void setNodeID(int no){id=no;}
		public void setResName(String res){ resName=res;}
		public void setAtomName(String atom){ atomName=atom;}
		public boolean getIsAssigned(){return isAsigned;}
		public void setCSLower(double cs_L){ csLower=cs_L;}
		public void setCSUpper(double cs_U){ csUpper=cs_U;}
		public void setAssignedID(int IDno){ asgedId=IDno;}
		public void addMappingNode(Node node){ vecMappingSet.add(node);}
		public Vector getMappingSet(){return vecMappingSet;}
		public int getAssignProtonID(){return asgedId;}
		public boolean getIsRestrained(){return isRestrained;}
		public void setIsRestrained(boolean isRsd){ isRestrained=isRsd;}
		public void EmptyMappingSet()
		{ 
			vecMappingSet=new Vector();
		}
		public void EmptyAdjSet()
		{ 
			vecAdj=new Vector();
		}
		public void AddCoord(Cartesian coord)
		{
			atomVec.add(coord);
		}
		public void DelAdj(int idNeigh)
		{
			Vector vecAdjTemp=new Vector();
			for (int i=0;i<vecAdj.size();i++)
			{
				Node nd= (Node) vecAdj.elementAt(i);
				int idT=nd.getID();
				if(idT==idNeigh)
					continue;
				vecAdjTemp.add(nd);
			}
			vecAdj=new Vector();
			vecAdj.addAll(vecAdjTemp);
		}//public void DelAdj(int idNeigh)
		
		public void AddAdjNOESY(Noesy noesy)
		{
			boolean isInPre=false;
			double csH1Cur=noesy.getH1();
			double csH2Cur=noesy.getH2();
			double csHeavyCur=noesy.getHeavy();
			for(int i=0;i<vecAdjNOESY.size();i++)
			{
				Noesy noesyT=(Noesy)vecAdjNOESY.elementAt(i);
				double csH1=noesyT.getH1();
				double csH2=noesyT.getH2();
				double csHeavy=noesyT.getHeavy();
				if(Math.abs(csH1Cur-csH1)<0.005 && Math.abs(csH2Cur-csH2)<0.005 && Math.abs(csHeavyCur-csHeavy)<0.005 )
					isInPre=true;
			}
			if(!isInPre)
				vecAdjNOESY.add(noesy);
		}
		public void AddCloseAdj(Node ndNeigh)
		{
			int IDT=ndNeigh.getID();
			boolean isInPres=false;
			for(int i=0;i<vecCloseAdj.size();i++)
			{
				Node nd=(Node)vecCloseAdj.elementAt(i);
				int IDPre=nd.getID();
				if(IDT==IDPre)
					isInPres=true;
			}
			if(!isInPres)
				vecCloseAdj.add(ndNeigh);
		}
	
	
		public void AddAdj(Node ndNeigh)
		{
			int IDT=ndNeigh.getID();
			boolean isInPres=false;
			for(int i=0;i<vecAdj.size();i++)
			{
				Node nd=(Node)vecAdj.elementAt(i);
				int IDPre=nd.getID();
				if(IDT==IDPre)
					isInPres=true;
			}
			if(!isInPres)
				vecAdj.add(ndNeigh);
		}
	}
	
	
	public Nasca()
	{
		
	}	
	   /**
     * rotamer selection without sidechain resonance assignment:
     * calculate the score of each rotamer, and return ensemble of rotamers with highest scores
     @param vecRot rotamers 
     @param assignVec resonance assignment
     @param vecBB backbone structure
     @param hnNoeVec hn Noe peak list
     @param cnoeVec ca noe peak list
     @param resNo residue number
     @param distBound distance bound to define noe distance
     @param pcikNum number of chosen rotamers with highest scores 
     @param vecRotStr rotamer vectors in the structure format.
     @param vecAsgBMRB: BMRB statistical information
     @return refined rotamer positions, and refined sd assignments
         */
    public Vector RefineRotLibWOSCAsg(double csErrH, double csErrN, double csErrCA,Vector vecRot, Vector vecBB,  Vector assignBBVec,
    		Vector NoesyVec,int resNo,double distBound, int pickNum,Vector vecRotStr,Vector vecResonGraph, Vector vecStructGraph, int[][]Map,Vector vecAsgBMRB)
    {
    	int i,j;
    	Vector vecNewRot=new Vector();
    	Pdb pp=new Pdb();    	
    	Pdb pdbRotamer,pdbRotamerStr;
    	H1CS h1cs=new H1CS();
    	Vector vecHdist;
    	String rotamResName="";
    	RotaPattern rotPattern=new RotaPattern();
    	Vector vecBackNoe;
    	Assign asg=new Assign();
    	double dbScore=0.0;
    	Vector vecRotTemp=new Vector();
    	int noeCounter=0;    
    
    	double maxSc=-999999.9;
    	int sumPeaks=0;
    	
    	//exact all possible sc assignments for this residue:
    	pdbRotamer=(Pdb)vecRot.elementAt(0);
    	Vector vecAtom=pdbRotamer.getAtomVec();
    	Vector vecSCProtons=new Vector();
    	for(int k=0;k<vecStructGraph.size();k++)
		{
			Node nodeStruct=(Node)vecStructGraph.elementAt(k);
			int res_no=nodeStruct.getResNo();
			String atom_name=nodeStruct.getAtomName();
			boolean isAssigned=nodeStruct.getIsAssigned();
			boolean isRestrained=nodeStruct.getIsRestrained();
			
			////////////////////
			String subAtomT=atom_name;
			if(atom_name.length()>=2)
				subAtomT=atom_name.substring(0,2);
			
			if(isAssigned)
				continue;
			
			boolean isIn=false;
			for(i=0;i<vecAtom.size();i++)
			{
	    		Cartesian cc = (Cartesian)vecAtom.elementAt(i);
	    		String atom=cc.getAtom();
	    			
    			
    				String subAtom=atom;
    			
	    			if(atom.length()>atom_name.length())
	    				subAtom=atom.substring(0,atom_name.length());
	    			
	    			if(resNo==res_no && subAtom.equalsIgnoreCase(atom_name))
	    			{
	    				isIn=true;
	    				break;
	    			}
	    			
    		   		
			}//for(i=0;i<vecAtom.size();i++)
			if(isIn)
				vecSCProtons.add(nodeStruct);
		}//for(int k=0;k<vecStructGraph.size();k++) 
    	
    	////////////////////////
    	Vector vecAllSCAsgs=EnumAllScAsgSaveIntoFile(vecSCProtons,vecResonGraph,vecStructGraph);
    	   	
    	int scAsgId=0;
    	int rotId=0;
    	for (int k=0;k<vecAllSCAsgs.size();k++)
    	{
    		Vector vecOneSCAsg=(Vector)vecAllSCAsgs.elementAt(k);
    		
    		double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsg,vecAsgBMRB);
    		Vector vecOneBBSCAsg=new Vector();
    		vecOneBBSCAsg.addAll(vecOneSCAsg);
    		vecOneBBSCAsg.addAll(assignBBVec);
    		
	    	for (i=0;i<vecRot.size();i++)
	    	{
	    		Vector vecRotamer =new Vector();
	    		pdbRotamer=(Pdb)vecRot.elementAt(i);
	    		pdbRotamerStr=(Pdb)vecRotStr.elementAt(i);
	    		
	    		vecRotamer.add(pdbRotamer);
	    		vecHdist=pp.compDistPatternForRotam(vecRotamer,resNo, vecBB,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
	    		
	    		rotamResName=pdbRotamer.getResidue();
	    		vecBackNoe=rotPattern.BackCompNoePattern(vecOneBBSCAsg,resNo,rotamResName,vecHdist);	    		
	    		BackNoe bk=new BackNoe();
	    		Vector vecBackNoeNew=bk.DeleteRepeat(vecBackNoe);	
	    		int [] numPeaks=new int[1];
	    		dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,NoesyVec, numPeaks,false);
	    		//dbScore=dbScore*vecBackNoe.size();
   	    		double asg_log=Math.log(asgScore);
	    		double noe_log=Math.log(dbScore);
	    		dbScore=1*asg_log+noe_log;		
	    		sumPeaks=sumPeaks+numPeaks[0];
	    		if (maxSc<dbScore)    	
	    		{
	    			maxSc=dbScore;
	    			scAsgId=k;
	    			rotId=i;
	    		}
	    		vecRotTemp.add(new RotaPattern(pdbRotamerStr, dbScore) );
	    	}//for (i=0;i<vecRot.size();i++)
    	}//for (int k=0;k<vecAllSCAsgs.size();k++)    	
	
     	Pdb pd=(Pdb)vecRot.elementAt(rotId);
    	vecNewRot.add(pd);
    	if(vecAllSCAsgs.size()>=1)
    	{
    		Vector vecScAsgT=(Vector)vecAllSCAsgs.elementAt(scAsgId);
    		Vector vecTemp=new Vector();
    		for(int k=0;k<vecScAsgT.size();k++)
    		{
    			H1CS h1csT=(H1CS)vecScAsgT.elementAt(k);
    			String atom=h1csT.getAtomName();
    			if(atom.equalsIgnoreCase("CB"))
    				continue;
    			assignBBVec.add(h1csT);
    		}//for(int k=0;k<vecScAsgT.size();k++)
    		
        	Collections.sort(assignBBVec, new H1CS.resNoComparator());
    	}
    	
     	return vecNewRot;///
    }

	   /**
     * Compute a structure without sidechain resonances, given the backbone and rotamers at all residue positions
      @param csErrH csErrN csErrCA error window size at each dimension
     @param pdbBBVec the PDB backbone structure
     @param asgVec    resonace list
     @param rotSrc  the path of rotmaer library
     @param NoesyVec  Noe vector including both ha and hn noes
     @param distBound  the distance bound for back computing distance pattern for each rotamer
     @param pickNum  the best number of rotamers picked for computation    
     @return final structure. and Updated asgVec.
     Note: the structure will be rotated for installing rotamers.
     So the structure need to be rotated back outside this function.
     */
    public Vector RotamSelectWOSCAsg(Vector pdbBBVec, Vector asgBBVec,
    		Vector NoesyVec, Vector vecResonGraph, Vector vecStructGraph, int[][]Map,Vector vecAsgBMRB)
    {
    	double csErrH=0.04; double csErrN=0.4; double csErrCA=0.4;
    	int pickNum=1;//number of  rotamers to be selected
    	double distBound=6.0;
    	String userDir = System.getProperty("user.dir");////
    	String rotSrc=userDir+"/system/rot-lib/";    	
    	int i,j;
    	Pdb pdb;
    	String resName;
    	int resNo;
    	String rotamFile;
    	Vector pdbRotam=new Vector();
    	Vector pdbRotamNewStr=new Vector();
    	Vector pdbRotamNewCS=new Vector();
    	Vector pdbRotamNewRef=new Vector();
    	Pdb pp=new Pdb();
    	RotaPattern rp=new RotaPattern();
    	Vector vecGrowPdb=new Vector();
    	vecGrowPdb=pdbBBVec;//initial structure to plug in rotamers
    
		
    	for (j=0;j<pdbBBVec.size();j++)
    	{ 	
        	pdb=(Pdb)vecGrowPdb.elementAt(j);
        	resName=pdb.getResidue();
        	resNo=pdb.getResidueNo();
        	Vector pdbRotVec=new Vector();
        	
        	
        	System.out.println("debugging resNo="+resNo);
        	
        		rotamFile=rotSrc+ resName.toLowerCase()+ ".pdb"; 
        		pdbRotam=pp.readRotamerPdb(rotamFile);
        		
        		pdbRotamNewStr=pp.nameConvert4SmallRotLibStr(pdbRotam);
        	
            	pdbRotVec=pdbRotam;//changed for the ff2 case//////////
    	
        		
        	int indRef = Collections.binarySearch(vecGrowPdb, new Pdb(resNo), new Pdb.PdbComparator());
        	int [] index=new int[2];        	
        	
        	if (!(indRef<0))    
        	{
            	pdbRotamNewRef=RefineRotLibWOSCAsg(csErrH,csErrN, csErrCA,pdbRotVec,vecGrowPdb,asgBBVec, NoesyVec,resNo,distBound, pickNum,pdbRotVec, vecResonGraph, vecStructGraph, Map,vecAsgBMRB);
        	
        	}
        	if(resName.equalsIgnoreCase("GLY"))
        	{
        		pdbRotamNewRef=new Vector();
        		pdbRotamNewRef.addAll(pdbRotVec);
        	}         	
        	
        	//here we need to be careful when bb is not accurate enough         	
        	if (index[0]<0.0)
        		index[0]=0;
        	
        	Vector vecSelectedRot=new Vector();
        	Pdb pdb_temp=(Pdb)pdbRotamNewStr.elementAt(index[0]);
        
        	vecSelectedRot.add(pdb_temp);        	
          	
        	//update the backbone structure, including previous fixed rotamer
 
        	Vector vecTemp=pp.CombinRotBB(pdbRotamNewRef,vecGrowPdb,resNo);    			
    		vecGrowPdb=vecTemp;
    		
    	}//for (j=0;j<pdbBBVec.size();j++)
    	
    	return vecGrowPdb;    	
    }
	/**
     * print out all possible for proton nodes
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs      
     @return 
     */
	public void PrintAllPossibleMappings(Vector vecResonGraph,Vector vecGraph, int [][] Map)
	{
		int i,j;
		for(j=0;j<vecGraph.size();j++)
    	{
    		Node node2=(Node)vecGraph.elementAt(j);
    		int resNo=node2.getResNo();
    		String resName=node2.getResName();
    		String atomName=node2.getAtomName();
    		String subAtomName=atomName;
    		if(atomName.length()>2)
    			subAtomName=atomName.substring(0,2);
    		if(atomName.equalsIgnoreCase("H")||subAtomName.equalsIgnoreCase("HA")||subAtomName.equalsIgnoreCase("HN"))
    			continue;
    			
    		System.out.println(resName+resNo+"-"+atomName+":");
    		
    		for(i=0;i<vecResonGraph.size();i++)
    		{
    			Node node=(Node)vecResonGraph.elementAt(i);
    			double cs_h=node.getProtonCS();
    			double cs_heavy=node.getHeavyCS();
    			if(Map[i][j]==1)
    				System.out.format("        %.3f ,  %.3f%n", cs_h, cs_heavy);
    			
    		}//for(i=0;i<vecResonGraph.size();i++)
    		
    	}//for(j=0;j<vecGraph.size();j++)
    	
	}
	/**
     * update stereo symmetry
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
   
     @return 
     */
	public void UpdateStereoSymSet(Vector vecResonGraph,Vector vecGraph, int [][] Map)
	{		
		double errHeavy=0.1;
		Peak pk=new Peak();
		//initialization
		for(int i=0;i<vecResonGraph.size();i++)
		{
			Node nodeReson=(Node)vecResonGraph.elementAt(i);
			nodeReson.vecStereoSymSet=new Vector();
		}
		UpdateMappingSet(vecResonGraph,vecGraph,Map);	
		
		//using only chemical shift information
		for(int i=0;i<vecResonGraph.size();i++)
		{
			Node nodeReson=(Node)vecResonGraph.elementAt(i);
			double cs_proton=nodeReson.getProtonCS();
			double cs_heavy=nodeReson.getHeavyCS();
			Vector vec_sym=nodeReson.vecSymSet;
			Vector vec_map=nodeReson.getMappingSet();
			
			for(int t=0;t<vec_sym.size();t++)
			{
				Node nodeReson2=(Node)vec_sym.elementAt(t);				
				double cs_heavy2=nodeReson2.getHeavyCS();			
				Vector vec_map2=nodeReson2.getMappingSet();
								
				boolean isMap=false;				
				
				//use the mapping information to refine stereo symmetry sets
				for(int k=0;k<vec_map.size();k++)
				{
					Node ndProton1=(Node)vec_map.elementAt(k);
					String atom1=ndProton1.getAtomName();
					String res1=ndProton1.getResName();
					String heavyName1=pk.GetHeavyAtomFromProton(res1,atom1);
					for(int w=0;w<vec_map2.size();w++)
					{
						Node ndProton2=(Node)vec_map2.elementAt(w);
						String atom2=ndProton2.getAtomName();
						String res2=ndProton2.getResName();
						String heavyName2=pk.GetHeavyAtomFromProton(res2,atom2);
						if(heavyName1.equalsIgnoreCase(heavyName2))
							isMap=true;						
					}//for(int w=0;w<vec_map2.size();w++)				
				}//for(int k=0;k<vec_map.size();k++)
				
				if(isMap && Math.abs(cs_heavy-cs_heavy2)< errHeavy  )
				{
					nodeReson.vecStereoSymSet.add(nodeReson2);
					nodeReson2.vecStereoSymSet.add(nodeReson);
					nodeReson.stereoID=nodeReson2.getID();
					nodeReson2.stereoID=nodeReson.getID();
				}

			}//for(int t=0;t<vec_adj.size();t++)
		}//for(i=0;i<vecResonGraph.size();i++)		
		
		//delete repeated stereo nodes;
		for(int i=0;i<vecResonGraph.size();i++)
		{
			Node nodeReson=(Node)vecResonGraph.elementAt(i);
			Vector vecStereoSet=nodeReson.vecStereoSymSet;
			for (int k=0;k<vecStereoSet.size();k++)
			{
				Node nodeStereo=(Node)vecStereoSet.elementAt(k);
				int curID=nodeStereo.getID();
				boolean isInPre=false;
				for(int t=0;t<vecStereoSet.size();t++)
				{
					Node nodeTemp=(Node)vecStereoSet.elementAt(t);
					if(t==k)
						continue;
					if(nodeTemp.getID()==curID)
						isInPre=true;
				}
				if(isInPre)
				{
					vecStereoSet.remove(k);
					k--;
				}
				
			}//for (int k=0;k<vecStereoSet.size();k++)
		}//for(int i=0;i<vecResonGraph.size();i++)
	}
	/**
     * update the mapping set based on the mapping matrix
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param 
     @param 
     @return 
     */
	public void UpdateIsRestrained(Vector vecResonGraph,Vector vecGraph, int [][] Map)
	{
		int i,j;
		for(i=0;i<vecResonGraph.size();i++)
		{
			Node nodeReson=(Node)vecResonGraph.elementAt(i);
			if(nodeReson.getMappingSet().size()<5)
				nodeReson.setIsRestrained(true);
		}	
	}
	/**
     * update the mapping set based on the mapping matrix
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param 
     @param 
     @return 
     */
	public void UpdateMappingSet(Vector vecResonGraph,Vector vecGraph, int [][] Map)
	{
		int i,j;
		for(i=0;i<vecResonGraph.size();i++)
		{
			Node nodeReson=(Node)vecResonGraph.elementAt(i);
			nodeReson.EmptyMappingSet();
		}			
		for(j=0;j<vecGraph.size();j++)
		{
			Node nodeStruct=(Node)vecGraph.elementAt(j);
			nodeStruct.EmptyMappingSet();
		}//for(int j=0;j<vecStructGraph.size();j++)			
			
		
		for(i=0;i<vecResonGraph.size();i++)
		{
			Node nodeReson=(Node)vecResonGraph.elementAt(i);
			int resonId=nodeReson.getID();
			
			for(j=0;j<vecGraph.size();j++)
			{
				Node nodeStruct=(Node)vecGraph.elementAt(j);
				if (Map[i][j]==1)
				{
					nodeReson.addMappingNode(nodeStruct);
					//if(!nodeReson.getIsAssigned())
					nodeStruct.addMappingNode(nodeReson);//we don't add assiged resonance node
				}
			}//for(int j=0;j<vecStructGraph.size();j++)			
		}//for(int i=0;i<vecResonGraph.size();i++)		
	}
	/**
     * refine mapping based on neighboring information
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param 
     @param 
     @return 
     */
	public void RefineMappingBasedNeighbor(Vector vecResonGraph,Vector vecGraph, int [][] Map)
	{
		Nasca isoGraph=new Nasca();
		int i,j;
		
    	for(i=0;i<vecResonGraph.size();i++)
    	{
    		Node node=(Node)vecResonGraph.elementAt(i);
    		Vector vecAdj=node.getAdjVec();
    		boolean isAssigned=node.getIsAssigned();
    		if(isAssigned)
    			continue;
    		
    		Vector vecNeighborSC=new Vector();
    		for(j=0;j<vecGraph.size();j++)
    		{
    			if(i==218 && j==3)
    				System.out.println("stop here...");
    			
    			Node node2=(Node)vecGraph.elementAt(j);
    			String resName=node2.getResName();
    			String atomName=node2.getAtomName();
    			String subAtom=atomName;
    			if (atomName.length()>2)
    				subAtom=atomName.substring(0,2);
    			
    			int threshold=0;
    			
    			if(resName.equalsIgnoreCase("GLN") && subAtom.equalsIgnoreCase("HG"))
    				threshold=16;
    			else if(resName.equalsIgnoreCase("GLN") && subAtom.equalsIgnoreCase("HE"))
    				threshold=2;
    			else if(resName.equalsIgnoreCase("GLU") && subAtom.equalsIgnoreCase("HG"))
    				threshold=11;
    			else if(resName.equalsIgnoreCase("TRP"))
    				threshold=6;
    			else if(resName.equalsIgnoreCase("PHE") && subAtom.equalsIgnoreCase("HE"))
    				threshold=4;
    			else
    				threshold=1;//1
    			if(resName.equalsIgnoreCase("LEU") && atomName.equalsIgnoreCase("HG"))
    				threshold=1;
    			
    			
    			Vector vecAdj2=node2.getAdjVec();
    			boolean isMapped=false;
    			
    			
    			//following are not useful
    			if( subAtom.equalsIgnoreCase("HB"))
    				threshold=1;//checked
    			if( subAtom.equalsIgnoreCase("HG")  )
    			{
    				if(resName.equalsIgnoreCase("GLU") )
    					threshold=15;//checked     				
    				if(resName.equalsIgnoreCase("LYS") )
    					threshold=15;//checked    				
    				if(resName.equalsIgnoreCase("LEU") )
    					threshold=9;//checked
    				if(resName.equalsIgnoreCase("MET") )
    					threshold=15;//checked for eta    				
    				if(resName.equalsIgnoreCase("PRO") )
    					threshold=15;//checked for eta    				
    				if(resName.equalsIgnoreCase("GLN") )
    					threshold=11;//checked for eta
    				if(resName.equalsIgnoreCase("VAL") )
    					threshold=15;//checked for eta
    			
    				
    				if(resName.equalsIgnoreCase("ARG") )
    					threshold=5;//unchecked
    				if(resName.equalsIgnoreCase("THR") )
    					threshold=5;//unchecked
    				if(resName.equalsIgnoreCase("ILE") )
    					threshold=5;//unchecked
    			}
    			
    			threshold=0;
    			if(vecAdj.size()<1 || vecAdj2.size()<1)
    			{
    				System.out.println("debugging...we found an empty neiborhood...");
    				continue;
    			}
    			
    			double[] neighborSC=new double[1];
    			isMapped=isoGraph.isMappedBetwTwoSets(vecAdj,vecAdj2,Map,false,threshold,neighborSC);
    			if(isMapped==false)
    				Map[i][j]=0;
    			
    		
    		}//for(j=0;j<vecGraph.size();j++)
    		
    		
    		
    	}//for(i=0;i<vecResonGraph.size();i++)
    	
	}
	/**
     * Initialization.
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecBBAsg backbone resonance assignment list
     @param 
     @param
     @return 
     */
	public void Initialization(Vector vecResonGraph,Vector vecGraph, int [][] Map,double [][] MapScore,Vector vecBBAsg)
	{
		int i,j;
		double errCB=0.25;//0.2 for eta
		Peak pk=new Peak();
		for(i=0;i<vecResonGraph.size();i++)
    		for(j=0;j<vecGraph.size();j++)
    		{
    			Map[i][j]=1;
    			MapScore[i][j]=-9999.9;
    		}
    	for(i=0;i<vecResonGraph.size();i++)
    	{
    		Node node=(Node)vecResonGraph.elementAt(i);
    		boolean isAsged=node.getIsAssigned();
    		if(isAsged==false)
    			continue;
    		int resNo=node.getResNo();
    		String atomName=node.getAtomName();
    		
    		for(j=0;j<vecGraph.size();j++)
    		{
    			Node nodeStruct=(Node)vecGraph.elementAt(j);
    			int resNoStruct=nodeStruct.getResNo();
    			String atomNameStuct=nodeStruct.getAtomName();
    			String resNameStruct=nodeStruct.getResName();
    			String subAtomNameStruct=atomNameStuct;    			
    			if(atomNameStuct.length()>2)
    				subAtomNameStruct=atomNameStuct.substring(0,2);
    			
    			if((resNoStruct==resNo && atomNameStuct.equalsIgnoreCase(atomName) ) ||
    					(resNoStruct==resNo && resNameStruct.equalsIgnoreCase("GLY") && subAtomNameStruct.equalsIgnoreCase(atomName) ))
    			{
    				if(nodeStruct.id==56)
    					System.out.println("debugging..");
    				
    				if(node.getIsAssigned())
    					nodeStruct.setIsAssigned(true);
    				Map[i][j]=1;
    				node.asgedId=j;
    				nodeStruct.asgedId=i;
    				for(int k=0;k<vecResonGraph.size();k++)
    				{
    					if(k==i)
    						continue;
    					Map[k][j]=0;
    				}
    			}//if(resNoStruct==resNo && atomNameStuct.equalsIgnoreCase(atomName))
    			else
    				Map[i][j]=0;    			
    		}//for(j=0;j<vecGraph.size();j++)
    	}//for(i=0;i<vecResonGraph.size();i++)
    	
    	//initialize the CB-HB connection:
       	for(i=0;i<vecResonGraph.size();i++)
    	{
    		Node node=(Node)vecResonGraph.elementAt(i);
    		double cs_heavy=node.getHeavyCS();
    		boolean isAsged=node.getIsAssigned();
    		if(isAsged==true)
    			continue;
    		int resNo=node.getResNo();
    		String atomName=node.getAtomName();
    		
    		for(j=0;j<vecGraph.size();j++)
    		{
    			Node nodeStruct=(Node)vecGraph.elementAt(j);
    			double csHeavyStruct=0.0;
    			int resNoStruct=nodeStruct.getResNo();
    			String atomNameStuct=nodeStruct.getAtomName();
    			String resNameStruct=nodeStruct.getResName();
    			String subAtomNameStruct=atomNameStuct;
    			String heavyNameStruct=pk.GetHeavyAtomFromProton(resNameStruct,atomNameStuct);
    			if(atomNameStuct.length()>2)
    				subAtomNameStruct=atomNameStuct.substring(0,2);
    			if(!subAtomNameStruct.equalsIgnoreCase("HB"))
    				continue;
    			if(!heavyNameStruct.equalsIgnoreCase("CB"))
    				continue;
    			//find the CB chemical shift
    			for(int k=0;k<vecBBAsg.size();k++)
    			{
    				H1CS h1cs=(H1CS)vecBBAsg.elementAt(k);
    				String CB_atom=h1cs.getAtomName();
    				int CB_resNo=h1cs.getResidueNo();
    				if(!CB_atom.equalsIgnoreCase("CB"))
    					continue;
    				if(CB_resNo==resNoStruct && CB_atom.equalsIgnoreCase("CB"))
    					csHeavyStruct=h1cs.getH1CS();
    			}//for(int k=0;k<vecBBAsg.size();k++)    			
    				
    			if(Math.abs(csHeavyStruct-cs_heavy)<errCB)
    			{
    				Map[i][j]=1;    				
    			}//if(resNoStruct==resNo && atomNameStuct.equalsIgnoreCase(atomName))
    			else
    				Map[i][j]=0;    			
    		}//for(j=0;j<vecGraph.size();j++)
    	}//for(i=0;i<vecResonGraph.size();i++)
       	
       	// set up stereo ID
		for(j=0;j<vecGraph.size();j++)
		{
			Node ndProton=(Node)vecGraph.elementAt(j);
			String atom=ndProton.getAtomName();
			String res=ndProton.getResName();
			String heavy=pk.GetHeavyAtomFromProton(res,atom);
			int resNo=ndProton.getResNo();			
			for(int k=0;k<vecGraph.size();k++)
			{
				if(j==k)
					continue;
				Node ndProton2=(Node)vecGraph.elementAt(k);
				String atom2=ndProton2.getAtomName();
				String res2=ndProton2.getResName();
				String heavy2=pk.GetHeavyAtomFromProton(res2,atom2);
				int resNo2=ndProton2.getResNo();
				if(heavy.equalsIgnoreCase(heavy2) && resNo==resNo2)
				{
					ndProton.stereoID=k;
					ndProton2.stereoID=j;
				}
			}//for(int k=0;k<vecGraph.size();k++)			
		}//for(int j=0;j<vecGraph.size();j++)
	}
	/**
     * refine mapping based on BMRB outliers
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public void RefineMappingBasedBMRBOutliers(Vector vecResonGraph,Vector vecGraph, int [][] Map,Vector vecBMRBNew)
	{
		int i,j;
		Peak pk=new Peak();
	       	
    	for(i=0;i<vecResonGraph.size();i++)
    	{
    		Node node=(Node)vecResonGraph.elementAt(i);
    		double csH=node.getProtonCS();
    		double csHeavy=node.getHeavyCS();//be care of -999.9
    		boolean isAsged=node.getIsAssigned();
    		if(isAsged) 
    			continue;
    		for(j=0;j<vecGraph.size();j++)
    		{    				
    			Node node2=(Node)vecGraph.elementAt(j);
    			String resName=node2.getResName();
    			String atomName=node2.getAtomName();
    			String subAtomName=atomName;
    			if(Math.abs(csH- 7.173  )<0.04 && Math.abs(csHeavy-133.407  )<0.4 && node2.getResNo()==3 && atomName.equalsIgnoreCase("HD1") )
    				System.out.println("stop here...");
    			
    			String heavyName=pk.GetHeavyAtomFromProton(resName,atomName);    			
    			for(int k=0;k<vecBMRBNew.size();k++)
    			{
    				
    				H1CS h1csB=(H1CS)vecBMRBNew.elementAt(k);
    				String resB=h1csB.getResidueType();
    				String atomB=h1csB.getAtomName();
    				double cs_L=h1csB.getCSLower();
    				double cs_U=h1csB.getCSUpper();
    				subAtomName=atomName;
    				if(subAtomName.length()>atomB.length())
    					subAtomName=atomName.substring(0,atomB.length());
    				if(resB.equalsIgnoreCase(resName) && atomB.equalsIgnoreCase(subAtomName) && atomB.substring(0,1).equalsIgnoreCase("H"))
    				{
    					if( (csH<cs_L) || (csH> cs_U))
    						Map[i][j]=0;
    				}
    				
    				if(resB.equalsIgnoreCase(resName) && atomB.equalsIgnoreCase(heavyName) )
    				{
    					if( (csHeavy<cs_L) || (csHeavy> cs_U))
    						Map[i][j]=0;
    				}
    			}//for(int k=0;k<vecBMRBNew.size();k++)
    		
    		}//for(j=0;j<vecGraph.size();j++)
    	}//for(i=0;i<vecResonGraph.size();i++)
    	
	}
	
	/**
     * refine mapping based on intra NOE patterns
     * Note: Sysmetry-only when distance<3.0A, and sidechain protons
     * 
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param vecSeq sequence list
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public void RefineMappingBasedRotPattern(Vector vecResonGraph,Vector vecGraph, int [][] Map)
	{		
		Peak pk=new Peak();
		UpdateStereoSymSet(vecResonGraph,vecGraph,Map);
		
		//set up stereo ID
		for(int j=0;j<vecGraph.size();j++)
		{
			Node ndProton=(Node)vecGraph.elementAt(j);
			String atom=ndProton.getAtomName();
			String res=ndProton.getResName();
			String heavy=pk.GetHeavyAtomFromProton(res,atom);
			int resNo=ndProton.getResNo();			
			for(int k=0;k<vecGraph.size();k++)
			{
				if(j==k)
					continue;
				Node ndProton2=(Node)vecGraph.elementAt(k);
				String atom2=ndProton2.getAtomName();
				String res2=ndProton2.getResName();
				String heavy2=pk.GetHeavyAtomFromProton(res2,atom2);
				int resNo2=ndProton2.getResNo();
				if(heavy.equalsIgnoreCase(heavy2) && resNo==resNo2)
				{
					ndProton.stereoID=k;
					ndProton2.stereoID=j;
				}
			}//for(int k=0;k<vecGraph.size();k++)			
		}//for(int j=0;j<vecGraph.size();j++)
		
		//refine HB...
		for(int j=0;j<vecGraph.size();j++)
		{
			Node ndProton=(Node)vecGraph.elementAt(j);
			String atom=ndProton.getAtomName();
			String res=ndProton.getResName();
			String heavy=pk.GetHeavyAtomFromProton(res,atom);
			//int resNo=ndProton.getResNo();
			Vector vecMapSet=ndProton.getMappingSet();
			if(!heavy.equalsIgnoreCase("CB"))
				continue;
			boolean isStereoSym=false;
			Vector vecStereoSym=new Vector();
			for(int k=0;k<vecMapSet.size();k++)
			{
				Node ndMapReson=(Node)vecMapSet.elementAt(k);
				if(ndMapReson.vecStereoSymSet.size()>1)
					isStereoSym=true;
				vecStereoSym.addAll(ndMapReson.vecStereoSymSet);
			}//for(int k=0;k<vecMapSet.size();k++)
			
			//update mapping set:
			if(!isStereoSym)
				continue;		
			if(ndProton.stereoID>=0)
			{
				for(int i=0;i<vecResonGraph.size();i++)
				{
					if(Map[i][j]==0)
						continue;
					boolean isInStereoSym=false;
					for(int k=0;k<vecStereoSym.size();k++)
					{
						Node ndStereoSym=(Node)vecStereoSym.elementAt(k);
						int id=ndStereoSym.getID();
						if(id==i)
							isInStereoSym=true;					
					}//for(int k=0;k<vecStereoSym.size();k++)
					if(!isInStereoSym)
					{
						Map[i][j]=0;
						Map[i][ndProton.stereoID]=0;
					}
				}//for(int i=0;i<vecResonGraph.size();i++)		
			}//if(ndProton.stereoID>=0)
			else
			{
				for(int i=0;i<vecResonGraph.size();i++)
				{
					if(Map[i][j]==0)
						continue;
					boolean isInStereoSym=false;
					for(int k=0;k<vecStereoSym.size();k++)
					{
						Node ndStereoSym=(Node)vecStereoSym.elementAt(k);
						int id=ndStereoSym.getID();
						if(id==i)
							isInStereoSym=true;					
					}//for(int k=0;k<vecStereoSym.size();k++)
					if(isInStereoSym)
						Map[i][j]=0;				
					
				}//for(int i=0;i<vecResonGraph.size();i++)		
			}//else
			
		}//for(int j=0;j<vecGraph.size();j++)
			
	}
	
	/**
     * refine mapping based on backbone assignment
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public void RefineMappingBasedAsg(Vector vecResonGraph,Vector vecGraph, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew)
	{
		int sizeReson=5;
		int sizeStruct=5;
		Nasca isoGraph=new Nasca();
		Peak pk=new Peak();
		int i,j;
    	for(i=0;i<vecResonGraph.size();i++)
    	{
    		Node node=(Node)vecResonGraph.elementAt(i);
    		double cs_proton=node.getProtonCS();
    		double cs_heavy=node.getHeavyCS();
    		Vector vecAdj=node.getAdjVec();
    		boolean isAssigned=node.getIsAssigned();
    		if(isAssigned)
    			continue;
    		
    		double maxMapping=0.0;
    		//for storing the mapping likelihood for all nodes
    		Vector vecNeighborSC=new Vector();    		    		
    		for(j=0;j<vecGraph.size();j++)
    		{
    			if(i==284 && j==24)
    				System.out.println("stop here...");
    			if(Map[i][j]==0)
        			continue;
    			Node node2=(Node)vecGraph.elementAt(j);
    			String resName=node2.getResName();
    			String atomName=node2.getAtomName();
    			String subAtom=atomName;
    			if (atomName.length()>2)
    				subAtom=atomName.substring(0,2);
    			
    			if(Math.abs(cs_proton-2.016    )<0.05 && Math.abs(cs_heavy-34.446 )<0.5  && node2.getResNo()==10 && atomName.equalsIgnoreCase("HB") )
    				System.out.println("stop here...");
    			
    			
    			String heavyName=pk.GetHeavyAtomFromProton(resName,atomName);
    			if( !(heavyName.substring(0,1).equalsIgnoreCase("N") || heavyName.substring(0,1).equalsIgnoreCase("C") ))
    				continue;
    			int threshold=0;    			
    			
    			/////////////////////////////////
    			Vector vecAdj2=node2.getAdjVec();
    			boolean isMapped=false;
    			if(vecAdj.size()<1 || vecAdj2.size()<1)
    			{
    				System.out.println("debugging...we found an empty neiborhood...");
    				continue;
    			} 
    		
    			double[] neighborSC=new double[1];
    			
    			
    			isMapped=isoGraph.isMappedBetwTwoSetsCalScore(node,node2,vecAdj,vecAdj2,Map,true,threshold,neighborSC,vecNewNoesy);
    			   			
    			RotaPattern rp=new RotaPattern(j,neighborSC[0]);
    			vecNeighborSC.add(rp);
    			
    		}//for(j=0;j<vecGraph.size();j++)
    		
    		for (int h=0;h<vecNeighborSC.size();h++)    		
    		{
    			RotaPattern rotPatn=(RotaPattern)vecNeighborSC.elementAt(h);    			
    			Node node2=(Node)vecGraph.elementAt(rotPatn.getResNo());
    			String resName=node2.getResName();
    			String atomName=node2.getAtomName();
    			int resNo=node2.getResNo();
    			Vector vecOneAsg=new Vector();
    			String heavyName=pk.GetHeavyAtomFromProton(resName, atomName);
    			vecOneAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
    			vecOneAsg.add(new H1CS(resNo,resName,heavyName,cs_heavy));
	    		    			
    			int graph_ID=rotPatn.getResNo();    			
    		   
    			double sc=rotPatn.getScore();
    			H1CS h1cs=new H1CS();
    			double asgScore=h1cs.BMRBSatisticsScore(vecOneAsg,vecBMRBNew);
    			double asg_log=Math.log(asgScore);
	    		double noe_log=Math.log(sc);
	    		if(asg_log<-10000.0)
	    			asg_log=-1000.0;
	    		if(noe_log<-10000.0)
	    			noe_log=-10000.0;
	    		double Score=0.2*asg_log+noe_log;
    			rotPatn.setScore(Score);
    			MapScore[i][graph_ID]=Score;
	    			    		
    		}//for (int h=0;h<vecNeighborSC.size();h++) 
    		
    		////////////////////////////
    		//we select the top k mapped nodes
    		Collections.sort(vecNeighborSC, new RotaPattern.scoreComparator());
    		
    		for(j=0;j<vecGraph.size();j++)
    		{
    			boolean isIn=false;
	    		for (int h=0;h<Math.min(sizeReson,vecNeighborSC.size());h++)//16
	    		{
	    			RotaPattern rotPatn=(RotaPattern)vecNeighborSC.elementAt(h);
	    			int graph_ID=rotPatn.getResNo();
	    			if(j==graph_ID)
	    				isIn=true;	    			
	    		}//for (int h=0;h<Math.min(3,vecNeighborSC.size());h++)
	    		if(!isIn)
	    			Map[i][j]=0;
    		}//for(j=0;j<vecGraph.size();j++)
    		
    		///////////////////////
    	}//for(i=0;i<vecResonGraph.size();i++)
    	
    	//phase 2.2:
    	//debugging: only output top k mapping candidates for each proton:
    	for(j=0;j<vecGraph.size();j++)
    	{
    		Node nodeStruct=(Node)vecGraph.elementAt(j);
    		Vector vecNeighborSC=new Vector();  
    		for(i=0;i<vecResonGraph.size();i++)
    		{
    			if(Map[i][j]==0)
    				continue;
    			RotaPattern rp=new RotaPattern(i,MapScore[i][j]);    			
    			vecNeighborSC.add(rp);    			
    		}//for(i=0;i<vecResonGraph.size();i++)
    		Collections.sort(vecNeighborSC, new RotaPattern.scoreComparator());
    		
    		for(i=0;i<vecResonGraph.size();i++)
    		{
    			boolean isIn=false;
	    		for (int h=0;h<Math.min(sizeStruct,vecNeighborSC.size());h++)//25
	    		{
	    			RotaPattern rotPatn=(RotaPattern)vecNeighborSC.elementAt(h);
	    			int graph_ID=rotPatn.getResNo();
	    			if(i==graph_ID)
	    				isIn=true;	    			
	    		}//for (int h=0;h<Math.min(3,vecNeighborSC.size());h++)
	    		if(!isIn) 
	    			Map[i][j]=0;
    		}//for(j=0;j<vecGraph.size();j++)
    	}//for(j=0;j<vecGraph.size();j++)
    	
	}
	/**
     * refine mapping based on previous restrained mapping sets,
     * modified from "RefineMappingBasedAsg"
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public void RefineMappingBasedRestrained(Vector vecResonGraph,Vector vecGraph, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew)
	{		
		int sizeReson=2;
		int sizeStruct=2;
		int restrainThreshold=5;
		Nasca isoGraph=new Nasca();
		Peak pk=new Peak();
		H1CS h1cs=new H1CS();
		int i,j;
		
		UpdateStereoSymSet(vecResonGraph,vecGraph,Map);
		for(j=0;j<vecGraph.size();j++)
		{    						
			Node node2=(Node)vecGraph.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			int resNo=node2.getResNo();
			boolean isRestrained=false;
			if(node2.getIsAssigned())
				continue;
			
			if(resNo==3 && atomName.equalsIgnoreCase("HB2"))
				System.out.println("debugging...");
			
			////////////////////
			String subAtom=atomName;
			if(atomName.length()>=2)
				subAtom=atomName.substring(0,2);
			if(!(subAtom.equalsIgnoreCase("HB") ) )
				continue;
			if(node2.stereoID>=0)
				sizeReson=1;//2
			else
				sizeReson=1;
			/////////////////////
			
			
			//for storing the mapping likelihood for all nodes
    		Vector vecNeighborSC=new Vector();    
	    	for(i=0;i<vecResonGraph.size();i++)
	    	{   
	    		
	    		if(Map[i][j]==0)
	    			continue;
	    		Node node=(Node)vecResonGraph.elementAt(i);
	    		double cs_proton=node.getProtonCS();
	    		double cs_heavy=node.getHeavyCS();
	    		Vector vecAdj=node.getAdjVec();
	    		boolean isAssigned=node.getIsAssigned();
	    		if(isAssigned)
	    			continue;	  
	    		
	    		////////////////////////////
	    		//
	    		if(Math.abs(cs_proton- 0.884   )<0.04 && Math.abs(cs_heavy-27.019   )<0.4  && node2.getResNo()==5 && atomName.equalsIgnoreCase("HG") )
    				System.out.println("stop here...");
	    		
	    		if(j==2 &&(i==305 ))
	    			System.out.println("stop here...");
	    		
    			String heavyName=pk.GetHeavyAtomFromProton(resName,atomName);
    			if( !(heavyName.substring(0,1).equalsIgnoreCase("N") || heavyName.substring(0,1).equalsIgnoreCase("C") ))
    				continue;
    			
    		  	/////////////////////////////////
    			Vector vecAdj2=node2.getAdjVec();
    			
    			if(vecAdj.size()<1 || vecAdj2.size()<1)
    			{
    				System.out.println("debugging...we found an empty neiborhood...");
    				continue;
    			} 
    		
    			double[] neighborSC=new double[1];    	
    			boolean isRestrainTemp=false;
    			isRestrainTemp=isoGraph.isMappedBetwTwoSetsCalScore(node,node2,vecAdj,vecAdj2,Map,true,restrainThreshold,neighborSC,vecNewNoesy);
    			Vector vecOneSCAsg=new Vector();
    			vecOneSCAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
    			vecOneSCAsg.add(new H1CS(resNo,resName,heavyName,cs_heavy));
    			double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsg,vecBMRBNew);
        		  			
    			neighborSC[0]=neighborSC[0]+asgScore;
    			MapScore[i][j]=neighborSC[0];
    						
    			if(isRestrainTemp)
		   			isRestrained=true;
		   		
    			RotaPattern rp=new RotaPattern(i,neighborSC[0]);
    			vecNeighborSC.add(rp);    			    	
    		}//for(i=0;i<vecResonGraph.size();i++)  		

	  
	    	  
	    	node2.setIsRestrained(true);
    		////////////////////////////
    		//we select the top k mapped nodes
    		Collections.sort(vecNeighborSC, new RotaPattern.scoreComparator());
    	    	
    		for(i=0;i<vecResonGraph.size();i++)
    		{    			
    			boolean isIn=false;
    			
	    		for (int h=0;h<Math.min(8,vecNeighborSC.size());h++)//6: good for HB#
	    		{
	    			RotaPattern rotPatn=(RotaPattern)vecNeighborSC.elementAt(h);
	    			int graph_ID=rotPatn.getResNo();
	    			
	    			Node nodeTemp=(Node)vecResonGraph.elementAt(graph_ID);
	    			int stereoIDTemp=nodeTemp.stereoID;
	    			if(i==graph_ID || i==stereoIDTemp)
	    				isIn=true;	    			
	    			
	    		}//for (int h=0;h<Math.min(3,vecNeighborSC.size());h++)
	    		
	    		if(!isIn)
	    			Map[i][j]=0;
    		}//for(j=0;j<vecGraph.size();j++)
    		
    		
    			
    	}//for(j=0;j<vecGraph.size();j++)
    	
	}
	/**random graph matching algorithm. allow two protons to be assigned to 
	 * the same resonance node.
     * 
     * note: unassigned case is possoble for a node; stereo protons can be assigned to 
     * the same resonance node.
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public Vector RandomGraphMatching(Vector vecResonGraph,Vector vecGraph, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew)
	{			
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;
		
		//1. Initialization: a random 1-1 assignment:
		//note: it is possible that an proton node is unassigned.
		UpdateMappingSet(vecResonGraph,vecGraph,Map);
		Random generator = new Random( 198221 );
		for(int j=0;j<vecGraph.size();j++)
		{    						
			Node node2=(Node)vecGraph.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			int resNo=node2.getResNo();
			boolean isRestrained=false;
			Vector vecMapSet=node2.getMappingSet();
			if(node2.getIsAssigned())
				continue;
						
			boolean isMapAllAsg=true;
			for(int k=0;k<vecMapSet.size();k++)
			{
				Node nodeMap=(Node)vecMapSet.elementAt(k);
				if(nodeMap.asgedId<0)
					isMapAllAsg=false;
				if(nodeMap.asgedId==node2.stereoID)//note: here we need to consider the stereo nodes.
					isMapAllAsg=false;
			}
			if(isMapAllAsg)//we don't assign current one
			{
				System.out.println("Here we found one unassigned proton node...");
			}
			
			
			int rdmInt=generator.nextInt(vecMapSet.size());
			Node nodeMap=(Node)vecMapSet.elementAt(rdmInt);
			
			node2.asgedId=nodeMap.getID();
			nodeMap.asgedId=node2.getID();			
		}//for(j=0;j<vecGraph.size();j++)
		
		//2. swap operations for resonance nodes (#of resonances > # of protons)
		
		int nSteps=200000;
		int count=0;
		System.out.println("nSteps="+nSteps);
		while(count<nSteps)
		{
			if( ((int)(count % 1000))==0 )
				System.out.println("current count="+count);
			//randomly choose two sc nodes:
			
			int rdmID=generator.nextInt(vecGraph.size());
			Node nodeChosen=(Node)vecGraph.elementAt(rdmID);		
			
			if(nodeChosen.getIsAssigned())
				continue;
			
			//calculate the old matching score, before swap:
			Vector vecNoesyTwo=new Vector();			
			Node nodeReson=new Node();			
			nodeReson=(Node)vecResonGraph.elementAt(nodeChosen.asgedId);			
			Vector vecNoesy=nodeReson.vecAdjNOESY;
			vecNoesyTwo.addAll(vecNoesy);
			
			Vector vecBackNoe=new Vector();
			Vector vecAdj=nodeChosen.getAdjVec();
			
			int resNo=-1;
			String res="";
			String atom="";
			String heavName="";
			double cs_proton=-999.9;
			double cs_heavy=-999.9;
			
			Vector vecOneSCAsg=new Vector();
			
		
			resNo=nodeChosen.getResNo();
			res=nodeChosen.getResName();
			atom=nodeChosen.getAtomName();
			heavName=pk.GetHeavyAtomFromProton(res,atom);
			cs_proton=nodeReson.getProtonCS();
			cs_heavy=nodeReson.getHeavyCS();
				
			vecOneSCAsg.add(new H1CS(resNo,res,atom,cs_proton));			
			vecOneSCAsg.add(new H1CS(resNo,res,heavName,cs_heavy));			
		 
			for(int k=0;k<vecAdj.size();k++)
			{
				Node nodeAdj=(Node)vecAdj.elementAt(k);
				Node nodeAdjReson=(Node)vecResonGraph.elementAt(nodeAdj.asgedId);
				int resNoSec=nodeAdj.getResNo();
				String resSec=nodeAdj.getResName();
				String atomSec=nodeAdj.getAtomName();
				String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
				double cs_protonSec=nodeAdjReson.getProtonCS();
				double cs_heavySec=nodeAdjReson.getHeavyCS();
				vecOneSCAsg.add(new H1CS(resNoSec,resSec,atomSec,cs_protonSec));
				vecOneSCAsg.add(new H1CS(resNoSec,resSec,heavyNameSec,cs_heavySec));
				
				vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,6.0,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
				vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,6.0,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
			
			}//for(int k=0;k<vecAdj1.size();k++)		
			
			Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
			int [] numPeaks=new int[1];
			double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
			dbScore=dbScore;//*vecBackNoeNew.size();//the more number of matched nodes, the better
			Vector vecOneSCAsgNew=h1cs.DeleteRepeatAsg(vecOneSCAsg);
			double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsgNew,vecBMRBNew);
			double score_old=dbScore+asgScore;//current score.			
			
			//compute the new score after swap:
			vecOneSCAsg=new Vector();	
			Vector vecMapSet=nodeChosen.getMappingSet();
			int rdmInt=generator.nextInt(vecMapSet.size());
			Node nodeResonMap=(Node)vecMapSet.elementAt(rdmInt); 
				
			resNo=nodeChosen.getResNo();
			res=nodeChosen.getResName();
			atom=nodeChosen.getAtomName();
			heavName=pk.GetHeavyAtomFromProton(res,atom);
			cs_proton=nodeResonMap.getProtonCS();//swap...
			cs_heavy=nodeResonMap.getHeavyCS();
			
			vecOneSCAsg.add(new H1CS(resNo,res,atom,cs_proton));			
			vecOneSCAsg.add(new H1CS(resNo,res,heavName,cs_heavy));			
			 
			for(int k=0;k<vecAdj.size();k++)
			{
				Node nodeAdj=(Node)vecAdj.elementAt(k);
				Node nodeAdjReson=(Node)vecResonGraph.elementAt(nodeAdj.asgedId);
				int resNoSec=nodeAdj.getResNo();
				String resSec=nodeAdj.getResName();
				String atomSec=nodeAdj.getAtomName();
				String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
				double cs_protonSec=nodeAdjReson.getProtonCS();
				double cs_heavySec=nodeAdjReson.getHeavyCS();
				
				vecOneSCAsg.add(new H1CS(resNoSec,resSec,atomSec,cs_protonSec));
				vecOneSCAsg.add(new H1CS(resNoSec,resSec,heavyNameSec,cs_heavySec));
				
				vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,6.0,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
				vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,6.0,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
			
			}//for(int k=0;k<vecAdj1.size();k++)		


			vecBackNoeNew=new Vector();
			vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);			
			dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
			dbScore=dbScore;//*vecBackNoeNew.size();//the more number of matched nodes, the better
			vecOneSCAsgNew=new Vector();
			vecOneSCAsgNew=h1cs.DeleteRepeatAsg(vecOneSCAsg);
			asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsgNew,vecBMRBNew);
			double score_new=dbScore+asgScore;//current score.
			
			if(score_new>score_old)//swap the assignments
			{
				nodeChosen.asgedId=nodeResonMap.getID();
				nodeResonMap.asgedId=nodeChosen.getID();			
			}
			count++;
		}//while(count<nSteps)
		
		Vector vecAsg=new Vector();
		for(int j=0;j<vecGraph.size();j++)
		{    						
			Node node2=(Node)vecGraph.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			String heavName=pk.GetHeavyAtomFromProton(resName,atomName);
			int resNo=node2.getResNo();
			
			int resonID=node2.asgedId;
			Node nodeResonAsg=(Node)vecResonGraph.elementAt(resonID);
			double cs_proton=nodeResonAsg.getProtonCS();
			double cs_heavy=nodeResonAsg.getHeavyCS();
			vecAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
			vecAsg.add(new H1CS(resNo,resName,heavName,cs_heavy));
		}//for(int j=0;j<vecGraph.size();j++)
		
		return vecAsg;
	}
	
	/**at each step, choose the most confident assignment for one proton node.
     * at each step, assign the whole residue.
     * note: unassigned case is possoble for a node; stereo protons can be assigned to 
     * the same resonance node.
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public Vector GreedyGraphMatchingResidue(Vector vecResonGraph,Vector vecGraph, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew)
	{			
		int maxResNo=300;
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Noesy noesy=new Noesy();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;				
		
		boolean [] isResidueAsg=new boolean[maxResNo];		
		Vector vecResidueGraph=new Vector();
		for (int k=0;k<maxResNo;k++)
		{
			isResidueAsg[k]=false;
			Vector vecCurRes=new Vector();
			for(int j=0;j<vecGraph.size();j++)
			{    						
				Node node2=(Node)vecGraph.elementAt(j);				
				int resNo=node2.getResNo();		
				if(resNo==k)
					vecCurRes.add(node2);
			}//for(int j=0;j<vecGraph.size();j++)
			if(vecCurRes.size()>0)
				vecResidueGraph.add(vecCurRes);
		}//for (int k=0;k<=maxResNo;k++)
		
		while(true)
		{
			UpdateMappingSet(vecResonGraph,vecGraph, Map);
			boolean isAllAsg=true;
			double maxScore=-99999.9;
			int maxResID=-1;
			int maxMapID=-1;
			
			for(int w=0;w<vecResidueGraph.size();w++)
			{
				if(isResidueAsg[w]==true)
					continue;
				isAllAsg=false;
				Vector vecCurResidue=(Vector)vecResidueGraph.elementAt(w);
				Vector vecAllSCAsgs=EnumAllScAsgSaveIntoFile(vecCurResidue,vecResonGraph,vecGraph);
		    	
				for(int x=0;x<vecAllSCAsgs.size();x++)//for all possible assignments for 1 residue
				{
					Vector vecOneAsg=(Vector)vecAllSCAsgs.elementAt(x);
					Vector vecOneSCAsg=new Vector();
					vecOneSCAsg.addAll(vecOneAsg);
					Vector vecBackNoe=new Vector();		
					Vector vecNoesyTwo=new Vector();
					for(int a=0;a<vecOneAsg.size();a++)
					{
						H1CS h1csCur=(H1CS)vecOneAsg.elementAt(a);
						Node nodeProton=(Node)vecGraph.elementAt(h1csCur.protonID);
						Node nodeReson=(Node)vecResonGraph.elementAt(h1csCur.resonID);
						Vector vecAdj=nodeProton.getAdjVec();
						int resNo=nodeProton.getResNo();
						String res=nodeProton.getResName();
						String atom=nodeProton.getAtomName();
						String heavName=pk.GetHeavyAtomFromProton(res,atom);
						double cs_proton=nodeReson.getProtonCS();
						double cs_heavy=nodeReson.getHeavyCS();
						Vector vecNoesy=nodeReson.vecAdjNOESY;
						vecNoesyTwo.addAll(vecNoesy);
						for(int t=0;t<vecAdj.size();t++)
						{
							Node nodeAdj=(Node)vecAdj.elementAt(t);
							if(!nodeAdj.getIsAssigned())
								continue;
							Node nodeAdjReson=(Node)vecResonGraph.elementAt(nodeAdj.asgedId);
							Vector vecNoesy2=nodeAdjReson.vecAdjNOESY;
							//vecNoesyTwo.addAll(vecNoesy2);
							int resNoSec=nodeAdj.getResNo();
							String resSec=nodeAdj.getResName();
							String atomSec=nodeAdj.getAtomName();
							String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
							double cs_protonSec=nodeAdjReson.getProtonCS();
							double cs_heavySec=nodeAdjReson.getHeavyCS();
							vecOneSCAsg.add(new H1CS(resNoSec,resSec,atomSec,cs_protonSec));
							vecOneSCAsg.add(new H1CS(resNoSec,resSec,heavyNameSec,cs_heavySec));
							
							vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,6.0,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
							vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,6.0,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
						
						}//for(int t=0;t<vecAdj.size();t++)	
					}//for(int a=0;a<vecOneAsg.size();a++)
					
					Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
					Vector vecNoesyTwoNew=noesy.DeleteRepeatedNoesy3D(vecNoesyTwo);
					int [] numPeaks=new int[1];
					double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwoNew, numPeaks,true);
					
					dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
					Vector vecOneSCAsgNew=h1cs.DeleteRepeatAsg(vecOneSCAsg);
					double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsgNew,vecBMRBNew);
					double score=dbScore+asgScore;//current score.		
					if(score>maxScore)
					{
						maxScore=score;
						maxResID=w;
						maxMapID=x; 
					}				
					
				}//for(int x=0;x<vecAllSCAsgs.size();x++)
			}//for(int w=0;w<vecResidueGraph.size();w++)	
			
			//System.out.println("maxID="+maxID);			
			Vector vecCurResidue=(Vector)vecResidueGraph.elementAt(maxResID);
			Vector vecAllSCAsgs=EnumAllScAsgSaveIntoFile(vecCurResidue,vecResonGraph,vecGraph);
			Vector vecOneAsg=(Vector)vecAllSCAsgs.elementAt(maxMapID);
			for(int a=0;a<vecOneAsg.size();a++)
			{
				H1CS h1csCur=(H1CS)vecOneAsg.elementAt(a);
				Node nodeProton=(Node)vecGraph.elementAt(h1csCur.protonID);
				Node nodeReson=(Node)vecResonGraph.elementAt(h1csCur.resonID);
				nodeProton.asgedId=nodeReson.getID();
				nodeReson.asgedId=nodeProton.getID();
				for(int i=0;i<vecResonGraph.size();i++)
				{
					if(i==nodeProton.asgedId)
						Map[i][nodeProton.getID()]=1;
					else
						Map[i][nodeProton.getID()]=0;
				}
				for(int j=0;j<vecGraph.size();j++)
				{
					if(j==nodeReson.asgedId)
						Map[nodeReson.getID()][j]=1;
					else
						Map[nodeReson.getID()][j]=0;
				}				
				nodeProton.setIsAssigned(true);
				nodeReson.setIsAssigned(true);
				System.out.println(nodeProton.getResNo()+nodeProton.getResName()+" - "+nodeProton.getAtomName()+": " + nodeReson.getProtonCS()+" , "+ nodeReson.getHeavyCS());
				
			}		
			System.out.println("---------------------");
			if(isAllAsg || maxResID<0)
				break;
		}//while(true)		
			
		Vector vecAsg=new Vector();
		for(int j=0;j<vecGraph.size();j++)
		{    						
			Node node2=(Node)vecGraph.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			String heavName=pk.GetHeavyAtomFromProton(resName,atomName);
			int resNo=node2.getResNo();
			
			int resonID=node2.asgedId;
			if(resonID<0)
				continue;
			Node nodeResonAsg=(Node)vecResonGraph.elementAt(resonID);
			double cs_proton=nodeResonAsg.getProtonCS();
			double cs_heavy=nodeResonAsg.getHeavyCS();
			vecAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
			vecAsg.add(new H1CS(resNo,resName,heavName,cs_heavy));
		}//for(int j=0;j<vecGraph.size();j++)
		
		return vecAsg;
	}
	/**Use the DEE algorithm to prune possible side-chain resonance assignments.
     @param vecResonGraph resonance graph 
     @param vecHPositions structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public void RefineMappingDeeCut(Vector vecResonGraph,Vector vecHPositions, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew)
	{			
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		Noesy noesy=new Noesy();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;				
		int counter=0;//count how many are pruned, for the purpose of debugging.
		
		for(int j=0;j<vecHPositions.size();j++)
		{    		
		//	System.out.println("current proton node ID : "+ j);	
			Node nodeProton=(Node)vecHPositions.elementAt(j);
			if(nodeProton.getIsAssigned())
				continue;
			String res=nodeProton.getResName();
			String atom=nodeProton.getAtomName();
			String heavyName=pk.GetHeavyAtomFromProton(res,atom);
			int resNo=nodeProton.getResNo();				
			Vector vecMapSet=nodeProton.getMappingSet();
			
			Vector vecAdj=nodeProton.getAdjVec();
			if(vecMapSet.size()<1 || vecAdj.size()<1)//rarely happen
				continue;
			
			if(nodeProton.getIsAssigned())
				continue;
			for(int k=0;k<vecMapSet.size();k++)
			{
				//System.out.println("   map ID: "+ k);	
				
				Node ndMapCur=(Node)vecMapSet.elementAt(k);
				boolean isPruned=false;
				double csCur_proton=ndMapCur.getProtonCS();
				double csCur_heavy=ndMapCur.getHeavyCS();
				
				//compute the best score for current assignment
				//compute the homo score:
				Vector vecOneSCAsg=new Vector();//for computing homo score.
    			vecOneSCAsg.add(new H1CS(resNo,res,atom,csCur_proton));
    			vecOneSCAsg.add(new H1CS(resNo,res,heavyName,csCur_heavy));
    			double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsg,vecBMRBNew);
        		double totalCurScore=asgScore;
				//compute the best pair score for current assignment
				for(int t=0;t<vecAdj.size();t++)
				{
					Node ndCurAdj=(Node)vecAdj.elementAt(t);
					if(ndCurAdj.getIsAssigned())
						continue;
					Vector vecNdCurAdjMap=ndCurAdj.getMappingSet();
					if(vecNdCurAdjMap.size()<1)//rarely happen
						continue;
					
					int resNoAdj=ndCurAdj.getResNo();
					String resAdj=ndCurAdj.getResName();
					String atomAdj=ndCurAdj.getAtomName();
					String heavyNameAdj=pk.GetHeavyAtomFromProton(resAdj,atomAdj);					
					int maxAdjID=-1;					
					double maxScore=-9999.9;
					
					Vector vecBackNoe=new Vector();//for computing pairwirse 
					for(int a=0;a<vecNdCurAdjMap.size();a++)
					{
						Node ndAdjMap=(Node)vecNdCurAdjMap.elementAt(a);
						Vector vecNoesyTwo=new Vector();	
						vecNoesyTwo.addAll(ndMapCur.vecAdjNOESY);
						vecNoesyTwo.addAll(ndAdjMap.vecAdjNOESY);
						
						double csAdjMapProton=ndAdjMap.getProtonCS();
						double csAdjMapHeavy=ndAdjMap.getHeavyCS();
						vecBackNoe.add(new BackNoe(csCur_proton,csCur_heavy,csAdjMapProton,6.0,0.0,resNo,resNoAdj,res,resAdj,atom,heavyName,atomAdj ));
						vecBackNoe.add(new BackNoe(csAdjMapProton,csAdjMapHeavy,csCur_proton,6.0,0.0,resNoAdj,resNo,resAdj,res,atomAdj,heavyNameAdj,atom ));
						
						Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
						int [] numPeaks=new int[1];
						double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
						dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
						if(dbScore>maxScore)
						{
							maxScore=dbScore;
							maxAdjID=a;
						}
					}//for(int a=0;a<vecNdCurAdjMap.size();a++)
					totalCurScore=totalCurScore+maxScore;					
				}//for(int t=0;t<vecAdj.size();t++)
				
				//compute the worse score for other assignments
				for(int h=0;h<vecMapSet.size();h++)
				{
					if(h==k)
						continue;				
					Node ndMapCurOther=(Node)vecMapSet.elementAt(h);					
					double csOther_proton=ndMapCurOther.getProtonCS();
					double csOther_heavy=ndMapCurOther.getHeavyCS();
					//compute the homo score:
					Vector vecOneSCAsgOther=new Vector();//for computing homo score.
					vecOneSCAsgOther.add(new H1CS(resNo,res,atom,csOther_proton));
					vecOneSCAsgOther.add(new H1CS(resNo,res,heavyName,csOther_heavy));
	    			double asgOtherScore=h1cs.BMRBSatisticsScore(vecOneSCAsgOther,vecBMRBNew);
	        		double totalOtherScore=asgOtherScore;
	        		
	        		//compute the worse pairwise score for other nodes:
	        		for(int t=0;t<vecAdj.size();t++)
					{
						Node ndCurAdj=(Node)vecAdj.elementAt(t);
						if(ndCurAdj.getIsAssigned())
							continue;
						Vector vecNdCurAdjMap=ndCurAdj.getMappingSet();
						if(vecNdCurAdjMap.size()<1)//rarely happen
							continue;
						
						int resNoAdj=ndCurAdj.getResNo();
						String resAdj=ndCurAdj.getResName();
						String atomAdj=ndCurAdj.getAtomName();
						String heavyNameAdj=pk.GetHeavyAtomFromProton(resAdj,atomAdj);					
						int minAdjID=-1;	
						
						double minScore=999999.9;
						Vector vecBackNoe=new Vector();//for computing pairwirse 
						for(int a=0;a<vecNdCurAdjMap.size();a++)
						{
							Node ndAdjMap=(Node)vecNdCurAdjMap.elementAt(a);
							Vector vecNoesyTwo=new Vector();	
							vecNoesyTwo.addAll(ndMapCurOther.vecAdjNOESY);
							vecNoesyTwo.addAll(ndAdjMap.vecAdjNOESY);
							
							double csAdjMapProton=ndAdjMap.getProtonCS();
							double csAdjMapHeavy=ndAdjMap.getHeavyCS();
							vecBackNoe.add(new BackNoe(csCur_proton,csCur_heavy,csAdjMapProton,6.0,0.0,resNo,resNoAdj,res,resAdj,atom,heavyName,atomAdj ));
							vecBackNoe.add(new BackNoe(csAdjMapProton,csAdjMapHeavy,csCur_proton,6.0,0.0,resNoAdj,resNo,resAdj,res,atomAdj,heavyNameAdj,atom ));
							
							Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
							int [] numPeaks=new int[1];
							double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
							dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
							if(dbScore<minScore)
							{
								minScore=dbScore;
								minAdjID=a;
							}
						}//for(int a=0;a<vecNdCurAdjMap.size();a++)
						totalOtherScore=totalOtherScore+minScore;			
					}//for(int t=0;t<vecAdj.size();t++)
	        		
	        		//check whether the DEE criterion is satisfied:
	        		if(totalCurScore<totalOtherScore)
	        		{
	        			isPruned=true;
	        			break;
	        		}
	        		
				}//for(int h=0;h<vecMapSet.size();h++)
				
				//remove the mapping if it is pruned:
				if(isPruned)
				{
					Map[ndMapCur.getID()][nodeProton.getID()]=0;
					counter++;
				}
			
			}//for(int k=0;k<vecMapSet.size();k++)
						
		}//for(int j=0;j<vecGraph.size();j++)			
	
		System.out.println("Total number of mappings pruned by DEE-CUT is: "+ counter);	
	
	}
	
	/**Use the DEE algorithm to prune possible mapping. Similar to RefineMappingDeeCut,
	 * but use the Goldstein criterion.
	 * Note: this function need to further test
    @param vecResonGraph resonance graph 
    @param vecStructGraph structure graph   
    @param Map mapping matrix between two graphs  
    @param MapScore mapping score matrix
    @param vecNewNoesy noesy cross peak list
    @param vecBMRBNew BMRB information
    @param
    @return 
    */
	public void RefineMappingDeeCutGoldstein(Vector vecResonGraph,Vector vecGraph, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew)
	{			
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		Noesy noesy=new Noesy();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;				
		int counter=0;//count how many are pruned, for the purpose of debugging.
		
		for(int j=0;j<vecGraph.size();j++)
		{	
			Node nodeProton=(Node)vecGraph.elementAt(j);
			if(nodeProton.getIsAssigned())
				continue;
			String res=nodeProton.getResName();
			String atom=nodeProton.getAtomName();
			String heavyName=pk.GetHeavyAtomFromProton(res,atom);
			int resNo=nodeProton.getResNo();				
			Vector vecMapSet=nodeProton.getMappingSet();
			
			Vector vecAdj=nodeProton.getAdjVec();
			if(vecMapSet.size()<1 || vecAdj.size()<1)//rarely happen
				continue;
			
			if(nodeProton.getIsAssigned())
				continue;
			for(int k=0;k<vecMapSet.size();k++)
			{				
				Node ndMapCur=(Node)vecMapSet.elementAt(k);
				boolean isPruned=false;
				double csCur_proton=ndMapCur.getProtonCS();
				double csCur_heavy=ndMapCur.getHeavyCS();
				
				//compute the best score for current assignment
				//compute the homo score:
				Vector vecOneSCAsg=new Vector();//for computing homo score.
				vecOneSCAsg.add(new H1CS(resNo,res,atom,csCur_proton));
				vecOneSCAsg.add(new H1CS(resNo,res,heavyName,csCur_heavy));
				double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsg,vecBMRBNew);
				double totalCurScore=asgScore;
				
				
				//compute the worse score for other assignments
				for(int h=0;h<vecMapSet.size();h++)
				{
					if(h==k)
						continue;				
					Node ndMapCurOther=(Node)vecMapSet.elementAt(h);					
					double csOther_proton=ndMapCurOther.getProtonCS();
					double csOther_heavy=ndMapCurOther.getHeavyCS();
					//compute the homo score:
					Vector vecOneSCAsgOther=new Vector();//for computing homo score.
					vecOneSCAsgOther.add(new H1CS(resNo,res,atom,csOther_proton));
					vecOneSCAsgOther.add(new H1CS(resNo,res,heavyName,csOther_heavy));
	    			double asgOtherScore=h1cs.BMRBSatisticsScore(vecOneSCAsgOther,vecBMRBNew);
	        		double totalOtherScore=asgOtherScore;
							
				
	        		double sumPairScore=0.0;
				
					//compute the best pair score
					for(int t=0;t<vecAdj.size();t++)
					{
						Node ndCurAdj=(Node)vecAdj.elementAt(t);
						if(ndCurAdj.getIsAssigned())
							continue;
						Vector vecNdCurAdjMap=ndCurAdj.getMappingSet();
						if(vecNdCurAdjMap.size()<1)//rarely happen
							continue;
						
						int resNoAdj=ndCurAdj.getResNo();
						String resAdj=ndCurAdj.getResName();
						String atomAdj=ndCurAdj.getAtomName();
						String heavyNameAdj=pk.GetHeavyAtomFromProton(resAdj,atomAdj);					
						int maxAdjID=-1;					
						double maxScore=-9999.9;
						
						double pairMaxScore=-999999.9;
						Vector vecBackNoe=new Vector();//for computing pairwirse 
						for(int a=0;a<vecNdCurAdjMap.size();a++)
						{
							Node ndAdjMap=(Node)vecNdCurAdjMap.elementAt(a);
							Vector vecNoesyTwo=new Vector();	
							vecNoesyTwo.addAll(ndMapCur.vecAdjNOESY);
							vecNoesyTwo.addAll(ndAdjMap.vecAdjNOESY);
							
							double csAdjMapProton=ndAdjMap.getProtonCS();
							double csAdjMapHeavy=ndAdjMap.getHeavyCS();
							vecBackNoe.add(new BackNoe(csCur_proton,csCur_heavy,csAdjMapProton,6.0,0.0,resNo,resNoAdj,res,resAdj,atom,heavyName,atomAdj ));
							vecBackNoe.add(new BackNoe(csAdjMapProton,csAdjMapHeavy,csCur_proton,6.0,0.0,resNoAdj,resNo,resAdj,res,atomAdj,heavyNameAdj,atom ));
							
							Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
							int [] numPeaks=new int[1];
							double dbCurScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
							dbCurScore=dbCurScore*vecBackNoeNew.size();//the more number of matched nodes, the better
							
							//compute other
							vecBackNoe.add(new BackNoe(csOther_proton,csOther_heavy,csAdjMapProton,6.0,0.0,resNo,resNoAdj,res,resAdj,atom,heavyName,atomAdj ));
							vecBackNoe.add(new BackNoe(csAdjMapProton,csAdjMapHeavy,csOther_proton,6.0,0.0,resNoAdj,resNo,resAdj,res,atomAdj,heavyNameAdj,atom ));
							double dbOtherScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
							dbOtherScore=dbOtherScore*vecBackNoeNew.size();//the more number of matched nodes, the better
							
							double pairScore=dbOtherScore-dbCurScore;
							
							
							if(pairScore>pairMaxScore)
							{
								pairMaxScore=pairScore;
								maxAdjID=a;
							}
						}//for(int a=0;a<vecNdCurAdjMap.size();a++)
						sumPairScore=sumPairScore+pairMaxScore;		
											
					}//for(int t=0;t<vecAdj.size();t++)
					
					if(totalOtherScore-asgScore+sumPairScore>0)						
						isPruned=true;
				}//for(int h=0;h<vecMapSet.size();h++)
				
				//remove the mapping if it is pruned:
				if(isPruned)
				{
					Map[ndMapCur.getID()][nodeProton.getID()]=0;
					counter++;
				}
			
			}//for(int k=0;k<vecMapSet.size();k++)
						
		}//for(int j=0;j<vecGraph.size();j++)			
	
		System.out.println("Total number of mappings pruned by DEE-CUT (Goldstein) is: "+ counter);	
	
	}
	
	/** Not used now
	 * at each step, choose the most confident assignment for one proton node. Use A* algorithm to find the best assignment at each step.
     * the defference between GreedyGraphMatching and here is that here the rotamers are allowed to change in the A* search
     * note: unassigned case is possoble for a node; stereo protons can be assigned to 
     * the same resonance node.
     @param vecResonGraph resonance graph 
     @param vecBBAsg backbone resonance list  
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param vecBBPdb bb pdb
     @param strRot rotamer lib file name
     @param caliConstant constant for calibration.
     @return 
     */
	public Vector GreedyGraphMatchingAstar(Vector vecResonGraph,Vector vecGraphOld, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew,Vector vecBBAsg,Vector vecBBPdb, String strRot,double caliConstant)
	{			
		int MAX_NODE_NUM=2000;
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Pdb pp=new Pdb();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		Noesy noesy=new Noesy();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;
		
		Vector vecRotStructure=pp.RotamSelectAndStructure(csErrH, csErrN, csErrCA,vecBBPdb,vecBBAsg,strRot,vecNewNoesy,
    			4.5, 1,caliConstant);
		double [][] distMatrix=new double[MAX_NODE_NUM][MAX_NODE_NUM];
		
		
		Vector vecGraph=UpdateStructGraphAdj(vecRotStructure,vecGraphOld,5.0,distMatrix);
		//update the mapping set
    	UpdateMappingSet(vecResonGraph,vecGraph,Map);
    	UpdateIsRestrained(vecResonGraph,vecGraph, Map);
		while(true)
		{
			boolean isAllAsg=true;
			double maxScore=-99999.9;
			int maxID=-1;
			int maxMapID=-1;
			for(int j=0;j<vecGraph.size();j++)
			{    						
				Node node2=(Node)vecGraph.elementAt(j);
				String res=node2.getResName();
				String atom=node2.getAtomName();
				String heavName=pk.GetHeavyAtomFromProton(res,atom);
				int resNo=node2.getResNo();				
				Vector vecMapSet=node2.getMappingSet();
				Vector vecAdj=node2.getAdjVec();
				if(node2.getIsAssigned())
					continue;
				
				for(int k=0;k<vecMapSet.size();k++)
				{
					Node nodeMap=(Node)vecMapSet.elementAt(k);					
					if(nodeMap.asgedId>=0 )//////////
					{
						Node nodeTemp=(Node)vecGraph.elementAt(nodeMap.asgedId);
						if(nodeTemp.stereoID!=node2.getID())
							continue;
						else if(nodeMap.vecStereoSymSet.size()>0)
							continue;							////
					}        ///////////////////////
					Vector vecOneSCAsg=new Vector();
					Vector vecBackNoe=new Vector();
					Vector vecNoesyTwo=new Vector();	
					Vector vecNoesy=nodeMap.vecAdjNOESY;
					vecNoesyTwo.addAll(vecNoesy);
					double cs_proton=nodeMap.getProtonCS();
					double cs_heavy=nodeMap.getHeavyCS();
					
					if(node2.stereoID>=0)
					{
						Node nodeStereo=(Node)vecGraph.elementAt(node2.stereoID);
					
						if(nodeStereo.asgedId>=0)
						{
							Node ndStereoReson=(Node)vecResonGraph.elementAt(nodeStereo.asgedId);
							if(Math.abs(ndStereoReson.getHeavyCS()-cs_heavy) >0.15)//0.15
								continue;
						}
					}//if(node2.stereoID>=0)
					vecOneSCAsg.add(new H1CS(resNo,res,atom,cs_proton));			
					vecOneSCAsg.add(new H1CS(resNo,res,heavName,cs_heavy));		
					
					for(int t=0;t<vecAdj.size();t++)
					{
						Node nodeAdj=(Node)vecAdj.elementAt(t);
						if(!nodeAdj.getIsAssigned())
							continue;
						Node nodeAdjReson=(Node)vecResonGraph.elementAt(nodeAdj.asgedId);
						vecNoesyTwo.addAll(nodeAdjReson.vecAdjNOESY);
						
						int resNoSec=nodeAdj.getResNo();
						String resSec=nodeAdj.getResName();
						String atomSec=nodeAdj.getAtomName();
						String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
						double cs_protonSec=nodeAdjReson.getProtonCS();
						double cs_heavySec=nodeAdjReson.getHeavyCS();
						vecOneSCAsg.add(new H1CS(resNoSec,resSec,atomSec,cs_protonSec));
						vecOneSCAsg.add(new H1CS(resNoSec,resSec,heavyNameSec,cs_heavySec));
						double distUp=6.0;
						if(isInCloseAdj(nodeAdj,node2))
							distUp=2.7;
						distUp=distMatrix[node2.getID()][nodeAdj.getID()];
						vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,distUp,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
						vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,distUp,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
					
					}//for(int t=0;t<vecAdj.size();t++)	
					
					Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
					
					int [] numPeaks=new int[1];
					double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,false);
					dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
					Vector vecOneSCAsgNew=h1cs.DeleteRepeatAsg(vecOneSCAsg);
					double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsgNew,vecBMRBNew);
					double score=dbScore+asgScore;//*0.5*vecBackNoeNew.size();//current score.		
					if(score>maxScore)
					{
						maxScore=score;
						maxID=j;
						maxMapID=k;
					}
				}//for(int k=0;k<vecMapSet.size();k++)
				isAllAsg=false;
				
			}//for(int j=0;j<vecGraph.size();j++)			
			if(isAllAsg || maxID<=0)
				break;		
			
			if(maxID==10)
				System.out.println("maxID="+maxID);
			//update the most confident node:
			System.out.println("maxID="+maxID);
			
			Node node2=(Node)vecGraph.elementAt(maxID);
			Vector vecMapSet=node2.getMappingSet();
			Node nodeMap=(Node)vecMapSet.elementAt(maxMapID);
			node2.asgedId=nodeMap.getID();
			nodeMap.asgedId=node2.getID();
			node2.setIsAssigned(true);
			nodeMap.setIsAssigned(true);
			System.out.println(node2.getResNo()+node2.getResName()+" - "+node2.getAtomName()+": " + nodeMap.getProtonCS()+" , "+ nodeMap.getHeavyCS());
			
			//try to assign the stereo proton node:
			int stereoProtonID=node2.stereoID;
			if(node2.stereoID>0)
			{
				Node ndStereoProton=(Node)vecGraph.elementAt(node2.stereoID);
				Vector vecStereoReson=nodeMap.vecStereoSymSet;
				if(vecStereoReson.size()>0)
				{
					if(vecStereoReson.size()==1)
					{
						Node ndStereoReson=(Node)vecStereoReson.elementAt(0);
						ndStereoProton.asgedId=ndStereoReson.getID();
						ndStereoReson.asgedId=ndStereoProton.getID();
						ndStereoProton.setIsAssigned(true);
						ndStereoReson.setIsAssigned(true);
					}
					else
					{
						Vector vecStereoProtonMap=ndStereoProton.getMappingSet();
						vecStereoProtonMap=new Vector();
						vecStereoProtonMap.addAll(vecMapSet);
					}
				}
			}//if(node2.stereoID>0)
			
		}//while(true)		
			
		Vector vecAsg=new Vector();
		for(int j=0;j<vecGraph.size();j++)
		{    						
			Node node2=(Node)vecGraph.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			String heavName=pk.GetHeavyAtomFromProton(resName,atomName);
			int resNo=node2.getResNo();
			
			int resonID=node2.asgedId;
			if(resonID<0)
				continue;
			Node nodeResonAsg=(Node)vecResonGraph.elementAt(resonID);
			double cs_proton=nodeResonAsg.getProtonCS();
			double cs_heavy=nodeResonAsg.getHeavyCS();
			vecAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
			vecAsg.add(new H1CS(resNo,resName,heavName,cs_heavy));
		}//for(int j=0;j<vecGraph.size();j++)
		
		return vecAsg;
	}

	/**at each step, choose the most confident assignment for one proton node.
     * 
     * note: unassigned case is possoble for a node; stereo protons can be assigned to 
     * the same resonance node.
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public Vector GreedyGraphMatching(Vector vecResonGraph,Vector vecGraph, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew)
	{			
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		Noesy noesy=new Noesy();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;
				
		while(true)
		{
			boolean isAllAsg=true;
			double maxScore=-99999.9;
			int maxID=-1;
			int maxMapID=-1;
			for(int j=0;j<vecGraph.size();j++)
			{    			
				
				Node node2=(Node)vecGraph.elementAt(j);
				String res=node2.getResName();
				String atom=node2.getAtomName();
				String heavName=pk.GetHeavyAtomFromProton(res,atom);
				int resNo=node2.getResNo();				
				Vector vecMapSet=node2.getMappingSet();
				Vector vecAdj=node2.getAdjVec();
				if(node2.getIsAssigned())
					continue;
				
				for(int k=0;k<vecMapSet.size();k++)
				{
					Node nodeMap=(Node)vecMapSet.elementAt(k);					
					if(nodeMap.asgedId>=0 )//////////
					{
						Node nodeTemp=(Node)vecGraph.elementAt(nodeMap.asgedId);
						if(nodeTemp.stereoID!=node2.getID())
							continue;
						else if(nodeMap.vecStereoSymSet.size()>0)
							continue;							////
					}        ///////////////////////
					Vector vecOneSCAsg=new Vector();
					Vector vecBackNoe=new Vector();
					Vector vecNoesyTwo=new Vector();	
					Vector vecNoesy=nodeMap.vecAdjNOESY;
					vecNoesyTwo.addAll(vecNoesy);
					double cs_proton=nodeMap.getProtonCS();
					double cs_heavy=nodeMap.getHeavyCS();
					
					if(node2.stereoID>=0)
					{
						Node nodeStereo=(Node)vecGraph.elementAt(node2.stereoID);
					
						if(nodeStereo.asgedId>=0)
						{
							Node ndStereoReson=(Node)vecResonGraph.elementAt(nodeStereo.asgedId);
							if(Math.abs(ndStereoReson.getHeavyCS()-cs_heavy) >0.15)//0.15
								continue;
						}
					}//if(node2.stereoID>=0)
					vecOneSCAsg.add(new H1CS(resNo,res,atom,cs_proton));			
					vecOneSCAsg.add(new H1CS(resNo,res,heavName,cs_heavy));		
					
					for(int t=0;t<vecAdj.size();t++)
					{
						Node nodeAdj=(Node)vecAdj.elementAt(t);
						if(!nodeAdj.getIsAssigned())
							continue;
						Node nodeAdjReson=(Node)vecResonGraph.elementAt(nodeAdj.asgedId);
						vecNoesyTwo.addAll(nodeAdjReson.vecAdjNOESY);
						
						int resNoSec=nodeAdj.getResNo();
						String resSec=nodeAdj.getResName();
						String atomSec=nodeAdj.getAtomName();
						String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
						double cs_protonSec=nodeAdjReson.getProtonCS();
						double cs_heavySec=nodeAdjReson.getHeavyCS();
						vecOneSCAsg.add(new H1CS(resNoSec,resSec,atomSec,cs_protonSec));
						vecOneSCAsg.add(new H1CS(resNoSec,resSec,heavyNameSec,cs_heavySec));
						double distUp=6.0;
						if(isInCloseAdj(nodeAdj,node2))
							distUp=2.7;
						vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,distUp,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
						vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,distUp,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
					
					}//for(int t=0;t<vecAdj.size();t++)	
					
					Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
					//Vector vecNoesyTwoNew=noesy.DeleteRepeatedNoesy3D(vecNoesyTwo);
					int [] numPeaks=new int[1];
					double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
					dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
					Vector vecOneSCAsgNew=h1cs.DeleteRepeatAsg(vecOneSCAsg);
					double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsgNew,vecBMRBNew);
					double score=dbScore+asgScore;//*0.5*vecBackNoeNew.size();//current score.		
					if(score>maxScore)
					{
						maxScore=score;
						maxID=j;
						maxMapID=k;
					}
				}//for(int k=0;k<vecMapSet.size();k++)
				isAllAsg=false;
				
			}//for(int j=0;j<vecGraph.size();j++)			
			if(isAllAsg || maxID<=0)
				break;		
			
			if(maxID==10)
				System.out.println("maxID="+maxID);
			//update the most confident node:
			System.out.println("maxID="+maxID);
			
			Node node2=(Node)vecGraph.elementAt(maxID);
			Vector vecMapSet=node2.getMappingSet();
			Node nodeMap=(Node)vecMapSet.elementAt(maxMapID);
			node2.asgedId=nodeMap.getID();
			nodeMap.asgedId=node2.getID();
			node2.setIsAssigned(true);
			nodeMap.setIsAssigned(true);
			System.out.println(node2.getResNo()+node2.getResName()+" - "+node2.getAtomName()+": " + nodeMap.getProtonCS()+" , "+ nodeMap.getHeavyCS());
			
			//try to assign the stereo proton node:
			int stereoProtonID=node2.stereoID;
			if(node2.stereoID>0)
			{
				Node ndStereoProton=(Node)vecGraph.elementAt(node2.stereoID);
				Vector vecStereoReson=nodeMap.vecStereoSymSet;
				if(vecStereoReson.size()>0)
				{
					if(vecStereoReson.size()==1)
					{
						Node ndStereoReson=(Node)vecStereoReson.elementAt(0);
						ndStereoProton.asgedId=ndStereoReson.getID();
						ndStereoReson.asgedId=ndStereoProton.getID();
						ndStereoProton.setIsAssigned(true);
						ndStereoReson.setIsAssigned(true);
					}
					else
					{
						Vector vecStereoProtonMap=ndStereoProton.getMappingSet();
						vecStereoProtonMap=new Vector();
						vecStereoProtonMap.addAll(vecMapSet);
					}
				}
			}//if(node2.stereoID>0)
			
		}//while(true)		
			
		Vector vecAsg=new Vector();
		for(int j=0;j<vecGraph.size();j++)
		{    						
			Node node2=(Node)vecGraph.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			String heavName=pk.GetHeavyAtomFromProton(resName,atomName);
			int resNo=node2.getResNo();
			
			int resonID=node2.asgedId;
			if(resonID<0)
				continue;
			Node nodeResonAsg=(Node)vecResonGraph.elementAt(resonID);
			double cs_proton=nodeResonAsg.getProtonCS();
			double cs_heavy=nodeResonAsg.getHeavyCS();
			vecAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
			vecAsg.add(new H1CS(resNo,resName,heavName,cs_heavy));
		}//for(int j=0;j<vecGraph.size();j++)
		
		return vecAsg;
	}

	/**compute the estimated cost in A* search.
	 @param vecResonGraph resonance graph. 
     @param vecStructGraph structure graph.
     @param vecBMRBNew BMRB information.  
     @param curIndex current index.
	 * @return
	 */
	public double ComputeAStarEstimatedCost(Vector vecResonGraph,Vector vecGraph,Vector vecBMRBNew, int curIndex)
	{
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;
		double est_cost=0.0;
		for(int j=0;j<vecGraph.size();j++)
		{    						
			Node node2=(Node)vecGraph.elementAt(j);
			String res=node2.getResName();
			String atom=node2.getAtomName();
			String heavName=pk.GetHeavyAtomFromProton(res,atom);
			int resNo=node2.getResNo();				
			Vector vecMapSet=node2.getMappingSet();
			Vector vecAdj=node2.getAdjVec();
			if(node2.getIsAssigned() || curIndex==j)
				continue;
			double maxScore=-99999.9;
			int maxMapID=-1;
			for(int k=0;k<vecMapSet.size();k++)
			{
				Node nodeMap=(Node)vecMapSet.elementAt(k);					
				if(nodeMap.asgedId>=0 )//////////
				{
					Node nodeTemp=(Node)vecGraph.elementAt(nodeMap.asgedId);
					if(nodeTemp.stereoID!=node2.getID())
						continue;
					else if(nodeMap.vecStereoSymSet.size()>0)
						continue;							////
				}        ///////////////////////
				Vector vecOneSCAsg=new Vector();
				Vector vecBackNoe=new Vector();
				Vector vecNoesyTwo=new Vector();	
				Vector vecNoesy=nodeMap.vecAdjNOESY;
				vecNoesyTwo.addAll(vecNoesy);
				double cs_proton=nodeMap.getProtonCS();
				double cs_heavy=nodeMap.getHeavyCS();
				
				if(node2.stereoID>=0)
				{
					Node nodeStereo=(Node)vecGraph.elementAt(node2.stereoID);
				
					if(nodeStereo.asgedId>=0)
					{
						Node ndStereoReson=(Node)vecResonGraph.elementAt(nodeStereo.asgedId);
						if(Math.abs(ndStereoReson.getHeavyCS()-cs_heavy) >0.15)//0.15
							continue;
					}
				}//if(node2.stereoID>=0)
				vecOneSCAsg.add(new H1CS(resNo,res,atom,cs_proton));			
				vecOneSCAsg.add(new H1CS(resNo,res,heavName,cs_heavy));		
				
				//Here we don't use the pairwise energy becoz it is quite slow to estimate it.
				/*
				for(int t=0;t<vecAdj.size();t++)
				{
					Node nodeAdj=(Node)vecAdj.elementAt(t);
					if(!nodeAdj.getIsAssigned())
						continue;
					Node nodeAdjReson=(Node)vecResonGraph.elementAt(nodeAdj.asgedId);
					vecNoesyTwo.addAll(nodeAdjReson.vecAdjNOESY);
					
					int resNoSec=nodeAdj.getResNo();
					String resSec=nodeAdj.getResName();
					String atomSec=nodeAdj.getAtomName();
					String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
					double cs_protonSec=nodeAdjReson.getProtonCS();
					double cs_heavySec=nodeAdjReson.getHeavyCS();
					vecOneSCAsg.add(new H1CS(resNoSec,resSec,atomSec,cs_protonSec));
					vecOneSCAsg.add(new H1CS(resNoSec,resSec,heavyNameSec,cs_heavySec));
					double distUp=6.0;
					if(isInCloseAdj(nodeAdj,node2))
						distUp=2.7;
					vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,distUp,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
					vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,distUp,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
				
				}//for(int t=0;t<vecAdj.size();t++)	
				*/
				//Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
				
				int [] numPeaks=new int[1];
				//double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
				//dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
				Vector vecOneSCAsgNew=h1cs.DeleteRepeatAsg(vecOneSCAsg);
				double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsgNew,vecBMRBNew);				
				
				double score=asgScore;//dbScore+asgScore;//*0.5*vecBackNoeNew.size();//current score.					
				if(score>maxScore)
				{
					maxScore=score;					
					maxMapID=k;
				}
			}//for(int k=0;k<vecMapSet.size();k++)
			if(maxMapID>=0)
				est_cost=est_cost+maxScore;
			
		}//for(int j=0;j<vecGraph.size();j++)			
		return est_cost;
	}
	
	
	

	/**Apply the A* search algorithm to find the optimal side-chain resonances.
     * 
     * note: unassigned case is possoble for a node; stereo protons can be assigned to 
     * the same resonance node.
     @param vecResonGraph resonance graph 
     @param vecHPositions structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public Vector AStarSearchForSCAsgs(Vector vecResonGraph,Vector vecHPositions, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew)
	{			
		Peak pk=new Peak();
		BackNoe bkNoe=new BackNoe();
		Assign asg=new Assign();
		H1CS h1cs=new H1CS();
		Noesy noesy=new Noesy();
		double csErrH=0.04; double csErrN=0.3;double csErrCA=0.3;
				
		while(true)
		{
			boolean isAllAsg=true;
			double maxScore=-99999.9;
			int maxID=-1;
			int maxMapID=-1;
			for(int j=0;j<vecHPositions.size();j++)
			{    						
				Node node2=(Node)vecHPositions.elementAt(j);
				String res=node2.getResName();
				String atom=node2.getAtomName();
				String heavName=pk.GetHeavyAtomFromProton(res,atom);
				int resNo=node2.getResNo();				
				Vector vecMapSet=node2.getMappingSet();
				Vector vecAdj=node2.getAdjVec();
				if(node2.getIsAssigned())
					continue;
				
				for(int k=0;k<vecMapSet.size();k++)
				{
					Node nodeMap=(Node)vecMapSet.elementAt(k);					
					if(nodeMap.asgedId>=0 )//////////
					{
						Node nodeTemp=(Node)vecHPositions.elementAt(nodeMap.asgedId);
						if(nodeTemp.stereoID!=node2.getID())
							continue;
						else if(nodeMap.vecStereoSymSet.size()>0)
							continue;							////
					}        ///////////////////////
					Vector vecOneSCAsg=new Vector();
					Vector vecBackNoe=new Vector();
					Vector vecNoesyTwo=new Vector();	
					Vector vecNoesy=nodeMap.vecAdjNOESY;
					vecNoesyTwo.addAll(vecNoesy);
					double cs_proton=nodeMap.getProtonCS();
					double cs_heavy=nodeMap.getHeavyCS();
					
					if(node2.stereoID>=0)
					{
						Node nodeStereo=(Node)vecHPositions.elementAt(node2.stereoID);
					
						if(nodeStereo.asgedId>=0)
						{
							Node ndStereoReson=(Node)vecResonGraph.elementAt(nodeStereo.asgedId);
							if(Math.abs(ndStereoReson.getHeavyCS()-cs_heavy) >0.15)//0.15
								continue;
						}
					}//if(node2.stereoID>=0)
					vecOneSCAsg.add(new H1CS(resNo,res,atom,cs_proton));			
					vecOneSCAsg.add(new H1CS(resNo,res,heavName,cs_heavy));		
					
					for(int t=0;t<vecAdj.size();t++)
					{
						Node nodeAdj=(Node)vecAdj.elementAt(t);
						if(!(nodeAdj.getIsAssigned() && (nodeAdj.asgedId>=0)  ))
							continue;
						
						Node nodeAdjReson=(Node)vecResonGraph.elementAt(nodeAdj.asgedId);
						vecNoesyTwo.addAll(nodeAdjReson.vecAdjNOESY);
						
						int resNoSec=nodeAdj.getResNo();
						String resSec=nodeAdj.getResName();
						String atomSec=nodeAdj.getAtomName();
						String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
						double cs_protonSec=nodeAdjReson.getProtonCS();
						double cs_heavySec=nodeAdjReson.getHeavyCS();
						vecOneSCAsg.add(new H1CS(resNoSec,resSec,atomSec,cs_protonSec));
						vecOneSCAsg.add(new H1CS(resNoSec,resSec,heavyNameSec,cs_heavySec));
						double distUp=6.0;
						if(isInCloseAdj(nodeAdj,node2))
							distUp=2.7;
						vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,distUp,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
						vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,distUp,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
					
					}//for(int t=0;t<vecAdj.size();t++)	
					
					Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
					//Vector vecNoesyTwoNew=noesy.DeleteRepeatedNoesy3D(vecNoesyTwo);
					int [] numPeaks=new int[1];
					double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesyTwo, numPeaks,true);
					dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
					Vector vecOneSCAsgNew=h1cs.DeleteRepeatAsg(vecOneSCAsg);
					double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsgNew,vecBMRBNew);
					
					//compute the actual cost, i.e. from first to current step:
					double gScore=dbScore+asgScore;//*0.5*vecBackNoeNew.size();//current score.		
					
					//////////////////////////////
					//compute the esitmated cost, i.e. from current step to the end:
					double hScore=ComputeAStarEstimatedCost(vecResonGraph,vecHPositions,vecBMRBNew,j);
					
					double fScore=gScore+hScore;
					//////////////////////////////
					
					if(fScore>maxScore)
					{
						maxScore=fScore;
						maxID=j;
						maxMapID=k;
					}
				}//for(int k=0;k<vecMapSet.size();k++)
				isAllAsg=false;
				
			}//for(int j=0;j<vecGraph.size();j++)			
			if(isAllAsg || maxID<=0)
				break;		
			
			//if(maxID==10)
				//System.out.println("maxID="+maxID);
			//update the most confident node:
			System.out.println("maxID="+maxID);
			
			Node node2=(Node)vecHPositions.elementAt(maxID);
			Vector vecMapSet=node2.getMappingSet();
			Node nodeMap=(Node)vecMapSet.elementAt(maxMapID);
			node2.asgedId=nodeMap.getID();
			nodeMap.asgedId=node2.getID();
			node2.setIsAssigned(true);
			nodeMap.setIsAssigned(true);
			System.out.println(node2.getResNo()+node2.getResName()+" - "+node2.getAtomName()+": " + nodeMap.getProtonCS()+" , "+ nodeMap.getHeavyCS());
			
			//try to assign the stereo proton node:
			int stereoProtonID=node2.stereoID;
			if(node2.stereoID>0)
			{
				Node ndStereoProton=(Node)vecHPositions.elementAt(node2.stereoID);
				Vector vecStereoReson=nodeMap.vecStereoSymSet;
				if(vecStereoReson.size()>0)
				{
					if(vecStereoReson.size()==1)
					{
						Node ndStereoReson=(Node)vecStereoReson.elementAt(0);
						ndStereoProton.asgedId=ndStereoReson.getID();
						ndStereoReson.asgedId=ndStereoProton.getID();
						ndStereoProton.setIsAssigned(true);
						ndStereoReson.setIsAssigned(true);
					}
					else
					{
						Vector vecStereoProtonMap=ndStereoProton.getMappingSet();
						vecStereoProtonMap=new Vector();
						vecStereoProtonMap.addAll(vecMapSet);
					}
				}
			}//if(node2.stereoID>0)
			
		}//while(true)		
			
		Vector vecAsg=new Vector();
		for(int j=0;j<vecHPositions.size();j++)
		{    						
			Node node2=(Node)vecHPositions.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			String heavName=pk.GetHeavyAtomFromProton(resName,atomName);
			int resNo=node2.getResNo();
			
			int resonID=node2.asgedId;
			if(resonID<0)
				continue;
			Node nodeResonAsg=(Node)vecResonGraph.elementAt(resonID);
			double cs_proton=nodeResonAsg.getProtonCS();
			double cs_heavy=nodeResonAsg.getHeavyCS();
			vecAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
			vecAsg.add(new H1CS(resNo,resName,heavName,cs_heavy));
		}//for(int j=0;j<vecGraph.size();j++)
		
		return vecAsg;
	}


	/**
     * refine mapping for HG protons
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param MapScore mapping score matrix
     @param vecNewNoesy noesy cross peak list
     @param vecBMRBNew BMRB information
     @param
     @return 
     */
	public void RefineMappingForHG(Vector vecResonGraph,Vector vecGraph, int [][] Map,double [][] MapScore, 
			Vector vecNewNoesy, Vector vecBMRBNew)
	{	
		Nasca isoGraph=new Nasca();
		Peak pk=new Peak();
		H1CS h1cs=new H1CS();
		int i,j;
		int restrainThreshold=15;
		
		UpdateStereoSymSet(vecResonGraph,vecGraph,Map);
		for(j=0;j<vecGraph.size();j++)
		{    						
			Node node2=(Node)vecGraph.elementAt(j);
			String resName=node2.getResName();
			String atomName=node2.getAtomName();
			int resNo=node2.getResNo();
			boolean isRestrained=false;
			if(node2.getIsAssigned())
				continue;
			
			if(resNo==3 && atomName.equalsIgnoreCase("HB2"))
				System.out.println("debugging...");
			
			////////////////////
			String subAtom=atomName;
			if(atomName.length()>=2)
				subAtom=atomName.substring(0,2);
			if(!(subAtom.equalsIgnoreCase("HG") || subAtom.equalsIgnoreCase("HD") ||subAtom.equalsIgnoreCase("HE")|| subAtom.equalsIgnoreCase("HH")||subAtom.equalsIgnoreCase("HZ") ) )
				continue;
			
			/////////////////////		
			
			//for storing the mapping likelihood for all nodes
    		Vector vecNeighborSC=new Vector();    
	    	for(i=0;i<vecResonGraph.size();i++)
	    	{   	    		
	    		if(Map[i][j]==0)
	    			continue;
	    		Node node=(Node)vecResonGraph.elementAt(i);
	    		double cs_proton=node.getProtonCS();
	    		double cs_heavy=node.getHeavyCS();
	    		Vector vecAdj=node.getAdjVec();
	    		boolean isAssigned=node.getIsAssigned();
	    		if(isAssigned)
	    			continue;	  
	    		
	    		////////////////////////////
	    		//
	    		if(Math.abs(cs_proton- 7.309    )<0.04 && Math.abs(cs_heavy-129.458   )<0.4  && node2.getResNo()==32 && atomName.equalsIgnoreCase("HE1") )
    				System.out.println("stop here...");
	    		
	    		if(j==2 &&(i==305 ))
	    			System.out.println("stop here...");
	    		
    			String heavyName=pk.GetHeavyAtomFromProton(resName,atomName);
    			if( !(heavyName.substring(0,1).equalsIgnoreCase("N") || heavyName.substring(0,1).equalsIgnoreCase("C") ))
    				continue;
    			
    		  	/////////////////////////////////
    			Vector vecAdj2=node2.getAdjVec();
    			
    			if(vecAdj.size()<1 || vecAdj2.size()<1)
    			{
    				System.out.println("debugging...we found an empty neiborhood...");
    				continue;
    			} 
    		
    			double[] neighborSC=new double[1];    	
    			boolean isRestrainTemp=false;
    			isRestrainTemp=isoGraph.isMappedBetwTwoSetsCalScore(node,node2,vecAdj,vecAdj2,Map,false,restrainThreshold,neighborSC,vecNewNoesy);
    			Vector vecOneSCAsg=new Vector();
    			vecOneSCAsg.add(new H1CS(resNo,resName,atomName,cs_proton));
    			vecOneSCAsg.add(new H1CS(resNo,resName,heavyName,cs_heavy));
    			double asgScore=h1cs.BMRBSatisticsScore(vecOneSCAsg,vecBMRBNew);
        		  			
    			neighborSC[0]=neighborSC[0]+asgScore;
    			MapScore[i][j]=neighborSC[0];
    						
    			if(isRestrainTemp)
		   			isRestrained=true;
		   		
    			RotaPattern rp=new RotaPattern(i,neighborSC[0]);
    			vecNeighborSC.add(rp);    			    	
    		}//for(i=0;i<vecResonGraph.size();i++)  		

	  
	    	node2.setIsRestrained(true);
    		////////////////////////////
    		//we select the top k mapped nodes
    		Collections.sort(vecNeighborSC, new RotaPattern.scoreComparator());
    	
    	
    		for(i=0;i<vecResonGraph.size();i++)
    		{    			
    			boolean isIn=false;
    			
	    		for (int h=0;h<Math.min(20,vecNeighborSC.size());h++)//6: good for HB#
	    		{
	    			RotaPattern rotPatn=(RotaPattern)vecNeighborSC.elementAt(h);
	    			int graph_ID=rotPatn.getResNo();
	    			
	    			Node nodeTemp=(Node)vecResonGraph.elementAt(graph_ID);
	    			int stereoIDTemp=nodeTemp.stereoID;
	    			if(i==graph_ID || i==stereoIDTemp)
	    				isIn=true;	    			
	    			
	    		}//for (int h=0;h<Math.min(3,vecNeighborSC.size());h++)
	    		
	    		if(!isIn)
	    			Map[i][j]=0;
    		}//for(j=0;j<vecGraph.size();j++)
    		
    		
  
    	}//for(j=0;j<vecGraph.size();j++)    	
	}

	/**
     * refine resonance graph based currrent possible mapping set
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param 
     @param 
     @return 
     */
	public void RefineResonGraph(Vector vecResonGraph,Vector vecStructGraph, int [][] Map)
	{
		int counter=0;
		for(int i=0;i<vecResonGraph.size();i++)
		{
			Node nodeReson=(Node)vecResonGraph.elementAt(i);
			Vector vecAdj=nodeReson.getAdjVec();
			Vector vecMap=nodeReson.getMappingSet();
			Vector vecAdjCopy=myVectorCopyOneLayer(vecAdj);
			for(int k=0;k<vecAdj.size();k++)
			{
				Node node_adj=(Node)vecAdj.elementAt(k);
				Vector vecMap_adj=node_adj.getMappingSet();
				boolean isExist=false;
				for(int h=0;h<vecMap.size();h++)
				{
					Node nodeTemp=(Node)vecMap.elementAt(h);
					Vector vecAdjTemp=nodeTemp.getAdjVec();
					for (int t=0;t<vecAdjTemp.size();t++)
					{
						Node nodeTemp2=(Node)vecAdjTemp.elementAt(t);
						int IdTemp2=nodeTemp2.getID();
						for(int w=0;w<vecMap_adj.size();w++)
						{
							Node nodeTemp3=(Node)vecMap_adj.elementAt(w);
							if(nodeTemp3.getID()==IdTemp2)
								isExist=true;
						}//for(int w=0;w<vecMap_adj.size();w++)
					}//for (int t=0;t<vecAdjTemp.size();t++)
				}//for(int h=0;h<vecMap.size();h++)
				if(!isExist)
				{
					counter++;
					vecAdj.remove(k);
					k--;
				}
			}//for(int k=0;k<vecAdj.size();k++)			
			
		}//for(int i=0;i<vecResonGraph.size();i++)
		
		System.out.println("Total number of edges has been removed: "+ counter);
	}
	/**
     * compute the exact one-to-one mapping between two graph
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param 
     @param 
     @return 
     */
	public void ExactSubgraphIsomorphism(Vector vecResonGraph,Vector vecStructGraph, int [][] Map)
	{
		boolean isDebug=true;
		     
		//compute the mxaimum interval:
		double maxInterval=0.0;
		for(int j=0;j<vecStructGraph.size();j++)
		{
			Node nodeStruct=(Node)vecStructGraph.elementAt(j);
			int resNo=nodeStruct.getResNo();
			String res=nodeStruct.getResName();
			String atom=nodeStruct.getAtomName();
			double cs_lower=999.9;
			double cs_upper=-999.9;
			for(int i=0;i<vecResonGraph.size();i++)
			{
				Node nodeReson=(Node)vecResonGraph.elementAt(i);
				double cs_h=nodeReson.getProtonCS();
				if(Map[i][j]==0)
					continue;
				if(cs_h>cs_upper)
					cs_upper=cs_h;
				if(cs_h<cs_lower)
					cs_lower=cs_h;				
			}//for(int i=0;i<vecResonGraph.size();i++)
			
			if(isDebug)
				System.out.println("The interval for "+resNo+ res+"-"+atom +" is: ["+cs_lower+ ", "+cs_upper+"] = "+ Math.abs(cs_upper-cs_lower));
			if(maxInterval< Math.abs(cs_upper -cs_lower) )
				maxInterval= Math.abs(cs_upper -cs_lower);
		}//for(int j=0;j<vecStructGraph.size();j++)
		
		if(isDebug)
			System.out.println("The maximum interval is:"+ maxInterval);
		
	
	
		CompOneToOneMapBtwInterval(vecResonGraph,vecStructGraph, Map,3,4.5);
		System.out.println("we finish one interval here:");
	}
	public Vector myVectorCopy(Vector vecSrc)
	{
		Vector vecFinal=new Vector();
		for(int i=0;i<vecSrc.size();i++)
		{
			Vector vecFianlOne=new Vector();
			Vector vecSrcOne=(Vector)vecSrc.elementAt(i);
			for (int j=0;j<vecSrcOne.size();j++)
			{
				Node node=(Node)vecSrcOne.elementAt(j);
				Node nodeCopy=new Node(node);				
				vecFianlOne.add(nodeCopy);
			}//for (int j=0;j<vecSrcOne.size();j++)
			vecFinal.add(vecFianlOne);
		}//for(int i=0;i<vecSrc.size();i++)
		return vecFinal;
	}
	public Vector myVectorCopyOneLayer(Vector vecSrc)
	{		
		Vector vecFianlOne=new Vector();			
		for (int j=0;j<vecSrc.size();j++)
		{
			Node node=(Node)vecSrc.elementAt(j);
			Node nodeCopy=new Node(node);				
			vecFianlOne.add(nodeCopy);
		}//for (int j=0;j<vecSrcOne.size();j++)	
		
		return vecFianlOne;
	}
	/**
     * enumerate all possible for assignment for sidechain protons in a residue   
     @param vecStructGraph structure graph, in which the mapping sets are given
     @param Map mapping matrix between two graphs  
     @param intvLower, intvUpper: bounds of interval
     @param 
     @return all possible sc assignments
     */
	public Vector EnumAllScAsg(Vector vecStructGraph)
	{
		Vector vecConfig=new Vector();
		for(int i=0;i<vecStructGraph.size();i++)
		{
			Node nodeStruct=(Node)vecStructGraph.elementAt(i);
			
			Vector vecMapSet=nodeStruct.getMappingSet();
			Vector vecConfigCopy=new Vector();	
			//vecConfigCopy=(Vector)vecConfig.clone();
			vecConfigCopy=myVectorCopy(vecConfig);
			
			vecConfig=new Vector();
			for(int j=0;j<vecMapSet.size();j++)
			{
				Node nodeMap=(Node)vecMapSet.elementAt(j);
				double cs_h=nodeMap.getProtonCS();
				double cs_heavy=nodeMap.getHeavyCS();
				int mapId=nodeMap.getID();
				Node ndStructCopy=new Node(nodeStruct);
				ndStructCopy.setProtonCS(cs_h);
				ndStructCopy.setHeavyCS(cs_heavy);
				ndStructCopy.setAssignedID(mapId);
				
				if(vecConfigCopy.size()<1)
				{
					Vector vecOneMap=new Vector();					
					vecOneMap.add(ndStructCopy);
					vecConfig.add(vecOneMap);					
					continue;
				}//if(vecConfig.size()<1)				
			
				//Vector vecPreCopy=new Vector(vecConfigCopy);
				Vector vecPreCopy=new Vector();
				vecPreCopy=myVectorCopy(vecConfigCopy);
				for(int k=0;k<vecPreCopy.size();k++)
				{
					Vector vecPreCopyOne=(Vector)vecPreCopy.elementAt(k);
					vecPreCopyOne.add(ndStructCopy);
					vecConfig.add(vecPreCopyOne);
				}//for(int k=0;k<vecPreCopy.size();k++)				
				
			}//for(int j=0;j<vecMapSet.size();j++)        	
			
		}//for(int i=0;i<vecResonIntvGraph.size();i++)
		
		Peak pk=new Peak();
		Vector vecAllH1CS=new Vector();
		for(int i=0;i<vecConfig.size();i++)
		{
			Vector vecOneH1CS=new Vector();
			Vector vecOneConfig=(Vector)vecConfig.elementAt(i);
			for(int j=0;j<vecOneConfig.size();j++)
			{
				Node node=(Node)vecOneConfig.elementAt(j);
				String resName=node.getResName();
				String atom=node.getAtomName();
				int res_no=node.getResNo();
				double cs_h=node.getProtonCS();
				
				if (atom.equalsIgnoreCase("H"))
					atom="HN";
		    		
		    	//////--------
		    	//for methyl group protons, need to be consistent with resonance list
		    	if(resName.equalsIgnoreCase("ALA") )
		    		if(atom.equalsIgnoreCase("HB1")||atom.equalsIgnoreCase("HB2")||atom.equalsIgnoreCase("HB3"))
		    			atom = "HB";
		    	if(resName.equalsIgnoreCase("ILE")  )
		    		if (atom.equalsIgnoreCase("HG21")||atom.equalsIgnoreCase("HG22")||atom.equalsIgnoreCase("HG23"))
		    			atom = "HG2";
		    	if(resName.equalsIgnoreCase("ILE"))
		    		if (atom.equalsIgnoreCase("HD11")||atom.equalsIgnoreCase("HD12")||atom.equalsIgnoreCase("HD13"))
		    			atom = "HD1";
		    	if(resName.equalsIgnoreCase("LEU") )
		    		if (atom.equalsIgnoreCase("HD11")||atom.equalsIgnoreCase("HD12")||atom.equalsIgnoreCase("HD13"))
		    			atom = "HD1";
		    	if(resName.equalsIgnoreCase("LEU"))
		    		if(atom.equalsIgnoreCase("HD21")||atom.equalsIgnoreCase("HD22")||atom.equalsIgnoreCase("HD23"))
		    			atom = "HD2";
		    	if(resName.equalsIgnoreCase("VAL"))
		    		if(atom.equalsIgnoreCase("HG11")||atom.equalsIgnoreCase("HG12")||atom.equalsIgnoreCase("HG13"))
		    			atom = "HG1";
		    	if(resName.equalsIgnoreCase("VAL") )
		    		if( atom.equalsIgnoreCase("HG21")||atom.equalsIgnoreCase("HG22")||atom.equalsIgnoreCase("HG23"))
		    			atom = "HG2";
		    	if(resName.equalsIgnoreCase("THR") )
		    		if (atom.equalsIgnoreCase("HG21")||atom.equalsIgnoreCase("HG22")||atom.equalsIgnoreCase("HG23"))
		    			atom = "HG2";
		    	if(resName.equalsIgnoreCase("MET") )
		    		if (atom.equalsIgnoreCase("HE1")||atom.equalsIgnoreCase("HE2")||atom.equalsIgnoreCase("HE3"))
		    			atom = "HE";
		    	//need to check for the same protons.....
		    	
				double cs_heavy=node.getHeavyCS();
				String atomHeavy=pk.GetHeavyAtomFromProton(resName,atom);
				vecOneH1CS.add(new H1CS(res_no,resName,atom,cs_h) );
				vecOneH1CS.add(new H1CS(res_no,resName,atomHeavy,cs_heavy) );
			}//for(int j=0;j<vecOneConfig.size();j++)
			vecAllH1CS.add(vecOneH1CS);
			
		}//for(int i=0;i<vecConfig.size();i++)
		
		return vecAllH1CS;
		
	}
		
	/**
     * compute all possible one-to-one mapping for all resonances within an interval
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param intvLower, intvUpper: bounds of interval
     @param 
     @return 
     */
	public void CompOneToOneMapBtwInterval(Vector vecResonGraph,Vector vecStructGraph, int [][] Map,double intvLower, double intvUpper)
	{
		
		//consider all resonance nodes in [-infinite, 3]
		String ss="";
		Vector vecResonIntvGraph=new Vector();
		int maxMaps=-999;
		for(int i=0;i<vecResonGraph.size();i++)
		{
			Node nodeReson=(Node)vecResonGraph.elementAt(i);
			if(nodeReson.getIsAssigned())
				continue;
			double cs_h=nodeReson.getProtonCS();
			if (cs_h<intvLower || cs_h > intvUpper)
				continue;
			Vector vecMapSet=nodeReson.getMappingSet();
			if(maxMaps<vecMapSet.size())
				maxMaps=vecMapSet.size();
			vecResonIntvGraph.add(nodeReson);
		}//for(int i=0;i<vecResonGraph.size();i++)
		
		String userDir = System.getProperty("user.dir");////
    	String srcTemp=userDir+"/system/config_temp.txt";    	
    	try
		{		
			PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(srcTemp)));
			out.close();
    	}catch (FileNotFoundException e)
    	{
    		System.out.println("File not found: " + srcTemp);
    	}catch (IOException e)
    	{
    	   System.out.println("IOException: the stack trace is:");
    	   e.printStackTrace();
    	}
        
		//Vector vecConfig=new Vector();
		for(int i=0;i<vecResonIntvGraph.size();i++)
		{
			Node nodeReson=(Node)vecResonIntvGraph.elementAt(i);
			
			Vector vecMapSet=nodeReson.getMappingSet();
			//Vector vecConfigCopy=new Vector(vecConfig);////
			String srcTempCopy=userDir+"/system/config_temp_copy.txt";  
			copyfile(srcTemp,srcTempCopy);
			
			try
			{
				
    			PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(srcTemp)));
    			    		
				for(int j=0;j<vecMapSet.size();j++)
				{
					Node nodeMap=(Node)vecMapSet.elementAt(j);
					int mapId=nodeMap.getID();
					
					
					if(i==0)
					{
						
						String str=nodeReson.getID()+"-"+nodeMap.getID();
				    	out.println(str);
						continue;
					}//if(vecConfig.size()<1)
					
					String srcTempPreCopy=userDir+"/system/config_temp_pre_copy.txt";
					copyfile(srcTempCopy,srcTempPreCopy);
					
					try
					{
					    BufferedReader in = new BufferedReader(new FileReader(srcTempPreCopy));			
					    while ((ss = in.readLine())!=null)  
					    {
					    	String str=ss+" , "+nodeReson.getID()+"-"+nodeMap.getID();
					    	out.println(str);					    	
					    }// while ((ss = in.readLine())!=null)  					    
					    in.close();    					    
					}catch (FileNotFoundException e) 
					{
						System.out.println("File not found: " + srcTempPreCopy);
			     	}catch (IOException e) 
			     	{
				    System.out.println("IOException: the stack trace is:");
				    e.printStackTrace();
			     	}
					
				}//for(int j=0;j<vecMapSet.size();j++)
				
			out.close();
        	}catch (FileNotFoundException e)
        	{
        		System.out.println("File not found: " + srcTemp);
        	}catch (IOException e)
        	{
        	   System.out.println("IOException: the stack trace is:");
        	   e.printStackTrace();
        	}
         	
			
		}//for(int i=0;i<vecResonIntvGraph.size();i++)
		
	}
	/**
     * compute all possible one-to-one mapping for all resonances within an interval
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param intvLower, intvUpper: bounds of interval
     @param 
     @return 
     */
	
	public Vector EnumAllScAsgSaveIntoFile(Vector vecSCProtons,Vector vecResonGraph,Vector vecStructGraph)
	{	
		String ss="";		
		int maxMaps=-999;		
		Vector vecConfig=new Vector();
		String userDir = System.getProperty("user.dir");////
    	String srcTemp=userDir+"/system/config_temp.txt";    	
    	try
		{		
			PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(srcTemp)));
			out.close();
    	}catch (FileNotFoundException e)
    	{
    		System.out.println("File not found: " + srcTemp);
    	}catch (IOException e)
    	{
    	   System.out.println("IOException: the stack trace is:");
    	   e.printStackTrace();
    	}
        
		//Vector vecConfig=new Vector();
		for(int i=0;i<vecSCProtons.size();i++)
		{
			Node nodeProton=(Node)vecSCProtons.elementAt(i);
			
			Vector vecMapSet=nodeProton.getMappingSet();
			
			String srcTempCopy=userDir+"/system/config_temp_copy.txt";  
			copyfile(srcTemp,srcTempCopy);
			
			try
			{
				
    			PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(srcTemp)));
    			
    			
				for(int j=0;j<vecMapSet.size();j++)
				{
					Node nodeMap=(Node)vecMapSet.elementAt(j);
					int mapId=nodeMap.getID();
					
					if(i==0)
					{
						
						String str=nodeProton.getID()+"-"+nodeMap.getID();
				    	out.println(str);
						continue;
					}//if(vecConfig.size()<1)
					
					String srcTempPreCopy=userDir+"/system/config_temp_pre_copy.txt";
					copyfile(srcTempCopy,srcTempPreCopy);
					
					try
					{
					    BufferedReader in = new BufferedReader(new FileReader(srcTempPreCopy));			
					    while ((ss = in.readLine())!=null)  
					    {
					    	String str=ss+","+nodeProton.getID()+"-"+nodeMap.getID();
					    	out.println(str);					    	
					    }// while ((ss = in.readLine())!=null)  					    
					    in.close();    					    
					}catch (FileNotFoundException e) 
					{
						System.out.println("File not found: " + srcTempPreCopy);
			     	}catch (IOException e) 
			     	{
				    System.out.println("IOException: the stack trace is:");
				    e.printStackTrace();
			     	}
					
				}//for(int j=0;j<vecMapSet.size();j++)
				
				if(vecMapSet.size()<1)
				{
					if(i==0)
					{						
						String str="9999999-9999999";
				    	out.println(str);
						//continue;
					}//if(vecConfig.size()<1)
					else
					{
						String srcTempPreCopy=userDir+"/system/config_temp_pre_copy.txt";
						copyfile(srcTempCopy,srcTempPreCopy);
					
						try
						{
						    BufferedReader in = new BufferedReader(new FileReader(srcTempPreCopy));			
						    while ((ss = in.readLine())!=null)  
						    {
						    	String str=ss+",9999999-9999999";
						    	out.println(str);					    	
						    }// while ((ss = in.readLine())!=null)  					    
						    in.close();    					    
						}catch (FileNotFoundException e) 
						{
							System.out.println("File not found: " + srcTempPreCopy);
				     	}catch (IOException e) 
				     	{
					    System.out.println("IOException: the stack trace is:");
					    e.printStackTrace();
				     	}
					}
				}//if(vecMapSet.size()<1)
				
			out.close();
        	}catch (FileNotFoundException e)
        	{
        		System.out.println("File not found: " + srcTemp);
        	}catch (IOException e)
        	{
        	   System.out.println("IOException: the stack trace is:");
        	   e.printStackTrace();
        	}
         	
			
		}//for(int i=0;i<vecResonIntvGraph.size();i++)
		
		//now read all configurations
		int index=0;
		Assign asg=new Assign();
		Peak pk=new Peak();
		H1CS h1cs=new H1CS();
		try
		{
		    BufferedReader in = new BufferedReader(new FileReader(srcTemp));			
		    while ((ss = in.readLine())!=null)  
		    {
		    	Vector vecConfigOne=new Vector();
		    	String ssTemp=ss;
		    	boolean isBreak=false;
		    	while(true)
		    	{		    		
		    		index = ssTemp.indexOf(",");
		    		String ssTemp2="";
		    		boolean isExisting=false;
		    		if(index==-1)
		    			ssTemp2=ssTemp;
		    		else
		    			ssTemp2=ssTemp.substring(0,index);
		    		Vector vecOder=asg.getOrder(ssTemp2);
		    		int ProtonID  =  Integer.parseInt((String)vecOder.elementAt(0) );
		    		int ResonID  =  Integer.parseInt((String)vecOder.elementAt(1) );
		    		if( !(ProtonID>99999 || ResonID> 99999))
		    		{
			    		Node nodeStruct=(Node)vecStructGraph.elementAt(ProtonID);
			    		Node nodeReson=(Node)vecResonGraph.elementAt(ResonID);
			    		String res=nodeStruct.getResName();
			    		int resNo=nodeStruct.getResNo();
			    		String atom=nodeStruct.getAtomName();
			    		String heavyAtom=pk.GetHeavyAtomFromProton(res,atom);
			    		double cs_h=nodeReson.getProtonCS();
			    		double cs_heavy=nodeReson.getHeavyCS();	    		
					
		    		
			    		double cs_heavyTemp=0.0;
			    		int saveId=0;
			    		for(int w=0;w<vecConfigOne.size();w++)
			    		{
			    			H1CS h1csTemp=(H1CS)vecConfigOne.elementAt(w);	
			    			cs_heavyTemp=h1csTemp.getH1CS();
			    			String heavyTemp=h1csTemp.getAtomName();//pk.GetHeavyAtomFromProton(h1csTemp.getResidueType(),h1csTemp.getAtomName() );
			    			if(heavyAtom.equalsIgnoreCase(heavyTemp))
			    			{
			    				isExisting=true;
			    				saveId=w;
			    				break;			    				
			    			}
			    		}//for(int w=0;w<vecConfigOne.size();w++)
			    		if(!isExisting)
			    			vecConfigOne.add(new H1CS(resNo,res,heavyAtom,cs_heavy,ProtonID,ResonID));
			    		else
			    		{
			    			if(Math.abs(cs_heavyTemp-cs_heavy)>0.2)
			    			{
			    				isBreak=true;
			    				vecConfigOne.remove(saveId);
			    				break;
			    			}
			    		}
			    		vecConfigOne.add(new H1CS(resNo,res,atom,cs_h,ProtonID,ResonID));
		    		}//if( !(ProtonID>99999999 || ResonID> 99999999))
		    		if(index==-1)
		    			break;
		    		else
		    			ssTemp=ssTemp.substring(index+1,ssTemp.length());
		    	}//while(index!=-1)
		    	if(isBreak)
		    		continue;
		    	boolean isSuccess=h1cs.checkCSBMRBOrder(vecConfigOne);
		    	if(isSuccess)
		    		vecConfig.add(vecConfigOne);
		    	else
		    		continue;		    	
		    }// while ((ss = in.readLine())!=null)  					    
		    in.close();    					    
		}catch (FileNotFoundException e) 
		{
			System.out.println("File not found: " + srcTemp);
     	}catch (IOException e) 
     	{
	    System.out.println("IOException: the stack trace is:");
	    e.printStackTrace();
     	}
		
		return vecConfig;
	}
	
	public static void copyfile(String srFile, String dtFile){
	    try{
	      File f1 = new File(srFile);
	      File f2 = new File(dtFile);
	      InputStream in = new FileInputStream(f1);
	      
	
	      OutputStream out = new FileOutputStream(f2);

	      byte[] buf = new byte[1024];
	      int len;
	      while ((len = in.read(buf)) > 0){
	        out.write(buf, 0, len);
	      }
	      in.close();
	      out.close();
	      //System.out.println("File copied.");
	    }
	    catch(FileNotFoundException ex){
	      System.out.println(ex.getMessage() + " in the specified directory.");
	      System.exit(0);
	    }
	    catch(IOException e){
	      System.out.println(e.getMessage());      
	    }
	  }

	/**New:
     * compute the statistical information of the sidechain assignment
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param vecRefAsg reference assignment table
     @param 
     @return 
     */
	public void StatisticsAllSCAssignment(Vector vecResonGraph,Vector vecStructGraph, int [][] Map,Vector vecRefAsg )
	{
		Peak pk=new Peak();
		int totalScH=0;
		int missing=0;
		int averSize=0,sum=0;
		int maxSize=0;
		String maxStr="";
		int[] counter=new int[200];
		Assign asg=new Assign();
		int notInRef=0;
		boolean [] isProtonFound=new boolean[vecStructGraph.size()];
		for(int j=0;j<vecStructGraph.size();j++)
			isProtonFound[j]=false;
		
		UpdateMappingSet(vecResonGraph,vecStructGraph, Map);
		for(int k=0;k<vecRefAsg.size();k++)
		{
			H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
			int resRefNo=h1cs.getResidueNo();
			String resRef=h1cs.getResidueType();
			String atomRef=h1cs.getAtomName();
			double csRef=h1cs.getH1CS();
			double csHeavyRef=-999.9;
			if(atomRef.equalsIgnoreCase("CA") ||atomRef.equalsIgnoreCase("N")|| 
					atomRef.equalsIgnoreCase("HN") ||atomRef.equalsIgnoreCase("H")||
					atomRef.equalsIgnoreCase("HA")||atomRef.equalsIgnoreCase("HA1")|| 
					atomRef.equalsIgnoreCase("HA2")||atomRef.equalsIgnoreCase("HA3")||
					atomRef.equalsIgnoreCase("C")||atomRef.equalsIgnoreCase("CO")||atomRef.equalsIgnoreCase("CB"))
				continue;			
						
			if(!atomRef.substring(0,1).equalsIgnoreCase("H"))
				continue;
			String heavyRef=pk.GetHeavyAtomFromProton(resRef,atomRef);
			totalScH++;
			//get the heavy atom chemcial shift in ref table
			for(int t=0;t<vecRefAsg.size();t++)
			{
				H1CS h1csTemp=(H1CS)vecRefAsg.elementAt(t);
				int resNoTemp=h1csTemp.getResidueNo();
				String atomTemp=h1csTemp.getAtomName();
				if(atomTemp.equalsIgnoreCase(heavyRef) && resNoTemp==resRefNo)
					csHeavyRef=h1csTemp.getH1CS();
			}//for(int t=0;t<vecRefAsg.size();t++)
			boolean isRefInCur=false;
			for(int j=0;j<vecStructGraph.size();j++)
			{
				Node nodeStr=(Node)vecStructGraph.elementAt(j);
				
				String atom=nodeStr.getAtomName();
				String subAtom=atom;
				if(atom.length()>atomRef.length())
					subAtom=atom.substring(0,atomRef.length());				
			
				int resNo=nodeStr.getResNo();							
				if( !(resNo==resRefNo && subAtom.equalsIgnoreCase(atomRef)) )
					continue;
				Vector vecMapSet=nodeStr.getMappingSet();
				for(int w=0;w<vecMapSet.size();w++)
				{
					Node resonNode=(Node)vecMapSet.elementAt(w);
					double csProton=resonNode.getProtonCS();
					double csHeavy=resonNode.getHeavyCS();
					if(Math.abs(csProton-csRef)<0.05 && Math.abs(csHeavy-csHeavyRef)<0.5 )
						isRefInCur=true;
				}//for(int w=0;w<vecMapSet.size();w++)				
			}//for(int j=0;j<vecStructGraph.size();j++)
			
			if(!isRefInCur)
			{
				missing++;
				System.out.println("Missing proton assignment: "+ resRef+" "+resRefNo+" - "+atomRef);
			}			
						
			
		}//for(int k=0;k<vecRefAsg.size();k++)
		
		
		System.out.println("Total number of sidechain protons in ref table: "+ totalScH);
		System.out.println("Total number of missing sidechain assignments: "+ missing);
		
	}
	/**This is an old function.
     * compute the statistical information of the sidechain assignment
     @param vecResonGraph resonance graph 
     @param vecStructGraph structure graph   
     @param Map mapping matrix between two graphs  
     @param vecRefAsg reference assignment table
     @param 
     @return 
     */
	public void StatisticsAllSCAssignmentOld(Vector vecResonGraph,Vector vecStructGraph, int [][] Map,Vector vecRefAsg )
	{
		Peak pk=new Peak();
		int totalScH=0;
		int missing=0;
		int averSize=0,sum=0;
		int maxSize=0;
		String maxStr="";
		int[] counter=new int[200];
		Assign asg=new Assign();
		int notInRef=0;
		for(int j=0;j<vecStructGraph.size();j++)
		{
			Node nodeStr=(Node)vecStructGraph.elementAt(j);
			String res=nodeStr.getResName();
			String atom=nodeStr.getAtomName();
			boolean isAtomInRefAsg=false,isHeavyInRefAsg=false;
					
			String atomHeavy=pk.GetHeavyAtomFromProton(res,atom);
			int resNo=nodeStr.getResNo();
			if(atom.equalsIgnoreCase("CA") ||atom.equalsIgnoreCase("N")|| 
					atom.equalsIgnoreCase("HN") ||atom.equalsIgnoreCase("H")||
					atom.equalsIgnoreCase("HA")||atom.equalsIgnoreCase("HA1")|| atom.equalsIgnoreCase("HA2")||atom.equalsIgnoreCase("HA3")||
					atom.equalsIgnoreCase("C")||atom.equalsIgnoreCase("CO")||atom.equalsIgnoreCase("CB"))
				continue;
			
			if(res.equalsIgnoreCase("CYS") && atom.equalsIgnoreCase("HG"))
				continue;
			boolean isProtonFound=false;
			boolean isHeavyFound=false;
			boolean isFound=false;
			int mappingSize=0;
						
			Vector vecRefAsgStereo=asg.SwapStereoAsg(vecRefAsg,resNo);
			
			//check whether the proton name is in assignment list
			for(int k=0;k<vecRefAsg.size();k++)
			{
				H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
				int resRefNo=h1cs.getResidueNo();
				String atomRef=h1cs.getAtomName();
				String subAtom=atom;
				if(atom.length()>atomRef.length())
					subAtom=atom.substring(0,atomRef.length());
				if(resNo==resRefNo && subAtom.equalsIgnoreCase(atomRef) )
					isAtomInRefAsg=true;								
			}//for(int k=0;k<vecRefAsg.size();k++)
			if(!isAtomInRefAsg)
			{
				notInRef++;
				continue;
			}
			
			//check whether the heavy atom name is in assignment list
			for(int k=0;k<vecRefAsg.size();k++)
			{
				H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
				int resRefNo=h1cs.getResidueNo();
				String atomRef=h1cs.getAtomName();					
				if(resNo==resRefNo && atomHeavy.equalsIgnoreCase(atomRef) )
					isHeavyInRefAsg=true;								
			}//for(int k=0;k<vecRefAsg.size();k++)
			
			///////////////////////////	
			for(int i=0;i<vecResonGraph.size();i++)
			{				
			//	if(i==204 && j==243)
				///	System.out.println("we stop");
				Node nodeReson=(Node)vecResonGraph.elementAt(i);
				double cs_h=nodeReson.getProtonCS();
				double cs_heavy=nodeReson.getHeavyCS();
				if (Map[i][j]==0)
					continue;
				
				//check the first prtoton
				for(int k=0;k<vecRefAsg.size();k++)
				{				
					H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
					int resRefNo=h1cs.getResidueNo();
					String atomRef=h1cs.getAtomName();
					String subAtom=atom;
					if(atom.length()>atomRef.length())
						subAtom=atom.substring(0,atomRef.length());
					if(resNo==resRefNo && subAtom.equalsIgnoreCase(atomRef) )
						isAtomInRefAsg=true;
					
					double csRef=h1cs.getH1CS();
					if(resNo==resRefNo && subAtom.equalsIgnoreCase(atomRef) && Math.abs(csRef-cs_h)<0.05 )
						isProtonFound=true;					
				}//for(int k=0;k<vecRefAsg.size();k++)
				
				//check the first prtoton, in the other stereospecific assignment case:
				for(int k=0;k<vecRefAsgStereo.size();k++)
				{
					H1CS h1cs=(H1CS)vecRefAsgStereo.elementAt(k);
					int resRefNo=h1cs.getResidueNo();
					String atomRef=h1cs.getAtomName();
					String subAtom=atom;
					if(atom.length()>atomRef.length())
						subAtom=atom.substring(0,atomRef.length());
					if(resNo==resRefNo && subAtom.equalsIgnoreCase(atomRef) )
						isAtomInRefAsg=true;
					
					double csRef=h1cs.getH1CS();
					if(resNo==resRefNo && subAtom.equalsIgnoreCase(atomRef) && Math.abs(csRef-cs_h)<0.05 )
						isProtonFound=true;					
				}//for(int k=0;k<vecRefAsgStereo.size();k++)
				
				if(!isHeavyInRefAsg)				
					isHeavyFound=true;
				
				//check the heavy atom
				for(int k=0;k<vecRefAsg.size();k++)
				{
					H1CS h1cs=(H1CS)vecRefAsg.elementAt(k);
					int resRefNo=h1cs.getResidueNo();
					String atomRef=h1cs.getAtomName();						
					double csRef=h1cs.getH1CS();
					if(resNo==resRefNo && atomHeavy.equalsIgnoreCase(atomRef) && Math.abs(csRef-cs_heavy)<0.5 )
						isHeavyFound=true;					
				}//for(int k=0;k<vecRefAsg.size();k++)
				
				mappingSize++;
			}//for(int i=0;i<vecResonGraph.size();i++)
			
			counter[mappingSize]++;
			if(isHeavyFound && isProtonFound )
				isFound=true;
			
			if(!isFound)
			{
				missing++;
				System.out.println("Missing sidechain assignment: "+ resNo +" "+ res+ " - " +atom); 
			}
				
			totalScH++;			
			if(mappingSize>maxSize)
			{
				maxSize=mappingSize;
				maxStr=resNo +" "+ res+ " - " +atom;
			}
			sum=sum+mappingSize;
			
		}//for(int i=0;i<vecResonGraph.size();i++)
		
		System.out.println("Total number of protons not in the reference table: "+ notInRef);
		System.out.println("Total number of sidechain protons: "+ totalScH);
		System.out.println("Total number of missing sidechain assignments: "+ missing);
		System.out.println("Average size of mappings: "+ sum/totalScH);
		System.out.println("Maximum size of mappings: "+maxSize+ "  happened in " +maxStr);
	
		for(int i=0;i<=20;i++)
			System.out.println("Total number of mapping sets with size "+ i+ " is : "+ counter[i]);
	}
	 
	/**
     * add the coordinates of a proton into the graph
     @param resID 
     @param resName  
     @param atomName 
     @param coord coordiante of the proton
     @param vecOldGraph old graph 
     @return new graph
     */
	public void AddProtonCoordinates(int resID,String resName, String atomName, Cartesian coord,Vector vecOldGraph )
	{
		boolean isInPres=false;
		for(int i=0;i<vecOldGraph.size();i++)
		{
			Node node=(Node)vecOldGraph.elementAt(i);
			int preResID=node.getResNo();
			String preAtomName=node.getAtomName();
			if( (resID==preResID) && (preAtomName.equalsIgnoreCase(atomName) ) )
			{
				node.AddCoord(coord);
				isInPres=true;
			}			
		}//for(int i=0;i<vecOldGraph.size();i++)
		if(!isInPres)
		{
			Vector atomVecT=new Vector();
			atomVecT.add(coord);			
			Node node =new Node(vecOldGraph.size(),resID,resName,atomName,atomVecT);
			vecOldGraph.add(node);
		}
		//return vecOldGraph;
	}
	   
	/**
     * put all rotamers at each redidue, and construct the label set (i.e. structure graph),
     * which includes all possible proton positions.
     * 
     * Note: this function was implemented in the early stage, which is slightly from what 
     * described in the paper. In prinple, it can reduce the running time of the original 
     * NASCA algorithm. In addition, the test has shown that it guarantee good results. 
     * Thus, we still keep using this function. 
     * 
     @param pdbBBVec the PDB backbone structure
     @param rotSrc  the path of rotmaer library
     @param noeUp upper limit for noe interaction, default:6A
     @return vecNodes vector of nodes.
     */
	public Vector ConstructProtonPositions(Vector pdbBBVec, String rotSrc, double noeUp)
	{
		double distUpClose=2.8;//distance bound for considering close neighbors
		Pdb pp=new Pdb();
		Vector vecPdbAllRots=pp.AllRotamersStructure(pdbBBVec,rotSrc);
		int curID=1;
		Vector vecAllNodes=new Vector();
		
		//initialization:
		for(int i=0;i<vecPdbAllRots.size();i++ )
		{
			Pdb pdb=(Pdb)vecPdbAllRots.elementAt(i);
			int resNo=pdb.getResidueNo();
			String resName=pdb.getResidue();
			Vector vecAtom=pdb.getAtomVec();
			for(int j=0;j<vecAtom.size();j++)
			{
				boolean isMethyl=false;
				Cartesian cc = (Cartesian)vecAtom.elementAt(j);
				String atomName=cc.getAtom();
				String subAtomName=atomName;
				if(atomName.length()>1)
					subAtomName=atomName.substring(0, (atomName.length()-1) );
				
				if(resName.equalsIgnoreCase("TYR") && atomName.equalsIgnoreCase("HH"))
					continue;
				if(resName.equalsIgnoreCase("CYS") && atomName.equalsIgnoreCase("HG"))
					continue;
				if(resName.equalsIgnoreCase("SER") && atomName.equalsIgnoreCase("HG"))
					continue;
				if(resName.equalsIgnoreCase("LYS") && (atomName.equalsIgnoreCase("HZ1")||atomName.equalsIgnoreCase("HZ2")||atomName.equalsIgnoreCase("HZ3") )   )
					continue;//
				if(resName.equalsIgnoreCase("ASP") && atomName.equalsIgnoreCase("HD2"))
					continue;
				if(resName.equalsIgnoreCase("GLU") && atomName.equalsIgnoreCase("HE2"))
					continue;
				if(resName.equalsIgnoreCase("ARG") && (atomName.equalsIgnoreCase("HE")||subAtomName.equalsIgnoreCase("HH1")||subAtomName.equalsIgnoreCase("HH2")  ))
					continue;
				if(resName.equalsIgnoreCase("THR") && atomName.equalsIgnoreCase("HG1"))
					continue;
				
				//for methyl group protons, need to be consistent with resonance list
		    	if(resName.equalsIgnoreCase("ALA") )
		    		if(atomName.equalsIgnoreCase("HB1")||atomName.equalsIgnoreCase("HB2")||atomName.equalsIgnoreCase("HB3"))
		    			atomName = "HB";
		    	if(resName.equalsIgnoreCase("ILE")  )
		    		if (atomName.equalsIgnoreCase("HG21")||atomName.equalsIgnoreCase("HG22")||atomName.equalsIgnoreCase("HG23"))
		    			atomName = "HG2";
		    	if(resName.equalsIgnoreCase("ILE"))
		    		if (atomName.equalsIgnoreCase("HD11")||atomName.equalsIgnoreCase("HD12")||atomName.equalsIgnoreCase("HD13"))
		    			atomName = "HD1";
		    	if(resName.equalsIgnoreCase("LEU") )
		    		if (atomName.equalsIgnoreCase("HD11")||atomName.equalsIgnoreCase("HD12")||atomName.equalsIgnoreCase("HD13"))
		    			atomName = "HD1";
		    	if(resName.equalsIgnoreCase("LEU"))
		    		if(atomName.equalsIgnoreCase("HD21")||atomName.equalsIgnoreCase("HD22")||atomName.equalsIgnoreCase("HD23"))
		    			atomName = "HD2";
		    	if(resName.equalsIgnoreCase("VAL"))
		    		if(atomName.equalsIgnoreCase("HG11")||atomName.equalsIgnoreCase("HG12")||atomName.equalsIgnoreCase("HG13"))
		    			atomName = "HG1";
		    	if(resName.equalsIgnoreCase("VAL") )
		    		if( atomName.equalsIgnoreCase("HG21")||atomName.equalsIgnoreCase("HG22")||atomName.equalsIgnoreCase("HG23"))
		    			atomName = "HG2";
		    	if(resName.equalsIgnoreCase("THR") )
		    		if (atomName.equalsIgnoreCase("HG21")||atomName.equalsIgnoreCase("HG22")||atomName.equalsIgnoreCase("HG23"))
		    			atomName = "HG2";	    	
		    	if(resName.equalsIgnoreCase("MET") )
		    		if (atomName.equalsIgnoreCase("HE1")||atomName.equalsIgnoreCase("HE2")||atomName.equalsIgnoreCase("HE3"))
		    			atomName = "HE";	    				
				
				if(atomName.substring(0,1).equalsIgnoreCase("H"))
					AddProtonCoordinates(resNo,resName,atomName, cc,vecAllNodes);
			}//for(int j=0;j<vecAtom.size();j++)			
		}//for(int i=0;i<vecPdbAllRots.size();i++ )
		
		//set up connection:
		double [] coordA = {0.0, 0.0, 0.0};
		double [] coordB = {0.0, 0.0, 0.0};
		for(int i=0;i<vecAllNodes.size();i++)
		{
			Node nodeA=(Node)vecAllNodes.elementAt(i);
			Vector vecAtomA=nodeA.getAtomVec();
			for(int i1=0;i1<vecAtomA.size();i1++)
			{
				Cartesian ccA = (Cartesian)vecAtomA.elementAt(i1);
				coordA=ccA.getXYZ();
				for (int j=0;j<vecAllNodes.size();j++)
				{
					Node nodeB=(Node)vecAllNodes.elementAt(j);
					if(i==j)
						continue;
					Vector vecAtomB=nodeB.getAtomVec();
					
					boolean isAllIn=true;
					boolean isAllInClose=true;
					for(int j1=0;j1<vecAtomB.size();j1++)
					{
						Cartesian ccB=(Cartesian)vecAtomB.elementAt(j1);
						coordB=ccB.getXYZ();
						double disTemp = Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
								+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
								+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
					
						if(disTemp>noeUp)
							isAllIn=false;
						if(disTemp>distUpClose)
							isAllInClose=false;
					}//for(int j1=0;j1<vecAtomB.size();j1++)
					
					if(isAllIn)
					{
						nodeA.AddAdj(nodeB);
						nodeB.AddAdj(nodeA);//here we let the graph be undirected
					}
					if(isAllInClose)
					{
						nodeA.AddCloseAdj(nodeB);
						nodeB.AddCloseAdj(nodeA);
					}
					
				}//for (int j=0;j<vecAllNodes.size();j++)
			}//for(i1=0;i1<vecAtom.size();i1++)
			
		}//for(int i=0;i<vecAllNodes.size();i++)
		
		return vecAllNodes;
	}
	/**
     *update the neighborhood of the structure graph, given a pdb including both bb and sc (i.e. one rotamer)
     @param pdbVec the PDB structure,including both bb and sc
     @param rotSrc  the path of rotmaer library
     @param noeUp upper limit for noe interaction, default:6A
     @param distMatrix distance matrix
     @return vecNodes vector of nodes.
     */
	public Vector UpdateStructGraphAdj(Vector pdbVec, Vector vecGraphOld,double noeUp, double [][] distMatrix)
	{
		double distUpClose=2.8;//distance bound for considering close neighbors
		Pdb pp=new Pdb();		
		int curID=1;	

		//set up connection:
		double [] coordA = {0.0, 0.0, 0.0};
		double [] coordB = {0.0, 0.0, 0.0};
		for(int i=0;i<vecGraphOld.size();i++)
		{
			Node node=(Node)vecGraphOld.elementAt(i);
			node.atomVec=new Vector();
		}
		for(int i=0;i<vecGraphOld.size();i++)
		{
			Node node=(Node)vecGraphOld.elementAt(i);
			int resNo=node.getResNo();
			String atom=node.getAtomName();
			
			for(int j=0;j<pdbVec.size();j++)
			{
				Pdb pdb=(Pdb)pdbVec.elementAt(j);
				int resNoPdb=pdb.getResidueNo();
				String resName=pdb.getResidue();
				Vector vecAtom=pdb.getAtomVec();
				for(int k=0;k<vecAtom.size();k++)
				{
					boolean isMethyl=false;
					Cartesian cc = (Cartesian)vecAtom.elementAt(k);
					String atomName=cc.getAtom();
					String subAtomName=atomName;
					if(atomName.length()>1)
						subAtomName=atomName.substring(0, (atomName.length()-1) );
					
					if(resName.equalsIgnoreCase("TYR") && atomName.equalsIgnoreCase("HH"))
						continue;
					if(resName.equalsIgnoreCase("CYS") && atomName.equalsIgnoreCase("HG"))
						continue;
					if(resName.equalsIgnoreCase("SER") && atomName.equalsIgnoreCase("HG"))
						continue;
					if(resName.equalsIgnoreCase("LYS") && (atomName.equalsIgnoreCase("HZ1")||atomName.equalsIgnoreCase("HZ2")||atomName.equalsIgnoreCase("HZ3") )   )
						continue;//
					if(resName.equalsIgnoreCase("ASP") && atomName.equalsIgnoreCase("HD2"))
						continue;
					if(resName.equalsIgnoreCase("GLU") && atomName.equalsIgnoreCase("HE2"))
						continue;
					if(resName.equalsIgnoreCase("ARG") && (atomName.equalsIgnoreCase("HE")||subAtomName.equalsIgnoreCase("HH1")||subAtomName.equalsIgnoreCase("HH2")  ))
						continue;
					if(resName.equalsIgnoreCase("THR") && atomName.equalsIgnoreCase("HG1"))
						continue;
					
					//for methyl group protons, need to be consistent with resonance list
			    	if(resName.equalsIgnoreCase("ALA") )
			    		if(atomName.equalsIgnoreCase("HB1")||atomName.equalsIgnoreCase("HB2")||atomName.equalsIgnoreCase("HB3"))
			    			atomName = "HB";
			    	if(resName.equalsIgnoreCase("ILE")  )
			    		if (atomName.equalsIgnoreCase("HG21")||atomName.equalsIgnoreCase("HG22")||atomName.equalsIgnoreCase("HG23"))
			    			atomName = "HG2";
			    	if(resName.equalsIgnoreCase("ILE"))
			    		if (atomName.equalsIgnoreCase("HD11")||atomName.equalsIgnoreCase("HD12")||atomName.equalsIgnoreCase("HD13"))
			    			atomName = "HD1";
			    	if(resName.equalsIgnoreCase("LEU") )
			    		if (atomName.equalsIgnoreCase("HD11")||atomName.equalsIgnoreCase("HD12")||atomName.equalsIgnoreCase("HD13"))
			    			atomName = "HD1";
			    	if(resName.equalsIgnoreCase("LEU"))
			    		if(atomName.equalsIgnoreCase("HD21")||atomName.equalsIgnoreCase("HD22")||atomName.equalsIgnoreCase("HD23"))
			    			atomName = "HD2";
			    	if(resName.equalsIgnoreCase("VAL"))
			    		if(atomName.equalsIgnoreCase("HG11")||atomName.equalsIgnoreCase("HG12")||atomName.equalsIgnoreCase("HG13"))
			    			atomName = "HG1";
			    	if(resName.equalsIgnoreCase("VAL") )
			    		if( atomName.equalsIgnoreCase("HG21")||atomName.equalsIgnoreCase("HG22")||atomName.equalsIgnoreCase("HG23"))
			    			atomName = "HG2";
			    	if(resName.equalsIgnoreCase("THR") )
			    		if (atomName.equalsIgnoreCase("HG21")||atomName.equalsIgnoreCase("HG22")||atomName.equalsIgnoreCase("HG23"))
			    			atomName = "HG2";	    	
			    	if(resName.equalsIgnoreCase("MET") )
			    		if (atomName.equalsIgnoreCase("HE1")||atomName.equalsIgnoreCase("HE2")||atomName.equalsIgnoreCase("HE3"))
			    			atomName = "HE";	    				
					if(resNo==resNoPdb && atomName.equalsIgnoreCase(atom))
					{						
						node.AddCoord(cc);//note:be careful about methyl protons.
					}
					
				}//for(int j=0;j<vecAtom.size();j++)			
			
			}//
		}//for(int i=0;i<vecGraphOld.size();i++)
		
		for(int i=0;i<vecGraphOld.size();i++)
		{
			Node nodeA=(Node)vecGraphOld.elementAt(i);
			
			Vector vecAtomA=nodeA.getAtomVec();
			for(int i1=0;i1<vecAtomA.size();i1++)
			{
				Cartesian ccA = (Cartesian)vecAtomA.elementAt(i1);
				coordA=ccA.getXYZ();
				for (int j=0;j<vecGraphOld.size();j++)
				{
					Node nodeB=(Node)vecGraphOld.elementAt(j);
					if(i==j)
						continue;
					Vector vecAtomB=nodeB.getAtomVec();
					
					boolean isAllIn=true;
					boolean isAllInClose=true;
					for(int j1=0;j1<vecAtomB.size();j1++)
					{
						Cartesian ccB=(Cartesian)vecAtomB.elementAt(j1);
						coordB=ccB.getXYZ();
						double disTemp = Math.sqrt((coordA[0] - coordB[0]) * (coordA[0] - coordB[0]) 
								+ (coordA[1] - coordB[1]) * (coordA[1] - coordB[1]) 
								+ (coordA[2] - coordB[2]) * (coordA[2] - coordB[2]));
					
						if(disTemp<=noeUp)
						{
							//isAllIn=false;
							nodeA.AddAdj(nodeB);
							nodeB.AddAdj(nodeA);
							distMatrix[nodeA.getID()][nodeB.getID()]=disTemp;
							distMatrix[nodeB.getID()][nodeA.getID()]=disTemp;
						}
						if(disTemp>distUpClose)
							isAllInClose=false;
					}//for(int j1=0;j1<vecAtomB.size();j1++)
					
					
					if(isAllInClose)
					{
						nodeA.AddCloseAdj(nodeB);
						nodeB.AddCloseAdj(nodeA);
					}
					
				}//for (int j=0;j<vecAllNodes.size();j++)
			}//for(i1=0;i1<vecAtom.size();i1++)
			
		}//for(int i=0;i<vecAllNodes.size();i++)
		Vector vecNewGraph=new Vector();
		vecNewGraph.addAll(vecGraphOld);
		
		return vecNewGraph;
	}
	/**
     * check whether one node is in another node's stereo set
     @param vecAsg all resonance assignments
     @param 
     @return 
     */
	public boolean isInStereoSet(Node nodeSrc, Node nodeObj)
	{
		boolean isIn=false;
		for(int i=0;i<nodeSrc.vecStereoSymSet.size();i++)
		{
			Node ndTemp=(Node)nodeSrc.vecStereoSymSet.elementAt(i);
			if(ndTemp.getID()==nodeObj.getID())
				isIn=true;
		}
		return isIn;
	}
	/**
     * check whether one node is in another node's close neighbors
     @param vecAsg all resonance assignments
     @param 
     @return 
     */
	public boolean isInCloseAdj(Node nodeSrc, Node nodeObj)
	{
		boolean isIn=false;
		for(int i=0;i<nodeSrc.vecCloseAdj.size();i++)
		{
			Node ndTemp=(Node)nodeSrc.vecCloseAdj.elementAt(i);
			if(ndTemp.getID()==nodeObj.getID())
				isIn=true;
		}
		return isIn;
	}
	/**
     * exact the backbone resonance assignment
     @param vecAsg all resonance assignments
     @param 
     @return backbone resonance assignment
     */
	public Vector ExtractBackboneAsignment(Vector vecAsg)
	{
		Vector vecBBAsg=new Vector();
		for(int i=0;i<vecAsg.size();i++)
		{
			H1CS h1cs=(H1CS)vecAsg.elementAt(i);
			String atom=h1cs.getAtomName();
			if(atom.equalsIgnoreCase("CA") ||atom.equalsIgnoreCase("N")|| 
					atom.equalsIgnoreCase("HN") ||atom.equalsIgnoreCase("H")||
					atom.equalsIgnoreCase("HA")||atom.equalsIgnoreCase("HA1")|| atom.equalsIgnoreCase("HA2")||atom.equalsIgnoreCase("HA3")||
					atom.equalsIgnoreCase("C")||atom.equalsIgnoreCase("CO")||atom.equalsIgnoreCase("CB"))
				vecBBAsg.add(h1cs);
		}
		return vecBBAsg;
	}

	public Vector ExtractSideChainAsignment(Vector vecAsg)
	{
		Vector vecBBAsg=new Vector();
		for(int i=0;i<vecAsg.size();i++)
		{
			H1CS h1cs=(H1CS)vecAsg.elementAt(i);
			String atom=h1cs.getAtomName();
			if(atom.equalsIgnoreCase("CA") ||atom.equalsIgnoreCase("N")|| 
					atom.equalsIgnoreCase("HN") ||atom.equalsIgnoreCase("H")||
					atom.equalsIgnoreCase("HA")||atom.equalsIgnoreCase("HA1")|| atom.equalsIgnoreCase("HA2")||atom.equalsIgnoreCase("HA3")||
					atom.equalsIgnoreCase("C")||atom.equalsIgnoreCase("CO")||atom.equalsIgnoreCase("CB"))				
				continue;			
		
			vecBBAsg.add(h1cs);
		}
		return vecBBAsg;
	}
	/**
     * add the reosnance node into the graph from given assigned resonance list
     @param vecNodeSet all resonance nodes
     @param vecNoesy Noesy peak list
     @param  
     @param 
     @param  
     @return the resonance graph after setting connection
     */
	public Vector ConstructResonGraph(Vector vecNodeSet, Vector vecNoesy,double errH1, double errH3, double errC, double errN )
	{	
		Vector vecResonGraph=new Vector();
		for(int i=0;i<vecNoesy.size();i++)
		{
			Noesy noesy=(Noesy)vecNoesy.elementAt(i);
			double errHeavy=0.0;
			double freH1=noesy.getH1();
			double freHeavy=noesy.getHeavy();
			double freH2=noesy.getH2();
			
			if(freHeavy>80)
				errHeavy=errN;
			else
				errHeavy=errC;
			
			for(int j=0;j<vecNodeSet.size();j++)
			{
				Node node=(Node)vecNodeSet.elementAt(j);
				double csH=node.getProtonCS();
				double csHeavy=node.getHeavyCS();
				if(Math.abs(csH-freH1)>errH1)
					continue;				
				if(Math.abs(csHeavy-freHeavy)>errHeavy)
					continue;
				
				
				for(int k=0;k<vecNodeSet.size();k++)
				{
					if(k==j)
						continue;
					Node node2=(Node)vecNodeSet.elementAt(k);
					double csH2=node2.getProtonCS();
					if(Math.abs(csH2-freH2)<errH3)
					{
						node.AddAdj(node2);//directed here
						node2.AddAdj(node);//will be considered in another loop
						
						node.AddAdjNOESY(noesy);
						node2.AddAdjNOESY(noesy);
					}
					
				}//for(int k=0;k<vecNodeSet.size();k++)
				
				
			}//for(int j=0;j<vecNodeSet.size();j++)
		}//for(int i=0;i<vecNoesy.size();i++)
		vecResonGraph.addAll(vecNodeSet);
		
		//remove isolated nodes
		/*for(int i=0;i<vecResonGraph.size();i++)
		{
			Node node=(Node)vecResonGraph.elementAt(i);
			Vector vecAdjT=node.getAdjVec();
			if(vecAdjT.size()<1)
			{
				vecResonGraph.remove(i);				
				i--;		
			}
		}//for(int i=0;i<vecResonGraph.size();i++)
		
		//reset the id
		for(int i=0;i<vecResonGraph.size();i++)
		{
			Node node=(Node)vecResonGraph.elementAt(i);
			node.setNodeID(i);		
		}//for(int i=0;i<vecResonGraph.size();i++)
		*/
		return vecResonGraph;		
	}
	/**
     * add the reosnance node into the graph from given assigned resonance list
     @param vecNodeSet all resonance nodes
     @param vecNoesy Noesy peak list
     @param  
     @param 
     @param  
     @return the resonance graph after setting connection
     */
	public void ConstructResonGraphSymmetry(Vector vecResonGraph, Vector vecNoesy,double errH1, double errH3, double errC, double errN )
	{			
		for(int i=0;i<vecNoesy.size();i++)
		{				
			Noesy noesy=(Noesy)vecNoesy.elementAt(i);
			double errHeavy=0.0;
			double freH1=noesy.getH1();
			double freHeavy=noesy.getHeavy();
			double freH2=noesy.getH2();
			
			if(freHeavy>80)
				errHeavy=errN;
			else
				errHeavy=errC;
			
			for(int j=0;j<vecResonGraph.size();j++)
			{
				Node node=(Node)vecResonGraph.elementAt(j);
				double csH=node.getProtonCS();
				double csHeavy=node.getHeavyCS();
				if(Math.abs(csH-freH1)>errH1)
					continue;				
				if(Math.abs(csHeavy-freHeavy)>errHeavy)
					continue;
					
				for(int k=0;k<vecResonGraph.size();k++)
				{
					if(k==j)
						continue;
					Node node2=(Node)vecResonGraph.elementAt(k);
					double csH2=node2.getProtonCS();
					double csHeavy2=node2.getHeavyCS();
														
					if(Math.abs(csH2-freH2)<errH3)
						node.vecSymSet.add(node2);		
					
				}//for(int k=0;k<vecAdj.size();k++)				
			}//for(int j=0;j<vecNodeSet.size();j++)
		}//for(int i=0;i<vecNoesy.size();i++)
		
		//delete non symmetric nodes
		for(int i=0;i<vecResonGraph.size();i++)
		{
			Node node=(Node)vecResonGraph.elementAt(i);
			Vector vec_sym=node.vecSymSet;
			
			for(int k=0;k<vec_sym.size();k++)
			{
				Node ndSym=(Node)vec_sym.elementAt(k);
				Vector vec_sym2=ndSym.vecSymSet;
				boolean isSymmetry=false;
				for(int w=0;w<vec_sym2.size();w++)
				{
					Node nd=(Node)vec_sym2.elementAt(w);
					int symID=nd.getID();
					if(symID==i)
						isSymmetry=true;						
				}//for(int w=0;w<vec_sym2.size();w++)
				if(!isSymmetry)
				{
					vec_sym.remove(k);
					k--;
				}//if(!isSymmetry)
				
			}//for(int k=0;k<vec_sym.size();k++)		
			
		}//for(int i=0;i<vecResonGraph.size();i++)
	
			
	}
	/**
     * add the reosnance node into the graph from given assigned resonance list
     @param csH,csHeavy: chemical shifts of proton and heavy atom attached to it, -999.9 if it is unkown
     @param csErrH,csErrC,csErrN: eror windows.
     @param  vecUnAsged unassigned resonance list
     @param 
     @param vecOldGraph old graph 
     @return new graph
     */
	public Vector ConstructResonGraphFromAsgedList(Vector vecAsged, Vector vecUnAsged )
	{
		Vector vecGraph=new Vector();
		Peak pk=new Peak();
		int curId=0;
		for (int i=0;i<vecAsged.size();i++)
		{
			double cs_heavy=-999.9;
			H1CS h1cs=(H1CS)vecAsged.elementAt(i);
			double cs_h=h1cs.getH1CS();
			String res=h1cs.getResidueType();
			String atom=h1cs.getAtomName();
			int resNo=h1cs.getResidueNo();
			if(!atom.substring(0,1).equalsIgnoreCase("H") )
				continue;
			String heavyName=pk.GetHeavyAtomFromProton(res,atom);
			for(int j=0;j<vecAsged.size();j++)
			{
				H1CS h1csHeavy=(H1CS)vecAsged.elementAt(j);
				double cs_Temp=h1csHeavy.getH1CS();
				String resT=h1csHeavy.getResidueType();
				String atomT=h1csHeavy.getAtomName();
				int resNoT=h1csHeavy.getResidueNo();
				if( (resNoT==resNo) && (atomT.equalsIgnoreCase(heavyName)) )
					cs_heavy=cs_Temp;
			}//for(int j=0;j<vecAsg.size();j++)
			
			Node node=new Node(curId,cs_h,cs_heavy,true);
			node.setAtomName(atom);
			node.setResName(res);
			node.setResNo(resNo);
			vecGraph.add(node);
			curId++;
			
		}//for (int i=0;i<vecAsg.size();i++)
		
		if(vecUnAsged==null)
			return vecGraph;
		///////////////
		for (int i=0;i<vecUnAsged.size();i++)
		{
			double cs_heavy=-999.9;
			H1CS h1cs=(H1CS)vecUnAsged.elementAt(i);
			double cs_h=h1cs.getH1CS();
			String res=h1cs.getResidueType();
			String atom=h1cs.getAtomName();
			int resNo=h1cs.getResidueNo();
			if(!atom.substring(0,1).equalsIgnoreCase("H") )
				continue;
			String heavyName=pk.GetHeavyAtomFromProton(res,atom);
			for(int j=0;j<vecUnAsged.size();j++)
			{
				H1CS h1csHeavy=(H1CS)vecUnAsged.elementAt(j);
				double cs_Temp=h1csHeavy.getH1CS();
				String resT=h1csHeavy.getResidueType();
				String atomT=h1csHeavy.getAtomName();
				int resNoT=h1csHeavy.getResidueNo();
				if( (resNoT==resNo) && (atomT.equalsIgnoreCase(heavyName)) )
					cs_heavy=cs_Temp;
			}//for(int j=0;j<vecAsg.size();j++)
			
			Node node=new Node(curId,cs_h,cs_heavy,false);
			node.setAtomName(" ");
			node.setResName(" ");
			node.setResNo(0);
			vecGraph.add(node);
			curId++;
			
		}//for (int i=0;i<vecAsg.size();i++)
		/////////////////
		return vecGraph;
	}
	/**
     * add the reosnance node into the graph
     @param csH,csHeavy: chemical shifts of proton and heavy atom attached to it, -999.9 if it is unkown
     @param csErrH,csErrC,csErrN: eror windows.
     @param  
     @param 
     @param vecOldGraph old graph 
     @return new graph
     */
	public Vector AddResonanceNode(Vector vecOldGraph,double csH, double csHeavy, double csErrH, double csErrC, double csErrN )
	{
		Vector vecNewGraph=new Vector();
		boolean isInPres=false;
		for(int i=0;i<vecOldGraph.size();i++)
		{
			Node node=(Node)vecOldGraph.elementAt(i);
			double cs_Proton=node.getProtonCS();
			double cs_Heavy=node.getHeavyCS();
			double csMaxHeavy=Math.max(cs_Heavy,csHeavy);
			double csErrHeavy=0.0;
			if(csMaxHeavy>80.0)
				csErrHeavy=csErrN;
			else
				csErrHeavy=csErrC;
			
			if(csHeavy<-100.0)
				continue;
			
			if(csHeavy>-100.0)
			{
				//if((Math.abs(csH-cs_Proton)<0.000001) && (Math.abs(csHeavy-cs_Heavy)<0.0000001) )
				if((Math.abs(csH-cs_Proton)<csErrH) && (Math.abs(csHeavy-cs_Heavy)<csErrHeavy) )						
				{
					isInPres=true;
					if (node.getIsAssigned()==false)
					{
						Node nodeNear=new Node(node.getID(),csH,csHeavy,false);
						//node.setProtonCS((csH+cs_Proton)/2 );
						//node.setHeavyCS((csHeavy+cs_Heavy)/2 );
					}
				}
			}//if(csHeavy>-100.0)
			else
			{
				
			}
			
		}//for(int i=0;i<vecOldGraph.size();i++)
		if(!isInPres)
		{					
			Node node =new Node(vecOldGraph.size(),csH,csHeavy,false);
			vecOldGraph.add(node);
		}
		vecNewGraph.addAll(vecOldGraph);
		return vecNewGraph;
	}

	/**
     * check whether there exists at least one mapping between two node sets
      @param vecNodeSetA: adj set for resonance node
     @param vecNodeSetB: adj set of proton node
     @param M previous mapping matrix
     @param isAssigned whether only consider the assigned backbone resonance nodes
     @return 
     */
	public boolean isMappedBetwTwoSetsCalScore(Node nodeReson, Node nodeStruct,Vector vecNodeSetA,Vector vecNodeSetB,int[][]M, boolean isConsiderAssigned,int threshold, double [] Score,Vector vecNoesy)
	{
		boolean isMapped=false;
		int counter=0;
		Peak pk=new Peak();
		double cs_proton=nodeReson.getProtonCS();
		double cs_heavy=nodeReson.getHeavyCS();
		String res=nodeStruct.getResName();
		String atom=nodeStruct.getAtomName();
		int resNo=nodeStruct.getResNo();
		String heavName=pk.GetHeavyAtomFromProton(res,atom);
		Vector vecBackNoe=new Vector();
		
		for(int i=0;i<vecNodeSetA.size();i++)
		{
			Node nodeA=(Node)vecNodeSetA.elementAt(i);	
			
			boolean isAsgedA=nodeA.getIsAssigned();
			int idA=nodeA.getID();
			if(isConsiderAssigned)
			{
				if(!isAsgedA)			
					continue;
			}
			
			
			for(int j=0;j<vecNodeSetB.size();j++)
			{
				Node nodeB=(Node)vecNodeSetB.elementAt(j);			
				boolean isAsgedB=nodeB.getIsAssigned();
				String atomName=nodeB.getAtomName();
				String subAtom=atomName;
				if(atomName.length()>=2)
					subAtom=atomName.substring(0,2);
			
				
				int idB=nodeB.getID();
				if(isConsiderAssigned)
				{
					if(!isAsgedB)			
						continue;
				}
				else
				{
					if(!(isAsgedB ||subAtom.equalsIgnoreCase("HB") ))			
						continue;
				}
				
				if (M[idA][idB]==1)
				{
					double cs_protonSec=nodeA.getProtonCS();
					double cs_heavySec=nodeA.getHeavyCS();
					String resSec=nodeB.getResName();
					String atomSec=nodeB.getAtomName();
					String heavyNameSec=pk.GetHeavyAtomFromProton(resSec,atomSec);
					int resNoSec=nodeB.getResNo();
					double distUp=6.0;
					if(isInCloseAdj(nodeB,nodeStruct))
						distUp=2.7;
					vecBackNoe.add(new BackNoe(cs_proton,cs_heavy,cs_protonSec,distUp,0.0,resNo,resNoSec,res,resSec,atom,heavName,atomSec ));
					vecBackNoe.add(new BackNoe(cs_protonSec,cs_heavySec,cs_proton,distUp,0.0,resNoSec,resNo,resSec,res,atomSec,heavyNameSec,atom ));
					counter++;
				}//if (M[idA][idB]==1)
					
			}//for(int j=0;j<vecNodeSetB.size();j++)			
		}//for(int i=0;i<vecNodeSetA.size();i++)
		if(counter>=threshold)
			isMapped=true;	
		
		Assign asg=new Assign();
		int [] numPeaks=new int[1];
		double csErrH=0.04, csErrN=0.4,csErrCA=0.4;
		BackNoe bkNoe=new BackNoe();
		Vector vecBackNoeNew=bkNoe.DeleteRepeat(vecBackNoe);
		double dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoeNew,vecNoesy, numPeaks,true);
		dbScore=dbScore*vecBackNoeNew.size();//the more number of matched nodes, the better
		if(vecBackNoe.size()>1)
			Score[0]=dbScore;
		else
			Score[0]=0.0;
		return isMapped;
	}
	/**
     * check whether there exists at least one mapping between two node sets
     @param vecNodeSetB: node sets
     @param M previous mapping matrix
     @param isAssigned whether only consider the assigned backbone resonance nodes
     @return 
     */
	public boolean isMappedBetwTwoSets(Vector vecNodeSetA,Vector vecNodeSetB,int[][]M, boolean isConsiderAssigned,int threshold, double [] Score)
	{
		boolean isMapped=false;
		int counter=0;
		Peak pk=new Peak();
		
		
		Vector vecBackNoe=new Vector();
		
		for(int i=0;i<vecNodeSetA.size();i++)
		{
			Node nodeA=(Node)vecNodeSetA.elementAt(i);	
			
			boolean isAsgedA=nodeA.getIsAssigned();
			int idA=nodeA.getID();
			if(isConsiderAssigned)
				if(!isAsgedA)
					continue;
			for(int j=0;j<vecNodeSetB.size();j++)
			{
				Node nodeB=(Node)vecNodeSetB.elementAt(j);			
				boolean isAsgedB=nodeB.getIsAssigned();
				int idB=nodeB.getID();
				if(isConsiderAssigned)
					if(!isAsgedB)
						continue;
				if (M[idA][idB]==1)
				{
					double cs_protonSec=nodeA.getProtonCS();
					double cs_heavySec=nodeA.getHeavyCS();
				
					int resNoSec=nodeB.getResNo();
				counter++;
				}//if (M[idA][idB]==1)
					
			}//for(int j=0;j<vecNodeSetB.size();j++)			
		}//for(int i=0;i<vecNodeSetA.size();i++)
		if(counter>=threshold)
			isMapped=true;	
		
		Assign asg=new Assign();
		int [] numPeaks=new int[1];
		double csErrH=0.05, csErrN=0.4,csErrCA=0.4;
		
		Score[0]=counter;
		return isMapped;
	}
	
	/**
     * exact the backbone resonance assignment
     @param vecNOESY NOESY peak list
     @param csErrH,csErrC,csErrN: error windows
     @return vector of unassigned resonances
     */
	public Vector ExtractAllResonFromNOESY(Vector vecNOESY,double csErrH, double csErrC, double csErrN)
	{
		Nasca isoGraph=new Nasca();
		Vector vecAllResonances=new Vector();
		for(int i=0;i<vecNOESY.size();i++)
		{
			Noesy noesy=(Noesy)vecNOESY.elementAt(i);
			double csHeavy=noesy.getHeavy();
    		double csH1=noesy.getH1();
    		double csH2=noesy.getH2();
    		double distUpper=noesy.getUpperDist();
    		
    		
		}//for(int i=0;i<vecNOESY.size();i++)
		
		return vecAllResonances;
	}
	
	  /** 
     *Implement the NOE assignment (NASCA) algorithm without using the side-chain resonance infomration.
     @param src location of the input file
     @param strOut location of the output file
     @param strInput  input file name
     @return void
     */       
    public void doNasca(String src, String strOut, String strInput)throws JampackException
	{
    	boolean isDebug=false;//whether outputing the debugging information
    	Hsqc hqc = new Hsqc();
    	Peak pk = new Peak();
    	
    	ModelRdc mdc = new ModelRdc();    	
    	Assign asg = new Assign();
    	long startTime = System.currentTimeMillis();    	
    	/////////////////////////////////////////////
    	/// 1. Read the input files
    	//
    	/////////////////////////////////////////////
    	int i, j;    	
    	Pdb  pp = new Pdb();    	
    	Vector<Map<String, String>> paraVec = asg.ParamReader(src+strInput);
    	double haErr  = 0.0;
    	double h1Err  = 0.0;
    	double c13Err = 0.0;
    	double hnErr  = 0.0;
    	double nErr   = 0.0;
    	
    	int nIs4DNoesy=0;
    	
    	String strReson="", strSeq="", strNoesy2D="", strHnNoesy3D="", strResNameScheme="",strIsCheckLongAA="",
    		strHaNoesy3D="", strResFormat="",strBB="",stroutName="",strNoesyFormat="",strOutOrFormat="",
    		strIsCheck="",strRefPdb="",strRefNameSchem="",strIsWholeStr="",strPdbNameScheme="",str4DNoesy="",
    		stroutResonName="",strOutResonFormat="";
    	int nIsoriginalUp=0; 
    	double noeLimit=0.0,metCor=0.0;
    	
    	for (i=0;i<paraVec.size();i++)
    	{
    		//get the error torlence for each dimension
    		Map paraMap = paraVec.elementAt(i);
    		if(paraMap.containsKey("HAERR"))
    			 haErr  =  Double.parseDouble((String)paraMap.get("HAERR"));
    		if(paraMap.containsKey("H1ERR"))
   			 	h1Err  =  Double.parseDouble((String)paraMap.get("H1ERR"));
    		if(paraMap.containsKey("C13ERR"))
    			c13Err  =  Double.parseDouble((String)paraMap.get("C13ERR"));
    		if(paraMap.containsKey("HNERR"))
    			hnErr  =  Double.parseDouble((String)paraMap.get("HNERR"));
    		if(paraMap.containsKey("NERR"))
    			nErr  =  Double.parseDouble((String)paraMap.get("NERR"));
    		//get the input file name   
    		if(paraMap.containsKey("SEQUENCE"))
    			strSeq  =  (String)paraMap.get("SEQUENCE");
    		if(paraMap.containsKey("RESFORMAT"))
    			strResFormat  =  (String)paraMap.get("RESFORMAT");
    		if(paraMap.containsKey("RESONANCE"))
    			strReson  =  (String)paraMap.get("RESONANCE");
    		strReson=strReson.toLowerCase();
    		if(paraMap.containsKey("2D-NOESY"))
    			strNoesy2D  =  (String)paraMap.get("2D-NOESY");
    		//strNoesy2D=strNoesy2D.toLowerCase();
    		if(paraMap.containsKey("3D-N15-NOESY"))
    			strHnNoesy3D  =  (String)paraMap.get("3D-N15-NOESY");
    		//strHnNoesy3D=strHnNoesy3D.toLowerCase();
    		if(paraMap.containsKey("3D-C13-NOESY"))
    			strHaNoesy3D  =  (String)paraMap.get("3D-C13-NOESY");
    		//strHaNoesy3D=strHaNoesy3D.toLowerCase();    		
    		if(paraMap.containsKey("PDBNAMESCHEME"))
    			strPdbNameScheme  =  (String)paraMap.get("PDBNAMESCHEME");	
    		if(paraMap.containsKey("OUTNOENAME"))
    			stroutName  =  (String)paraMap.get("OUTNOENAME");   	
    		if(paraMap.containsKey("BACKBONE"))
    			strBB  =  (String)paraMap.get("BACKBONE");
    		if(paraMap.containsKey("NOELIMIT"))
    			noeLimit  =  Double.parseDouble((String)paraMap.get("NOELIMIT"));
    		if(paraMap.containsKey("IS4DNOESY"))
    			nIs4DNoesy  =   Integer.parseInt((String)paraMap.get("IS4DNOESY"));
    		
    		if(paraMap.containsKey("4D-NOESY"))
    			str4DNoesy  =  (String)paraMap.get("4D-NOESY");
    		
    		if(paraMap.containsKey("ISORIGINALUP"))
    			nIsoriginalUp  =   Integer.parseInt((String)paraMap.get("ISORIGINALUP"));	
    		if(paraMap.containsKey("RESNAMESCHEME"))
    			strResNameScheme  =  (String)paraMap.get("RESNAMESCHEME");	
    		if(paraMap.containsKey("NOESY-FORMAT"))
    			strNoesyFormat  =  (String)paraMap.get("NOESY-FORMAT");   	
    		if(paraMap.containsKey("ISCHECKLONGAA"))
    			strIsCheckLongAA =  (String)paraMap.get("ISCHECKLONGAA"); 	
    		if(paraMap.containsKey("ISCHECK"))
    			strIsCheck  =  (String)paraMap.get("ISCHECK");   	
    		if(paraMap.containsKey("REFPDB"))
    			strRefPdb  =  (String)paraMap.get("REFPDB");  
    		if(paraMap.containsKey("METHYL-CORRECTION"))
    			metCor  =  Double.parseDouble((String)paraMap.get("METHYL-CORRECTION"));    		
    		if(paraMap.containsKey("REFNAMESCHEME"))
    			strRefNameSchem  =  (String)paraMap.get("REFNAMESCHEME");  
    		if(paraMap.containsKey("ISWHOLESTRUCTURE"))
    			strIsWholeStr  =  (String)paraMap.get("ISWHOLESTRUCTURE");  
    		if(paraMap.containsKey("ISOUTORFORMAT"))
    			strOutOrFormat  =  (String)paraMap.get("ISOUTORFORMAT");  
    		
    		if(paraMap.containsKey("OUTRESONANCENAME"))
    			stroutResonName  =  (String)paraMap.get("OUTRESONANCENAME"); 
    		if(paraMap.containsKey("OUTRESONANCEFORMAT"))
    			strOutResonFormat  =  (String)paraMap.get("OUTRESONANCEFORMAT");
    	}       	   	
    	//-------------------------------------
    	// (1.1) Read the protein sequence
    	// 
    	//-------------------------------------    	
    	String seqFile = src + strSeq;
    	//this is the vector of residues (protein sequence)
    	Vector vecSeq=asg.ReaderSeq(seqFile);    	
    	
    	//-------------------------------------
    	// (1.2) Read the resonance list 
    	//     	
    	//following is for the general case
    	H1CS h1CS=new H1CS();    	
    	String assignFile = src+ strReson;
    	//this is the vector that stores resonance assignments;
    	//each one includes resNo, resName, atomName, csH1(chemical shift value)
    	Vector assignVec=new Vector();
    	if(strResFormat.equalsIgnoreCase("CYANA"))
    		assignVec=h1CS.h1CSReader(assignFile,vecSeq); 
    	else if(strResFormat.equalsIgnoreCase("BMRB"))
    		assignVec=h1CS.h1CSReader_BMRB(assignFile);  
    	else 
    	{
    		System.out.println("unknown format of the resonance file...");
    		System.exit(0);
    	}
    	
    	//all sorted proton resonances
    	Vector allH1Vec = pk.allProtonSorted(assignVec);
    	Collections.sort(allH1Vec, new Peak.csComparator());     	
    	//-------------------------------------
    	// (1.3) Read the NOESY peak list 
    	// in the xeasy format
    	// including 2D NOESY peak list, 3D N15 NOESY and 3D C13 NOESY peak lists
    	//-------------------------------------
    	Noesy noesy = new Noesy();
    	
    	String strNoeFile = ""; 
    	Vector hnNoeVec=new Vector();//for storing 3d n15 NOESY peaks
    	Vector cnoeVec=new Vector();//for storing 3d c13 noesy peaks
    	Vector Noe4DVec=new Vector();//for storing 4d NOESY peaks
    	if (!strNoesy2D.equalsIgnoreCase("NULL"))
    	{
    		//to do: extension to 2d case...
    	}
    	if (!strHnNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHnNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			hnNoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			hnNoeVec = noesy.NoesyReaderNMRView(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	if (!strHaNoesy3D.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+strHaNoesy3D;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			cnoeVec = noesy.NoesyReader(strNoeFile);
    		else
    			cnoeVec=noesy.NoesyReaderNMRView(strNoeFile);
    	}
    	
    	//4D NOESY data
    	if (!str4DNoesy.equalsIgnoreCase("NULL"))
    	{
    		strNoeFile=src+str4DNoesy;
    		if(strNoesyFormat.equalsIgnoreCase("XEASY"))
    			Noe4DVec = noesy.NoesyReader(strNoeFile);// to be changed....
    		else
    			Noe4DVec = noesy.NoesyReaderNMRView4D(strNoeFile);    		
    	}//if (!strNoesy3D.equalsIgnoreCase("NULL"))
    	
    	Vector vecNoesy=new Vector();
    	vecNoesy.addAll(hnNoeVec);
    	vecNoesy.addAll(cnoeVec);
    	
    	Vector vecNoesy4D=new Vector();
    	vecNoesy4D.addAll(Noe4DVec);//for 4D noesy    	

    	////////////////////////////////////    	    	
    	//intensity calibration    	
    	Vector vecNewNoesy=new Vector();
    	if(vecNoesy.size()>0)
    		vecNewNoesy=noesy.SetCalibration(vecNoesy); ////  	
    	Vector vecNewNoesy4D=new Vector();
    	if(vecNoesy4D.size()>0)
    		vecNewNoesy4D=noesy.SetCalibration(vecNoesy4D);   
    	double[] constant=new double[1];
    	Vector vecSSE=new Vector();//not used in the following function:
    	noesy.SetCalibrationN15(hnNoeVec,vecSSE,assignVec,constant);//change from vecNoesy to hnNoeVec
    	
    	// -------------------------------------
    	// (1.4) Read the backbone structure     	
    	//-------------------------------------    	
    	String pdbFile = src+strBB;
    	Vector pdbVecBB = pp.readPdb(pdbFile);
    	Vector pdbVecNew=new Vector();
    	
    	//update backbone atom names according to protein sequence input
    	//residue name in previous backbone may be all in "ALA".
    	if(strIsWholeStr.equalsIgnoreCase("1"))    	
    		pdbVecNew= pp.residueNameUpdateNoStr(vecSeq, pdbVecBB); 
    	else
    		pdbVecNew = pp.residueNameUpdate(vecSeq, pdbVecBB);
    	
    	Vector pdbVecNewSec=new Vector();
		if(strPdbNameScheme.equalsIgnoreCase("PDB-OLD"))
			pdbVecNewSec=pp.nameConvertOrder(pdbVecNew); 
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-NEW"))
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecNew);//need to double check
		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-OLD"))
		{
			Vector pdbVecSSE22=pp.nameConvertOrder(pdbVecNew); 
			pdbVecNewSec=pp.nameConvert2PDBNew(pdbVecSSE22);//need to double check
		}
		else
			pdbVecNewSec.addAll(pdbVecNew);  
    	
		Vector vecStructureAll=new Vector();
		vecStructureAll.addAll(pdbVecNewSec);
		
		Vector vecBBTemp=pp.OutputBackbone(pdbVecNewSec);
		pdbVecNewSec=new Vector();
		pdbVecNewSec.addAll(vecBBTemp);    	
    	
		/////////////////////////////////////////////
    	/// 2. Implement the NASCA algorithm
    	//
    	/////////////////////////////////////////////
		Nasca nasca=new Nasca();
		
		String userDir = System.getProperty("user.dir");////
    	String srcRot=userDir+"/system/rot-lib/";//path of rotamer library    	
    	
    	//------------------------------------------
    	//2.1 construct the set of proton positions:
    	//------------------------------------------
    	//construct the label set (here we also call the structure graph), which includes
    	//the set of proton positions after placing rotamers on the RDC-defined backbone
      	Vector vecLabels=nasca.ConstructProtonPositions(pdbVecNewSec,srcRot,5.0);
    	
      	//------------------------------------------
    	//2.2 construct the NOESY graph:
    	//------------------------------------------    	
    	Vector vecAsgNew=new Vector();		
		vecAsgNew=h1CS.CSNameConvertFromBMRBNewToPDBNew(assignVec);//assume that the default naming scheme of cs is BMRB
		assignVec=new Vector();
		assignVec.addAll(vecAsgNew);
    	Vector vecBBAsg=nasca.ExtractBackboneAsignment(assignVec);		
    	
		//sidechain resonance nodes from NOESY
		Vector vecResonNodeSet=nasca.ConstructResonGraphFromAsgedList(vecBBAsg,null);
		
		for(i=0;i<vecNewNoesy.size();i++)
		{
			Noesy ne=(Noesy)vecNewNoesy.elementAt(i);
			double csH1=ne.getH1();
			double csHeavy=ne.getHeavy();
			double csH2=ne.getH2();
			Vector vecResGraphTemp=new Vector();
			vecResGraphTemp=nasca.AddResonanceNode(vecResonNodeSet,csH1,csHeavy,0.02, 0.1, 0.1);//error windows: by half
			
			vecResonNodeSet=new Vector();
			vecResonNodeSet.addAll(vecResGraphTemp);
		}
		//group each resonance cluster:
		for(i=0;i<vecResonNodeSet.size();i++)
		{
			Node node=(Node)vecResonNodeSet.elementAt(i);
			Vector vecAdj=node.getAdjVec();
			int counter=1;
			double csProtonSum=node.getProtonCS();
			double csHeavySum=node.getHeavyCS();
			for(int k=0;k<vecAdj.size();k++)
			{
				Node ndAdj=(Node)vecAdj.elementAt(k);
				csProtonSum=csProtonSum+ndAdj.getProtonCS();
				csHeavySum=csHeavySum+ndAdj.getHeavyCS();
				counter++;
			}
			csProtonSum=csProtonSum/counter;
			csHeavySum=csHeavySum/counter;
			node.setProtonCS(csProtonSum);
			node.setHeavyCS(csHeavySum);
			node.EmptyAdjSet();
		}		
		//set up the edge connections for resonances:
		Vector vecResonGraph=nasca.ConstructResonGraph(vecResonNodeSet,vecNewNoesy,0.04,0.04,0.1,0.1);//0.04,0.03,0.2,0.2
		
		//set up symmetry set
		ConstructResonGraphSymmetry(vecResonNodeSet,vecNewNoesy,0.03,0.03,0.1,0.1);//0.04,0.03,0.2,0.2
			
		//read BMRB statistical information:
		String srcBMRB=userDir+"/system/BMRB_CS.txt";    	
    	Vector vecBMRB=h1CS.ReadBMRBSatistics(srcBMRB);
    	Vector vecBMRBNew=new Vector();		
    	vecBMRBNew=h1CS.CSNameConvertFromBMRBNewToPDBNew(vecBMRB);//assume that the default naming scheme of cs is BMRB
    	
    	//--------------------------------------------------------
    	//2.3 pruning infeasable side-chain resonance assignments
    	//    using BMRB outliers and backbone chemical shifts.
    	//--------------------------------------------------------   
    	int [][]Map;//mapping matrix, for storing the mappings between NOESY graph and proton positions
    	double [][] MapScore;
    	Map=new int[vecResonGraph.size()][vecLabels.size()];
    	MapScore=new double[vecResonGraph.size()][vecLabels.size()];
    	
    	//initially set up all possible mappings
    	Initialization(vecResonGraph,vecLabels,Map,MapScore,vecBBAsg);
    	
    	/////////////////////////
    	//pruning side-chain resonance assignments that fall outside the BMRB intervals
    	RefineMappingBasedBMRBOutliers(vecResonGraph,vecLabels,Map,vecBMRBNew);
    	
    	//update the mapping set
    	UpdateMappingSet(vecResonGraph,vecLabels,Map);
    	UpdateIsRestrained(vecResonGraph,vecLabels, Map);    	
    	
    	//Refinement based on the bb resonance assignments:
        //Note: This function was implemented in the early stage as a preposcessing of the 
    	//side-chain resonance assignments. Now we 
    	//still keep it, since in practice it might speed up the pruning process. 
    	//In principle, the assignments pruned from this function can be also 
    	//pruned by the DEE assignment approach.
    	RefineMappingBasedRestrained(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy, vecBMRBNew);
    	
    	//Refinement based on previously assigned HG resonances:
        //Note: This function was implemented in the early stage as a preposcessing of the 
    	//side-chain resonance assignments. Now we 
    	//still keep it, since in practice it helps reduce the running time. 
    	//In principle, the assignments pruned from this function can be also 
    	//pruned by the DEE assignment approach.
       	RefineMappingForHG(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy, vecBMRBNew);       	
     	UpdateMappingSet(vecResonGraph,vecLabels,Map);
     	
     	//-------------------------------------------
     	//2.4 Use the DEE assignment algorithm to prune side-chain 
     	//    resonance assignments that are provably not part of 
     	//    the optimal solution.
      	RefineMappingDeeCut(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy, vecBMRBNew);//new     
     	UpdateMappingSet(vecResonGraph,vecLabels,Map);     	
       	
        //print out all remaining assignments, after pruning. 
     	//This information is for debugging.
       	PrintAllPossibleMappings(vecResonGraph,vecLabels,Map);
       	
       	//output the statistical summary of the remaining assignments after pruning.
       	//This information is for debugging.
       	nasca.StatisticsAllSCAssignment(vecResonGraph,vecLabels, Map,assignVec ); 
    	
    	//update the mapping set
    	UpdateMappingSet(vecResonGraph,vecLabels,Map);
    	H1CS h1cs=new H1CS();
    	
    	//----------------------------------------------------
    	//2.5 Use the A* search algorithm to find the optimal 
    	//    side-chain resonance assignments.    	
    	//----------------------------------------------------
    	Vector vecAsgNewT0=AStarSearchForSCAsgs(vecResonGraph,vecLabels,Map,MapScore,vecNewNoesy,vecBMRBNew);
    	Vector vecAsgNewT=h1cs.DeleteRepeatAsg(vecAsgNewT0);//delete repeated assignments.
    	
    	//print out the resonance assignments in different formats:
    	//This information will be useful for debugging.
    	h1cs.PrintAllAssignment(vecAsgNewT);  //In BMRB format    	
    	h1cs.PrintAllAssignmentCyana(vecAsgNewT); //in CYANA format
    	
    	//////////////////////////////
    	//print out the resonance assignments into files:
    	String outResFileName=userDir+stroutName;    	
    	try{
    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(outResFileName)));
    		
    		if(strOutResonFormat.equalsIgnoreCase("CYANA"))        	
    			h1cs.PrintAllAssignmentToFileCyana(vecAsgNewT,out);        	
        	else        	
        		h1cs.PrintAllAssignmentToFileBMRB(vecAsgNewT,out);      	
	       
	    	out.close();
	    	System.out.println("The NOE assignment table has been generated in "+ outResFileName); 
    	}catch (FileNotFoundException e)
		{
			System.out.println("File not found: " + outResFileName);
		}catch (IOException e)
		{
		   System.out.println("IOException: the stack trace is:");
		   e.printStackTrace();
		}

    	///////////////////////////////////////   	
    	//compare with the reference resonance assignment list:
    	//this information is for debugging, and evaluating the performance.
    	h1cs.CompareSCAsgsWRefAsgs(vecAsgNewT,assignVec,vecBBAsg);    	
    	
    	//compute the completeness of the reference resonance assignment table:
    	double counter=0.0;
    	for(int t=0;t<vecLabels.size();t++)
    	{
    		Node node=(Node)vecLabels.elementAt(t);
    		String ndAtom=node.getAtomName();
    		int ndNo=node.getResNo();
    		
	    	for (int k=0;k<assignVec.size();k++)
	    	{	    		
	    		H1CS h1csT=(H1CS)assignVec.elementAt(k);	    		
	    		int refNo=h1csT.getResidueNo();
	    		String atom=h1csT.getAtomName();
	    		String subNdAtom=ndAtom;
	    		if(ndAtom.length()>atom.length())
	    			subNdAtom=ndAtom.substring(0,atom.length());
	    		
	    		if( !(atom.substring(0,1).equalsIgnoreCase("H")) )
	    			continue;
	    		if(ndNo==refNo && subNdAtom.equalsIgnoreCase(atom))
	    		{
	    			counter=counter+1;    	
	    			break;
	    		}
	    	}//end for (int k=0;k<assignVec.size();k++)
    	}//end for(int t=0;t<vecGraph.size();t++)
    	System.out.println("Completeness of reference assignment table is: "+(double) (counter/vecLabels.size()));
    
    	//compute the completeness of the our assignment table.
    	counter=0;
    	for(int k=0;k<vecAsgNewT.size();k++)
    	{
    		H1CS h1csT=(H1CS)vecAsgNewT.elementAt(k);
    		String atom=h1csT.getAtomName();
    		if(atom.substring(0,1).equalsIgnoreCase("H"))
    			counter=counter+1;     		
    	}
    	System.out.println("Completeness of our assignment table is: "+ (double)(counter/vecLabels.size()));
            
    	///vecBBAsg=new Vector();
    	///vecBBAsg.addAll(vecAsgNewT);
    	
    	////////////////////////////////////////////////    	
    	/*for(int k=0;k<vecBBAsg.size();k++)
    	{
    		H1CS h1_cs=(H1CS)vecBBAsg.elementAt(k);
    		
    		boolean isHydrophobic=false;
    		
    	}//for(int k=0;k<vecBBAsg.size();k++)    	
    	*/
    	//--------------------------------------------
    	//2.6 NOE assignment process
    	//
    	//--------------------------------------------
    	Vector vecAmbgAsg=new Vector();    	
    	//all sorted proton resonances
    	allH1Vec=new Vector();
    	allH1Vec = pk.allProtonSorted(vecAsgNewT);
    	Collections.sort(allH1Vec, new Peak.csComparator()); 
    	
    	//compute the initial ambiguous NOE assignments. 
    	//In principle we could use the NOESY graph to generate the set of 
    	//initial ambiugous assignments. But here we re-use the previously-written function.
    	//The result would be the same. 
    	vecAmbgAsg=asg.InitAmbiAssignment(nErr,c13Err,hnErr,haErr,h1Err,vecAsgNewT,allH1Vec,vecNewNoesy );        	
    	
    	//calibration for all initial NOE assignments
    	Vector vecAmbgAsgTemp=noesy.SetCalibrationCyana(vecAmbgAsg,vecNewNoesy,constant[0]);    	
    	vecAmbgAsg=new Vector();
    	vecAmbgAsg.addAll(vecAmbgAsgTemp);
     	 	
    	
     	//delete symmetric NOEs (which are redundent constraints):
     	Vector vecNoeRefine2=new Vector();     	
     	for(i=0;i<vecAmbgAsg.size();i++)
     	{
     		Noe noe=(Noe)vecAmbgAsg.elementAt(i);
     		int noA=noe.getResidueNoA();
     		int noB=noe.getResidueNoB();
     		String atomA=noe.getAtomA();
     		String atomB=noe.getAtomB();
     		if(noA==noB && atomA.equalsIgnoreCase(atomB))
     			continue;
     		
     		boolean isIn=true;
     		if(strOutOrFormat.equalsIgnoreCase("0"))
     		{
	     		for (j=0;j<vecNoeRefine2.size();j++)
	     		{
	     			Noe noeT=(Noe)vecNoeRefine2.elementAt(j);
	     			int noAT=noeT.getResidueNoA();
	     			int noBT=noeT.getResidueNoB();
	     			String atomAT=noeT.getAtomA();
	     			String atomBT=noeT.getAtomB();
	     			if(noA==noAT && atomA.equalsIgnoreCase(atomAT) && noB==noBT && atomB.equalsIgnoreCase(atomBT))
	     				isIn=false;
	     			if(noA==noBT && atomA.equalsIgnoreCase(atomBT) && noB==noAT && atomB.equalsIgnoreCase(atomAT))
	     				isIn=false;
	     		}//for (j=0;j<vecNoeRefine2.size();j++)
     		}//if(strOutOrFormat.equalsIgnoreCase("0"))
     		if(	isIn)
     			vecNoeRefine2.add(noe);
     	}
     	
     	vecAmbgAsg=new Vector();     	
     	vecAmbgAsg.addAll(vecNoeRefine2);
     	
     	//place the rotamers onto the RDC-defined backbone. 
     	Vector vecRotStructure=pp.RotamSelectAndStructure(hnErr, nErr, c13Err,pdbVecNewSec,vecAsgNewT,srcRot,vecNewNoesy,
    			4.5, 1,constant[0]);
     	
     	//We prune ambiguous NOE assignments based on the structural information. 
     	//Note: The NOE pruning process is a little different from what described in the paper
     	//but in the principle the results should be similar, since the structural information
     	//combined from RDC-defined backbone and rotamer library is used in a similar way.
     	Vector vecRefinedNoes=new Vector();    	
    	vecRefinedNoes=asg.PrunAmbgNoeAsg(vecRotStructure,vecAmbgAsg,noeLimit,0.0,nIsoriginalUp);
    	
     	Collections.sort(vecRefinedNoes, new Noe.NoeComparator()); 
     	
    	//sort the NOE table
    	if(!strOutOrFormat.equalsIgnoreCase("1"))
    	{
	    	for(i=0;i<vecAmbgAsg.size();i++)
	    	{
	    		Noe noe=(Noe)vecAmbgAsg.elementAt(i);
	    		int firstResNo=noe.getResidueNoA();
	    		int secondResNo=noe.getResidueNoB();
	    		String firstResName=noe.getResidueA();
	    		String secondResName=noe.getResidueB();
	    		String firstAtomName=noe.getAtomA();///
	    		String secondAtomName=noe.getAtomB();     		
	    		
	    		
	    		if(secondResNo<firstResNo)
	    		{
	    			noe.setResNoA(secondResNo);
	    			noe.setResNoB(firstResNo);
	    			noe.setAtom1Name(secondAtomName);
	    			noe.setAtom2Name(firstAtomName);
	    			noe.setResNameA(secondResName);
	    			noe.setResNameB(firstResName);
	    		}
	    	}//for(i=0;i<vecManAsgNew.size();i++)
    	
	    	Collections.sort(vecAmbgAsg, new Noe.noeComparatorC());
    	}    	      	
     	
    	if(strIsCheckLongAA.equalsIgnoreCase("1"))
    	{    		
    		Vector vecTemp=new Vector();
    		for(i=0;i<vecRefinedNoes.size();i++)
        	{
        		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
        		int firstResNo=noe.getResidueNoA();
        		int secondResNo=noe.getResidueNoB();
        		String firstResName=noe.getResidueA();
        		String secondResName=noe.getResidueB();
        		String atomA=noe.getAtomA();
        		String atomB=noe.getAtomB();        		
        		String subAtomA=atomA;
        		String subAtomB=atomB;
        		if (atomA.length()>=2)
        			subAtomA=atomA.substring(0,2);
        		if(atomB.length()>=2) 
        			subAtomB=atomB.substring(0,2);
        			
        		if(Math.abs(firstResNo-secondResNo)<=4 )
        		{
        			vecTemp.add(noe);
        			continue;
        		}
        	
        		boolean isKeep=true;
        		if ( pk.isCharged(firstResName) && pk.isHydrophobic(secondResName) )
        			isKeep=false;
        		if ( pk.isCharged(secondResName) && pk.isHydrophobic(firstResName) )
        			isKeep=false;
        		if( (subAtomB.equalsIgnoreCase("HN") || subAtomB.equalsIgnoreCase("H")||subAtomB.equalsIgnoreCase("HA")||subAtomB.equalsIgnoreCase("HB")) &&
        				(subAtomA.equalsIgnoreCase("HN")||subAtomA.equalsIgnoreCase("H")||subAtomA.equalsIgnoreCase("HA")|| subAtomA.equalsIgnoreCase("HB")   ))
        			isKeep=true;
        		
        		if(isKeep)
        			vecTemp.add(noe);
        	}//for(i=0;i<vecManAsgNew.size();i++)
    		
    		vecRefinedNoes=new Vector();
    		vecRefinedNoes.addAll(vecTemp);
    	}//if(strIsCheckLongAA.equalsIgnoreCase("1"))    	
    	
    	//sort the NOE restraints, and updated the atom names for metyl groups:
    	Collections.sort(vecRefinedNoes, new Noe.NoeComparator()); 
        Noe ne=new Noe();
        Vector vecNoeNewT=ne.NameConvertFromPDBNewToPDBNew(vecRefinedNoes);
        vecRefinedNoes=new Vector();
        vecRefinedNoes.addAll(vecNoeNewT);             
       
     	//compute the numbers of unique and multiple assignments:
     	vecNoeRefine2=new Vector();
     	Vector vecNoeUniq=new Vector();
     	for(i=0;i<vecRefinedNoes.size();i++)
     	{
     		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
     		int pkID=noe.getPeakID();
     		boolean isUniq=true;
     		for(j=0;j<vecRefinedNoes.size();j++)
     		{
     			if (i==j)
     				continue;
     			Noe noeT=(Noe)vecRefinedNoes.elementAt(j);
     			int pkIDT=noeT.getPeakID();
     			if(pkID==pkIDT)
     				isUniq=false;
     		}
     		if(isUniq)
     			vecNoeUniq.add(noe);
     	}
     	
    	//compare the NOE assignments with the reference structure and output the summary.
     	double[] number=new double[4];
     	if(strIsCheck.equalsIgnoreCase("1"))
     	{
     		Noe noe_temp=new Noe();
     		
     		String pdbFileXray = src+strRefPdb;
        	Vector vecTempPdbBB=pp.readPdb(pdbFileXray);
        	Vector vecTempPdbBB2=new Vector();        	
        	
        	if(strRefNameSchem.equalsIgnoreCase("PDB-OLD")) 	
        		vecTempPdbBB2 = pp.nameConvertOrder(vecTempPdbBB);
        	else if(strRefNameSchem.equalsIgnoreCase("BMRB-NEW"))
        		vecTempPdbBB2=pp.nameConvert2PDBNew(vecTempPdbBB);
    		else if(strPdbNameScheme.equalsIgnoreCase("BMRB-OLD"))
    		{
    			Vector pdbVecSSE22=pp.nameConvertOrder(vecTempPdbBB); 
    			vecTempPdbBB2=pp.nameConvert2PDBNew(pdbVecSSE22);
    		}
        	else
        		vecTempPdbBB2.addAll(vecTempPdbBB);
        	System.out.println("============================================ ");
        	System.out.println("====Comparisons with reference structure===== ");
        	noe_temp.CompareAsgToRefStr(vecRefinedNoes,vecTempPdbBB2,6.0,0.0,number);
        	//System.out.println("Total number of manual NOEs = "+ vecManAsg.size());
        	System.out.println("Number of correct NOE assignments in reference structure=  "+ number[0] );
        	System.out.println("Number of wrong NOE assignments in reference structure=  "+ number[1] );    	
        	System.out.println("Number of correct Long-Rang NOE assignments in reference structure=  "+ number[2] );
        	System.out.println("Number of wrong Long-Rang NOE assignments in reference structure=  "+ number[3] );
        	
        	System.out.println("============================================ ");
     	}//if(strIsCheck.equalsIgnoreCase("1"))
     	/////////////////////////////////////////////
     	
    	//Output the final NOE assignments in standard xplor format, and statistical summary    	    	
     	String userDirTemp = System.getProperty("user.dir");
     	if(strOutOrFormat.equalsIgnoreCase("0"))
    	{
	     	String fileName=userDirTemp+stroutName;    	
	    	
	    	String xplorNoeStr="";
	    	double[] disUpper=new double[1];
	    	try{
	    		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
	    		out.println("!REMARK: Total number of NOESY cross peaks is "+ (vecNoesy.size()+vecNoesy4D.size()));    		
	    		out.println("!REMARK: Total number of NOEs is "+vecRefinedNoes.size());
	    		out.println("!REMARK: the number of unique NOE assignments is "+vecNoeUniq.size());
	    		out.println("!REMARK: the number of multiple NOE assignments is "+(vecRefinedNoes.size()-vecNoeUniq.size()) );
	    		    		
	    		out.println("!REMARK: Number of correct assignments in reference structure=  "+ number[0] );
	        	out.println("!REMARK: Number of wrong assignments in reference structure=  "+ number[1] );    	
	        	out.println("!REMARK: Number of correct Long-Rang NOE assignments in reference structure=  "+ number[2] );
	        	out.println("!REMARK: Number of wrong Long-Rang NOE assignments in reference structure=  "+ number[3] );
	    		out.println("");
	    		out.println("");
	    		
		        for (i=0;i<vecRefinedNoes.size();i++)		       	
		    	{
		    		Noe noe=(Noe)vecRefinedNoes.elementAt(i);
		    		int resNo1=noe.getResidueNoA();
		    		int resNo2=noe.getResidueNoB();
		    		String res1=noe.getResidueA();
		    		String res2=noe.getResidueB();
		    		String atom1=noe.getAtomA();
		    		String atom2=noe.getAtomB();
		    		double distUpper=noe.getUpper();
		    		
		    		//xplor format:	    		
					xplorNoeStr =pk.xplorNoeStatementNew(resNo1, res1, atom1,resNo2, res2, atom2, distUpper);
					if(isDebug)
						System.out.println(xplorNoeStr); 
					out.println(xplorNoeStr);
		    	}//for (i=0;i<vecRefinedNoes.size();i++)
		    	out.close();
		    	System.out.println("The NOE assignment table has been generated in "+ fileName); 
	    	}catch (FileNotFoundException e)
			{
				System.out.println("File not found: " + fileName);
			}catch (IOException e)
			{
			   System.out.println("IOException: the stack trace is:");
			   e.printStackTrace();
			}
    	}
     	else if(strOutOrFormat.equalsIgnoreCase("1")) //we output the NOEs in OR format:
    	{
    		Noe noeTemp=new Noe();    		
    		Vector vecMultiAsg=noeTemp.ConvertSingleToMultiAsg(vecRefinedNoes);
    		String fileName=userDirTemp+stroutName;  
    		noeTemp.OutputMultiNOEAsg(vecMultiAsg,fileName);    		
    	}
     	
     	///////////////////////////////////////
     	long endTime = System.currentTimeMillis();
    	double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
    	System.out.println("Time= "+ totalTime +" minutes");
	}
}
