package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	PdbRmsd.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//    JMZ		 Jianyang (Michael) Zeng	       Duke University			zengjy@cs.duke.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

     
import java.io. *;
import java.util. *;
import Jampack.JampackException;

// TODO: Auto-generated Javadoc
/** * 
 *  
   A program for determining the relative positions of a helix and a beta sheet 
 * based on the NOE distances between them.
 * The main idea is: 
 * (1) Compute the centers of the two groups of nuclei with NOEs.
 * Compute the RMSD between two PDBs based on SVD.
 * In the simple case their AA sequence are the same except that
 * one can be only a partial of the other (shorter than the other).
 * When do the rotation the center is very important so we need to
 * compute the center first as in the algorithm. 
 * Written by Lincong Wang (2001-2005) and Jianyang (Michael) Zeng (2005-2009).
*/
public class PdbRmsd 
{
    
    /** The pdb vec a. */
    private Vector pdbVecA = new Vector();
    
    /** The pdb vec b. */
    private Vector pdbVecB = new Vector();
    
    /** The Constant atomNameArrA. */
    static final String[] atomNameArrA = {"N", "C"}; //atoms compared
    
    /** The Constant atomNameArrB. */
    static final String[] atomNameArrB = {"N", "C"};  //A and B can, in principle, be different
    
    /** The is skip out ensembe. */
    public boolean isSkipOutEnsembe=false;
    
    /**
     * Instantiates a new pdb rmsd.
     */
    public PdbRmsd(){
	pdbVecA = new Vector();
	pdbVecB = new Vector();
    }
    
    /**
     * Gets the pdb vec a.
     * 
     * @return the pdb vec a
     */
    public Vector getPdbVecA(){
	return pdbVecA;
    }
    
    /**
     * Gets the pdb vec b.
     * 
     * @return the pdb vec b
     */
    public Vector getPdbVecB(){
	return pdbVecB;
    } 
    
    /**
     * calculate the angles between two vectors v1 and v2
     * The returned angle is in the [0,Pi] range or the cosine itself.
     * or just return the interAngle cosine
     * 
     * @param v1 the v1
     * @param v2 the v2
     * 
     * @return the double
     */
    public double interAngle(double[] v1, double[] v2){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	double v2Len = Math.sqrt(v2[0]*v2[0] + v2[1]*v2[1] + v2[2]*v2[2]);
	return ( (v1[0]*v2[0] + v1[1]*v2[1] + v1[2]*v2[2]) / (v1Len * v2Len) );

    }

    /**
     * cal the directioal cosines of the vector vec.
     * 
     * @param vec the vec
     * 
     * @return the double[]
     */
    public double[] dirCos(double [] vec) {
	double len = 0.0;
	double [] dirs = new double[vec.length];
	for (int i=0; i<vec.length; i++)
	    len += vec[i] * vec[i];
	for (int j=0; j<vec.length; j++)
	    dirs[j] = vec[j] / Math.sqrt(len);
	return dirs;
    }

    /**
     * compute the center of a PDB object (the 3D coordinates).
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the double[]
     */
    public double[] centerCal(Vector pdbVec){
	int i, j, k;
	Pdb pp = new Pdb();
	Cartesian cc = new Cartesian();
	Vector atomVec = new Vector();
	double xC1=0.0, yC1=0.0, zC1=0.0;
	int mm = pdbVec.size();
	int m = 0;
	String atom = "";
	for (i=0; i<pdbVec.size(); i++){
	    pp = (Pdb)pdbVec.elementAt(i); 
	    atomVec = pp.getAtomVec();
	    for (j=0; j<atomVec.size(); j++){ //save all those of B with atom Name as in the array
		cc = (Cartesian)atomVec.elementAt(j);
		atom = cc.getAtom();
		for (k =0; k< atomNameArrA.length; k++){ //since this is supposely very short so linear search OK
		    if (atom.equals(atomNameArrA[k])){
			cc = (Cartesian)atomVec.elementAt(j);
			xC1 += cc.getX(); 
			yC1 += cc.getY();
			zC1 += cc.getZ();
			m++;
		    }
		}
	    }
	}
	// Compute the center
	if (m > 0){
	    xC1 /= m;
	    yC1 /= m;
	    zC1 /= m;
	}
	return new double[]{xC1, yC1, zC1};
    }

    /**
     * compute the center of a PDB coordinates when the coordinates are represented as an array of nx3.
     * 
     * @param pdbVec the pdb vec
     * 
     * @return the double[]
     */
    public double[] centerCal(double[][]  pdbVec){
	double xC1=0.0, yC1=0.0, zC1=0.0;
	int mm = pdbVec.length;
	for (int i=0; i<mm; i++){
	    xC1 += pdbVec[i][0];
	    yC1 += pdbVec[i][1];
	    zC1 += pdbVec[i][2];
	}
	// Compute the center
	if (mm > 0){
	    xC1 /= mm;
	    yC1 /= mm;
	    zC1 /= mm;
	}
	return new double[]{xC1, yC1, zC1};
    }
    
    /**
     * compute the center of a PDB coordinates when the coordinates are represented as an array of nx3.
     * 
     * @param vecCC the vec cc
     * 
     * @return the double[]
     */
    public double[] centerCalNew(Vector vecCC)
    {
		double xC1=0.0, yC1=0.0, zC1=0.0;
		int mm = vecCC.size();
		for (int i=0; i<vecCC.size(); i++)
		{
			Cartesian ccTemp=(Cartesian)vecCC.elementAt(i);
			double [] coord = {0.0, 0.0, 0.0};
			coord=ccTemp.getXYZ();
		    xC1 += coord[0];
		    yC1 += coord[1];
		    zC1 += coord[2];
		}
	// Compute the center
	if (mm > 0){
	    xC1 /= mm;
	    yC1 /= mm;
	    zC1 /= mm;
	}
	return new double[]{xC1, yC1, zC1};
    }

    /**
     * compute the H-bond energy term between the two paired strands.
     * 
     * @param pdbVecA     strand A
     * @param pdbVecB     strand B, A's partner
     * @param hbVec       the vector of Hbond information
     * @param debug if true it print the information for debuging
     * 
     * @return the double
     */
    public double centerFit(Vector pdbVecA, Vector pdbVecB, Vector hbVec, boolean debug){
	boolean debug2 = false;
	final int  noOfAtoms = hbVec.size(); 
	int i, j, k;
	//One pair (N->H...O or O...H<-N), their partner's coordinates are computed based on perfect H-bond.
	double[][] hPartnerOfA = new double[noOfAtoms][3]; //Coodinates of its H-bond partner,
	double[][] coordOfA    = new double[noOfAtoms][3]; //Coodinates of atoms N and O, in the order of N, O
	double[][] dirsOfA     = new double[noOfAtoms][3]; //and directional cosines

	double[][] hPartnerOfB = new double[noOfAtoms][3]; //the corresponding ones for strand B
	double[][] coordOfB    = new double[noOfAtoms][3]; 
	double[][] dirsOfB     = new double[noOfAtoms][3]; 

	Pdb pp = new Pdb();
	Cartesian cc = new Cartesian();
	Vector atomVec = new Vector();
	String atom = "";
	double [] amide = new double[3];
	double [] nh = new double[3];
	double [] n2NHVec  = new double[3];
	double [] co = new double[3];
	double [] o  = new double[3];
	double [] co2OVec = new double[3];
	int residueNo = 0;
	int residueNoA;   //the Sequence No of one of the two partners of the NOE
	int residueNoB;   //the Sequence No of Another partner of the NOE
	String atomA;     //the name of Atom A
	String atomB;     //the name of Atom B
	String type;      //the name of A
	Hbond hb = new  Hbond();
	//Extract atoms NH2O and O2HN of strandA, and compute the directions of N2H and CO2O for both strand A and B
	boolean isN = false;
	int index  =  -1;
	for (i = 0; i < noOfAtoms; i++){
	    hb = (Hbond)hbVec.elementAt(i);
	    residueNoA = hb.getResidueNoA();
	    atomA = hb.getAtomA();
	    index = Collections.binarySearch(pdbVecA, new Pdb(residueNoA), new Pdb.PdbComparator());
	    //Extract the coordinates and direction of the H-bond partners, one partner:
	    if ( index > -1 ){
		pp = (Pdb)pdbVecA.elementAt(index);
		residueNo = pp.getResidueNo();
		atomVec = pp.getAtomVec();
		if  (atomA.equals("O")){
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("C"))
			    co = cc.getXYZ();
			else if (atom.equals("O")){
			    o = cc.getXYZ();
			    coordOfA[i] = new double[]{o[0], o[1], o[2]};
			}
		    }
		    //The coordinate of the H-bond partner of the atom O 
		    hPartnerOfA[i] = new double[]{o[0] + (o[0] - co[0]) * Const.dN2COHBond / Const.dCO2O, 
						  o[1] + (o[1] - co[1]) * Const.dN2COHBond / Const.dCO2O,
						  o[2] + (o[2] - co[2]) * Const.dN2COHBond / Const.dCO2O};
		    dirsOfA[i] = new double[]{o[0] - co[0],  o[1]-co[1], o[2]-co[2]};
		    if (debug2){
			System.out.println(residueNoA);
			System.out.println("O: "+i+"  "+coordOfA[i][0]+"  "+coordOfA[i][1]+"   "+coordOfA[i][2]);
			System.out.println("A: "+i+"  "+dirsOfA[i][0]+"  "+dirsOfA[i][1]+"   "+dirsOfA[i][2]);
		    }
		} else if (atomA.equals("N")) {
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("N")){
			    amide = cc.getXYZ();
			    coordOfA[i] = new double[]{amide[0], amide[1], amide[2]};
			}else if (atom.equals("H"))
			    nh = cc.getXYZ();
		    }
		    //The coordinate of the H-bond partner of the atom N
		    hPartnerOfA[i] = new double[]{amide[0] + (nh[0] - amide[0]) * Const.dN2COHBond / Const.dN2H, 
						  amide[1] + (nh[1] - amide[1]) * Const.dN2COHBond / Const.dN2H, 
						  amide[2] + (nh[2] - amide[2]) * Const.dN2COHBond / Const.dN2H};
		    dirsOfA[i] = new double[]{nh[0]-amide[0], nh[1]-amide[1], nh[2]-amide[2]};
		    if (debug2){
			System.out.println(residueNoA);
			System.out.println("N: "+i+"  "+coordOfA[i][0]+"  "+coordOfA[i][1]+"   "+coordOfA[i][2]);
			System.out.println("A: "+i+"  "+dirsOfA[i][0]+"  "+dirsOfA[i][1]+"   "+dirsOfA[i][2]);
		    }
		}
	    }
	    //Extract the coordinates and direction of the H-bond partners, the corresponding partner:
	    residueNoB = hb.getResidueNoB();
	    atomB = hb.getAtomB();
	    index = Collections.binarySearch(pdbVecB, new Pdb(residueNoB), new Pdb.PdbComparator());
	    if ( index > -1 ){
		pp = (Pdb)pdbVecB.elementAt(index);
		residueNo = pp.getResidueNo();
		atomVec = pp.getAtomVec();
		if  (atomB.equals("O")){
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("C"))
			    co = cc.getXYZ();
			else if (atom.equals("O")){
			    o = cc.getXYZ();
			    coordOfB[i] = new double[]{o[0], o[1], o[2]};
			}
		    }
		    dirsOfB[i] = new double[]{o[0] - co[0], o[1]-co[1], o[2]-co[2]};
		    if (debug2){
			System.out.println(residueNoB);
			System.out.println("O: "+i+"  "+coordOfB[i][0]+" : "+coordOfB[i][1]+"   "+coordOfB[i][2]);
		    }
		}else if (atomB.equals("N")) {
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("N")){
			    amide = cc.getXYZ();
			    coordOfB[i] = new double[]{amide[0], amide[1], amide[2]};
			}else if (atom.equals("H"))
			    nh = cc.getXYZ();
		    }
		    //The coordinate of the atom O of the H-bond partner of the NH vector
		    hPartnerOfB[i] = new double[]{amide[0] + (nh[0] - amide[0]) * Const.dN2COHBond / Const.dN2H, 
						  amide[1] + (nh[1] - amide[1]) * Const.dN2COHBond / Const.dN2H, 
						  amide[2] + (nh[2] - amide[2]) * Const.dN2COHBond / Const.dN2H};
		    dirsOfB[i] = new double[]{nh[0]-amide[0], nh[1]-amide[1], nh[2]-amide[2]};
		    if (debug2){
			System.out.println(residueNoB);
			System.out.println("N: "+i+"  "+coordOfB[i][0]+" : "+coordOfB[i][1]+"   "+coordOfB[i][2]);
		    }
		}
	    }
	}
	double[] centerOfPartnersOfA = centerCal(hPartnerOfA); //the center of the coords of the H-bond partners of A.
	double x1, y1, z1, x2, y2, z2;  //for compute the H-bond distances.
	double disHBond;

	double[] centerOfB = centerCal(coordOfB); //the center of the coords of B.
	double[] cT={centerOfPartnersOfA[0]-centerOfB[0], centerOfPartnersOfA[1]-centerOfB[1], centerOfPartnersOfA[2]-centerOfB[2]};
	Vector pdbVec2N = pp.newPdbByTranslation(pdbVecB, cT); //the new coordinate for fragment B after the fitting
	double cosAngle, disRms = 0.0, angleRms = 0.0;
	double[][] coordOfBN  = new double[noOfAtoms][3]; 

	for (i=0; i<coordOfB.length; i++){
	    coordOfBN[i] = new double[]{coordOfB[i][0]+cT[0], coordOfB[i][1]+cT[1], coordOfB[i][2]+cT[2]};
 	    x2 = coordOfBN[i][0];   
 	    y2 = coordOfBN[i][1];
 	    z2 = coordOfBN[i][2];
   	    x1 = coordOfA[i][0];   
   	    y1 = coordOfA[i][1];
   	    z1 = coordOfA[i][2];
  	    disHBond =  Math.sqrt((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1) + (z2-z1)*(z2-z1));
	    disRms += (disHBond - Const.dN2COHBond) * (disHBond - Const.dN2COHBond);  //deviation from the ideal value
	    cosAngle = interAngle(dirsOfA[i], dirsOfB[i]);
	    angleRms += (cosAngle + 1.0) * (cosAngle + 1.0); //- (-1.0) since if they are colinear
	    if (debug)
		System.out.println("Distance = "+ disHBond +"  Angle = "+cosAngle);  //distance between N and O of an H-bond
	}
	if (noOfAtoms > 0){
	    angleRms = Math.sqrt(angleRms / coordOfB.length ); 
	    disRms   = Math.sqrt(disRms / coordOfB.length ); 
	}
	if (debug)
	    System.out.println(disRms+"  "+angleRms);
	return (angleRms+disRms);
    }

  

    /**
     * compute the H-bond energy term between the two paired strands.
     * Identical to the centerFit method except that the newly paired
     * Pdb vector are returned
     * 
     * @param pdbVecA     strand A
     * @param pdbVecB     strand B, A's partner
     * @param hbVec       the vector of Hbond information
     * @param debug the debug
     * 
     * @return the translated and paired Pdb vector
     */
    public Vector centerFitVec(Vector pdbVecA, Vector pdbVecB, Vector hbVec, boolean debug){
	boolean debug2 = false;
	final int  noOfAtoms = hbVec.size(); 
	int i, j, k;
	//One pair (N->H...O or O...H<-N), their partner's coordinates are computed based on perfect H-bond.
	double[][] hPartnerOfA = new double[noOfAtoms][3]; //Coodinates of its H-bond partner,
	double[][] coordOfA    = new double[noOfAtoms][3]; //Coodinates of atoms N and O, in the order of N, O
	double[][] dirsOfA     = new double[noOfAtoms][3]; //and directional cosines

	double[][] hPartnerOfB = new double[noOfAtoms][3]; //the corresponding ones for strand B
	double[][] coordOfB    = new double[noOfAtoms][3]; 
	double[][] dirsOfB     = new double[noOfAtoms][3]; 

	Pdb pp = new Pdb();
	Cartesian cc = new Cartesian();
	Vector atomVec = new Vector();
	String atom = "";
	double [] amide = new double[3];
	double [] nh = new double[3];
	double [] n2NHVec  = new double[3];
	double [] co = new double[3];
	double [] o  = new double[3];
	double [] co2OVec = new double[3];
	int residueNo = 0;
	int residueNoA;   //the Sequence No of one of the two partners of the NOE
	int residueNoB;   //the Sequence No of Another partner of the NOE
	String atomA;     //the name of Atom A
	String atomB;     //the name of Atom B
	String type;      //the name of A
	Hbond hb = new  Hbond();
	//Extract atoms NH2O and O2HN of strandA, and compute the directions of N2H and CO2O for both strand A and B
	boolean isN = false;
	int index  =  -1;
	for (i = 0; i < noOfAtoms; i++){
	    hb = (Hbond)hbVec.elementAt(i);
	    residueNoA = hb.getResidueNoA();
	    atomA = hb.getAtomA();
	    index = Collections.binarySearch(pdbVecA, new Pdb(residueNoA), new Pdb.PdbComparator());
	    //Extract the coordinates and direction of the H-bond partners, one partner:
	    if ( index > -1 ){
		pp = (Pdb)pdbVecA.elementAt(index);
		residueNo = pp.getResidueNo();
		atomVec = pp.getAtomVec();
		if  (atomA.equals("O")){
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("C"))
			    co = cc.getXYZ();
			else if (atom.equals("O")){
			    o = cc.getXYZ();
			    coordOfA[i] = new double[]{o[0], o[1], o[2]};
			}
		    }
		    //The coordinate of the H-bond partner of the atom O 
		    hPartnerOfA[i] = new double[]{o[0] + (o[0] - co[0]) * Const.dN2COHBond / Const.dCO2O, 
						  o[1] + (o[1] - co[1]) * Const.dN2COHBond / Const.dCO2O,
						  o[2] + (o[2] - co[2]) * Const.dN2COHBond / Const.dCO2O};
		    dirsOfA[i] = new double[]{o[0] - co[0],  o[1]-co[1], o[2]-co[2]};
		    if (debug2){
			System.out.println(residueNoA);
			System.out.println("O: "+i+"  "+coordOfA[i][0]+"  "+coordOfA[i][1]+"   "+coordOfA[i][2]);
			System.out.println("A: "+i+"  "+dirsOfA[i][0]+"  "+dirsOfA[i][1]+"   "+dirsOfA[i][2]);
		    }
		} else if (atomA.equals("N")) {
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("N")){
			    amide = cc.getXYZ();
			    coordOfA[i] = new double[]{amide[0], amide[1], amide[2]};
			}else if (atom.equals("H"))
			    nh = cc.getXYZ();
		    }
		    //The coordinate of the H-bond partner of the atom N
		    hPartnerOfA[i] = new double[]{amide[0] + (nh[0] - amide[0]) * Const.dN2COHBond / Const.dN2H, 
						  amide[1] + (nh[1] - amide[1]) * Const.dN2COHBond / Const.dN2H, 
						  amide[2] + (nh[2] - amide[2]) * Const.dN2COHBond / Const.dN2H};
		    dirsOfA[i] = new double[]{nh[0]-amide[0], nh[1]-amide[1], nh[2]-amide[2]};
		    if (debug2){
			System.out.println(residueNoA);
			System.out.println("N: "+i+"  "+coordOfA[i][0]+"  "+coordOfA[i][1]+"   "+coordOfA[i][2]);
			System.out.println("A: "+i+"  "+dirsOfA[i][0]+"  "+dirsOfA[i][1]+"   "+dirsOfA[i][2]);
		    }
		}
	    }
	    //Extract the coordinates and direction of the H-bond partners, the corresponding partner:
	    residueNoB = hb.getResidueNoB();
	    atomB = hb.getAtomB();
	    index = Collections.binarySearch(pdbVecB, new Pdb(residueNoB), new Pdb.PdbComparator());
	    if ( index > -1 ){
		pp = (Pdb)pdbVecB.elementAt(index);
		residueNo = pp.getResidueNo();
		atomVec = pp.getAtomVec();
		if  (atomB.equals("O")){
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("C"))
			    co = cc.getXYZ();
			else if (atom.equals("O")){
			    o = cc.getXYZ();
			    coordOfB[i] = new double[]{o[0], o[1], o[2]};
			}
		    }
		    dirsOfB[i] = new double[]{o[0] - co[0], o[1]-co[1], o[2]-co[2]};
		    if (debug2){
			System.out.println(residueNoB);
			System.out.println("O: "+i+"  "+coordOfB[i][0]+" : "+coordOfB[i][1]+"   "+coordOfB[i][2]);
		    }
		}else if (atomB.equals("N")) {
		    for (j=0; j<atomVec.size(); j++){
			cc = (Cartesian)atomVec.elementAt(j);
			atom = cc.getAtom();
			if (atom.equals("N")){
			    amide = cc.getXYZ();
			    coordOfB[i] = new double[]{amide[0], amide[1], amide[2]};
			}else if (atom.equals("H"))
			    nh = cc.getXYZ();
		    }
		    //The coordinate of the atom O of the H-bond partner of the NH vector
		    hPartnerOfB[i] = new double[]{amide[0] + (nh[0] - amide[0]) * Const.dN2COHBond / Const.dN2H, 
						  amide[1] + (nh[1] - amide[1]) * Const.dN2COHBond / Const.dN2H, 
						  amide[2] + (nh[2] - amide[2]) * Const.dN2COHBond / Const.dN2H};
		    dirsOfB[i] = new double[]{nh[0]-amide[0], nh[1]-amide[1], nh[2]-amide[2]};
		    if (debug2){
			System.out.println(residueNoB);
			System.out.println("N: "+i+"  "+coordOfB[i][0]+" : "+coordOfB[i][1]+"   "+coordOfB[i][2]);
		    }
		}
	    }
	}
	double[] centerOfPartnersOfA = centerCal(hPartnerOfA); //the center of the coords of the H-bond partners of A.
	double x1, y1, z1, x2, y2, z2;  //for compute the H-bond distances.
	double disHBond;

	double[] centerOfB = centerCal(coordOfB); //the center of the coords of B.
	double[] cT1={centerOfPartnersOfA[0]-centerOfB[0], centerOfPartnersOfA[1]-centerOfB[1], centerOfPartnersOfA[2]-centerOfB[2]};
	double cosAngle = 0.0, disRms = 0.0, angleRms = 0.0;
	double[][] coordOfBN  = new double[noOfAtoms][3]; 
	double [] cT = new double[3];
	double [] cTSave = new double[3];
	long seed = 9752864; //419057923;
	Random rr = new Random(seed);
	int nCycle = 1;//4096;
	double rT = 0, rTotal = 100.0;
	double 	disRmsS = 0.0;
	double 	angleRmsS = 0.0;
	double range = 0.0;
	for (int mm = 0; mm< nCycle; mm++){
	    disRms = 0.0;
	    angleRms = 0.0;
	    cT[0] = cT1[0]+  range * (0.50-rr.nextDouble());
	    cT[1] = cT1[1]+  range * (0.50-rr.nextDouble());
	    cT[2] = cT1[2]+  range * (0.50-rr.nextDouble());
	    for (i=0; i<coordOfB.length; i++){
		coordOfBN[i] = new double[]{coordOfB[i][0]+cT[0], coordOfB[i][1]+cT[1], coordOfB[i][2]+cT[2]};
		x2 = coordOfBN[i][0];   
		y2 = coordOfBN[i][1];
		z2 = coordOfBN[i][2];
		x1 = coordOfA[i][0];   
		y1 = coordOfA[i][1];
		z1 = coordOfA[i][2];
		disHBond =  Math.sqrt((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1) + (z2-z1)*(z2-z1));
		disRms += (disHBond - Const.dN2COHBond) * (disHBond - Const.dN2COHBond);  //deviation from the ideal value
		cosAngle = interAngle(dirsOfA[i], dirsOfB[i]);
		angleRms += (cosAngle + 1.0) * (cosAngle + 1.0); //- (-1.0) since if they are colinear
	    }
	    rT = disRms;
	    if (rT < rTotal){
		if (debug)
		    System.out.println("DisRms= "+Math.sqrt(disRms/coordOfB.length )+"; AngleRms="+Math.sqrt(angleRms/coordOfB.length));
		rTotal = rT;
		System.arraycopy(cT, 0, cTSave, 0, 3);
		disRmsS = disRms;
		angleRmsS = angleRms;
	    }
	}

	for (i=0; i<coordOfB.length; i++){
	    coordOfBN[i] = new double[]{coordOfB[i][0]+cTSave[0], coordOfB[i][1]+cTSave[1], coordOfB[i][2]+cTSave[2]};
 	    x2 = coordOfBN[i][0];   
 	    y2 = coordOfBN[i][1];
 	    z2 = coordOfBN[i][2];
   	    x1 = coordOfA[i][0];   
   	    y1 = coordOfA[i][1];
   	    z1 = coordOfA[i][2];
  	    disHBond =  Math.sqrt((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1) + (z2-z1)*(z2-z1));
	    disRms += (disHBond - Const.dN2COHBond) * (disHBond - Const.dN2COHBond);  //deviation from the ideal value
	    cosAngle = interAngle(dirsOfA[i], dirsOfB[i]);
	    angleRms += (cosAngle + 1.0) * (cosAngle + 1.0); //- (-1.0) since if they are colinear
	    if (debug)
		System.out.println("Dis= "+ disHBond +"  Angle = "+cosAngle);  //distance between N and O of an H-bond
	}
	if (noOfAtoms > 0){
	    angleRms = Math.sqrt(angleRmsS / coordOfB.length ); 
	    disRms   = Math.sqrt(disRmsS / coordOfB.length ); 
	}
	Vector pdbVec2N = pp.newPdbByTranslation(pdbVecB, cTSave); //the new coordinate for fragment B after the fitting
	if (debug)
	    System.out.println(disRms+"  "+angleRms);
	return pdbVec2N;
    }

  
    /**
     * Note: all rotamers are put in both structures.
     * Compute the new center of atoms involved in NOE distances.
     * pdbA and pdbB are the original coordinates noeDistance: the noe
     * distances between the two fragments. format (low, upper).
     * arrange the three arrays with the same order so that the
     * distance i is between nuclei i of A and i of B.
     * 
     * @param noeVec the NOE distance restraints between the fragment A and B
     * @param pdbVecA one fragment
     * @param pdbVecB another fragment
     * @param noeRms saving the min NOE rmsd
     * @param debugNOE if true it will print out the NOE fitting Rmsds
     * @param vecUpdateNOE the vec update noe
     * 
     * @return the Pdb vector for translated fragment which is B while
     * Pdb vector for A is NOT changed
     */
    public Vector positionByNOENew(final Vector noeVec, Vector pdbVecA, Vector pdbVecB, 
    		double[] noeRms, boolean debugNOE, Vector vecUpdateNOE)
    {
       	int i, j;
    	Vector atomVec = new Vector();
    	Cartesian cc = new Cartesian();
    	String atom  = "";
    	Pdb pp=new Pdb();
    	
    	double [][] newArrB = new double[pdbVecB.size()][3];        	
    	
    	int noA, noB;
    	int index = -1;    	
    	String atomA = "";
    	String atomB = "";
    	int sizeA = 0;
    	int sizeB = 0;
    	double[][] pdbA2 = new double[noeVec.size()][3];
    	double[][] pdbB2 = new double[noeVec.size()][3];
    	Vector id4A = new Vector();
    	Vector id4B = new Vector();
    	String id = "";  //Combining the residueNO and Atom name for easy comparion.
    	double [][] noeDistance = new double[noeVec.size()][2];
    	
    	for (i=0; i<noeVec.size(); i++)
    	{
    	    noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
    	    
    	    id = String.valueOf(noA)+"CA";
    	    index = Collections.binarySearch(pdbVecA, new Pdb(noA), new Pdb.PdbComparator() );
    	    if ( index > -1 )
    	    {
    	    	pp = (Pdb)pdbVecA.elementAt(index);
    	    	atomVec = pp.getAtomVec();
    	    	for (j=0; j<atomVec.size(); j++)
    	    	{
    			    cc = (Cartesian)atomVec.elementAt(j);
    			    atom = cc.getAtom();
    			    if ( atom.equalsIgnoreCase("CA") && !id4A.contains(id) ) 
    			    {  
    			    	pdbA2[sizeA] = cc.getXYZ();	
    			    	id4A.add(id);
    			    	sizeA++;
    			    }
    	    	}
    	    }//if ( index > -1 )
    	    noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
    	
    	    id = String.valueOf(noB)+"CA";
    	    index = Collections.binarySearch(pdbVecB, new Pdb(noB), new Pdb.PdbComparator() );
    	    if ( index > -1 )
    	    {
    			pp = (Pdb)pdbVecB.elementAt(index);
    			atomVec = pp.getAtomVec();
    			for (j=0; j<atomVec.size(); j++)
    			{
    			    cc = (Cartesian)atomVec.elementAt(j);
    			    atom = cc.getAtom();
    			  
    			    if ( atom.equalsIgnoreCase("CA") && !id4B.contains(id) ) 
    			    {
    					pdbB2[sizeB] = cc.getXYZ();	
    					id4B.add(id);
    					sizeB++;
    			    }
    			}
    	    }//if ( index > -1 )
    	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
    	}

    
    	// Compute the center    	  	
    	double xC1 = 0.0, yC1 = 0.0, zC1 = 0.0, xC2 = 0.0, yC2 = 0.0, zC2 = 0.0; 
    	
    	for (i=0; i<pdbA2.length; i++)
    	{
    	    xC1 += pdbA2[i][0];
    	    yC1 += pdbA2[i][1];
    	    zC1 += pdbA2[i][2];
    	}
    	if (pdbA2.length > 0)
    	{ // Compute the center
     	    xC1 /=sizeA;
     	    yC1 /= sizeA;
     	    zC1 /= sizeA;
    	}
    	for (i=0; i<pdbB2.length; i++)
    	{
    	    xC2 += pdbB2[i][0]; 
    	    yC2 += pdbB2[i][1];
    	    zC2 += pdbB2[i][2];
    	}
    	if (pdbB2.length> 0)
    	{ // Compute the center
     	    xC2 /= sizeB;
     	    yC2 /= sizeB;
     	    zC2 /=sizeB;
    	}
    	
    	double lengthR = 0.0;
    	double [] trans_vecA = new double[3];
    	double [] trans_vecB = new double[3];
    	
    	double [] centerA = {xC1, yC1, zC1};
    	double [] centerB = {xC2, yC2, zC2};
    	double [] centerO={0.0, 0.0, 0.0};
    
    	trans_vecA= pp.internuclearVec(centerA,centerO);	
    	trans_vecB= pp.internuclearVec(centerB,centerO);
    	//move centers of both pdbs to the origin
    	Vector<Pdb> pdbVecNewA=pp.newPdbByTranslation(pdbVecA,trans_vecA);
    	Vector<Pdb> pdbVecNewB=pp.newPdbByTranslation(pdbVecB,trans_vecB);
    	
    	double x1, y1, z1, x2, y2, z2;
    	double ranTheta, ranPhi,  ranThetaS = 0.0, ranPhiS = 0.0;
    	
    	double low = 0.0, upper = 0.0;
    	int resolution = 2;
    	int radiusBound =40;
    
     	double lenS = 0.0;
     
     	double dbScore=0.0;
     	Assign asg=new Assign();
     	double maxSc=-999999.9;
    
     	Vector vecSSEPack=new Vector();
     
     	double disRms = 0.0;
     	int mm=noeVec.size();
     	double disRmsd = 1000.0;
    	double[] noeDis = new double[mm];
     	for (int m = 0; m < (180/resolution); m++)
    	{
    	    ranTheta = m * resolution * Math.PI / 180.0;
    	    for (j=0; j< (360 / resolution); j++)
    	    {
    	    	ranPhi = j * resolution * Math.PI / 360.0;
    	    	for (int k =radiusBound ; k> 0; k--)
    	    	{
    	    		lengthR  =2.0 + k *0.2;    	    		
    	    		double[] centerBNew = { lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
    	 			      lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
    	 			      lengthR * Math.cos(ranTheta)};
    	    		double[] trans_vec = new double[3];
    	    		trans_vec=pp.internuclearVec(centerO,centerBNew);
    	    		
    	    		Vector pdbVecNewB2=pp.newPdbByTranslation(pdbVecNewB,trans_vec);
    	 	    	
    			    disRms = 0.0;
    			    for (i = 0; i < noeVec.size(); i++)
    			    {
    			    	noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
    			    	atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
    			    	
    			    	noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
    			 	    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
    			 	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
    					
    			 	    double [] dis=new double[1];
    			 	    double [] distance =new double[1];
    			 	 	pp.measurePackDisAllRotamers(pdbVecNewA,pdbVecNewB2,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
    			 	 	noeDis[i]=dis[0];
    			 	 	disRms += noeDis[i] * noeDis[i];
    			    }
    			    disRms = Math.sqrt(disRms / mm);
    			    
    			    if (Math.abs(disRms - 0.0) < Const.eps)
    			    {
    			    	System.out.println("here we have one packing satisfying noe constraints:");
    			    	System.out.println(disRms+" ... "+ranTheta+"  "+ranPhi+"  "+lengthR); //Check the progress
    	    			
    			    	double[] centerBNewTemp=new double [3];
    			    	centerBNewTemp =new double[] {lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
    			    			lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
    			    			lengthR * Math.cos(ranTheta)};
    			    	trans_vecB= pp.internuclearVec(centerO,centerBNewTemp);
    			    	Vector<Pdb> pdbVecNewBTemp=pp.newPdbByTranslation(pdbVecNewB,trans_vecB);
    			    	Vector ppVecT = new Vector();
    			    	ppVecT.addAll(pdbVecNewA);
    			    	ppVecT.addAll(pdbVecNewBTemp);
    			    	Collections.sort(ppVecT, new Pdb.PdbComparator());
    			    	pp.print(ppVecT);
    			    	System.out.println("TER");
    		         	System.out.println("END");    
    			    
    			    }
    			    
    			    if ( disRms < disRmsd )
    			    {
    					ranThetaS = ranTheta;
    					ranPhiS =  ranPhi;
    					disRmsd = disRms;
    					lenS = lengthR;
    					System.out.println(disRmsd+" ... "+ranThetaS+"  "+ranPhiS+"  "+lenS); //Check the progress
    			    }     	    		
    		    	
    	    	}//for (int k =0 ; k< radiusBound; k++)
    	    }//for (j=0; j< (360 / resolution); j++)
    	}//for (int m = 0; m < (180/resolution); m++) 
   
    	vecSSEPack.add(new SSEPacking(ranThetaS,ranPhiS,lenS,disRmsd ));
    	
        Collections.sort(vecSSEPack, new SSEPacking.SSEPackingComparator());   
    	
        double[] centerBNew2=new double [3];
        Vector vecPackEnsemble=new Vector();
        for(i=0;i<vecSSEPack.size();i++)
        {
			SSEPacking ssePack=(SSEPacking)vecSSEPack.elementAt(i);
			ranThetaS=ssePack.getTheta();
			ranPhiS=ssePack.getPhi();
			lenS=ssePack.getLengthR();
			
			centerBNew2 =new double[] {lenS * Math.sin(ranThetaS) * Math.cos(ranPhiS),
				        lenS * Math.sin(ranThetaS) * Math.sin(ranPhiS),
				       lenS * Math.cos(ranThetaS)};
			trans_vecB= pp.internuclearVec(centerO,centerBNew2);
			Vector<Pdb> pdbVecNewBFinal=pp.newPdbByTranslation(pdbVecNewB,trans_vecB);
			Vector ppVecT = new Vector();
			ppVecT.addAll(pdbVecNewA);
			ppVecT.addAll(pdbVecNewBFinal);
			Collections.sort(ppVecT, new Pdb.PdbComparator());
			vecPackEnsemble.add(ppVecT);
			 
		
        }//for(i=0;i<vecSSEPack.size();i++)    	
   
        noeRms[0]=((SSEPacking)vecSSEPack.elementAt(0)).getScore();
        return vecPackEnsemble;
    }
    
    /**
     * use heavy atom to replace pseduo protons
     * Compute the new center of atoms involved in NOE distances.
     * pdbA and pdbB are the original coordinates noeDistance: the noe
     * distances between the two fragments. format (low, upper).
     * arrange the three arrays with the same order so that the
     * distance i is between nuclei i of A and i of B.
     * 
     * @param noeVec the NOE distance restraints between the fragment A and B
     * @param pdbVecA one fragment
     * @param pdbVecB another fragment
     * @param noeRms saving the NOE rmsd
     * @param debugNOE if true it will print out the NOE fitting Rmsds
     * @param vecUpdateNOE the vec update noe
     * 
     * @return the Pdb vector for translated fragment which is B while
     * Pdb vector for A is NOT changed
     */
    public Vector positionByNOE_old(final Vector noeVec, Vector pdbVecA, Vector pdbVecB, 
    		double[] noeRms, boolean debugNOE, Vector vecUpdateNOE){
	//Extract neccessary information from the NOE table.
	int i, j;
	int noA, noB;
	int index = -1;
	String atom  = "";
	String atomA = "";
	String atomB = "";
	Vector atomVec = new Vector();
	Cartesian cc = new Cartesian();
	Pdb pp = new Pdb();
	int sizeA = 0;
	int sizeB = 0;
	double[][] pdbA2 = new double[noeVec.size()][3];
	double[][] pdbB2 = new double[noeVec.size()][3];
	Vector id4A = new Vector();
	Vector id4B = new Vector();
	String id = "";  //Combining the residueNO and Atom name for easy comparion.
	double[][] pdbA = new double[noeVec.size()][3];
	double[][] pdbB = new double[noeVec.size()][3];
	double [][] noeDistance = new double[noeVec.size()][2];
	for (i=0; i<noeVec.size(); i++)
	{
	    noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
	    atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
	    id = String.valueOf(noA)+atomA;
	    index = Collections.binarySearch(pdbVecA, new Pdb(noA), new Pdb.PdbComparator() );
	    if ( index > -1 )
	    {
	    	pp = (Pdb)pdbVecA.elementAt(index);
	    	atomVec = pp.getAtomVec();
	    	for (j=0; j<atomVec.size(); j++)
	    	{
			    cc = (Cartesian)atomVec.elementAt(j);
			    atom = cc.getAtom();
			    if ( atom.equalsIgnoreCase(atomA) )
				pdbA[i] = cc.getXYZ();
			    //for deleting repeatation in NOEs when computing the centers 
			    if ( atom.equalsIgnoreCase(atomA) && !id4A.contains(id) ) 
			    {  
			    	pdbA2[sizeA] = cc.getXYZ();	
			    	id4A.add(id);
			    	sizeA++;
			    }
	    	}
	    }//if ( index > -1 )
	    noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
	    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
	    id = String.valueOf(noB)+atomB;
	    index = Collections.binarySearch(pdbVecB, new Pdb(noB), new Pdb.PdbComparator() );
	    if ( index > -1 )
	    {
			pp = (Pdb)pdbVecB.elementAt(index);
			atomVec = pp.getAtomVec();
			for (j=0; j<atomVec.size(); j++)
			{
			    cc = (Cartesian)atomVec.elementAt(j);
			    atom = cc.getAtom();
			    if (atom.equalsIgnoreCase(atomB))
				pdbB[i] = cc.getXYZ();
			    if ( atom.equalsIgnoreCase(atomB) && !id4B.contains(id) ) 
			    {
					pdbB2[sizeB] = cc.getXYZ();	
					id4B.add(id);
					sizeB++;
			    }
			}
	    }//if ( index > -1 )
	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
	}
	// Compute the center
	final int n = 3;
	int mm = pdbA.length;   
	double xC1 = 0.0, yC1 = 0.0, zC1 = 0.0, xC2 = 0.0, yC2 = 0.0, zC2 = 0.0; 
	double noeAve    = 0.0;
	for (i=0; i<sizeA; i++){
	    xC1 += pdbA2[i][0];
	    yC1 += pdbA2[i][1];
	    zC1 += pdbA2[i][2];
	}
	if (sizeA > 0){ // Compute the center
 	    xC1 /= sizeA;
 	    yC1 /= sizeA;
 	    zC1 /= sizeA;
	}
	for (i=0; i<sizeB; i++){
	    xC2 += pdbB2[i][0]; 
	    yC2 += pdbB2[i][1];
	    zC2 += pdbB2[i][2];
	}
	if (sizeB > 0){ // Compute the center
 	    xC2 /= sizeB;
 	    yC2 /= sizeB;
 	    zC2 /= sizeB;
	}
	//compute average NOEs
	for (i=0; i<noeVec.size(); i++)
	    noeAve += (noeDistance[i][0] + noeDistance[i][1]) / 2.0;
	//noeAve /= noeVec.size();
	noeAve /= mm;
	
	double lengthR = noeAve;
	double [][] arrA = new double[mm][n];
	double [][] arrB = new double[mm][n];
	double [][] newArrB = new double[mm][n];
	double [] centerA = {xC1, yC1, zC1};
 	double diffCenter = Math.sqrt((xC1-xC2)* (xC1-xC2)+(yC1-yC2) * (yC1-yC2)+(zC1-zC2) * (zC1-zC2)); 	
	double disRms = 0.0;
	double[] noeDis = new double[mm];
	for (i=0; i<mm; i++)
	{  //move both centers to the origin
	    arrA[i][0] = pdbA[i][0] - xC1; 
	    arrA[i][1] = pdbA[i][1] - yC1;
	    arrA[i][2] = pdbA[i][2] - zC1;
	    arrB[i][0] = pdbB[i][0] - xC2; 
	    arrB[i][1] = pdbB[i][1] - yC2;
	    arrB[i][2] = pdbB[i][2] - zC2; 
	    noeDis[i]  = Math.sqrt(  (arrA[i][0] - arrB[i][0]) * (arrA[i][0] - arrB[i][0]) 
			       + (arrA[i][1] - arrB[i][1]) * (arrA[i][1] - arrB[i][1]) 
			       + (arrA[i][2] - arrB[i][2]) * (arrA[i][2] - arrB[i][2]));
	}
	
 	double x1, y1, z1, x2, y2, z2;
	double ranTheta, ranPhi,  ranThetaS = 0.0, ranPhiS = 0.0;
	double disRmsd = 1000.0;
	long seed = 9752864; //419057923;
	Random rr = new Random(seed);
	double low = 0.0, upper = 0.0;
	int resolution = 5;//changed here....
	int noeRangeRes = 100;//100;
 	double lenS = 0.0;
 	double [] trans_vecA = new double[3];
	double [] trans_vecB = new double[3];
	String userDir = System.getProperty("user.dir");////
	String src=userDir+"/inputFiles/";
	String fileName =src+ "packing.pdb";
	
	try{
		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
		
		
	for (int m = 0; m < (180/resolution); m++)
	{
	    ranTheta = m * resolution * Math.PI / 180.0;
	    for (j=0; j< (360 / resolution); j++)
	    {
	    	ranPhi = j * resolution * Math.PI / 360.0;
	    	//for (int k =0 ; k< noeRangeRes; k++)
	    	//{
	    	for (int k =noeRangeRes ; k>=0 ; k--)
		    {
			    lengthR  = noeAve + 6.0*(0.50-rr.nextDouble());
			    //lengthR  = 2.0 + 0.2*k;
			    //move the center of B to a new position on the upper sphere, recompute B'coordinates
			    disRms = 0.0;
			    for (i = 0; i < mm; i++)
			    {
					newArrB[i][0] = arrB[i][0] + lengthR * Math.sin(ranTheta) * Math.cos(ranPhi); 
					newArrB[i][1] = arrB[i][1] + lengthR * Math.sin(ranTheta) * Math.sin(ranPhi);
					newArrB[i][2] = arrB[i][2] + lengthR * Math.cos(ranTheta);
					noeDis[i]  = Math.sqrt((arrA[i][0] - newArrB[i][0]) * (arrA[i][0] - newArrB[i][0]) 
							       +(arrA[i][1] - newArrB[i][1]) * (arrA[i][1] -newArrB[i][1]) 
							       +(arrA[i][2] - newArrB[i][2]) * (arrA[i][2] -newArrB[i][2]));
					if (noeDis[i] > noeDistance[i][0] && noeDis[i] < noeDistance[i][1])
					    disRms += 0.0;
					else if(noeDis[i] <= noeDistance[i][0] )
					    disRms += noeDistance[i][0] - noeDis[i];
					else if(noeDis[i] >= noeDistance[i][1])
					    disRms += noeDis[i] - noeDistance[i][1];
			    }
			    disRms = Math.sqrt(disRms / mm);
			    
			    if (Math.abs(disRms - 0.0) < Const.eps)
			    {
			    	
			    	System.out.println("here we have one packing satisfying noe constraints:");
			    	System.out.println(disRms+" ... "+ranTheta+"  "+ranPhi+"  "+lengthR); //Check the progress
	    			 //vecSSEPack.add(new SSEPacking(ranTheta,ranPhi,lengthR,disRms ));
			    	double[] centerBNewTemp=new double [3];
			    	centerBNewTemp =new double[] {-xC2+lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
			    			-yC2 +lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
			    			-zC2 + lengthR * Math.cos(ranTheta)};
			    	//trans_vecB= pp.internuclearVec(centerO,centerBNewTemp);
			    	Vector<Pdb> pdbVecNewBTemp=pp.newPdbByTranslation(pdbVecB,centerBNewTemp);
			    	Vector ppVecT = new Vector();
			    	double [] centerANewTemp = {-centerA[0], -centerA[1], -centerA[2]};
			    	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANewTemp);
			    	
			    	ppVecT.addAll(pdbANew);
			    	ppVecT.addAll(pdbVecNewBTemp);
			    	Collections.sort(ppVecT, new Pdb.PdbComparator());
			    	
			    	pp.print(ppVecT);
			    	out.println("MODEL"+m+j+k);///to file
			    	pp.printToFile(ppVecT,fileName, out);
			    	System.out.println("TER");
		         	System.out.println("END");   
		         	
		            out.println("TER");
		         	out.println("ENDMDL");   
		         	
			    
			
			    
			    }
			    if ( disRms < disRmsd  )
			    {
					ranThetaS = ranTheta;
					ranPhiS =  ranPhi;
					disRmsd = disRms;
					lenS = lengthR;
					//System.out.println(disRmsd+" ... "+ranThetaS+"  "+ranPhiS+"  "+lenS); //Check the progress
			    }
	    	}//for (int k =0 ; k< noeRangeRes; k++)
	    }//for (j=0; j< (360 / resolution); j++)
	}//for (int m = 0; m < (180/resolution); m++)
	
	out.println("END");  
	out.close();
	}catch (FileNotFoundException e)
	{
		System.out.println("File not found: " + fileName);
	}catch (IOException e)
	{
	   System.out.println("IOException: the stack trace is:");
	   e.printStackTrace();
	}
	
	
	
	
	noeRms[0] = disRmsd;
	if (debugNOE)
	{
	    System.out.println("NOE rmsd = "+disRmsd);
	    System.out.println("NOE distances difference for individual ones:");
	}
	double[] centerBNew = {-xC2 + lenS * Math.sin(ranThetaS) * Math.cos(ranPhiS),
			       -yC2 + lenS * Math.sin(ranThetaS) * Math.sin(ranPhiS),
			       -zC2 + lenS * Math.cos(ranThetaS)};
	for (i=0; i<mm; i++)
	{  //move both centers to the origin
	    arrB[i][0] = pdbB[i][0] + centerBNew[0]; 
	    arrB[i][1] = pdbB[i][1] + centerBNew[1];
	    arrB[i][2] = pdbB[i][2] + centerBNew[2];
	    noeDis[i]  = Math.sqrt( (arrA[i][0] - arrB[i][0]) * (arrA[i][0] - arrB[i][0]) 
				   +(arrA[i][1] - arrB[i][1]) * (arrA[i][1] - arrB[i][1]) 
				   +(arrA[i][2] - arrB[i][2]) * (arrA[i][2] - arrB[i][2]));
	    if (debugNOE)  	    //Print the NOE distance after the grid-search.
		System.out.println(noeDis[i]);
	    
	    //we added some lines here to filter "wrong" NOEs
	   // if(noeDis[i]<8.0)
	    	//vecUpdateNOE.add(noeVec.elementAt(i));
	}
	Vector pdbBNew   = pp.newPdbByTranslation(pdbVecB, centerBNew);
	double [] centerANew = {-centerA[0], -centerA[1], -centerA[2]};
	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANew);
	//Print the new PDBs
	Vector ppVecT = new Vector();
	ppVecT.addAll(pdbANew);
	ppVecT.addAll(pdbBNew);
	Collections.sort(ppVecT, new Pdb.PdbComparator());
	return ppVecT;
    }
    
    
    /**
     * use exact name for pseduo protons
     * Compute the new center of atoms involved in NOE distances.
     * pdbA and pdbB are the original coordinates noeDistance: the noe
     * distances between the two fragments. format (low, upper).
     * arrange the three arrays with the same order so that the
     * distance i is between nuclei i of A and i of B.
     * 
     * @param noeVec the NOE distance restraints between the fragment A and B
     * @param pdbVecA one fragment
     * @param pdbVecB another fragment
     * @param noeRms saving the NOE rmsd
     * @param debugNOE if true it will print out the NOE fitting Rmsds
     * @param vecUpdateNOE the vec update noe
     * 
     * @return the Pdb vector for translated fragment which is B while
     * Pdb vector for A is NOT changed
     */
    public Vector positionByNOE(final Vector noeVec, Vector pdbVecA, Vector pdbVecB, 
    		double[] noeRms, boolean debugNOE, Vector vecUpdateNOE)
    {
		//Extract neccessary information from the NOE table.
		int i, j;
		int noA, noB;
		int index = -1;
		String atom  = "";
		String atomA = "";
		String atomB = "";
		Vector atomVec = new Vector();
		Cartesian cc = new Cartesian();
		Pdb pp = new Pdb();
		int sizeA = 0;
		int sizeB = 0;
		double[][] pdbA2 = new double[noeVec.size()*6][3];
		double[][] pdbB2 = new double[noeVec.size()*6][3];
		Vector id4A = new Vector();
		Vector id4B = new Vector();
		String id = "";  //Combining the residueNO and Atom name for easy comparion.
		//double[][] pdbA = new double[noeVec.size()][3];
		//double[][] pdbB = new double[noeVec.size()][3];
		double [][] noeDistance = new double[noeVec.size()][2];	
	
		//compute the centers of atom coordinates involed in NOEs
		boolean isQ=true;
		for (i=0; i<noeVec.size(); i++)
		{
		    noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
		    atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
		 
		    ////////
		    String tempA="";
		    isQ=true;
		    if(atomA.length()>=2)
		    {
			    if(atomA.substring(0,1).equalsIgnoreCase("Q") && (!atomA.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempA="H"+atomA.substring(1,atomA.length());
			    else if(atomA.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempA="H"+atomA.substring(2,atomA.length());
			    else 
			    {
			    	tempA=atomA;	
			    	isQ=false;
			    }
		    }//if(atomA.length()>=2)
		    else 
		    {
		    	tempA=atomA;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecA, new Pdb(noA), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
		    	pp = (Pdb)pdbVecA.elementAt(index);
		    	atomVec = pp.getAtomVec();
		    	for (j=0; j<atomVec.size(); j++)
		    	{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    
				    if(atom.length()>=tempA.length())
				    {
				    	String debug_str=atom.substring(0,tempA.length() );
				    	//if (atom.substring(0,tempA.length() ).equalsIgnoreCase(tempA) )
				    		//pdbA[i] = cc.getXYZ();
				    	id = String.valueOf(noA)+atom;
					    //for deleting repeatation in NOEs when computing the centers 
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp=atom.substring(0,tempA.length() );
				    	else
				    		atom_temp=atom;
				    	
					    if ( atom_temp.equalsIgnoreCase(tempA) && !id4A.contains(id) ) 
					    {  
					    	pdbA2[sizeA] = cc.getXYZ();	
					    	id4A.add(id);
					    	sizeA++;
					    }
				    }// if(atom.length()>=tempA.length())
				    
				    
		    	}//for (j=0; j<atomVec.size(); j++)
		    }//if ( index > -1 )
		    
		    noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
		    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
		    //////	//
		    String tempB="";
		    isQ=true;
		    if(atomB.length()>=2)
		    {
			    if(atomB.substring(0,1).equalsIgnoreCase("Q") && (!atomB.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempB="H"+atomB.substring(1,atomB.length());
			    else if(atomB.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempB="H"+atomB.substring(2,atomB.length());
			    else 
			    {
			    	tempB=atomB;  
			    	isQ=false;
			    }
		    }//if(atomB.length()>=2)
		    else
		    {
		    	tempB=atomB;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecB, new Pdb(noB), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
				pp = (Pdb)pdbVecB.elementAt(index);
				atomVec = pp.getAtomVec();
				for (j=0; j<atomVec.size(); j++)
				{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    if (atom.length()>=tempB.length())
				    {
				    	id = String.valueOf(noB)+atom;
				    
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp= atom.substring(0,tempB.length());
				    	else
				    		atom_temp=atom;
					    if (atom_temp.equalsIgnoreCase(tempB) && !id4B.contains(id) ) 
					    {
							pdbB2[sizeB] = cc.getXYZ();	
							id4B.add(id);
							sizeB++;
					    }
				    }//if (atom.length()>=tempB.length())
				    
				}
		    }//if ( index > -1 )
		    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
		}
	// Compute the center
	final int n = 3;
	//int mm = pdbA.length;   
	double xC1 = 0.0, yC1 = 0.0, zC1 = 0.0, xC2 = 0.0, yC2 = 0.0, zC2 = 0.0; 
	double noeAve    = 0.0;
	for (i=0; i<sizeA; i++)
	{
	    xC1 += pdbA2[i][0];
	    yC1 += pdbA2[i][1];
	    zC1 += pdbA2[i][2];
	}
	if (sizeA > 0)
	{ // Compute the center
 	    xC1 /= sizeA;
 	    yC1 /= sizeA;
 	    zC1 /= sizeA;
	}
	for (i=0; i<sizeB; i++)
	{
	    xC2 += pdbB2[i][0]; 
	    yC2 += pdbB2[i][1];
	    zC2 += pdbB2[i][2];
	}
	if (sizeB > 0)
	{ // Compute the center
 	    xC2 /= sizeB;
 	    yC2 /= sizeB;
 	    zC2 /= sizeB;
	}
	//compute average NOEs
	for (i=0; i<noeVec.size(); i++)
	    noeAve += (noeDistance[i][0] + noeDistance[i][1]) / 2.0;
	noeAve /= noeVec.size();
	//noeAve /= mm;
	
	double lengthR = noeAve;
	
 	double diffCenter = Math.sqrt((xC1-xC2)* (xC1-xC2)+(yC1-yC2) * (yC1-yC2)+(zC1-zC2) * (zC1-zC2)); 	
	double disRms = 0.0;
	double noeHarm=0.0;
	double[] noeDis = new double[noeVec.size()];
	double[] noeBackDis = new double[noeVec.size()];
	double[] dis_save = new double[noeVec.size()];
	
	
	double [] trans_vecA = new double[3];
	double [] trans_vecB = new double[3];
	
	double [] centerA = {xC1, yC1, zC1};
	double [] centerB = {xC2, yC2, zC2};
	double [] centerO={0.0, 0.0, 0.0};
	
	trans_vecA= pp.internuclearVec(centerA,centerO);	
	trans_vecB= pp.internuclearVec(centerB,centerO);
	//	move centers of both pdbs to the origin
	Vector<Pdb> pdbVecNewA=pp.newPdbByTranslation(pdbVecA,trans_vecA);
	Vector<Pdb> pdbVecNewB=pp.newPdbByTranslation(pdbVecB,trans_vecB);
		
 	double x1, y1, z1, x2, y2, z2;
	double ranTheta, ranPhi,  ranThetaS = 0.0, ranPhiS = 0.0;
	double disRmsd = 1000.0;
	long seed = 9754; //419057923;
	Random rr = new Random(seed);
	double low = 0.0, upper = 0.0;
	int resolution = 5;//changed here....
	int noeRangeRes = 20;
 	double lenS = 0.0;
 
	String userDir = System.getProperty("user.dir");
	String src=userDir+"/inputFiles/";
	String fileName =src+ "packing0.pdb";
	int counter=0;
	int nTemp=0;
	Vector vecEnsemblePdbs=new Vector();
	
	vdw vander = new vdw();
	Vector vdwVec = new Vector();
	
	try{
		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
		
		
	for (int m = 0; m < (180/resolution); m++)
	{
	    ranTheta = m * resolution * Math.PI / 180.0;
	    for (j=0; j< (360 / resolution); j++)
	    {
	    	ranPhi = j * resolution * Math.PI / 360.0;
	    	
	    	for (int k =noeRangeRes ; k>=0 ; k--)
		    {
			    lengthR  = 0.0 + 0.5*k;
			    //move the center of B to a new position on the upper sphere, recompute B'coordinates
			    double[] centerBNew = { lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
  	 			      lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
  	 			      lengthR * Math.cos(ranTheta)};
			    double[] trans_vec = new double[3];
	    		trans_vec=pp.internuclearVec(centerO,centerBNew);
	    		
	    		Vector pdbVecNewB2=pp.newPdbByTranslation(pdbVecNewB,trans_vec);
			    
			    disRms = 0.0;
			    noeHarm=0.0;
			   
			    for (i = 0; i < noeVec.size(); i++)
			    {
			    	noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
			    	atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
			    	
			    	noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
			 	    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
			 	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
					
			 	    double [] dis=new double[1];
			 	    double [] distance=new double[1];
			 	 	pp.measurePackDisAllRotamers(pdbVecNewA,pdbVecNewB2,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
			 	 	noeDis[i]=dis[0];
			 	 	noeBackDis[i]=distance[0];
			 	 	disRms += noeDis[i] * noeDis[i];
			 	 	noeHarm+=noeDis[i] * noeDis[i];
			    }
			    noeHarm=noeHarm/noeVec.size();		    
			    disRms = Math.sqrt(disRms / noeVec.size());
			   
			 
			    if ( disRms  < 1.0 )
			    {
			    	counter+=1;
			    	if ( (counter/40)>nTemp )
			    	{
			    		out.close();
			    		fileName =src+ "packing"+ (int)(counter/40)+".pdb";
			    		out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
			    	}//if ( (counter/150)>counter )
			    	nTemp=counter/40;
			    	
			    	System.out.println("here we have one packing satisfying noe constraints:");
			    	System.out.println(disRms+" ... "+ranTheta+"  "+ranPhi+"  "+lengthR); //Check the progress
	    			 //vecSSEPack.add(new SSEPacking(ranTheta,ranPhi,lengthR,disRms ));
			    	double[] centerBNewTemp=new double [3];
			    	centerBNewTemp =new double[] {-xC2+lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
			    			-yC2 +lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
			    			-zC2 + lengthR * Math.cos(ranTheta)};
			    	//trans_vecB= pp.internuclearVec(centerO,centerBNewTemp);
			    	Vector<Pdb> pdbVecNewBTemp=pp.newPdbByTranslation(pdbVecB,centerBNewTemp);
			    	Vector ppVecT = new Vector();
			    	double [] centerANewTemp = {-centerA[0], -centerA[1], -centerA[2]};
			    	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANewTemp);
			    	
			    	ppVecT.addAll(pdbANew);
			    	ppVecT.addAll(pdbVecNewBTemp);
			    	Collections.sort(ppVecT, new Pdb.PdbComparator());
			    	System.out.println("MODEL"+m+" "+j+" "+k);
			    	
			    	
//			    	compute vdw energy:
			    	double[] vdwValue = new double[1];
			    	boolean hasVDW = false;
			    	double vdwLevel = 0.05;
			    	boolean printVDWViolation = false;
			    	vdwVec = vander.convert2VDW(ppVecT);
		        
		        	int numClashes=vander.countStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,0.8);
			    	System.out.println("the number of steric clashes: "+ numClashes);
		        	if(numClashes>60)
			    		continue;
			    	
			        boolean isInPreClusters=pp.isInPdbClusters(vecEnsemblePdbs,ppVecT,0.5);///////////
			    	if(! isInPreClusters)////////
			    	{
				    	out.println("MODEL"+m+" "+j+" "+k);;///to file
				    	
				    	out.println("REMARK :   NOE RMSD=:"+disRms);
				    	out.println("REMARK :   NOE HARM=:"+noeHarm);
				    	out.println("REMARK :   numClashes=:"+numClashes);
				    	
				    	pp.printToFile(ppVecT,fileName, out);
				    	
				    	System.out.println("TER");
			         	System.out.println("END");   
			         	
			            out.println("TER");
			         	out.println("ENDMDL");
			    	
			         	vecEnsemblePdbs.add(ppVecT);///////
			    	}/////
			    }
			    
			    
			    if ( disRms < disRmsd  )
			    {
					ranThetaS = ranTheta;
					ranPhiS =  ranPhi;
					disRmsd = disRms;
					lenS = lengthR;
					for (i = 0; i < noeVec.size(); i++)
						dis_save[i]=noeBackDis[i];		    		 
				 	 	
					System.out.println(disRmsd+" ... "+ranThetaS+"  "+ranPhiS+"  "+lenS); //Check the progress
			    }
	    	}//for (int k =0 ; k< noeRangeRes; k++)
	    }//for (j=0; j< (360 / resolution); j++)
	}//for (int m = 0; m < (180/resolution); m++)
	
	out.println("END");  
	out.close();
	}catch (FileNotFoundException e)
	{
		System.out.println("File not found: " + fileName);
	}catch (IOException e)
	{
	   System.out.println("IOException: the stack trace is:");
	   e.printStackTrace();
	}
		
	noeRms[0] = disRmsd;
	if (debugNOE)
	{
	    System.out.println("NOE rmsd = "+disRmsd);
	    System.out.println("NOE distances difference for individual ones:");
	}
	double[] centerBNew = {-xC2 + lenS * Math.sin(ranThetaS) * Math.cos(ranPhiS),
			       -yC2 + lenS * Math.sin(ranThetaS) * Math.sin(ranPhiS),
			       -zC2 + lenS * Math.cos(ranThetaS)};
	for (i = 0; i < noeVec.size(); i++)
		System.out.println(dis_save[i]);
	
	Vector pdbBNew   = pp.newPdbByTranslation(pdbVecB, centerBNew);
	double [] centerANew = {-centerA[0], -centerA[1], -centerA[2]};
	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANew);
	//Print the new PDBs
	Vector ppVecT = new Vector();
	ppVecT.addAll(pdbANew);
	ppVecT.addAll(pdbBNew);
	Collections.sort(ppVecT, new Pdb.PdbComparator());
	return ppVecT;
    }
    
    /**
     * pack two SSEs based on NOE restratints
     * In both SSEs, all rotamers are placed.
     * 
     * @param noeVec the NOE distance restraints between the fragment A and B
     * @param pdbVecA one fragment
     * @param pdbVecB another fragment
     * @param noeRms saving the NOE rmsd
     * @param debugNOE if true it will print out the NOE fitting Rmsds
     * @param vecSeq sequence of residue names
     * @param resol_cluster the resolution of each cluster
     * @param vecUpdateNOE the vec update noe
     * @param strPass the str pass
     * 
     * @return the Pdb vector for translated fragment which is B while
     * Pdb vector for A is NOT changed
     * 
     * @throws JampackException the jampack exception
     */
    public Vector positionByNOEAllRotamers(final Vector noeVec, Vector pdbVecA, Vector pdbVecB, 
    		double[] noeRms, boolean debugNOE, Vector vecUpdateNOE, String strPass,Vector vecSeq, double resol_cluster)throws JampackException
    {
		//Extract neccessary information from the NOE table.
		int i, j;
		int noA, noB;
		int index = -1;
		String atom  = "";
		String atomA = "";
		String atomB = "";
		Vector atomVec = new Vector();
		Cartesian cc = new Cartesian();
		Pdb pp = new Pdb();
		int sizeA = 0;
		int sizeB = 0;
		double[][] pdbA2 = new double[noeVec.size()*6][3];
		double[][] pdbB2 = new double[noeVec.size()*6][3];
		Vector id4A = new Vector();
		Vector id4B = new Vector();
		String id = "";  //Combining the residueNO and Atom name for easy comparion.
		
		double [][] noeDistance = new double[noeVec.size()][2];	
		
		//compute the centers of atom coordinates involed in NOEs
		boolean isQ=true;
		for (i=0; i<noeVec.size(); i++)
		{
		    noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
		    atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
		 
		    ////////
		    String tempA="";
		    isQ=true;
		    if(atomA.length()>=2)
		    {
			    if(atomA.substring(0,1).equalsIgnoreCase("Q") && (!atomA.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempA="H"+atomA.substring(1,atomA.length());
			    else if(atomA.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempA="H"+atomA.substring(2,atomA.length());
			    else 
			    {
			    	tempA=atomA;	
			    	isQ=false;
			    }
		    }//if(atomA.length()>=2)
		    else 
		    {
		    	tempA=atomA;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecA, new Pdb(noA), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
		    	pp = (Pdb)pdbVecA.elementAt(index);
		    	atomVec = pp.getAtomVec();
		    	for (j=0; j<atomVec.size(); j++)
		    	{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    
				    if(atom.length()>=tempA.length())
				    {
				    	String debug_str=atom.substring(0,tempA.length() );
				    
				    	id = String.valueOf(noA)+atom;
					    //for deleting repeatation in NOEs when computing the centers 
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp=atom.substring(0,tempA.length() );
				    	else
				    		atom_temp=atom;
				    	
					    if ( atom_temp.equalsIgnoreCase(tempA) && !id4A.contains(id) ) 
					    {  
					    	pdbA2[sizeA] = cc.getXYZ();	
					    	id4A.add(id);
					    	sizeA++;
					    }
				    }// if(atom.length()>=tempA.length())
				    
				    
		    	}//for (j=0; j<atomVec.size(); j++)
		    }//if ( index > -1 )
		    
		    noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
		    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
		    //////	//
		    String tempB="";
		    isQ=true;
		    if(atomB.length()>=2)
		    {
			    if(atomB.substring(0,1).equalsIgnoreCase("Q") && (!atomB.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempB="H"+atomB.substring(1,atomB.length());
			    else if(atomB.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempB="H"+atomB.substring(2,atomB.length());
			    else 
			    {
			    	tempB=atomB;  
			    	isQ=false;
			    }
		    }//if(atomB.length()>=2)
		    else
		    {
		    	tempB=atomB;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecB, new Pdb(noB), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
				pp = (Pdb)pdbVecB.elementAt(index);
				atomVec = pp.getAtomVec();
				for (j=0; j<atomVec.size(); j++)
				{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    if (atom.length()>=tempB.length())
				    {
				    	id = String.valueOf(noB)+atom;
				    
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp= atom.substring(0,tempB.length());
				    	else
				    		atom_temp=atom;
					    if (atom_temp.equalsIgnoreCase(tempB) && !id4B.contains(id) ) 
					    {
							pdbB2[sizeB] = cc.getXYZ();	
							id4B.add(id);
							sizeB++;
					    }
				    }//if (atom.length()>=tempB.length())
				    
				}
		    }//if ( index > -1 )
		    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
		}
	// Compute the center
	final int n = 3;
	//int mm = pdbA.length;   
	double xC1 = 0.0, yC1 = 0.0, zC1 = 0.0, xC2 = 0.0, yC2 = 0.0, zC2 = 0.0; 
	double noeAve    = 0.0;
	for (i=0; i<sizeA; i++)
	{
	    xC1 += pdbA2[i][0];
	    yC1 += pdbA2[i][1];
	    zC1 += pdbA2[i][2];
	}
	if (sizeA > 0)
	{ // Compute the center
 	    xC1 /= sizeA;
 	    yC1 /= sizeA;
 	    zC1 /= sizeA;
	}
	for (i=0; i<sizeB; i++)
	{
	    xC2 += pdbB2[i][0]; 
	    yC2 += pdbB2[i][1];
	    zC2 += pdbB2[i][2];
	}
	if (sizeB > 0)
	{ // Compute the center
 	    xC2 /= sizeB;
 	    yC2 /= sizeB;
 	    zC2 /= sizeB;
	}
	//compute average NOEs
	for (i=0; i<noeVec.size(); i++)
	    noeAve += (noeDistance[i][0] + noeDistance[i][1]) / 2.0;
	noeAve /= noeVec.size();
	//noeAve /= mm;
	
	double lengthR = noeAve;
	
 	double diffCenter = Math.sqrt((xC1-xC2)* (xC1-xC2)+(yC1-yC2) * (yC1-yC2)+(zC1-zC2) * (zC1-zC2)); 	
	double disRms = 0.0;
	double noeHarm=0.0;
	double[] noeDis = new double[noeVec.size()];
	double[] noeBackDis = new double[noeVec.size()];
	double[] dis_save = new double[noeVec.size()];
		
	double [] trans_vecA = new double[3];
	double [] trans_vecB = new double[3];
	//double [][] arrA = new double[pdbA.length][n];
	//double [][] arrB = new double[pdbB.length][n];
	//double [][] newArrB = new double[mm][n];
	double [] centerA = {xC1, yC1, zC1};
	double [] centerB = {xC2, yC2, zC2};
	double [] centerO={0.0, 0.0, 0.0};
	
	trans_vecA= pp.internuclearVec(centerA,centerO);	
	trans_vecB= pp.internuclearVec(centerB,centerO);
	//	move centers of both pdbs to the origin
	Vector<Pdb> pdbVecNewA=pp.newPdbByTranslation(pdbVecA,trans_vecA);
	Vector<Pdb> pdbVecNewB=pp.newPdbByTranslation(pdbVecB,trans_vecB);
		
 	double x1, y1, z1, x2, y2, z2;
	double ranTheta, ranPhi,  ranThetaS = 0.0, ranPhiS = 0.0;
	double disRmsd = 1000.0;
	long seed = 9754; //419057923;
	Random rr = new Random(seed);
	double low = 0.0, upper = 0.0;
	int resolution = 5;//5;//changed here....
	int noeRangeRes = 20;//100;
 	double lenS = 0.0;
 	
	String userDir = System.getProperty("user.dir");////
	String src=userDir+"/inputFiles/";
	String fileName =strPass+"0.pdb";
	
	int counter=0;
	int nTemp=0;
	Vector vecEnsemblePdbs=new Vector();
	
	vdw vander = new vdw();
	Vector vdwVec = new Vector();
	Vector vdwVecA = new Vector();
	Vector vdwVecB = new Vector();
	Cartesian ccCenterA=new Cartesian();
	Vector vecCenterB=new Vector();
	
	
	for (int m = 0; m < (180/resolution); m++)
	{
	    ranTheta = m * resolution * Math.PI / 180.0;
	    for (j=0; j< (360 / resolution); j++)
	    {
	    	
	    	ranPhi = j * resolution * Math.PI / 180.0;
	    	
	    	for (int k =noeRangeRes ; k>=0 ; k--)
		    {
	    		
			    lengthR  = 0.0 + 0.5*k;
			    //move the center of B to a new position on the upper sphere, recompute B'coordinates
			    double[] centerBNew = { lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
  	 			      lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
  	 			      lengthR * Math.cos(ranTheta)};
			    double[] trans_vec = new double[3];
	    		trans_vec=pp.internuclearVec(centerO,centerBNew);
	    		
	    		Vector pdbVecNewB2=pp.newPdbByTranslation(pdbVecNewB,trans_vec);
			    
			    disRms = 0.0;
			    noeHarm=0.0;
			   
			    for (i = 0; i < noeVec.size(); i++)
			    {
			    	noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
			    	atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
			    	
			    	noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
			 	    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
			 	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
					
			 	    double [] dis=new double[1];
			 	    double [] distance=new double[1];
			 	 	pp.measurePackDisAllRotamers(pdbVecNewA,pdbVecNewB2,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
			 	 	noeDis[i]=dis[0];
			 	 	noeBackDis[i]=distance[0];
			 	 	disRms += noeDis[i] * noeDis[i];
			 	 	noeHarm+=noeDis[i] * noeDis[i];
			    }
			    noeHarm=noeHarm/noeVec.size();		    
			    disRms = Math.sqrt(disRms / noeVec.size());
			   
			  
			    if ( Math.abs(disRms - 0.0) < Const.eps )			   
			    {			    				    	
			    	System.out.println("here we have one packing satisfying noe constraints:");
			    	System.out.println(disRms+" ... "+ranTheta+"  "+ranPhi+"  "+lengthR); //Check the progress
	    			
			    	double[] centerBNewTemp=new double [3];
			    	centerBNewTemp =new double[] {-xC2+lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
			    			-yC2 +lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
			    			-zC2 + lengthR * Math.cos(ranTheta)};
			    	
			    	Vector<Pdb> pdbVecNewBTemp=pp.newPdbByTranslation(pdbVecB,centerBNewTemp);
			    	Vector ppVecT = new Vector();
			    	double [] centerANewTemp = {-centerA[0], -centerA[1], -centerA[2]};
			    	
			    	
			    	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANewTemp);		    	
			    	
			    	ppVecT.addAll(pdbANew);
			    	ppVecT.addAll(pdbVecNewBTemp);
			    	Collections.sort(ppVecT, new Pdb.PdbComparator());
			    	System.out.println("MODEL"+m+" "+j+" "+k);
			    	
			    	ccCenterA=new Cartesian("",-centerA[0], -centerA[1], -centerA[2] );
			    	
			    	Cartesian ccCenterB=new Cartesian("",centerBNewTemp[0], centerBNewTemp[1], centerBNewTemp[2] );
			    
//			    	compute vdw energy:
			    	double[] vdwValue = new double[1];
			    	boolean hasVDW = false;
			    	double vdwLevel = 0.05;
			    	boolean printVDWViolation = false;
			    	
			    	String rotSrc= userDir+"/system/rot-lib/";// src+ "rotasamp-small/";  
	    		   
			    	boolean[] resIndex=new boolean[ppVecT.size()];
			    	for (int kk=0;kk<resIndex.length;kk++)
			    		resIndex[kk]=false;
	    		    Vector pdbVecSSE_temp2 = pp.residueNameUpdate(vecSeq, ppVecT);
	    		    Vector  ppVecTALA=pp.AlaninizeStructureResName(pdbVecSSE_temp2,rotSrc);  			    	
			    	vdwVec = vander.convert2VDW(ppVecTALA);
			    	
			    	///////////			    	
	    		    Vector pdbVecSSE_tempA = pp.residueNameUpdate(vecSeq, pdbANew);
	    		    Vector  ppVecTALA_A=pp.AlaninizeStructureResName(pdbVecSSE_tempA,rotSrc);  			    	
			    	vdwVecA = vander.convert2VDW(ppVecTALA_A);
			    	//////////			    	
	    		    Vector pdbVecSSE_tempB = pp.residueNameUpdate(vecSeq, pdbVecNewBTemp);
	    		    Vector  ppVecTALA_B=pp.AlaninizeStructureResName(pdbVecSSE_tempB,rotSrc);  			    	
			    	vdwVecB = vander.convert2VDW(ppVecTALA_B);
			    	//////////
		        	
			    	int numClashes=vander.countStericClashBetweenTwo(vdwVecA,vdwVecB, vdwValue, vdwLevel, printVDWViolation, true,0.5);
		        	
		        	System.out.println("the number of steric clashes: "+ numClashes);
			    	
		        	if(numClashes>0) ///changed for ubq,need to change back to 0...
			    		continue;
			    	
			       
			        boolean isInPreClusters=pp.isInPointClusters(vecCenterB,ccCenterB,resol_cluster*2);//////////0.4 for ubq, 0.8 used for eta
			    	if(! isInPreClusters)////////
			    	{/////////
			    		counter+=1;
			    		if(!isSkipOutEnsembe)
			    		{
				    		try
				    		{
				    			fileName = strPass+ (int)(counter)+".pdb";	
				    			PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
				    			
						    	out.println("MODEL"+m+" "+j+" "+k);;///to file
						    	
						    	out.println("REMARK :   NOE RMSD=:"+disRms);
						    	out.println("REMARK :   NOE HARM=:"+noeHarm);
						    	out.println("REMARK :   numClashes in Alaninized strcuture=:"+numClashes);
						    	
						    	Vector vecFirstBB=pp.OutputBackbone(ppVecT);
						    	pp.printToFile(vecFirstBB,fileName, out);
						    	
						    	System.out.println("TER");
					         	System.out.println("END");   
					         	
					            out.println("TER");
					         	out.println("ENDMDL");
					         	out.println("END");  
					        	out.close();
				        	}catch (FileNotFoundException e)
				        	{
				        		System.out.println("File not found: " + fileName);
				        	}catch (IOException e)
				        	{
				        	   System.out.println("IOException: the stack trace is:");
				        	   e.printStackTrace();
				        	}
			    		}//end of if(!isSkipOutEnsembe)
			         	
			         	vecCenterB.add(ccCenterB);
			    	}/////end of if(! isInPreClusters)
			    }//end of if ( Math.abs(disRms - 0.0) < Const.eps )	
			    
			    
			    if ( disRms < disRmsd  )
			    {
					ranThetaS = ranTheta;
					ranPhiS =  ranPhi;
					disRmsd = disRms;
					lenS = lengthR;
					for (i = 0; i < noeVec.size(); i++)
						dis_save[i]=noeBackDis[i];		    		 
				 	 	
					System.out.println(disRmsd+" ... "+ranThetaS+"  "+ranPhiS+"  "+lenS); //Check the progress
			    }
	    	}//for (int k =0 ; k< noeRangeRes; k++)
	    }//for (j=0; j< (360 / resolution); j++)
	}//for (int m = 0; m < (180/resolution); m++)
		
	noeRms[0] = disRmsd;
	if (debugNOE)
	{
	    System.out.println("NOE rmsd = "+disRmsd);
	    System.out.println("NOE distances difference for individual ones:");
	}
	
	double[] centerBNew=centerCalNew(vecCenterB);
	
	for (i = 0; i < noeVec.size(); i++)
		System.out.println(dis_save[i]);
	
	Vector pdbBNew   = pp.newPdbByTranslation(pdbVecB, centerBNew);
	//double [] centerANew = {-centerA[0], -centerA[1], -centerA[2]};
	double [] centerANew=ccCenterA.getXYZ();
	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANew);
	//Print the new PDBs
	Vector ppVecT = new Vector();
	ppVecT.addAll(pdbANew);
	ppVecT.addAll(pdbBNew);
	Collections.sort(ppVecT, new Pdb.PdbComparator());
	return ppVecT;
    }
    
    /**
     * pack two SSEs based on Amgibuous NOE restratints. (The translation should at least satisfy a minimum
     * number (3-5) of NOE constraints).
     * In both SSEs, all rotamers are placed
     * Written in 1/27/09--1/27/09.
     * 
     * @param noeVec the NOE distance restraints between the fragment A and B
     * @param pdbVecA one fragment
     * @param pdbVecB another fragment
     * @param noeRms saving the NOE rmsd
     * @param debugNOE if true it will print out the NOE fitting Rmsds
     * @param vecSeq sequence of residue names
     * @param resol_cluster the resolution of each cluster
     * @param vecUpdateNOE the vec update noe
     * @param strPass the str pass
     * 
     * @return the Pdb vector for translated fragment which is B while
     * Pdb vector for A is NOT changed
     * 
     * @throws JampackException the jampack exception
     */
    public Vector positionByAmbiNOEAllRotamers(final Vector noeVec, Vector pdbVecA, Vector pdbVecB, 
    		double[] noeRms, boolean debugNOE, Vector vecUpdateNOE, String strPass,Vector vecSeq, double resol_cluster)throws JampackException
    {
		//Extract neccessary information from the NOE table.
		int i, j;
		int noA, noB;
		int index = -1;
		String atom  = "";
		String atomA = "";
		String atomB = "";
		Vector atomVec = new Vector();
		Cartesian cc = new Cartesian();
		Pdb pp = new Pdb();
		int sizeA = 0;
		int sizeB = 0;
		double[][] pdbA2 = new double[noeVec.size()*6][3];
		double[][] pdbB2 = new double[noeVec.size()*6][3];
		Vector id4A = new Vector();
		Vector id4B = new Vector();
		String id = "";  //Combining the residueNO and Atom name for easy comparion.
		
		double [][] noeDistance = new double[noeVec.size()][2];	
		
		//compute the centers of atom coordinates involed in NOEs
		boolean isQ=true;
		for (i=0; i<noeVec.size(); i++)
		{
		    noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
		    atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
		 
		    ////////
		    String tempA="";
		    isQ=true;
		    if(atomA.length()>=2)
		    {
			    if(atomA.substring(0,1).equalsIgnoreCase("Q") && (!atomA.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempA="H"+atomA.substring(1,atomA.length());
			    else if(atomA.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempA="H"+atomA.substring(2,atomA.length());
			    else 
			    {
			    	tempA=atomA;	
			    	isQ=false;
			    }
		    }//if(atomA.length()>=2)
		    else 
		    {
		    	tempA=atomA;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecA, new Pdb(noA), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
		    	pp = (Pdb)pdbVecA.elementAt(index);
		    	atomVec = pp.getAtomVec();
		    	for (j=0; j<atomVec.size(); j++)
		    	{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    
				    if(atom.length()>=tempA.length())
				    {
				    	String debug_str=atom.substring(0,tempA.length() );
				    
				    	id = String.valueOf(noA)+atom;
					    //for deleting repeatation in NOEs when computing the centers 
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp=atom.substring(0,tempA.length() );
				    	else
				    		atom_temp=atom;
				    	
					    if ( atom_temp.equalsIgnoreCase(tempA) && !id4A.contains(id) ) 
					    {  
					    	pdbA2[sizeA] = cc.getXYZ();	
					    	id4A.add(id);
					    	sizeA++;
					    }
				    }// if(atom.length()>=tempA.length())
				    
				    
		    	}//for (j=0; j<atomVec.size(); j++)
		    }//if ( index > -1 )
		    
		    noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
		    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
		    //////	//
		    String tempB="";
		    isQ=true;
		    if(atomB.length()>=2)
		    {
			    if(atomB.substring(0,1).equalsIgnoreCase("Q") && (!atomB.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempB="H"+atomB.substring(1,atomB.length());
			    else if(atomB.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempB="H"+atomB.substring(2,atomB.length());
			    else 
			    {
			    	tempB=atomB;  
			    	isQ=false;
			    }
		    }//if(atomB.length()>=2)
		    else
		    {
		    	tempB=atomB;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecB, new Pdb(noB), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
				pp = (Pdb)pdbVecB.elementAt(index);
				atomVec = pp.getAtomVec();
				for (j=0; j<atomVec.size(); j++)
				{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    if (atom.length()>=tempB.length())
				    {
				    	id = String.valueOf(noB)+atom;
				    
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp= atom.substring(0,tempB.length());
				    	else
				    		atom_temp=atom;
					    if (atom_temp.equalsIgnoreCase(tempB) && !id4B.contains(id) ) 
					    {
							pdbB2[sizeB] = cc.getXYZ();	
							id4B.add(id);
							sizeB++;
					    }
				    }//if (atom.length()>=tempB.length())
				    
				}
		    }//if ( index > -1 )
		    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
		}
	// Compute the center
	final int n = 3;
	//int mm = pdbA.length;   
	double xC1 = 0.0, yC1 = 0.0, zC1 = 0.0, xC2 = 0.0, yC2 = 0.0, zC2 = 0.0; 
	double noeAve    = 0.0;
	for (i=0; i<sizeA; i++)
	{
	    xC1 += pdbA2[i][0];
	    yC1 += pdbA2[i][1];
	    zC1 += pdbA2[i][2];
	}
	if (sizeA > 0)
	{ // Compute the center
 	    xC1 /= sizeA;
 	    yC1 /= sizeA;
 	    zC1 /= sizeA;
	}
	for (i=0; i<sizeB; i++)
	{
	    xC2 += pdbB2[i][0]; 
	    yC2 += pdbB2[i][1];
	    zC2 += pdbB2[i][2];
	}
	if (sizeB > 0)
	{ // Compute the center
 	    xC2 /= sizeB;
 	    yC2 /= sizeB;
 	    zC2 /= sizeB;
	}
	//compute average NOEs
	for (i=0; i<noeVec.size(); i++)
	    noeAve += (noeDistance[i][0] + noeDistance[i][1]) / 2.0;
	noeAve /= noeVec.size();
	//noeAve /= mm;
	
	double lengthR = noeAve;
	
 	double diffCenter = Math.sqrt((xC1-xC2)* (xC1-xC2)+(yC1-yC2) * (yC1-yC2)+(zC1-zC2) * (zC1-zC2)); 	
	double disRms = 0.0;
	double noeHarm=0.0;
	double[] noeDis = new double[noeVec.size()];
	double[] noeBackDis = new double[noeVec.size()];
	double[] dis_save = new double[noeVec.size()];
		
	double [] trans_vecA = new double[3];
	double [] trans_vecB = new double[3];
	//double [][] arrA = new double[pdbA.length][n];
	//double [][] arrB = new double[pdbB.length][n];
	//double [][] newArrB = new double[mm][n];
	double [] centerA = {xC1, yC1, zC1};
	double [] centerB = {xC2, yC2, zC2};
	double [] centerO={0.0, 0.0, 0.0};
	
	trans_vecA= pp.internuclearVec(centerA,centerO);	
	trans_vecB= pp.internuclearVec(centerB,centerO);
	//	move centers of both pdbs to the origin
	Vector<Pdb> pdbVecNewA=pp.newPdbByTranslation(pdbVecA,trans_vecA);
	Vector<Pdb> pdbVecNewB=pp.newPdbByTranslation(pdbVecB,trans_vecB);
		
 	double x1, y1, z1, x2, y2, z2;
	double ranTheta, ranPhi,  ranThetaS = 0.0, ranPhiS = 0.0;
	double disRmsd = 1000.0;
	long seed = 9754; //419057923;
	Random rr = new Random(seed);
	double low = 0.0, upper = 0.0;
	int resolution = 5;//5;//changed here....
	int noeRangeRes = 20;//100;
 	double lenS = 0.0;
 	
	String userDir = System.getProperty("user.dir");////
	String src=userDir+"/inputFiles/";
	String fileName =strPass+"0.pdb";
	
	int counter=0;
	int nTemp=0;
	Vector vecEnsemblePdbs=new Vector();
	
	vdw vander = new vdw();
	Vector vdwVec = new Vector();
	Vector vdwVecA = new Vector();
	Vector vdwVecB = new Vector();
	Cartesian ccCenterA=new Cartesian();
	Vector vecCenterB=new Vector();
	
	
	for (int m = 0; m < (180/resolution); m++)
	{
	    ranTheta = m * resolution * Math.PI / 180.0;
	    for (j=0; j< (360 / resolution); j++)
	    {
	    	
	    	ranPhi = j * resolution * Math.PI / 180.0;
	    	
	    	for (int k =noeRangeRes ; k>=0 ; k--)
		    {
	    		
			    lengthR  = 0.0 + 0.5*k;
			    //move the center of B to a new position on the upper sphere, recompute B'coordinates
			    double[] centerBNew = { lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
  	 			      lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
  	 			      lengthR * Math.cos(ranTheta)};
			    double[] trans_vec = new double[3];
	    		trans_vec=pp.internuclearVec(centerO,centerBNew);
	    		
	    		Vector pdbVecNewB2=pp.newPdbByTranslation(pdbVecNewB,trans_vec);
			    
			    disRms = 0.0;
			    noeHarm=0.0;
			   
			    int numSatisfiedNoes=0;
			    for (i = 0; i < noeVec.size(); i++)
			    {
			    	noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
			    	atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
			    	
			    	noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
			 	    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
			 	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
					
			 	    double [] dis=new double[1];
			 	    double [] distance=new double[1];
			 	 	boolean isSatisfied=pp.measurePackDisAllRotamers(pdbVecNewA,pdbVecNewB2,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
			 	 	if(distance[0]<noeDistance[i][1])
			 	 		numSatisfiedNoes++;
			 	 	noeDis[i]=dis[0];
			 	 	noeBackDis[i]=distance[0];
			 	 	disRms += noeDis[i] * noeDis[i];
			 	 	noeHarm+=noeDis[i] * noeDis[i];
			    }
			    noeHarm=noeHarm/noeVec.size();		    
			    disRms = Math.sqrt(disRms / noeVec.size());
			   
			  
			    //if ( Math.abs(disRms - 0.0) < Const.eps )			   
			   // if(numSatisfiedNoes>=(noeVec.size() * 0.25)) //more than 25% NOE restraints are satisfied.
			    	if(numSatisfiedNoes>=1)
			    {			    				    	
			    	System.out.println("here we have one packing satisfying noe constraints:");
			    	System.out.println(disRms+" ... "+ranTheta+"  "+ranPhi+"  "+lengthR); //Check the progress
	    			
			    	double[] centerBNewTemp=new double [3];
			    	centerBNewTemp =new double[] {-xC2+lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
			    			-yC2 +lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
			    			-zC2 + lengthR * Math.cos(ranTheta)};
			    	
			    	Vector<Pdb> pdbVecNewBTemp=pp.newPdbByTranslation(pdbVecB,centerBNewTemp);
			    	Vector ppVecT = new Vector();
			    	double [] centerANewTemp = {-centerA[0], -centerA[1], -centerA[2]};
			    	
			    	
			    	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANewTemp);		    	
			    	
			    	ppVecT.addAll(pdbANew);
			    	ppVecT.addAll(pdbVecNewBTemp);
			    	Collections.sort(ppVecT, new Pdb.PdbComparator());
			    	System.out.println("MODEL"+m+" "+j+" "+k);
			    	
			    	ccCenterA=new Cartesian("",-centerA[0], -centerA[1], -centerA[2] );
			    	
			    	Cartesian ccCenterB=new Cartesian("",centerBNewTemp[0], centerBNewTemp[1], centerBNewTemp[2] );
			    
			    	//compute the (approximate) vdw energy:
			    	double[] vdwValue = new double[1];
			    	boolean hasVDW = false;
			    	double vdwLevel = 0.05;
			    	boolean printVDWViolation = false;
			    	
			    	String rotSrc=userDir+"/system/rot-lib/";//src+ "rotasamp-small/";  ///////
			    	
			    	boolean[] resIndex=new boolean[ppVecT.size()];
			    	for (int kk=0;kk<resIndex.length;kk++)
			    		resIndex[kk]=false;
	    		    Vector pdbVecSSE_temp2 = pp.residueNameUpdate(vecSeq, ppVecT);
	    		    Vector  ppVecTALA=pp.AlaninizeStructureResName(pdbVecSSE_temp2,rotSrc);  			    	
			    	vdwVec = vander.convert2VDW(ppVecTALA);
			    	
			    	///////////			    	
	    		    Vector pdbVecSSE_tempA = pp.residueNameUpdate(vecSeq, pdbANew);
	    		    Vector  ppVecTALA_A=pp.AlaninizeStructureResName(pdbVecSSE_tempA,rotSrc);  			    	
			    	vdwVecA = vander.convert2VDW(ppVecTALA_A);
			    	//////////			    	
	    		    Vector pdbVecSSE_tempB = pp.residueNameUpdate(vecSeq, pdbVecNewBTemp);
	    		    Vector  ppVecTALA_B=pp.AlaninizeStructureResName(pdbVecSSE_tempB,rotSrc);  			    	
			    	vdwVecB = vander.convert2VDW(ppVecTALA_B);
			    	//////////
		        	
			    	int numClashes=vander.countStericClashBetweenTwo(vdwVecA,vdwVecB, vdwValue, vdwLevel, printVDWViolation, true,0.5);
		        	
		        	System.out.println("the number of steric clashes: "+ numClashes);
			    	
		        	if(numClashes>0) ///changed for ubq,need to change back to 0...
			    		continue;
			    	
			       
			        boolean isInPreClusters=pp.isInPointClusters(vecCenterB,ccCenterB,resol_cluster*2);//////////0.4 for ubq, 0.8 used for eta
			    	if(! isInPreClusters)////////
			    	{/////////
			    		counter+=1;
			    		if(!isSkipOutEnsembe)
			    		{
				    		try
				    		{
				    			fileName = strPass+ (int)(counter)+".pdb";	
				    			PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
				    			
						    	out.println("MODEL"+m+" "+j+" "+k);;///to file
						    	
						    	out.println("REMARK :   NOE RMSD=:"+disRms);
						    	out.println("REMARK :   NOE HARM=:"+noeHarm);
						    	out.println("REMARK :   numClashes in Alaninized strcuture=:"+numClashes);
						    	
						    	Vector vecFirstBB=pp.OutputBackbone(ppVecT);
						    	pp.printToFile(vecFirstBB,fileName, out);
						    	
						    	System.out.println("TER");
					         	System.out.println("END");   
					         	
					            out.println("TER");
					         	out.println("ENDMDL");
					         	out.println("END");  
					        	out.close();
				        	}catch (FileNotFoundException e)
				        	{
				        		System.out.println("File not found: " + fileName);
				        	}catch (IOException e)
				        	{
				        	   System.out.println("IOException: the stack trace is:");
				        	   e.printStackTrace();
				        	}
			    		}//end of if(!isSkipOutEnsembe)
			         	
			         	vecCenterB.add(ccCenterB);
			    	}/////end of if(! isInPreClusters)
			    }//end of if ( Math.abs(disRms - 0.0) < Const.eps )	
			    
			    
			    if ( disRms < disRmsd  )
			    {
					ranThetaS = ranTheta;
					ranPhiS =  ranPhi;
					disRmsd = disRms;
					lenS = lengthR;
					for (i = 0; i < noeVec.size(); i++)
						dis_save[i]=noeBackDis[i];		    		 
				 	 	
					System.out.println(disRmsd+" ... "+ranThetaS+"  "+ranPhiS+"  "+lenS); //Check the progress
			    }
	    	}//for (int k =0 ; k< noeRangeRes; k++)
	    }//for (j=0; j< (360 / resolution); j++)
	}//for (int m = 0; m < (180/resolution); m++)
		
	noeRms[0] = disRmsd;
	if (debugNOE)
	{
	    System.out.println("NOE rmsd = "+disRmsd);
	    System.out.println("NOE distances difference for individual ones:");
	}
	
	double[] centerBNew=centerCalNew(vecCenterB);
	
	for (i = 0; i < noeVec.size(); i++)
		System.out.println(dis_save[i]);
	
	Vector pdbBNew   = pp.newPdbByTranslation(pdbVecB, centerBNew);
	//double [] centerANew = {-centerA[0], -centerA[1], -centerA[2]};
	double [] centerANew=ccCenterA.getXYZ();
	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANew);
	//Print the new PDBs
	Vector ppVecT = new Vector();
	ppVecT.addAll(pdbANew);
	ppVecT.addAll(pdbBNew);
	Collections.sort(ppVecT, new Pdb.PdbComparator());
	return ppVecT;
    }
       
    //this is an old function. Can be deleted.
    /**
     * Position by noe all rotamers saved.
     * 
     * @param noeVec the noe vec
     * @param pdbVecA the pdb vec a
     * @param pdbVecB the pdb vec b
     * @param noeRms the noe rms
     * @param debugNOE the debug noe
     * @param vecUpdateNOE the vec update noe
     * @param strPass the str pass
     * @param vecSeq the vec seq
     * 
     * @return the vector
     * 
     * @throws JampackException the jampack exception
     */
    public Vector positionByNOEAllRotamersSaved(final Vector noeVec, Vector pdbVecA, Vector pdbVecB, 
    		double[] noeRms, boolean debugNOE, Vector vecUpdateNOE, String strPass,Vector vecSeq)throws JampackException
    {
		//Extract neccessary information from the NOE table.
		int i, j;
		int noA, noB;
		int index = -1;
		String atom  = "";
		String atomA = "";
		String atomB = "";
		Vector atomVec = new Vector();
		Cartesian cc = new Cartesian();
		Pdb pp = new Pdb();
		int sizeA = 0;
		int sizeB = 0;
		double[][] pdbA2 = new double[noeVec.size()*6][3];
		double[][] pdbB2 = new double[noeVec.size()*6][3];
		Vector id4A = new Vector();
		Vector id4B = new Vector();
		String id = "";  //Combining the residueNO and Atom name for easy comparion.
		//double[][] pdbA = new double[noeVec.size()][3];
		//double[][] pdbB = new double[noeVec.size()][3];
		double [][] noeDistance = new double[noeVec.size()][2];	
		
		//compute the centers of atom coordinates involed in NOEs
		boolean isQ=true;
		for (i=0; i<noeVec.size(); i++)
		{
		    noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
		    atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
		 
		    ////////
		    String tempA="";
		    isQ=true;
		    if(atomA.length()>=2)
		    {
			    if(atomA.substring(0,1).equalsIgnoreCase("Q") && (!atomA.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempA="H"+atomA.substring(1,atomA.length());
			    else if(atomA.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempA="H"+atomA.substring(2,atomA.length());
			    else 
			    {
			    	tempA=atomA;	
			    	isQ=false;
			    }
		    }//if(atomA.length()>=2)
		    else 
		    {
		    	tempA=atomA;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecA, new Pdb(noA), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
		    	pp = (Pdb)pdbVecA.elementAt(index);
		    	atomVec = pp.getAtomVec();
		    	for (j=0; j<atomVec.size(); j++)
		    	{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    
				    if(atom.length()>=tempA.length())
				    {
				    	String debug_str=atom.substring(0,tempA.length() );
				    	//if (atom.substring(0,tempA.length() ).equalsIgnoreCase(tempA) )
				    		//pdbA[i] = cc.getXYZ();
				    	id = String.valueOf(noA)+atom;
					    //for deleting repeatation in NOEs when computing the centers 
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp=atom.substring(0,tempA.length() );
				    	else
				    		atom_temp=atom;
				    	
					    if ( atom_temp.equalsIgnoreCase(tempA) && !id4A.contains(id) ) 
					    {  
					    	pdbA2[sizeA] = cc.getXYZ();	
					    	id4A.add(id);
					    	sizeA++;
					    }
				    }// if(atom.length()>=tempA.length())
				    
				    
		    	}//for (j=0; j<atomVec.size(); j++)
		    }//if ( index > -1 )
		    
		    noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
		    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
		    //////	//
		    String tempB="";
		    isQ=true;
		    if(atomB.length()>=2)
		    {
			    if(atomB.substring(0,1).equalsIgnoreCase("Q") && (!atomB.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempB="H"+atomB.substring(1,atomB.length());
			    else if(atomB.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempB="H"+atomB.substring(2,atomB.length());
			    else 
			    {
			    	tempB=atomB;  
			    	isQ=false;
			    }
		    }//if(atomB.length()>=2)
		    else
		    {
		    	tempB=atomB;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecB, new Pdb(noB), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
				pp = (Pdb)pdbVecB.elementAt(index);
				atomVec = pp.getAtomVec();
				for (j=0; j<atomVec.size(); j++)
				{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    if (atom.length()>=tempB.length())
				    {
				    	id = String.valueOf(noB)+atom;
				    	//if (atom.equalsIgnoreCase(atomB))
					    	//pdbB[i] = cc.getXYZ();
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp= atom.substring(0,tempB.length());
				    	else
				    		atom_temp=atom;
					    if (atom_temp.equalsIgnoreCase(tempB) && !id4B.contains(id) ) 
					    {
							pdbB2[sizeB] = cc.getXYZ();	
							id4B.add(id);
							sizeB++;
					    }
				    }//if (atom.length()>=tempB.length())
				    
				}
		    }//if ( index > -1 )
		    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
		}
	// Compute the center
	final int n = 3;
	//int mm = pdbA.length;   
	double xC1 = 0.0, yC1 = 0.0, zC1 = 0.0, xC2 = 0.0, yC2 = 0.0, zC2 = 0.0; 
	double noeAve    = 0.0;
	for (i=0; i<sizeA; i++)
	{
	    xC1 += pdbA2[i][0];
	    yC1 += pdbA2[i][1];
	    zC1 += pdbA2[i][2];
	}
	if (sizeA > 0)
	{ // Compute the center
 	    xC1 /= sizeA;
 	    yC1 /= sizeA;
 	    zC1 /= sizeA;
	}
	for (i=0; i<sizeB; i++)
	{
	    xC2 += pdbB2[i][0]; 
	    yC2 += pdbB2[i][1];
	    zC2 += pdbB2[i][2];
	}
	if (sizeB > 0)
	{ // Compute the center
 	    xC2 /= sizeB;
 	    yC2 /= sizeB;
 	    zC2 /= sizeB;
	}
	//compute average NOEs
	for (i=0; i<noeVec.size(); i++)
	    noeAve += (noeDistance[i][0] + noeDistance[i][1]) / 2.0;
	noeAve /= noeVec.size();
	//noeAve /= mm;
	
	double lengthR = noeAve;
	//double [][] arrA = new double[mm][n];
	//double [][] arrB = new double[mm][n];
	//double [][] newArrB = new double[mm][n];
	//double [] centerA = {xC1, yC1, zC1};
 	double diffCenter = Math.sqrt((xC1-xC2)* (xC1-xC2)+(yC1-yC2) * (yC1-yC2)+(zC1-zC2) * (zC1-zC2)); 	
	double disRms = 0.0;
	double noeHarm=0.0;
	double[] noeDis = new double[noeVec.size()];
	double[] noeBackDis = new double[noeVec.size()];
	double[] dis_save = new double[noeVec.size()];
	
	
	/*for (i=0; i<mm; i++)
	{  //move both centers to the origin
	    arrA[i][0] = pdbA[i][0] - xC1; 
	    arrA[i][1] = pdbA[i][1] - yC1;
	    arrA[i][2] = pdbA[i][2] - zC1;
	    arrB[i][0] = pdbB[i][0] - xC2; 
	    arrB[i][1] = pdbB[i][1] - yC2;
	    arrB[i][2] = pdbB[i][2] - zC2; 
	    noeDis[i]  = Math.sqrt(  (arrA[i][0] - arrB[i][0]) * (arrA[i][0] - arrB[i][0]) 
			       + (arrA[i][1] - arrB[i][1]) * (arrA[i][1] - arrB[i][1]) 
			       + (arrA[i][2] - arrB[i][2]) * (arrA[i][2] - arrB[i][2]));
	}*/
	double [] trans_vecA = new double[3];
	double [] trans_vecB = new double[3];
	//double [][] arrA = new double[pdbA.length][n];
	//double [][] arrB = new double[pdbB.length][n];
	//double [][] newArrB = new double[mm][n];
	double [] centerA = {xC1, yC1, zC1};
	double [] centerB = {xC2, yC2, zC2};
	double [] centerO={0.0, 0.0, 0.0};
	//double[] centerBNew={0.0, 0.0, 0.0};
	trans_vecA= pp.internuclearVec(centerA,centerO);	
	trans_vecB= pp.internuclearVec(centerB,centerO);
	//	move centers of both pdbs to the origin
	Vector<Pdb> pdbVecNewA=pp.newPdbByTranslation(pdbVecA,trans_vecA);
	Vector<Pdb> pdbVecNewB=pp.newPdbByTranslation(pdbVecB,trans_vecB);
		
 	double x1, y1, z1, x2, y2, z2;
	double ranTheta, ranPhi,  ranThetaS = 0.0, ranPhiS = 0.0;
	double disRmsd = 1000.0;
	long seed = 9754; //419057923;
	Random rr = new Random(seed);
	double low = 0.0, upper = 0.0;
	int resolution = 5;//5;//changed here....
	int noeRangeRes = 20;//100;
 	double lenS = 0.0;
 	//double [] trans_vecA = new double[3];
	//double [] trans_vecB = new double[3];
	String userDir = System.getProperty("user.dir");////
	String src=userDir+"/inputFiles/";
	String fileName =strPass+"0.pdb";
	//String fileName =src+ "packing0.pdb";
	int counter=0;
	int nTemp=0;
	Vector vecEnsemblePdbs=new Vector();
	
	vdw vander = new vdw();
	Vector vdwVec = new Vector();
	
	try{
		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
		
		
	for (int m = 0; m < (180/resolution); m++)
	{
	    ranTheta = m * resolution * Math.PI / 180.0;
	    for (j=0; j< (360 / resolution); j++)
	    {
	    	//ranPhi = j * resolution * Math.PI / 360.0;//Lincong's original
	    	ranPhi = j * resolution * Math.PI / 180.0;
	    	
	    	//for (int k =0 ; k< noeRangeRes; k++)
	    	//{
	    	for (int k =noeRangeRes ; k>=0 ; k--)
		    {
	    		//System.out.println("m="+m+" j="+j + " k="+k);
			    //lengthR  = noeAve + 6.0*(0.50-rr.nextDouble());
			    lengthR  = 0.0 + 0.5*k;
			    //move the center of B to a new position on the upper sphere, recompute B'coordinates
			    double[] centerBNew = { lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
  	 			      lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
  	 			      lengthR * Math.cos(ranTheta)};
			    double[] trans_vec = new double[3];
	    		trans_vec=pp.internuclearVec(centerO,centerBNew);
	    		
	    		Vector pdbVecNewB2=pp.newPdbByTranslation(pdbVecNewB,trans_vec);
			    
			    disRms = 0.0;
			    noeHarm=0.0;
			    /*
			    //here is what lincong did:
			    for (i = 0; i < mm; i++)
			    {
					newArrB[i][0] = arrB[i][0] + lengthR * Math.sin(ranTheta) * Math.cos(ranPhi); 
					newArrB[i][1] = arrB[i][1] + lengthR * Math.sin(ranTheta) * Math.sin(ranPhi);
					newArrB[i][2] = arrB[i][2] + lengthR * Math.cos(ranTheta);
					noeDis[i]  = Math.sqrt((arrA[i][0] - newArrB[i][0]) * (arrA[i][0] - newArrB[i][0]) 
							       +(arrA[i][1] - newArrB[i][1]) * (arrA[i][1] -newArrB[i][1]) 
							       +(arrA[i][2] - newArrB[i][2]) * (arrA[i][2] -newArrB[i][2]));
					if (noeDis[i] > noeDistance[i][0] && noeDis[i] < noeDistance[i][1])
					    disRms += 0.0;
					else if(noeDis[i] <= noeDistance[i][0] )
					    disRms += noeDistance[i][0] - noeDis[i];
					else if(noeDis[i] >= noeDistance[i][1])
					    disRms += noeDis[i] - noeDistance[i][1];
			    }*/
			    for (i = 0; i < noeVec.size(); i++)
			    {
			    	noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
			    	atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
			    	
			    	noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
			 	    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
			 	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
					
			 	    double [] dis=new double[1];
			 	    double [] distance=new double[1];
			 	 	pp.measurePackDisAllRotamers(pdbVecNewA,pdbVecNewB2,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
			 	 	noeDis[i]=dis[0];
			 	 	noeBackDis[i]=distance[0];
			 	 	disRms += noeDis[i] * noeDis[i];
			 	 	noeHarm+=noeDis[i] * noeDis[i];
			    }
			    noeHarm=noeHarm/noeVec.size();		    
			    disRms = Math.sqrt(disRms / noeVec.size());
			   
			    //if ( disRms  < 3.0 )
			    if ( Math.abs(disRms - 0.0) < Const.eps )			   
			    {
			    	counter+=1;
			    	if ( (counter/40)>nTemp )
			    	{
			    		out.close();
			    		//fileName =src+ "packing"+ (int)(counter/40)+".pdb";
			    		//fileName =src+ strPass+ (int)(counter/40)+".pdb";
			    		fileName = strPass+ (int)(counter/40)+".pdb";			    		
			    		
			    		out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
			    	}//if ( (counter/150)>counter )
			    	nTemp=counter/40;
			    	
			    	System.out.println("here we have one packing satisfying noe constraints:");
			    	System.out.println(disRms+" ... "+ranTheta+"  "+ranPhi+"  "+lengthR); //Check the progress
	    			 //vecSSEPack.add(new SSEPacking(ranTheta,ranPhi,lengthR,disRms ));
			    	double[] centerBNewTemp=new double [3];
			    	centerBNewTemp =new double[] {-xC2+lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
			    			-yC2 +lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
			    			-zC2 + lengthR * Math.cos(ranTheta)};
			    	//trans_vecB= pp.internuclearVec(centerO,centerBNewTemp);
			    	Vector<Pdb> pdbVecNewBTemp=pp.newPdbByTranslation(pdbVecB,centerBNewTemp);
			    	Vector ppVecT = new Vector();
			    	double [] centerANewTemp = {-centerA[0], -centerA[1], -centerA[2]};
			    	
		    	
			    	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANewTemp);		    	
			    	
			    	ppVecT.addAll(pdbANew);
			    	ppVecT.addAll(pdbVecNewBTemp);
			    	Collections.sort(ppVecT, new Pdb.PdbComparator());
			    	System.out.println("MODEL"+m+" "+j+" "+k);
			    	//pp.print(ppVecT);
			    	
//			    	compute vdw energy:
			    	double[] vdwValue = new double[1];
			    	boolean hasVDW = false;
			    	double vdwLevel = 0.05;
			    	boolean printVDWViolation = false;
			    	
			    	
			    	
			    	boolean[] resIndex=new boolean[ppVecT.size()];
			    	for (int kk=0;kk<resIndex.length;kk++)
			    		resIndex[kk]=false;
			    	String rotSrc=  src+ "rotasamp-small/";  
	    		   ///// Vector  ppVecTALA=pp.AlaninizeStructure(ppVecT,resIndex,rotSrc);
	    		
	    		    Vector pdbVecSSE_temp2 = pp.residueNameUpdate(vecSeq, ppVecT);
	    		    Vector  ppVecTALA=pp.AlaninizeStructureResName(pdbVecSSE_temp2,rotSrc);
			    	
			    	
			    	vdwVec = vander.convert2VDW(ppVecTALA);
		        	//hasVDW = vander.computeVDW(vdwVec, vdwValue, vdwLevel, printVDWViolation, true);  
		        	int numClashes=vander.countStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,0.5);
			    	System.out.println("the number of steric clashes: "+ numClashes);
			    	
		        	if(numClashes>0) ///changed for ubq,need to change back to 0...
			    		continue;
			    	
			        boolean isInPreClusters=pp.isInPdbClusters(vecEnsemblePdbs,ppVecT,0.8);///////////
			    	if(! isInPreClusters)////////
			    	{/////////
			    		//counter+=1;
				    	out.println("MODEL"+m+" "+j+" "+k);;///to file
				    	
				    	out.println("REMARK :   NOE RMSD=:"+disRms);
				    	out.println("REMARK :   NOE HARM=:"+noeHarm);
				    	out.println("REMARK :   numClashes in Alaninized strcuture=:"+numClashes);
				    	//compute vdw energy:
				    	/*double[] vdwValue = new double[1];
				    	boolean hasVDW = false;
				    	double vdwLevel = 0.05;
				    	boolean printVDWViolation = false;
				    	//vdwVec = vander.convert2VDW(ppVecT);
			        	//hasVDW = vander.computeVDW(vdwVec, vdwValue, vdwLevel, printVDWViolation, false);  
			        	*/
				    	
			        	//out.println("REMARK :   vdw energy=:"+ vdwValue[0]);
				    	Vector vecFirstBB=pp.OutputBackbone(ppVecT);
				    	pp.printToFile(vecFirstBB,fileName, out);
				    	
				    	System.out.println("TER");
			         	System.out.println("END");   
			         	
			            out.println("TER");
			         	out.println("ENDMDL");
			    	
			         	vecEnsemblePdbs.add(vecFirstBB);///////
			    	}/////
			    }
			    
			    
			    if ( disRms < disRmsd  )
			    {
					ranThetaS = ranTheta;
					ranPhiS =  ranPhi;
					disRmsd = disRms;
					lenS = lengthR;
					for (i = 0; i < noeVec.size(); i++)
						dis_save[i]=noeBackDis[i];		    		 
				 	 	
					System.out.println(disRmsd+" ... "+ranThetaS+"  "+ranPhiS+"  "+lenS); //Check the progress
			    }
	    	}//for (int k =0 ; k< noeRangeRes; k++)
	    }//for (j=0; j< (360 / resolution); j++)
	}//for (int m = 0; m < (180/resolution); m++)
	
	out.println("END");  
	out.close();
	}catch (FileNotFoundException e)
	{
		System.out.println("File not found: " + fileName);
	}catch (IOException e)
	{
	   System.out.println("IOException: the stack trace is:");
	   e.printStackTrace();
	}
		
	noeRms[0] = disRmsd;
	if (debugNOE)
	{
	    System.out.println("NOE rmsd = "+disRmsd);
	    System.out.println("NOE distances difference for individual ones:");
	}
	double[] centerBNew = {-xC2 + lenS * Math.sin(ranThetaS) * Math.cos(ranPhiS),
			       -yC2 + lenS * Math.sin(ranThetaS) * Math.sin(ranPhiS),
			       -zC2 + lenS * Math.cos(ranThetaS)};
	for (i = 0; i < noeVec.size(); i++)
		System.out.println(dis_save[i]);
	/*for (i=0; i<mm; i++)
	{  //move both centers to the origin
	    arrB[i][0] = pdbB[i][0] + centerBNew[0]; 
	    arrB[i][1] = pdbB[i][1] + centerBNew[1];
	    arrB[i][2] = pdbB[i][2] + centerBNew[2];
	    noeDis[i]  = Math.sqrt( (arrA[i][0] - arrB[i][0]) * (arrA[i][0] - arrB[i][0]) 
				   +(arrA[i][1] - arrB[i][1]) * (arrA[i][1] - arrB[i][1]) 
				   +(arrA[i][2] - arrB[i][2]) * (arrA[i][2] - arrB[i][2]));
	    if (debugNOE)  	    //Print the NOE distance after the grid-search.
		System.out.println(noeDis[i]);
	    
	    //we added some lines here to filter "wrong" NOEs
	   // if(noeDis[i]<8.0)
	    	//vecUpdateNOE.add(noeVec.elementAt(i));
	}*/
	Vector pdbBNew   = pp.newPdbByTranslation(pdbVecB, centerBNew);
	double [] centerANew = {-centerA[0], -centerA[1], -centerA[2]};
	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANew);
	//Print the new PDBs
	Vector ppVecT = new Vector();
	ppVecT.addAll(pdbANew);
	ppVecT.addAll(pdbBNew);
	Collections.sort(ppVecT, new Pdb.PdbComparator());
	return ppVecT;
    }

    /**
     * pack two SSEs based on NOE restratints
     * In both SSEs, all rotamers are places.
     * similar to positionByNOEAllRotamers, but use cube grid search approach.
     * 
     * @param noeVec the NOE distance restraints between the fragment A and B
     * @param pdbVecA one fragment
     * @param pdbVecB another fragment
     * @param noeRms saving the NOE rmsd
     * @param debugNOE if true it will print out the NOE fitting Rmsds
     * @param vecUpdateNOE the vec update noe
     * 
     * @return the Pdb vector for translated fragment which is B while
     * Pdb vector for A is NOT changed
     * 
     * @throws JampackException the jampack exception
     */
    public Vector positionByNOEAllRotamersGrid(final Vector noeVec, Vector pdbVecA, Vector pdbVecB, 
    		double[] noeRms, boolean debugNOE, Vector vecUpdateNOE)throws JampackException
    {
		//Extract neccessary information from the NOE table.
		int i, j;
		int noA, noB;
		int index = -1;
		String atom  = "";
		String atomA = "";
		String atomB = "";
		Vector atomVec = new Vector();
		Cartesian cc = new Cartesian();
		Pdb pp = new Pdb();
		int sizeA = 0;
		int sizeB = 0;
		double[][] pdbA2 = new double[noeVec.size()*6][3];
		double[][] pdbB2 = new double[noeVec.size()*6][3];
		Vector id4A = new Vector();
		Vector id4B = new Vector();
		String id = "";  //Combining the residueNO and Atom name for easy comparion.
		
		double [][] noeDistance = new double[noeVec.size()][2];	
	
		//compute the centers of atom coordinates involed in NOEs
		boolean isQ=true;
		for (i=0; i<noeVec.size(); i++)
		{
		    noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
		    atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
		 
		    ////////
		    String tempA="";
		    isQ=true;
		    if(atomA.length()>=2)
		    {
			    if(atomA.substring(0,1).equalsIgnoreCase("Q") && (!atomA.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempA="H"+atomA.substring(1,atomA.length());
			    else if(atomA.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempA="H"+atomA.substring(2,atomA.length());
			    else 
			    {
			    	tempA=atomA;	
			    	isQ=false;
			    }
		    }//if(atomA.length()>=2)
		    else 
		    {
		    	tempA=atomA;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecA, new Pdb(noA), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
		    	pp = (Pdb)pdbVecA.elementAt(index);
		    	atomVec = pp.getAtomVec();
		    	for (j=0; j<atomVec.size(); j++)
		    	{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    
				    if(atom.length()>=tempA.length())
				    {
				    	String debug_str=atom.substring(0,tempA.length() );
				    	
				    	id = String.valueOf(noA)+atom;
					    //for deleting repeatation in NOEs when computing the centers 
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp=atom.substring(0,tempA.length() );
				    	else
				    		atom_temp=atom;
				    	
					    if ( atom_temp.equalsIgnoreCase(tempA) && !id4A.contains(id) ) 
					    {  
					    	pdbA2[sizeA] = cc.getXYZ();	
					    	id4A.add(id);
					    	sizeA++;
					    }
				    }// if(atom.length()>=tempA.length())
				    
				    
		    	}//for (j=0; j<atomVec.size(); j++)
		    }//if ( index > -1 )
		    
		    noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
		    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
		    //////	//
		    String tempB="";
		    isQ=true;
		    if(atomB.length()>=2)
		    {
			    if(atomB.substring(0,1).equalsIgnoreCase("Q") && (!atomB.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempB="H"+atomB.substring(1,atomB.length());
			    else if(atomB.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempB="H"+atomB.substring(2,atomB.length());
			    else 
			    {
			    	tempB=atomB;  
			    	isQ=false;
			    }
		    }//if(atomB.length()>=2)
		    else
		    {
		    	tempB=atomB;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecB, new Pdb(noB), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
				pp = (Pdb)pdbVecB.elementAt(index);
				atomVec = pp.getAtomVec();
				for (j=0; j<atomVec.size(); j++)
				{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    if (atom.length()>=tempB.length())
				    {
				    	id = String.valueOf(noB)+atom;
				    	
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp= atom.substring(0,tempB.length());
				    	else
				    		atom_temp=atom;
					    if (atom_temp.equalsIgnoreCase(tempB) && !id4B.contains(id) ) 
					    {
							pdbB2[sizeB] = cc.getXYZ();	
							id4B.add(id);
							sizeB++;
					    }
				    }//if (atom.length()>=tempB.length())
				    
				}
		    }//if ( index > -1 )
		    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
		}
	// Compute the center
	final int n = 3;
	//int mm = pdbA.length;   
	double xC1 = 0.0, yC1 = 0.0, zC1 = 0.0, xC2 = 0.0, yC2 = 0.0, zC2 = 0.0; 
	double noeAve    = 0.0;
	for (i=0; i<sizeA; i++)
	{
	    xC1 += pdbA2[i][0];
	    yC1 += pdbA2[i][1];
	    zC1 += pdbA2[i][2];
	}
	if (sizeA > 0)
	{ // Compute the center
 	    xC1 /= sizeA;
 	    yC1 /= sizeA;
 	    zC1 /= sizeA;
	}
	for (i=0; i<sizeB; i++)
	{
	    xC2 += pdbB2[i][0]; 
	    yC2 += pdbB2[i][1];
	    zC2 += pdbB2[i][2];
	}
	if (sizeB > 0)
	{ // Compute the center
 	    xC2 /= sizeB;
 	    yC2 /= sizeB;
 	    zC2 /= sizeB;
	}
	//compute average NOEs
	for (i=0; i<noeVec.size(); i++)
	    noeAve += (noeDistance[i][0] + noeDistance[i][1]) / 2.0;
	noeAve /= noeVec.size();
	//noeAve /= mm;
	
	double lengthR = noeAve;
	
 	double diffCenter = Math.sqrt((xC1-xC2)* (xC1-xC2)+(yC1-yC2) * (yC1-yC2)+(zC1-zC2) * (zC1-zC2)); 	
	double disRms = 0.0;
	double noeHarm=0.0;
	double[] noeDis = new double[noeVec.size()];
	double[] noeBackDis = new double[noeVec.size()];
	double[] dis_save = new double[noeVec.size()];
	
	
	double [] trans_vecA = new double[3];
	double [] trans_vecB = new double[3];
	
	double [] centerA = {xC1, yC1, zC1};
	double [] centerB = {xC2, yC2, zC2};
	double [] centerO={0.0, 0.0, 0.0};
	
	trans_vecA= pp.internuclearVec(centerA,centerO);	
	trans_vecB= pp.internuclearVec(centerB,centerO);
	//	move centers of both pdbs to the origin
	Vector<Pdb> pdbVecNewA=pp.newPdbByTranslation(pdbVecA,trans_vecA);
	Vector<Pdb> pdbVecNewB=pp.newPdbByTranslation(pdbVecB,trans_vecB);
		
 	double x1, y1, z1, x2, y2, z2;
	double ranTheta, ranPhi,  ranThetaS = 0.0, ranPhiS = 0.0;
	double disRmsd = 1000.0;
	long seed = 9754; //419057923;
	Random rr = new Random(seed);
	double low = 0.0, upper = 0.0;
	int resolution = 5;//5;//changed here....
	int noeRangeRes = 80;//100;
 	double lenS = 0.0;
 
	String userDir = System.getProperty("user.dir");////
	String src=userDir+"/inputFiles/";
	String fileName =src+ "packing0.pdb";
	int counter=0;
	int nTemp=0;
	Vector vecEnsemblePdbs=new Vector();
	
	vdw vander = new vdw();
	Vector vdwVec = new Vector();
	
	try{
		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
		
		
	for (int m = -noeRangeRes; m <=noeRangeRes; m++)
	{
	    ranTheta =  0.0 + 0.3*m;
	    for (j=-noeRangeRes; j<=noeRangeRes; j++)
	    {
	    	ranPhi =  0.0 + 0.3*j;
	    	
	    	for (int k =-noeRangeRes ; k<=noeRangeRes ; k++)
		    {
	    		
			    lengthR  = 0.0 + 0.3*k;
			    //move the center of B to a new position on the upper sphere, recompute B'coordinates
			    double[] centerBNew = { ranTheta, ranPhi, lengthR};
			    double[] trans_vec = new double[3];
	    		trans_vec=pp.internuclearVec(centerO,centerBNew);
	    		
	    		Vector pdbVecNewB2=pp.newPdbByTranslation(pdbVecNewB,trans_vec);
			    
			    disRms = 0.0;
			    noeHarm=0.0;
			  
			    for (i = 0; i < noeVec.size(); i++)
			    {
			    	noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
			    	atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
			    	
			    	noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
			 	    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
			 	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
					
			 	    double [] dis=new double[1];
			 	    double [] distance=new double[1];
			 	 	pp.measurePackDisAllRotamers(pdbVecNewA,pdbVecNewB2,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
			 	 	noeDis[i]=dis[0];
			 	 	noeBackDis[i]=distance[0];
			 	 	disRms += noeDis[i] * noeDis[i];
			 	 	noeHarm+=noeDis[i] * noeDis[i];
			    }
			    noeHarm=noeHarm/noeVec.size();		    
			    disRms = Math.sqrt(disRms / noeVec.size());
			   
			  
			    if ( Math.abs(disRms - 0.0) < Const.eps )			   
			    {
			    	counter+=1;
			    	if ( (counter/40)>nTemp )
			    	{
			    		out.close();
			    		fileName =src+ "packing"+ (int)(counter/40)+".pdb";
			    		out = new PrintWriter(new BufferedWriter(new FileWriter(fileName)));
			    	}//if ( (counter/150)>counter )
			    	nTemp=counter/40;
			    	
			    	System.out.println("here we have one packing satisfying noe constraints:");
			    	System.out.println(disRms+" ... "+ranTheta+"  "+ranPhi+"  "+lengthR); //Check the progress
	    			 //vecSSEPack.add(new SSEPacking(ranTheta,ranPhi,lengthR,disRms ));
			    	double[] centerBNewTemp=new double [3];
			    	centerBNewTemp =new double[] {-xC2+ranTheta,-yC2 +ranPhi,-zC2 + lengthR};
			    	//trans_vecB= pp.internuclearVec(centerO,centerBNewTemp);
			    	Vector<Pdb> pdbVecNewBTemp=pp.newPdbByTranslation(pdbVecB,centerBNewTemp);
			    	Vector ppVecT = new Vector();
			    	double [] centerANewTemp = {-centerA[0], -centerA[1], -centerA[2]};
			    	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANewTemp);
			    	
			    	ppVecT.addAll(pdbANew);
			    	ppVecT.addAll(pdbVecNewBTemp);
			    	Collections.sort(ppVecT, new Pdb.PdbComparator());
			    	System.out.println("MODEL"+m+" "+j+" "+k);
			    	
//			    	compute vdw energy:
			    	double[] vdwValue = new double[1];
			    	boolean hasVDW = false;
			    	double vdwLevel = 0.05;
			    	boolean printVDWViolation = false;
			    	
			    	boolean[] resIndex=new boolean[ppVecT.size()];
			    	for (int kk=0;kk<resIndex.length;kk++)
			    		resIndex[kk]=false;
			    	String rotSrc=  src+ "rotasamp-small/";  
	    		    Vector  ppVecTALA=pp.AlaninizeStructure(ppVecT,resIndex,rotSrc);
	    		  
			    	vdwVec = vander.convert2VDW(ppVecTALA);
		        	
		        	int numClashes=vander.countStericClash(vdwVec, vdwValue, vdwLevel, printVDWViolation, true,0.9);
			    	System.out.println("the number of steric clashes: "+ numClashes);
		        	if(numClashes>0)
			    		continue;
			    	
			        boolean isInPreClusters=pp.isInPdbClusters(vecEnsemblePdbs,ppVecT,0.6);///////////
			    	if(! isInPreClusters)////////
			    	{/////////
			    		
				    	out.println("MODEL"+m+" "+j+" "+k);;///to file
				    	
				    	out.println("REMARK :   NOE RMSD=:"+disRms);
				    	out.println("REMARK :   NOE HARM=:"+noeHarm);
				    	out.println("REMARK :   numClashes in Alaninized strcuture=:"+numClashes);
				    	
				    	Vector vecFirstBB=pp.OutputBackbone(ppVecT);
				    	pp.printToFile(vecFirstBB,fileName, out);
				    	
				    	System.out.println("TER");
			         	System.out.println("END");   
			         	
			            out.println("TER");
			         	out.println("ENDMDL");
			    	
			         	vecEnsemblePdbs.add(vecFirstBB);///////
			    	}/////
			    }
			    
			    
			    if ( disRms < disRmsd  )
			    {
					ranThetaS = ranTheta;
					ranPhiS =  ranPhi;
					disRmsd = disRms;
					lenS = lengthR;
					for (i = 0; i < noeVec.size(); i++)
						dis_save[i]=noeBackDis[i];		    		 
				 	 	
					System.out.println(disRmsd+" ... "+ranThetaS+"  "+ranPhiS+"  "+lenS); //Check the progress
			    }
	    	}//for (int k =0 ; k< noeRangeRes; k++)
	    }//for (j=0; j< (360 / resolution); j++)
	}//for (int m = 0; m < (180/resolution); m++)
	
	out.println("END");  
	out.close();
	}catch (FileNotFoundException e)
	{
		System.out.println("File not found: " + fileName);
	}catch (IOException e)
	{
	   System.out.println("IOException: the stack trace is:");
	   e.printStackTrace();
	}
		
	noeRms[0] = disRmsd;
	if (debugNOE)
	{
	    System.out.println("NOE rmsd = "+disRmsd);
	    System.out.println("NOE distances difference for individual ones:");
	}
	double[] centerBNew = {-xC2 + ranThetaS,  -yC2 + ranPhiS,  -zC2 +lenS };
	for (i = 0; i < noeVec.size(); i++)
		System.out.println(dis_save[i]);
	
	Vector pdbBNew   = pp.newPdbByTranslation(pdbVecB, centerBNew);
	double [] centerANew = {-centerA[0], -centerA[1], -centerA[2]};
	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANew);
	//Print the new PDBs
	Vector ppVecT = new Vector();
	ppVecT.addAll(pdbANew);
	ppVecT.addAll(pdbBNew);
	Collections.sort(ppVecT, new Pdb.PdbComparator());
	return ppVecT;
    }
    
   
    /**
     * pack two sses based on back-computed NOE patterns.
     * pdbA and pdbB are the original coordinates noeDistance: the noe
     * distances between the two fragments. format (low, upper).
     * arrange the three arrays with the same order so that the
     * distance i is between nuclei i of A and i of B.
     * 
     * @param pdbVecA one fragment
     * @param pdbVecB another fragment
     * @param assignVec vector of resonane assignment list
     * @param NoesyVec vector of experimental NOESY peak list
     * @param distBound NOE upper distance limit
     * @param pickNum size of packing ensemble
     * @param max_score store the maximum score in the ensemble of packings
     * @param csErrH the cs err h
     * @param csErrN the cs err n
     * @param csErrCA the cs err ca
     * 
     * @return ensemble of translations with top scores
     */
    public Vector positionByNOEPatterns(Vector pdbVecA, Vector pdbVecB,Vector assignVec,Vector NoesyVec,
    		double csErrH, double csErrN, double csErrCA, double distBound, int pickNum, double[] max_score)
    {
    	int i, j;
    	Vector atomVec = new Vector();
    	Cartesian cc = new Cartesian();
    	String atom  = "";
    	Pdb pp=new Pdb();
    	
    	//compute the center of pdbVecA based on CA coordinates
    	double[][] pdbA = new double[pdbVecA.size()][3];
    	double[][] pdbB = new double[pdbVecB.size()][3];
    	double [][] newArrB = new double[pdbVecB.size()][3];
    	for (i=0;i<pdbVecA.size();i++)
    	{
    		Pdb pdb=(Pdb)pdbVecA.elementAt(i);
    		atomVec=pdb.getAtomVec();
    		for (j=0; j<atomVec.size(); j++)
    		{
    		    cc = (Cartesian)atomVec.elementAt(j);
    		    atom = cc.getAtom();
    		    if ( atom.equalsIgnoreCase("CA") )
    		    	pdbA[i] = cc.getXYZ();
    		}
    	}//for (i=0;i<pdbVecA.size();i++)
    	
    	for(i=0;i< pdbVecB.size();i++)
    	{
    		Pdb pdb=(Pdb)pdbVecB.elementAt(i);
    		atomVec=pdb.getAtomVec();
    		for (j=0; j<atomVec.size(); j++)
    		{
    		    cc = (Cartesian)atomVec.elementAt(j);
    		    atom = cc.getAtom();
    		    if ( atom.equalsIgnoreCase("CA") )
    		    	pdbB[i] = cc.getXYZ();
    		}    		
    	}//for(i=0;i<(Pdb)pdbVecB.size();i++)
    	
    	// Compute the center    	  	
    	double xC1 = 0.0, yC1 = 0.0, zC1 = 0.0, xC2 = 0.0, yC2 = 0.0, zC2 = 0.0; 
    	
    	for (i=0; i<pdbA.length; i++)
    	{
    	    xC1 += pdbA[i][0];
    	    yC1 += pdbA[i][1];
    	    zC1 += pdbA[i][2];
    	}
    	if (pdbA.length > 0)
    	{ // Compute the center
     	    xC1 /=pdbA.length;
     	    yC1 /= pdbA.length;
     	    zC1 /= pdbA.length;
    	}
    	for (i=0; i<pdbB.length; i++)
    	{
    	    xC2 += pdbB[i][0]; 
    	    yC2 += pdbB[i][1];
    	    zC2 += pdbB[i][2];
    	}
    	if (pdbB.length> 0)
    	{ // Compute the center
     	    xC2 /= pdbB.length;
     	    yC2 /= pdbB.length;
     	    zC2 /= pdbB.length;
    	}
    	
    	double lengthR = 0.0;
    	double [] trans_vecA = new double[3];
    	double [] trans_vecB = new double[3];
    	//double [][] arrA = new double[pdbA.length][n];
    	//double [][] arrB = new double[pdbB.length][n];
    	//double [][] newArrB = new double[mm][n];
    	double [] centerA = {xC1, yC1, zC1};
    	double [] centerB = {xC2, yC2, zC2};
    	double [] centerO={0.0, 0.0, 0.0};
    	//double[] centerBNew={0.0, 0.0, 0.0};
    	trans_vecA= pp.internuclearVec(centerA,centerO);	
    	trans_vecB= pp.internuclearVec(centerB,centerO);
    	//move centers of both pdbs to the origin
    	Vector<Pdb> pdbVecNewA=pp.newPdbByTranslation(pdbVecA,trans_vecA);
    	Vector<Pdb> pdbVecNewB=pp.newPdbByTranslation(pdbVecB,trans_vecB);
    	
    	double x1, y1, z1, x2, y2, z2;
    	double ranTheta, ranPhi,  ranThetaS = 0.0, ranPhiS = 0.0;
    	//double disRmsd = 1000.0;
    	//long seed = 9752864; //419057923;
    	//Random rr = new Random(seed);
    	double low = 0.0, upper = 0.0;
    	int resolution = 1;
    	int radiusBound =100;
    	double len_resolution=1.0;//0.5;
     	double lenS = 0.0;
     	RotaPattern rotPattern= new RotaPattern();
     	double dbScore=0.0;
     	Assign asg=new Assign();
     	double maxSc=-999999.9;
     	//int MAX_SIZE=500;
     	Vector vecSSEPack=new Vector();
     	//SSEPacking ssePack=new SSEPacking();
     	
    	for (int m = 0; m < (180/resolution); m++)
    	{
    	    ranTheta = m * resolution * Math.PI / 180.0;
    	    for (j=0; j< (360 / resolution); j++)
    	    {
    	    	ranPhi = j * resolution * Math.PI / 360.0;
    	    	for (int k =radiusBound ; k> 0; k--)
    	    	{
    	    		//System.out.println("m = "+ m + " j= " +j+ " k= "+ k);
    	    		
    	    		//lengthR  = noeAve + 4.0*(0.50-rr.nextDouble());
    	    		lengthR  =2.0 + k *0.2;    	    		
    	    		double[] centerBNew = { lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
    	 			      lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
    	 			      lengthR * Math.cos(ranTheta)};
    	    		double[] trans_vec = new double[3];
    	    		trans_vec=pp.internuclearVec(centerO,centerBNew);
    	    		
    	    		Vector pdbVecNewB2=pp.newPdbByTranslation(pdbVecNewB,trans_vec);
    	    		
    	    		//compute the distance patterns for sses
    	    		Vector vecHdist=pp.compDistPatternForSSEPacking(pdbVecNewA,pdbVecNewB2,distBound);//vecRotamerRmsd,pdbVecNRmsd);//distance bound
    	    		    	
    	    		Vector vecBackNoe=rotPattern.BackCompNoePattern(assignVec,0,"",vecHdist);
    	    		
    	    		int [] numPeaks=new int[1];
    	    		
    	    		
    	    		dbScore=asg.NoePatternMatchScore(csErrH,csErrN, csErrCA,vecBackNoe,NoesyVec, numPeaks,false);
    	    		
    	    		//System.out.println("match number = "+ numPeaks[0] + " score= " +dbScore);
    	    		//sumPeaks=sumPeaks+numPeaks[0];
    	    		if(vecSSEPack.size()<pickNum)
    	    		{
    	    			vecSSEPack.add(new SSEPacking(ranTheta,ranPhi,lengthR,dbScore ));
    	    			System.out.println(" ranTheta = "+ ranTheta + " ranPhi= " +ranPhi+ "LengthR= "+ lengthR+ " maxSc="+maxSc);
        	    		
    	    		}
    	    		else
    	    		{
    	    			SSEPacking ssePack=(SSEPacking)vecSSEPack.elementAt(pickNum-1);
    	    			maxSc=ssePack.getScore();
    	    			if(maxSc<dbScore)
    	    			{
    	    				
    	    				vecSSEPack.remove(pickNum-1);
    	    				vecSSEPack.add(new SSEPacking(ranTheta,ranPhi,lengthR,dbScore ));
    	    				System.out.println(" ranTheta = "+ ranTheta + " ranPhi= " +ranPhi+ "LengthR= "+ lengthR+ " maxSc="+maxSc);
    	    			}//if(maxSc<dbScore)
    	    			
    	    		}//else
    	    		
    	    		Collections.sort(vecSSEPack, new SSEPacking.SSEPackingComparator());   
    		    	
    	    	}//for (int k =0 ; k< radiusBound; k++)
    	    }//for (j=0; j< (360 / resolution); j++)
    	}//for (int m = 0; m < (180/resolution); m++) 
    	
    	SSEPacking vecPackTemp=(SSEPacking)vecSSEPack.elementAt(0);
    	max_score[0]=vecPackTemp.getScore();

    double[] centerBNew2=new double [3];
    Vector vecPackEnsemble=new Vector();
	for(i=0;i<vecSSEPack.size();i++)
	{
		SSEPacking ssePack=(SSEPacking)vecSSEPack.elementAt(i);
		ranThetaS=ssePack.getTheta();
		ranPhiS=ssePack.getPhi();
		lenS=ssePack.getLengthR();
		
		centerBNew2 =new double[] {lenS * Math.sin(ranThetaS) * Math.cos(ranPhiS),
			        lenS * Math.sin(ranThetaS) * Math.sin(ranPhiS),
			       lenS * Math.cos(ranThetaS)};
		trans_vecB= pp.internuclearVec(centerO,centerBNew2);
		Vector<Pdb> pdbVecNewBFinal=pp.newPdbByTranslation(pdbVecNewB,trans_vecB);
		Vector ppVecT = new Vector();
		ppVecT.addAll(pdbVecNewA);
		ppVecT.addAll(pdbVecNewBFinal);
		Collections.sort(ppVecT, new Pdb.PdbComparator());
		vecPackEnsemble.add(ppVecT);
		
	}//for(i=0;i<vecSSEPack.size();i++)    	
   
	return vecPackEnsemble;
    }

    /**
     * calculate the minimum rms between two sses, for checking packing symmetry
     * In both SSEs, all rotamers are places.
     * 
     * @param noeVec the NOE distance restraints between the fragment A and B
     * @param pdbVecA one fragment
     * @param pdbVecB another fragment
     * @param noeRms saving the NOE rmsd
     * @param debugNOE if true it will print out the NOE fitting Rmsds
     * 
     * @return the symmetry ID from 0-3.
     * 
     * @throws JampackException the jampack exception
     */
    public Vector calcMinPackingRms(final Vector noeVec, Vector pdbVecA, Vector pdbVecB, 
    		double[] noeRms, boolean debugNOE)throws JampackException
    {
		//Extract neccessary information from the NOE table.
		int i, j;
		int noA, noB;
		int index = -1;
		String atom  = "";
		String atomA = "";
		String atomB = "";
		Vector atomVec = new Vector();
		Cartesian cc = new Cartesian();
		Pdb pp = new Pdb();
		int sizeA = 0;
		int sizeB = 0;
		double[][] pdbA2 = new double[noeVec.size()*6][3];
		double[][] pdbB2 = new double[noeVec.size()*6][3];
		Vector id4A = new Vector();
		Vector id4B = new Vector();
		String id = "";  //Combining the residueNO and Atom name for easy comparion.
		
		double [][] noeDistance = new double[noeVec.size()][2];	
		
		//compute the centers of atom coordinates involed in NOEs
		boolean isQ=true;
		for (i=0; i<noeVec.size(); i++)
		{
		    noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
		    atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
		 
		    ////////
		    String tempA="";
		    isQ=true;
		    if(atomA.length()>=2)
		    {
			    if(atomA.substring(0,1).equalsIgnoreCase("Q") && (!atomA.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempA="H"+atomA.substring(1,atomA.length());
			    else if(atomA.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempA="H"+atomA.substring(2,atomA.length());
			    else 
			    {
			    	tempA=atomA;	
			    	isQ=false;
			    }
		    }//if(atomA.length()>=2)
		    else 
		    {
		    	tempA=atomA;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecA, new Pdb(noA), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
		    	pp = (Pdb)pdbVecA.elementAt(index);
		    	atomVec = pp.getAtomVec();
		    	for (j=0; j<atomVec.size(); j++)
		    	{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    
				    if(atom.length()>=tempA.length())
				    {
				    	String debug_str=atom.substring(0,tempA.length() );
				    	//if (atom.substring(0,tempA.length() ).equalsIgnoreCase(tempA) )
				    		//pdbA[i] = cc.getXYZ();
				    	id = String.valueOf(noA)+atom;
					    //for deleting repeatation in NOEs when computing the centers 
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp=atom.substring(0,tempA.length() );
				    	else
				    		atom_temp=atom;
				    	
					    if ( atom_temp.equalsIgnoreCase(tempA) && !id4A.contains(id) ) 
					    {  
					    	pdbA2[sizeA] = cc.getXYZ();	
					    	id4A.add(id);
					    	sizeA++;
					    }
				    }// if(atom.length()>=tempA.length())
				    
				    
		    	}//for (j=0; j<atomVec.size(); j++)
		    }//if ( index > -1 )
		    
		    noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
		    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
		    //////	//
		    String tempB="";
		    isQ=true;
		    if(atomB.length()>=2)
		    {
			    if(atomB.substring(0,1).equalsIgnoreCase("Q") && (!atomB.substring(1,2).equalsIgnoreCase("Q")) )
			      	tempB="H"+atomB.substring(1,atomB.length());
			    else if(atomB.substring(0,2).equalsIgnoreCase("QQ")) 
			    	tempB="H"+atomB.substring(2,atomB.length());
			    else 
			    {
			    	tempB=atomB;  
			    	isQ=false;
			    }
		    }//if(atomB.length()>=2)
		    else
		    {
		    	tempB=atomB;
		    	isQ=false;
		    }
		    
		    index = Collections.binarySearch(pdbVecB, new Pdb(noB), new Pdb.PdbComparator() );
		    if ( index > -1 )
		    {
				pp = (Pdb)pdbVecB.elementAt(index);
				atomVec = pp.getAtomVec();
				for (j=0; j<atomVec.size(); j++)
				{
				    cc = (Cartesian)atomVec.elementAt(j);
				    atom = cc.getAtom();
				    if (atom.length()>=tempB.length())
				    {
				    	id = String.valueOf(noB)+atom;
				    	//if (atom.equalsIgnoreCase(atomB))
					    	//pdbB[i] = cc.getXYZ();
				    	String atom_temp="";
				    	if(isQ)
				    		atom_temp= atom.substring(0,tempB.length());
				    	else
				    		atom_temp=atom;
					    if (atom_temp.equalsIgnoreCase(tempB) && !id4B.contains(id) ) 
					    {
							pdbB2[sizeB] = cc.getXYZ();	
							id4B.add(id);
							sizeB++;
					    }
				    }//if (atom.length()>=tempB.length())
				    
				}
		    }//if ( index > -1 )
		    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
		}
	// Compute the center
	final int n = 3;
	//int mm = pdbA.length;   
	double xC1 = 0.0, yC1 = 0.0, zC1 = 0.0, xC2 = 0.0, yC2 = 0.0, zC2 = 0.0; 
	double noeAve    = 0.0;
	for (i=0; i<sizeA; i++)
	{
	    xC1 += pdbA2[i][0];
	    yC1 += pdbA2[i][1];
	    zC1 += pdbA2[i][2];
	}
	if (sizeA > 0)
	{ // Compute the center
 	    xC1 /= sizeA;
 	    yC1 /= sizeA;
 	    zC1 /= sizeA;
	}
	for (i=0; i<sizeB; i++)
	{
	    xC2 += pdbB2[i][0]; 
	    yC2 += pdbB2[i][1];
	    zC2 += pdbB2[i][2];
	}
	if (sizeB > 0)
	{ // Compute the center
 	    xC2 /= sizeB;
 	    yC2 /= sizeB;
 	    zC2 /= sizeB;
	}
	//compute average NOEs
	for (i=0; i<noeVec.size(); i++)
	    noeAve += (noeDistance[i][0] + noeDistance[i][1]) / 2.0;
	noeAve /= noeVec.size();
	
	
	double lengthR = noeAve;
	
 	double diffCenter = Math.sqrt((xC1-xC2)* (xC1-xC2)+(yC1-yC2) * (yC1-yC2)+(zC1-zC2) * (zC1-zC2)); 	
	double disRms = 0.0;
	double noeHarm=0.0;
	double[] noeDis = new double[noeVec.size()];
	double[] noeBackDis = new double[noeVec.size()];
	double[] dis_save = new double[noeVec.size()];
	
	
	double [] trans_vecA = new double[3];
	double [] trans_vecB = new double[3];
	
	double [] centerA = {xC1, yC1, zC1};
	double [] centerB = {xC2, yC2, zC2};
	double [] centerO={0.0, 0.0, 0.0};
	
	trans_vecA= pp.internuclearVec(centerA,centerO);	
	trans_vecB= pp.internuclearVec(centerB,centerO);
	//	move centers of both pdbs to the origin
	Vector<Pdb> pdbVecNewA=pp.newPdbByTranslation(pdbVecA,trans_vecA);
	Vector<Pdb> pdbVecNewB=pp.newPdbByTranslation(pdbVecB,trans_vecB);
		
 	double x1, y1, z1, x2, y2, z2;
	double ranTheta, ranPhi,  ranThetaS = 0.0, ranPhiS = 0.0;
	double disRmsd = 1000.0;
	long seed = 9754; //419057923;
	Random rr = new Random(seed);
	double low = 0.0, upper = 0.0;
	int resolution = 4;//5;//changed here....
	int noeRangeRes = 20;//100;
 	double lenS = 0.0;
 
		
	for (int m = 0; m < (180/resolution); m++)
	{
	    ranTheta = m * resolution * Math.PI / 180.0;
	    for (j=0; j< (360 / resolution); j++)
	    {
	    	ranPhi = j * resolution * Math.PI / 180.0;
	    		    	
	    	for (int k =noeRangeRes ; k>=0 ; k--)
		    {
	    		
			    lengthR  = 0.0 + 0.5*k;
			    //move the center of B to a new position on the upper sphere, recompute B'coordinates
			    double[] centerBNew = { lengthR * Math.sin(ranTheta) * Math.cos(ranPhi),
  	 			      lengthR * Math.sin(ranTheta) * Math.sin(ranPhi),
  	 			      lengthR * Math.cos(ranTheta)};
			    double[] trans_vec = new double[3];
	    		trans_vec=pp.internuclearVec(centerO,centerBNew);
	    		
	    		Vector pdbVecNewB2=pp.newPdbByTranslation(pdbVecNewB,trans_vec);
			    
			    disRms = 0.0;
			    noeHarm=0.0;
			   
			    for (i = 0; i < noeVec.size(); i++)
			    {
			    	noA = ((Noe)noeVec.elementAt(i)).getResidueNoA();
			    	atomA = ((Noe)noeVec.elementAt(i)).getAtomA();
			    	
			    	noB = ((Noe)noeVec.elementAt(i)).getResidueNoB();
			 	    atomB = ((Noe)noeVec.elementAt(i)).getAtomB();
			 	    noeDistance[i] = ((Noe)noeVec.elementAt(i)).getRange();
					
			 	    double [] dis=new double[1];
			 	    double [] distance=new double[1];
			 	 	pp.measurePackDisAllRotamers(pdbVecNewA,pdbVecNewB2,noA,atomA,noB, atomB,noeDistance[i][0], noeDistance[i][1],dis,distance);
			 	 	noeDis[i]=dis[0];
			 	 	noeBackDis[i]=distance[0];
			 	 	disRms += noeDis[i] * noeDis[i];
			 	 	noeHarm+=noeDis[i] * noeDis[i];
			    }
			    noeHarm=noeHarm/noeVec.size();		    
			    disRms = Math.sqrt(disRms / noeVec.size()); 
			    			    
			    if ( disRms < disRmsd  )
			    {
					ranThetaS = ranTheta;
					ranPhiS =  ranPhi;
					disRmsd = disRms;
					lenS = lengthR;
					for (i = 0; i < noeVec.size(); i++)
						dis_save[i]=noeBackDis[i];		    		 
				 	 	
					System.out.println(disRmsd+" ... "+ranThetaS+"  "+ranPhiS+"  "+lenS); //Check the progress
			    }
	    	}//for (int k =0 ; k< noeRangeRes; k++)
	    }//for (j=0; j< (360 / resolution); j++)
	}//for (int m = 0; m < (180/resolution); m++)
	
	
		
	noeRms[0] = disRmsd;
	if (debugNOE)
	{
	    System.out.println("NOE rmsd = "+disRmsd);
	    System.out.println("NOE distances difference for individual ones:");
	}
	double[] centerBNew = {-xC2 + lenS * Math.sin(ranThetaS) * Math.cos(ranPhiS),
			       -yC2 + lenS * Math.sin(ranThetaS) * Math.sin(ranPhiS),
			       -zC2 + lenS * Math.cos(ranThetaS)};
	for (i = 0; i < noeVec.size(); i++)
		System.out.println(dis_save[i]);
	
	Vector pdbBNew   = pp.newPdbByTranslation(pdbVecB, centerBNew);
	double [] centerANew = {-centerA[0], -centerA[1], -centerA[2]};
	Vector pdbANew   = pp.newPdbByTranslation(pdbVecA, centerANew);
	//Print the new PDBs
	Vector ppVecT = new Vector();
	ppVecT.addAll(pdbANew);
	ppVecT.addAll(pdbBNew);
	Collections.sort(ppVecT, new Pdb.PdbComparator());
	return ppVecT;
    }
 
    
    /**
     * private utility routines *.
     * 
     * @param x the x
     * @param y the y
     */
    /** Check magnitude of difference of scalars. **/
    private static void check(double x, double y) {
	double eps = Math.pow(2.0, -52.0);
	if (x == 0 & Math.abs(y) < 10*eps) return;
	if (y == 0 & Math.abs(x) < 10*eps) return;
	if (Math.abs(x-y) > 10*eps*Math.max(Math.abs(x),Math.abs(y))) {
	    throw new RuntimeException("The difference x-y is too large: x = " 
				       + Double.toString(x) + "  y = " + Double.toString(y));
	}
    }

    /**
     * Check norm of difference of "vectors". *
     * 
     * @param x the x
     * @param y the y
     */
    private static void check(double[] x, double[] y) {
	if (x.length == y.length ) {
	    for (int i=0;i<x.length;i++) {
		check(x[i],y[i]);
	    } 
	} else {
	    throw new RuntimeException("Attempt to compare vectors of different lengths");
	}
    }

    /**
     * Check norm of difference of arrays. *
     * 
     * @param x the x
     * @param y the y
     */
    private static void check(double[][] x, double[][] y) {
	Matrix A = new Matrix(x);
	Matrix B = new Matrix(y);
	check(A,B);
    }

    /**
     * Check norm of difference of Matrices. *
     * 
     * @param X the x
     * @param Y the y
     */
    private static void check(Matrix X, Matrix Y) {
	double eps = Math.pow(2.0,-52.0);
	if (X.norm1() == 0. & Y.norm1() < 10*eps) return;
	if (Y.norm1() == 0. & X.norm1() < 10*eps) return;
	if (X.minus(Y).norm1() > 1000*eps*Math.max(X.norm1(),Y.norm1())) {
	    throw new RuntimeException("The norm of (X-Y) is too large: " +  Double.toString(X.minus(Y).norm1()));
	}
    }

    /**
     * Shorten spelling of print. *
     * 
     * @param s the s
     */
    private static void print (String s) {
	System.out.print(s);
    }

    /**
     * Print appropriate messages for successful outcome try *.
     * 
     * @param s the s
     * @param e the e
     */
    private static void try_success (String s, String e) {
	print(">    " + s + "success\n");
	if ( e != "" ) {
	    print(">      Message: " + e + "\n");
	}
    }
    
    /**
     * Print appropriate messages for unsuccessful outcome try *.
     * 
     * @param count the count
     * @param s the s
     * @param e the e
     * 
     * @return the int
     */
    private static int try_failure (int count,String s,String e) {
	print(">    " + s + "*** failure ***\n>      Message: " + e + "\n");
	return ++count;
    }
}
		    
