package rdcPanda;

///////////////////////////////////////////////////////////////////////////////////////////////
//	ppGlobal.java
//
//	  Version:           0.1
//
//
//	  authors:
// 	  initials            name                      organization               email
//	 ---------   -----------------------        ------------------------    ------------------
//	  LW            Lincong Wang                  Dartmouth College       wlincong@cs.dartmouth.edu
//
///////////////////////////////////////////////////////////////////////////////////////////////



/*
	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.
	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	Lesser General Public License for more details.
	
	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
	USA
	
	Contact Info:
		Bruce Donald
		Duke University
		Department of Computer Science
		Levine Science Research Center (LSRC)
		Durham
		NC 27708-0129 
		USA
		brd@cs.duke.edu
	
	If you use or publish any results derived from the use of this program please cite:
	J. Zeng, J. Boyles, C. Tripathy, L. Wang, A. Yan, P. Zhou and B.R. Donald. 
	"High-Resolution Protein Structure Determination Starting with a Global Fold 
	Calculated from Exact Solutions to the RDC Equations." Submitted For Review.

	Copyright (C) 2009 Jianyang (Michael) Zeng, Lincong Wang and Bruce R. Donald		
	<signature of Bruce Donald>, June 2008 and January 2009
	Bruce Donald, Professor of Computer Science
 */

import java.io. *;
import java.util. *;

// TODO: Auto-generated Javadoc
/** * 
 * 
   * A class for computing the first peptide plane from an NCA and NH vector.
 * Several common methods used by other classes are implemented here. 
 *  Written by Lincong Wang (2001-2005).
*/
public class ppGlobal implements Cloneable{

    /**
     * Instantiates a new pp global.
     */
    public ppGlobal(){
    } 

    /**
     * calculate the length between two vectors v1 and v2
     * The returned angle is the [0,Pi] region.
     * 
     * @param v1 the v1
     * 
     * @return the double
     */
    public double length(double[] v1){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	return v1Len;
    }
    
    /**
     * calculate the angles between two vectors v1 and v2
     * The returned angle is in the [0,Pi] range.
     * 
     * @param v1 the v1
     * @param v2 the v2
     * 
     * @return the double
     */
    public double interAngle(double[] v1, double[] v2){
	double v1Len = Math.sqrt(v1[0]*v1[0] + v1[1]*v1[1] + v1[2]*v1[2]);
	double v2Len = Math.sqrt(v2[0]*v2[0] + v2[1]*v2[1] + v2[2]*v2[2]);
	double c = (v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2]) / (v1Len * v2Len);
	return Math.acos(c);  //acos [0, PI] so should be OK
    }
    
    /**
     * cal the directioal cosines of the vector vec.
     * 
     * @param vec the vec
     * 
     * @return the double[]
     */
    public double[] dirCos(double [] vec) {
	double len = 0.0;
	double [] dirs = new double[vec.length];
	for (int i=0; i<vec.length; i++)
	    len += vec[i] * vec[i];
	for (int j=0; j<vec.length; j++)
	    dirs[j] = vec[j] / Math.sqrt(len);
	return dirs;
    }

    /**
     * compute global rotation (Rg) with two noncolinear vectors, Rg
     * is the rotation matrix for rotating the PDB coordinate system
     * into the following system:
     * +z-axis along the bond NH(i)->N(i), "->" means from atom N(i) to atom NH(i).
     * +y-axis in the peptide plane i,and the angle  between +y and the N(i)->CA(i) < 90 degree.
     * +x-axis is defined based on the right-handedness.
     * 
     * @param nToh the directional cosine of N->NH vector
     * @param nToca the directional cosine of N->CA vector
     * 
     * @return All the rotational Matrix, not just one
     */
    public Matrix RgCal(double[] nToh, double[] nToca){
	double[] n2h = dirCos(nToh); //cal directional cosines
	double[] n2ca = dirCos(nToca);
	double theta = Math.PI - interAngle(nToh, nToca);
	double xNH = n2h[0], yNH = n2h[1], zNH = n2h[2];
	double xNCA = n2ca[0], yNCA = n2ca[1], zNCA = n2ca[2];
	double[][] mat = new double[3][3];
	double sinTheta = 9.99;
	Matrix A = new Matrix(3,3);
	double alpha1=9.99, alpha2=9.99, beta1=9.99, beta2=9.99, gamma1=9.99, gamma2=9.99;
	if (zNH == 1.0){ //cal from the rotation matrix
	    System.out.println("zNH is 1");
	    if ( !(zNCA == 1.0) && !(zNCA == -1.0) ){
		sinTheta = Math.sqrt(1.0-zNCA*zNCA); //under such condition cosTheta = zNCA
		mat[0][0] = yNCA / sinTheta;
		mat[0][1] = xNCA / sinTheta;
		mat[0][2] = 0.0;
		mat[1][0] = -xNCA / sinTheta;
		mat[1][1] = yNCA / sinTheta;
		mat[1][2] = 0.0;
		mat[2][0] = 0.0;
		mat[2][1] = 0.0;
		mat[2][2] = 1.0;
		return (new Matrix(mat)).transpose();
	    } else { //impossible for normal PDB file
		System.out.println("zNCA is 1 or -1");
		System.exit(0);
	    }
	}else if (zNH == -1.0){
	    System.out.println("zNH is -1");
	    mat[0][0] = -yNCA / sinTheta;
	    mat[0][1] =  xNCA / sinTheta;
	    mat[0][2] = 0.0;
	    mat[1][0] = xNCA / sinTheta;
	    mat[1][1] = yNCA / sinTheta;
	    mat[1][2] = 0.0;
	    mat[2][0] = 0.0;
	    mat[2][1] = 0.0;
	    mat[2][2] = 1.0;
	    return (new Matrix(mat)).transpose(); //return directly need transpose
	}else {
	    beta1 = Math.acos(zNH);  //Should be minus
	    double sinAlpha = yNH / Math.sin(beta1);
	    double cosAlpha = xNH / Math.sin(beta1);
	    if (( cosAlpha > 0) || (cosAlpha == 0) )
		alpha1 = Math.asin(sinAlpha);
	    else if ( cosAlpha < 0)
		alpha1 = Math.PI-Math.asin(sinAlpha);
	    //the second set
	    beta2 = -beta1;
	    sinAlpha = yNH / Math.sin(beta2);
	    cosAlpha = xNH / Math.sin(beta2);
	    if (( cosAlpha > 0) || (cosAlpha == 0) )
		alpha2 = Math.asin(sinAlpha);
	    else if ( cosAlpha < 0)
		alpha2 = Math.PI-Math.asin(sinAlpha);
	    //The first set
	    double sinGamma = (zNCA+Math.cos(beta1)*Math.cos(theta))/(Math.sin(theta)*Math.sin(beta1));
	    double cosGamma = (yNCA*Math.cos(alpha1)-xNCA*Math.sin(alpha1)) / Math.sin(theta);
	    if (( cosGamma > 0) || (cosGamma == 0) )
		gamma1 = Math.asin(sinGamma);
	    else if ( cosGamma < 0 )
		gamma1 = Math.PI-Math.asin(sinGamma);
	    //the second set
	    sinGamma = (zNCA+Math.cos(beta2)*Math.cos(theta)) / (Math.sin(theta) *Math.sin(beta2));
	    cosGamma = (yNCA*Math.cos(alpha2)-xNCA*Math.sin(alpha2))/Math.sin(theta);
	    if (( cosGamma > 0) || (cosGamma == 0) )
		gamma2 = Math.asin(sinGamma);
	    else if ( cosGamma < 0)
		gamma2 = Math.PI - Math.asin(sinGamma);
	}//make a euler rotation matrix from three Euler angles
	Matrix MM = A.rotationMat(Math.PI, "+y");
 	Matrix mm = A.eulerMat(alpha1, beta1, gamma1);
 	return (MM.times(mm));
    }

     /* compute global rotation (Rg) with two noncolinear vectors, Rg
      * is the rotation matrix for rotating the PDB coordinate system
      * into the following system:
      * +z-axis along the bond NH(i)->N(i), "->" means from atom N(i) to atom NH(i). 
      * +y-axis in the peptide plane i,and the angle  between +y and the N(i)->CA(i) < 90 degree.  
      * +x-axis is defined based on the right-handedness.
      @param nToh the directional cosine of N->NH vector
      @param nToca the directional cosine of N->CA vector
      @param rightHand to compare with the rotation matrix from the SVD method. 
      @return All the rotational Matrix, not just one
     */
    /**
      * Rg cal.
      * 
      * @param nToh the n toh
      * @param nToca the n toca
      * @param rightHand the right hand
      * 
      * @return the matrix
      */
     public Matrix RgCal(double[] nToh, double[] nToca, boolean rightHand){
	double[] n2h = dirCos(nToh); //cal directional cosines
	double[] n2ca = dirCos(nToca);
 	double theta = Math.PI - interAngle(nToh, nToca);
	double xNH = n2h[0], yNH = n2h[1], zNH = n2h[2];
	double xNCA = n2ca[0], yNCA = n2ca[1], zNCA = n2ca[2];
	double[][] mat = new double[3][3];
	double sinTheta = 9.99;
	Matrix A = new Matrix(3,3);
	double alpha1=9.99,  beta1=9.99, gamma1=9.99;
	if (zNH == 1.0){ //cal from the rotation matrix
	    System.out.println("zNH is +1");
	    if ( !(zNCA == 1.0) && !(zNCA == -1.0) ){
		sinTheta = Math.sqrt(1.0-zNCA*zNCA); //under such condition cosTheta = -zNCA
		mat[0][0] = - yNCA / sinTheta;
		mat[0][1] = xNCA / sinTheta;
		mat[0][2] = 0.0;
		mat[1][0] = xNCA / sinTheta;
		mat[1][1] = yNCA / sinTheta;
		mat[1][2] = 0.0;
		mat[2][0] = 0.0;
		mat[2][1] = 0.0;
		mat[2][2] = 1.0;
		return (new Matrix(mat)).transpose();
	    } else { //impossible for normal PDB file
		System.out.println("zNCA is 1 or -1");
		System.exit(0);
	    }
	}else if (zNH == -1.0){
	    System.out.println("zNH is -1");
	    sinTheta = Math.sqrt(1.0-zNCA*zNCA); //under such condition cosTheta = zNCA
	    mat[0][0] =  yNCA / sinTheta;
	    mat[0][1] =  xNCA / sinTheta;
	    mat[0][2] = 0.0;
	    mat[1][0] = - xNCA / sinTheta;
	    mat[1][1] = yNCA / sinTheta;
	    mat[1][2] = 0.0;
	    mat[2][0] = 0.0;
	    mat[2][1] = 0.0;
	    mat[2][2] = 1.0;
	    return (new Matrix(mat)).transpose(); //return directly need transpose
	}else {
	    beta1 = Math.acos(zNH);  //Should be minus
	    double sinAlpha = yNH / Math.sin(beta1);
	    double cosAlpha = xNH / Math.sin(beta1);
	    if (( cosAlpha > 0) || (cosAlpha == 0) )
		alpha1 = Math.asin(sinAlpha);
	    else if ( cosAlpha < 0)
		alpha1 = Math.PI - Math.asin(sinAlpha);

	    double sinGamma = (zNCA + Math.cos(beta1) * Math.cos(theta)) / (Math.sin(theta)*Math.sin(beta1));
	    double cosGamma = (yNCA * Math.cos(alpha1) - xNCA*Math.sin(alpha1)) / Math.sin(theta);
	    if (( cosGamma > 0) || (cosGamma == 0) )
		gamma1 = Math.asin(sinGamma);
	    else if ( cosGamma < 0 )
		gamma1 = Math.PI - Math.asin(sinGamma);
	}//make a euler rotation matrix from three Euler angles
 	Matrix MM = A.rotationMat(Math.PI, "+y"); //Before this rotation +Z is along N->NH
 	Matrix mm = A.eulerMat(alpha1, beta1, gamma1);
 	if (rightHand)
 	    return (MM.times(mm));
 	return ((Const.mLeftHand).times(MM.times(mm)));
    }

    /**
     * compute global rotation (Rg) with two no colinear vectors
     * Rg is the same as the last method
     * Here the input are the NH vector and one Euler angle gamma for the N(i)->CA(i) vector
     * accounting the rotation of the peptide plane along the NH vector.
     * 
     * @param nToh the n toh
     * @param gamma the gamma
     * 
     * @return the matrix
     */
    public Matrix RgCal(double[] nToh, double gamma){
	double[] n2h = dirCos(nToh); //cal directional cosines
	double xNH = n2h[0], yNH = n2h[1], zNH=n2h[2];
	double[][] mat = new double[3][3];
	double sinTheta = 9.99;
	Matrix A = new Matrix(3,3);
	double alpha1=9.99, alpha2=9.99, beta1=9.99, beta2=9.99, gamma1=9.99, gamma2=9.99;

	beta1 = Math.acos(zNH);
	double sinAlpha = yNH / Math.sin(beta1);
	double cosAlpha = xNH / Math.sin(beta1);
	if (( cosAlpha > 0) || (cosAlpha == 0) )
	    alpha1 = Math.asin(sinAlpha);
	else if ( cosAlpha < 0)
	    alpha1 = Math.PI-Math.asin(sinAlpha);
	//the second set
	beta2 = -beta1;
	sinAlpha = yNH / Math.sin(beta2);
	cosAlpha = xNH / Math.sin(beta2);
	if (( cosAlpha > 0) || (cosAlpha == 0) )
	    alpha2 = Math.asin(sinAlpha);
	else if ( cosAlpha < 0)
	    alpha2 = Math.PI-Math.asin(sinAlpha);
	//make a euler rotation matrix from three Euler angles
	Matrix mm = A.eulerMat(alpha1, beta1, gamma);
	//invert the +z axis: point along the NH->N direction for the sake of easy math
	Matrix MM = A.rotationMat(Math.PI, "+y");
	return (MM.times(mm));
    }
}
